unit untTopicListView;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls,
  ComCtrls, Contnrs, SyncObjs,
  untGlobal, untBoard, untTopic;

type
  TTopicListView = class(TListView)
  private
    { Private 錾 }
    FListItemToClear: TObjectList;
    FTopicList: TList;
    FSelectedBoard: TBoard;
  protected
    { Protected 錾 }
    procedure Board_TopicListReceived(Sender: TObject);
    procedure TopicListViewData(Sender: TObject; Item: TListItem);
    procedure TopicListViewCustomDrawItem(Sender: TCustomListView;
                                          Item: TListItem;
                                          State: TCustomDrawState;
                                          var DefaultDraw: Boolean);
    procedure TopicListViewColumnClick(Sender: TObject;
                                       Column: TListColumn);
    procedure ClearTopic;
    procedure SetOnError;
    procedure SetOnComplete;
    procedure SetOnStatus;
    procedure SyncOnComplete;
  public
    { Public 錾 }
    CompleteEvent: TSimpleEvent;
    ForceToOpenBoard: boolean;
    procedure TopicListRefresh;
    procedure OpenBoard(Board: TBoard);
    procedure Search(const searchText: string);
    function SelectedTopic: TTopic;
    procedure RemoveTopic(tp: TTopic);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published 錾 }
  end;

procedure Register;

implementation

uses
  Math, StrUtils, Graphics,
  untMainForm, untConfig, untOnlineBoard, untBoardCategory,
  untAllSubscribedTopics, untOfflineBoard, untBoard2ch, untBoardList,
  untTaskManager;

var
  FTopicListSortOrder: boolean;
  FTopicListSortIndex: integer;

function GetTopicStateNumber(tp: TTopic): Integer;
begin
  if tp.State = dsNone then
  begin
    if tsFavorite in tp.Status then
    begin
      Result := 1
    end else if tsFreezed in tp.Status then
    begin
      Result := 3
    end else if tsError in tp.Status then
    begin
      Result := 2
    end else if (tp.DatSize > 0) and
                (tp.MessageCount > tp.GotMessageCount) then
    begin
      Result := 5
    end else if (tp.DatSize > 0) and
                (tp.GotMessageCount > 0) then
    begin
      Result := 4
    end else
    begin
      Result := -1;
    end;
  end else begin
    Result := 0
  end;
end;

function TopicListSortCompare(Item1, Item2: Pointer): Integer;
var
  compare: integer;
  topic1, Topic2: TTopic;
  noChange: boolean;

begin
  topic1 := Item1;
  topic2 := Item2;
  noChange := false;
  compare  := 0;

  case FTopicListSortIndex of
    0:
      Compare := CompareValue(GetTopicStateNumber(topic1),
                              GetTopicStateNumber(topic2));
    1:
      Compare := CompareValue(topic1.Priority,
                              topic2.Priority);
    2:
      Compare := CompareValue(topic1.Index,
                              topic2.Index);
    3:
      Compare := AnsiCompareStr(topic1.Title,
                                topic2.Title);
    4:
      Compare := CompareValue(topic1.MessageCount,
                              topic2.MessageCount);
    5:
      Compare := CompareValue(topic1.GotMessageCount,
                              topic2.GotMessageCount);
    6:
      Compare := CompareValue(topic1.NewMessageCount,
                              topic2.NewMessageCount);
    7:
      Compare := AnsiCompareStr(topic1.LastReadDate,
                                topic2.LastReadDate);
    8:
      Compare := AnsiCompareStr(topic1.LastWriteDate,
                                topic2.LastWriteDate);
    9:
      Compare := AnsiCompareStr((topic1.OnlineBoard as TBoard).DisplayName,
                                (topic2.OnlineBoard as TBoard).DisplayName);
    else
      noChange := true;
  end;

  if FTopicListSortOrder then
  begin
    compare := -compare;
  end;

  if noChange then
  begin
    compare := CompareValue(topic1.Index, topic2.Index);
  end;
  result := compare;
end;

procedure TTopicListView.SyncOnComplete;
begin
  MainForm.Board_Complete(self);
end;

procedure TTopicListView.RemoveTopic(tp: TTopic);
begin
  FTopicList.Remove(tp);
end;

procedure TTopicListView.Search(const searchText: string);
var
  newTopicList: TList;
  i: integer;

begin
  newTopicList := TList.Create;
  // gsbNXg猟
  for i := 0 to FTopicList.Count - 1 do
  begin
    if AnsiContainsText(TTopic(FTopicList[i]).Title, searchText) then
    begin
      newTopicList.Add(FTopicList[i]);
    end
  end;
  for i := 0 to FTopicList.Count - 1 do
  begin
    if not AnsiContainsText(TTopic(FTopicList[i]).Title, searchText) then
    begin
      // 
      //newTopicList.Add(FTopicList[i]);
    end;
  end;
  FTopicList.Clear;
  FTopicList.Assign(newTopicList);
  newTopicList.Free;
  Items.Count := FTopicList.Count;
  TopicListRefresh;
  ForceToOpenBoard := true;
end;

function TTopicListView.SelectedTopic: TTopic;
begin
  if ItemIndex = -1 then
  begin
    result := nil;
  end else
  begin
    result := FTopicList[ItemIndex];
  end;
end;

procedure TTopicListView.TopicListRefresh;
var
  i: integer;
  item: TListItem;

begin
  Items.BeginUpdate;
  Items.Clear;
  FListItemToClear.Clear;
  for i := 0 to FTopicList.Count - 1 do
  begin
    item := Items.Add;
    FListItemToClear.Add(item);
  end;
  Items.EndUpdate;
end;

// gsbNM
procedure TTopicListView.Board_TopicListReceived(Sender: TObject);
var
  i: integer;
  item: TListItem;

begin
  Items.BeginUpdate;
  FTopicList.Capacity := FSelectedBoard.TopicList.Count;
  for i := FSelectedBoard.ReceivedIndex to FSelectedBoard.TopicList.Count -1 do
  begin
    FTopicList.Add(FSelectedBoard.TopicList[i]);
    item := Items.Add;
    FListItemToClear.Add(item);
  end;
  Items.EndUpdate;
  // ܂ɍĕ`悵Ă݂
  DoubleBuffered := true;
  Repaint;
  DoubleBuffered := false;
end;

procedure TTopicListView.SetOnError;
begin
  FSelectedBoard.OnError    := MainForm.Board_Error;
end;

procedure TTopicListView.SetOnComplete;
begin
  FSelectedBoard.OnComplete := MainForm.Board_Complete;
end;

procedure TTopicListView.SetOnStatus;
begin
  (FSelectedBoard as TOnlineBoard).OnStateChange := MainForm.Board_StatusChange;
end;

{ --------------------------------------------------------
  pr  : {[hJ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TTopicListView.OpenBoard(Board: TBoard);
var
  subjectFile: string;
  fileStream: TFileStream;
  modifiedTime: TDateTime;
  doClear, hasChange: boolean;

begin
  CompleteEvent.ResetEvent;
  if not Assigned(Board) then
  begin
    CompleteEvent.SetEvent;
    exit;
  end;

  doClear := true;
  hasChange := true;

  if Assigned(FSelectedBoard) then
  begin
    if FSelectedBoard is TOfflineBoard then
    begin
      (FSelectedBoard as TOfflineBoard).Save;
    end else if FSelectedBoard is TOnlineBoard then
    begin
      (FSelectedBoard as TOnlineBoard).CancelDownload;
    end;

    if FSelectedBoard = Board then
    begin
       if FSelectedBoard.IsDownloading then
       begin
          doClear := false;
       end;
       if Board is TBoard2ch and not ForceToOpenBoard then
       begin
          hasChange := (Board as TBoard2ch).HasTopicListChange;
          if not hasChange then
          begin
            doClear := false;
          end;
       end;
    end;

    if doClear then
    begin
      gSynchronizer.DoSynchronize(ClearTopic);
      FTopicList.Clear;
      FListItemToClear.Clear;

      FSelectedBoard.FreeAllTopic;
      (FSelectedBoard.BoardList as TBoardList).GC;
    end;

    FSelectedBoard.OnReceived := nil;
  end;

  FSelectedBoard := Board;
  FSelectedBoard.OnReceived := Board_TopicListReceived;
  gSynchronizer.DoSynchronize(SetOnComplete);
  gSynchronizer.DoSynchronize(SetOnError);  
  if FSelectedBoard is TOnlineBoard then
  begin
    gSynchronizer.DoSynchronize(SetOnStatus);
  end;

  // VXmp
  if gConfig.TopicListColorUse and (FSelectedBoard is TOnlineBoard) then
  begin
    if not gFolderAlias.GetFolderPath((FSelectedBoard as TOnlineBoard).Server,
                                      (FSelectedBoard as TOnlineBoard).BoardName,
                                      'subjects',
                                      subjectFile) then
    begin
      subjectFile := subjectFile + 'subjects';
    end;

    if FileExists(subjectFile) then
    begin
      fileStream := TFileStream.Create(subjectFile, fmOpenRead or fmShareDenyNone);
      modifiedTime := FileDateToDateTime(FileGetDate(FileStream.Handle));
      fileStream.Free;
      // datetime2unixtime
      // 32400 = 60 * 60 * 9 => JST=GMT+9  T[oɂĈႤ
      (FSelectedBoard as TOnlineBoard).ModTime := Trunc((modifiedTime - EncodeDate(1970, 1, 1)) * 86400) - 32400;
    end;
  end;

  if hasChange or ForceToOpenBoard then
  begin
    ForceToOpenBoard := false;
    FSelectedBoard.SendTopicList;
  end else
  begin
    gSynchronizer.DoSynchronize(ClearTopic);
    gSynchronizer.DoSynchronize(TopicListRefresh);
    gSynchronizer.DoSynchronize(SyncOnComplete);
  end;
  CompleteEvent.SetEvent;
end;

procedure TTopicListView.ClearTopic;
begin
  Items.Clear;
  DoubleBuffered := true;
  Repaint;
  DoubleBuffered := false;
end;

procedure TTopicListView.TopicListViewData(Sender: TObject; Item: TListItem);
var
  topic: TTopic;
  gotMsg, newMsg: string;

begin
  if (Item.Index < 0) or (Item.Index > (FTopicList.Count - 1)) then
  begin
    exit;
  end;

  topic := TTopic(FTopicList[Item.Index]);

  if not Assigned(topic) then
  begin
    exit;
  end;

  gotMsg := IntToStr(topic.GotMessageCount);
  if topic.GotMessageCount <= 0 then
  begin
    gotMsg := '';
  end;

  newMsg := IntToStr(topic.NewMessageCount);
  if topic.NewMessageCount < 0 then
  begin
    newMsg := '';
  end;

  Item.Caption  := '';
  if topic.Priority < 0 then
  begin
    Item.SubItems.Add(IntToStr(topic.Priority))
  end else if topic.Priority > 0 then
  begin
    Item.SubItems.Add('+' + IntToStr(topic.Priority))
  end else
  begin
    Item.SubItems.Add('');
  end;
  Item.SubItems.Add(IntToStr(topic.Index));

  Item.SubItems.Add(topic.Title);
  Item.SubItems.Add(IntToStr(topic.MessageCount));
  Item.SubItems.Add(gotMsg);
  Item.SubItems.Add(newMsg);
  Item.SubItems.Add(topic.LastReadDate);
  Item.SubItems.Add(topic.LastWriteDate);

  if topic.OnlineBoard <> nil then
  begin
    Item.SubItems.Add((topic.OnlineBoard as TBoard).DisplayName)
  end else
  begin
    Item.SubItems.Add('');
  end;

  Item.SubItems.Add(topic.Memo);

  // ACR
  Item.ImageIndex := GetTopicStateNumber(topic);
end;

procedure TTopicListView.TopicListViewCustomDrawItem(Sender: TCustomListView;
                                                     Item: TListItem;
                                                     State: TCustomDrawState;
                                                     var DefaultDraw: Boolean);
var
  color: TColor;
  imgIndex: Integer;
  topic: TTopic;

begin
  DefaultDraw := true;
  if not gConfig.TopicListColorUse then
  begin
    exit;
  end;
  color := clBlack;

  if Item <> nil then
  begin
    topic := TTopic(FTopicList[Item.Index]);
    imgIndex := Item.ImageIndex;

    case imgIndex of
     -1:
      begin
        color := gConfig.TopicListNotGotColor;
      end;
      1:
      begin
        // Cɓ
        color := gConfig.TopicListFavoriteColor;
        if (Topic.DatSize > 0) then
        begin
          if (topic.MessageCount > topic.GotMessageCount) or
             (topic.NewMessageCount > 0) then
          begin
            color := gConfig.TopicListNewMesColor;
          end;
        end;
      end;
      4:
      begin
        // VȂ
        color := gConfig.TopicListNoNewMesColor;
      end;
      5:
      begin
        // V
        color := gConfig.TopicListNewMesColor;
      end;
    end;
    Canvas.Font.Color := color;
  end;
end;

// Xgr[\[g
procedure TTopicListView.TopicListViewColumnClick(Sender: TObject;
                                                  Column: TListColumn);
begin
  if Column.Index = FTopicListSortIndex then
  begin
    FTopicListSortOrder := not FTopicListSortOrder;
  end else
  begin
    FTopicListSortOrder := true;
    FTopicListSortIndex := Column.id;
  end;
  FTopicList.Sort(TopicListSortCompare);

  // Xbhqɂ̒g̃\[g
  if FSelectedBoard is TOfflineBoard then
  begin
    (FSelectedBoard as TOfflineBoard).Sort(TopicListSortCompare);
  end;
  DoubleBuffered := true;
  Repaint;
  DoubleBuffered := false;
end;

constructor TTopicListView.Create;
begin
  inherited;

  FListItemToClear := TObjectList.Create;
  FTopicList       := TList.Create;
  CompleteEvent := TSimpleEvent.Create;
  OnColumnClick    := TopicListViewColumnClick;
  OnCustomDrawItem := TopicListViewCustomDrawItem;
  OnData           := TopicListViewData;
end;

destructor TTopicListView.Destroy;
begin
  CompleteEvent.Free;
  FTopicList.Free;
  FListItemToClear.Free;

  inherited;
end;

procedure Register;
begin
  RegisterComponents('Samples', [TTopicListView]);
end;


end.
