#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

"""
<comment-ja>
アップデートライブラリを定義する
</comment-ja>
<comment-en>
Define the update lib.
</comment-en>

@file:   updater.py

@author: Kei Funagayama <kei@karesansui-project.info>

@copyright: GPL v2

"""

import os
import tempfile
import warnings

import yum # GPLv2
import yum.Errors # GPLv2
import yum.plugins # GPLv2

import rpm # GPLv2
RPM_VERIFY = False

import karesansui
from karesansui.lib.const import PROXY_ENABLE, PROXY_DISABLE
from karesansui.lib.net.http import proxies

class KaresansuiYumException(karesansui.KaresansuiPlusException):
    """<comment-ja>
    Yum関連の基底例外クラス
    </comment-ja>
    <comment-en>
    Base Exception Class for yum stuff
    </comment-en>
    """
    pass 

class KaresansuiYumRepoError(KaresansuiYumException):
    """<comment-ja>
    Yumリポジトリ関連の例外クラス
    </comment-ja>
    <comment-en>
    Exception class for yum repository stuff
    </comment-en>
    """
    pass

class KaresansuiYumBase(yum.YumBase):
    """<comment-ja>
    Karesansui用にカスタマイズされたYumライブラリクラス
    </comment-ja>
    <comment-en>
    Customized YumBase class for Karesansui
    </comment-en>
    """

    def __init__(self, ftemp):
        self._ftemp = ftemp
        yum.YumBase.__init__(self)

    def doConfigSetup(self, fn='/etc/yum/yum.conf', root='/', init_plugins=True,
                      plugin_types=(yum.plugins.TYPE_CORE,), optparser=None, debuglevel=None,
                      errorlevel=None):
        """<comment-ja>
        yum.confの初期化処理 # Override
        </comment-ja>
        <comment-en>
        yum.conf initialization # Override
        </comment-en>
        """
        if hasattr(self, 'preconf'):
            #self.preconf.fn = fn
            #self.preconf.fn = "/etc/opt/karesansui/yum.conf"
            self.preconf.fn = self._ftemp
            self.preconf.root = root
            self.preconf.init_plugins = init_plugins
            self.preconf.plugin_types = plugin_types
            self.preconf.optparser = optparser
            self.preconf.debuglevel = debuglevel
            self.preconf.errorlevel = errorlevel
            
        return self.conf

class RpmLib:
    """<comment-ja>
    Karesansui用にカスタマイズされたRPMライブラリクラス
    </comment-ja>
    <comment-en>
    Rpmlib customized for Karesansui
    </comment-en>
    """

    def __init__(self, callback=None):
        #global logfile

        self.callback = callback
        self.fd = 0
        #if logfile is None:
        #    logfile = "/dev/stdout"
        #self.logfile = logfile
        self.rpm_callback_fd = 0
        self.cnt = 0
        self.output = None

        self.ts = rpm.TransactionSet()
        if RPM_VERIFY is False:
            #self.ts.setVSFlags(rpm._RPMVSF_NOSIGNATURES)
            self.ts.setVSFlags(-1)

    def __del__(self):
        try:
            self.ts.closeDB()
            if self.fd != 0:
                os.close(self.fd)
        except:
            pass

    def readHeader(self, filename):
        self.fd = os.open(filename, os.O_RDONLY)

        try:
            hdr = self.ts.hdrFromFdno(self.fd)
        except rpm.error, e:
            print str(e)
            hdr = None

        os.close(self.fd)
        return hdr

    def doQuery(self, name=None):
        ret = {}

        if type(name) == str:
            names = [name]
        else:
            names = name

        for _pkg in names:
            mi = self.ts.dbMatch()
            if mi:
                #mi.pattern('name', rpm.RPMMIRE_REGEX, _pkg)
                mi.pattern('name', rpm.RPMMIRE_STRCMP, _pkg)
                # rpm --querytags
                for info in mi:
                    ret[info['name']] = {"name":info["name"],
                                         "version":info["version"],
                                         "release":info["release"],
                                         "epoch":info["epoch"],
                                         "summary":info["summary"],
                                         "description":info["description"],
                                         "buildtime":info["buildtime"],
                                         "buildhost":info["buildhost"],
                                         "installtime":info["installtime"],
                                         "size":info["size"],
                                         "vendor":info["vendor"],
                                         "license":info["license"],
                                         "copyright":info["copyright"],
                                         "packager":info["packager"],
                                         "group":info["group"],
                                         "url":info["url"],
                                         "os":info["os"],
                                         "arch":info["arch"],
                                         "requires":info["requires"],
                                         }
            del mi
        return ret

class Updater:
    """<comment-ja>
    Karesansuiのアップデート処理用基底クラス
    </comment-ja>
    <comment-en>
    Base clas for Karesansui updating stuff
    </comment-en>
    """

    pass

class YumUpdater(Updater):
    """<comment-ja>
    Karesansuiのアップデート処理用クラス(YUM)
    </comment-ja>
    <comment-en>
    Update processing class for Karesansui (YUM)
    </comment-en>
    """

    def __init__(self, config):
        """<comment-ja>
        @param config: karesansui.config
        @type config: dict
        </comment-ja>
        <comment-en>
        @param config: karesansui.config
        @type config: dict
        </comment-en>
        """
        self.config = config
        # yum
        self._ftemp = "%s.conf" % tempfile.mkstemp()[1]
        # Set proxy
        proxy = None
        if config['application.proxy.status'] is PROXY_ENABLE:
            proxy = proxies(config['application.proxy.server'],
                            config['application.proxy.port'],
                            config['application.proxy.user'],
                            config['application.proxy.password'],
                            'http')

        from ConfigParser import SafeConfigParser
        parser = SafeConfigParser()
        parser.add_section('main')
        parser.set('main', 'cachedir', config['application.updater.yum.cachedir'])
        parser.set('main', 'keepcache', '0')
        parser.set('main', 'debuglevel', '2')
        parser.set('main', 'logfile', config['application.updater.yum.log.file'])
        parser.set('main', 'distroverpkg', 'redhat-release')
        parser.set('main', 'tolerant', '1')
        parser.set('main', 'exactarch', '1')
        parser.set('main', 'obsoletes', '1')
        parser.set('main', 'gpgcheck', '1')
        parser.set('main', 'plugins', '1')
        parser.set('main', 'metadata_expire', '1h')
        parser.set('main', 'installonly_limit', '5')
        if proxy is not None:
            parser.set('main', 'proxy', proxy['http'])

        """
        prevent overwriting karesansui source under development environment.
        """
        cwd = os.path.dirname(os.path.abspath(__file__))
        if os.path.exists(cwd+"/../../.git/"):
            parser.set('main', 'exclude', 'karesansui*')

        fp = open(self._ftemp, "w")
        parser.write(fp)
        fp.close()
        
        self.base = KaresansuiYumBase(self._ftemp)
        self.base.doConfigSetup()

        # rpm
        self.rpmlib = RpmLib()
        
    def __del__(self):
        os.unlink(self._ftemp)
        
    def _pre(self, repos=None):
        # config repos
        if repos is None:
            repos = []
            for y in [x.strip() for x in self.config['application.updater.yum.target.repos'].split(',') if x]:
                repos.append(y)

        self.repos = repos
        if not repos:
            raise KaresansuiYumException('Update repository has not been set.')
        
        # Disable all repos
        for x in self.base.repos.listEnabled():
            self.base.repos.disableRepo(x.id)

        # Enable target repo
        for x in repos:
            self.base.repos.enableRepo(x)

    def _open(self):
        try:
            self.base.doTsSetup()
            self.base.doLock()
            return True
        except yum.Errors.LockError, le:
            raise KaresansuiYumException(le.value)
        except yum.Errors.RepoError, msg:
            raise KaresansuiYumRepoError(msg.value)
        except yum.Errors.YumBaseError, ybe:
            raise KaresansuiYumException(ybe.value)

    def _close(self):
        self.base.closeRpmDB()
        self.base.doUnlock()


    def pkgs(self, repos=None):
        """<comment-ja>
        指定したYumリポジトリからパッケージリストを取得します。
        @param repos: リポジトリ名リスト
        @type repos: list
        @return: Yumリポジトリ別のパッケージリスト
        @rtype: dict
        </comment-ja>
        <comment-en>
        Get package list from specified yum repository
        @param repos: list of repositories
        @type repos: list
        @return: Package list for each yum repository
        @rtype: dict
        </comment-en>
        """
        ret = {}
        try:
            self._pre(repos)
            self._open()
            for repo in self.repos:
                try:
                    ret[repo] = self.base.repos.getPackageSack().simplePkgList()
                except yum.Errors.YumBaseError, ybe:
                    raise KaresansuiYumException(ybe.value)

            return ret
        finally:
            self._close()

    def update_pkgs(self, repos=None):
        """<comment-ja>
        指定したYumリポジトリからアップデート可能なパッケージリストを取得します。
        @param repos: リポジトリ名リスト
        @type repos: list
        @return: Yumリポジトリ別アップデート可能なパッケージリスト
        @rtype: dict
        </comment-ja>
        <comment-en>
        Get updatable package list from specified yum repository.
        @param repos: list of repositories
        @type repos: list
        @return: Updatable package list for each yum repository
        @rtype: dict
        </comment-en>
        """
        try:
            try:
                ret = {}
                self._pre(repos)
                self._open()
                for repo in self.repos:
                    #return self.base.doPackageLists('updates').updates
                    pkg = self.base.doPackageLists('updates').updates
                    pkgs = []
                    for x in pkg:
                        pkgs.append((x.name,
                                     x.arch,
                                     x.epoch,
                                     x.version,
                                     x.release))
                    ret[repo] = pkgs
                return ret
            except yum.Errors.YumBaseError, ybe:
                raise KaresansuiYumException(ybe.value)
        finally:
            self._close()


    def update(self, repos=None, pkg=None):
        """<comment-ja>
        指定されたYumリポジトリ＋パッケージ名を元に[yum update]を行います。
        @param repos:Yumリポジトリ名リスト
        @type repos: list
        @param pkg: パッケージ名
        @type pkg: str
        @return: 実行されたアップデートパッケージリスト
        @rtype: list
        </comment-ja>
        <comment-en>
        Do [yum update] using specified yum repository and package name.
        @param repos: list of yum repositories
        @type repos: list
        @param pkg: package name
        @type pkg: str
        @return: list of updated packages
        @rtype: list
        </comment-en>
        """
        try:
            try:
                self._pre()
                self._open()
                #self.base.doGenericSetup()
                #self.base.doRepoSetup()
                if pkg == None:
                    ret = self.base.update()
                else:
                    ret = self.base.update(pattern=pkg)

                self.base.processTransaction()
                return ret
            except yum.Errors.PackageSackError, pse:
                raise KaresansuiYumException(pse.value)
            except yum.Errors.YumBaseError, ybe:
                raise KaresansuiYumException(ybe.value)

        finally:
            self._close()

    def rpm_detail(self, name):
        """<comment-ja>
        rpmデータベースから指定したrpmパッケージ情報を取得します。
        @param name: rpmパッケージ名
        @type name: str
        @return: rpmパッケージ情報
        @rtype: dict
        </comment-ja>
        <comment-en>
        Get rpm package information for the specified rpm package from rpm database.
        @param name: rpm package name
        @type name: str
        @return: rpm package information
        @rtype: dict
        </comment-en>
        """
        if type(name) is not str:
            return None
        return self.rpmlib.doQuery(name)

    def rpm_details(self, names):
        """<comment-ja>
        rpmデータベースから指定したrpmパッケージ情報を取得します。
        @param names: rpmパッケージ名
        @type names: str
        @return: rpmパッケージ情報
        @rtype: dict
        </comment-ja>
        <comment-en>
        Get rpm package information for the specified rpm package from rpm database.
        @param name: rpm package name
        @type name: str
        @return: rpm package information
        @rtype: dict
        </comment-en>
        """
        if type(name) is not list:
            return None

        if not names:
            return []
        ret = []
        for name in names:
            ret.append(self.rpmlib.doQuery(name))

        return ret

    def repos2rpms_details(self, repos=None):
        """<comment-ja>
        Yumリポジトリ別のrpmパッケージ情報リストを取得します。
        @param repos: Yumリポジトリリスト
        @type repos: list
        @return: rpmパッケージ情報リスト
        @rtype: dict
        </comment-ja>
        <comment-en>
        Get rpm package information list for each yum repositories.
        @param repos: list of yum repositories
        @type repos: list
        @return: rpm package information
        @rtype: dict
        </comment-en>
        """
        _pkgs = self.pkgs(repos)
        rets = {}
        for x in self.repos:
            ret = []
            for y in _pkgs[x]:
                r = self.rpm_detail(y[0])
                if r:
                    ret.append(r) # installed
                else:
                    ret.append(None) # not installed

            rets[x] = ret
        return rets

    def refresh(self, repos=None):
        """<comment-ja>
        指定したYumリポジトリ名を元に、ローカルYumリポジトリを最新にします。
        @param repos: Yumリポジトリリスト
        @type repos: list
        </comment-ja>
        <comment-en>
        Refresh local yum repository using specified yum repository.
        @param repos: list of yum repositories
        @type repos: list
        </comment-en>
        """
        ret = {}
        try:
            self._pre(repos)
            self._open()
            try:
                for repo in self.base.repos.findRepos('*'):
                    repo.metadata_expire = 0
                    repo.mdpolicy = "group:all"
                for repo in self.base.repos.listEnabled():
                    repo.repoXML

                self.base.repos.populateSack(mdtype='metadata', cacheonly=1)
                self.base.repos.populateSack(mdtype='filelists', cacheonly=1)
                self.base.repos.populateSack(mdtype='otherdata', cacheonly=1)
            except yum.Errors.YumBaseError, ybe:
                raise KaresansuiYumException(ybe.value)
        finally:
            self._close()
        

