#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import os
import sys
import re
import grp
import traceback
import time
import glob

from installer.const import *
from installer.trans import _, N_
from installer.utils import *
from installer.rpmlib import RpmLib

karesansui_group = "kss"

def get_domain_data_dir():
    retval = None

    try:
        search_path = sh_config_read(DEFAULT_KARESANSUI_CONF)["application.search.path"]
    except:
        return retval

    """ Set search paths """
    for y in [x.strip() for x in search_path.split(',') if x]:
        if (y in sys.path) is False: sys.path.insert(0, y)

    try:
        import karesansui
        from karesansui.lib.const import VIRT_LIBVIRT_DATA_DIR, VIRT_DOMAINS_DIR
    except ImportError, e:
        return retval

    return VIRT_DOMAINS_DIR


def migrate_10_11(opts,mode=0,log=None,test=False,verbose=False):
    retval = True

    if log is None:
        if verbose is True:
            log = "/dev/stdout"
        else:
            log = "/dev/null"

    logf = open(log,"a")

    domain_data_dir = get_domain_data_dir()
    if domain_data_dir is None:
        retval = False
        if verbose is True:
            sys.stderr.write("Error: VIRT_DOMAINS_DIR is undefined.")
        return retval

    """ pre: define variables """
    data_dir = os.path.dirname(domain_data_dir)
    old_images_dir     = "%s/images"   % (data_dir,)
    old_boot_dir       = "%s/boot"     % (data_dir,)
    old_disk_dir       = "%s/disk"     % (data_dir,)
    old_snapshot_dir   = "%s/snapshot" % (data_dir,)
    old_xml_config_dir = "%s/xen/xml"      % (VENDOR_SYSCONFDIR,)
    new_xml_config_dir = "%s/libvirt/qemu" % (VENDOR_SYSCONFDIR,)
    backup_dir         = "%s/v1.0_save"% (data_dir,)
    kss_config_file    = DEFAULT_KARESANSUI_CONF
    whitelist_file     = DEFAULT_SILHOUETTE_WHITELIST

    if os.path.exists(backup_dir):
        if verbose is True:
            logf.write("Notice: the backup directory for v1.0 old data exists. [%s]\n" % backup_dir)
        return retval

    if verbose is True:
        logf.write("old_images_dir  :%s\n" % (old_images_dir,))
        logf.write("old_boot_dir    :%s\n" % (old_boot_dir,))
        logf.write("old_disk_dir    :%s\n" % (old_disk_dir,))
        logf.write("old_snapshot_dir:%s\n" % (old_snapshot_dir,))
        logf.write("old_xml_config_dir :%s\n" % (old_xml_config_dir,))
        logf.write("new_xml_config_dir :%s\n" % (new_xml_config_dir,))
        logf.write("kss_config_file :%s\n" % (kss_config_file,))
        logf.write("whitelist_file  :%s\n" % (whitelist_file,))


    """ 1, os disk image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the old disk image directory...\n")
    if os.path.exists(old_images_dir):
        if verbose is True:
            logf.write(">> old_images_dir '%s' is found.\n" % (old_images_dir,))

        for file in os.listdir(old_images_dir):
            if file[-4:] == ".img":
                domain_name = file[0:-4]

                if verbose is True:
                    logf.write(">> image file '%s' is found.\n" % (file,))

                new_images_dir = "%s/%s/images" % (domain_data_dir,domain_name,)
                new_images_path = "%s/%s.img" % (new_images_dir,domain_name,)
                old_images_path = "%s/%s"     % (old_images_dir,file,)

                if not os.path.exists(new_images_dir):
                    if test is True:
                        logf.write(">>>[test] os.makedirs('%s')\n" % (new_images_dir,))
                    else:
                        if verbose is True:
                            logf.write(">>> os.makedirs('%s')... " % (new_images_dir,))
                        os.makedirs(new_images_dir)
                        if verbose is True:
                            logf.write("done\n")

                if os.path.exists(new_images_path):
                    if verbose is True:
                        logf.write(">>> new image file '%s' is found.\n" % (new_images_path,))
                else:
                    if test is True:
                        logf.write(">>>[test] os.renames('%s','%s')\n" % (old_images_path,new_images_path,))
                    else:
                        if verbose is True:
                            logf.write(">>> os.renames('%s','%s')... " % (old_images_path,new_images_path,))
                        os.renames(old_images_path,new_images_path)
                        if verbose is True:
                            logf.write("done\n")

        """ 古いディレクトリはバックアップへ移動する """
        if os.path.exists(old_images_dir):
            if test is True:
                logf.write(">>[test] os.renames('%s','%s')\n" % (old_images_dir,backup_dir,))
            else:
                if not os.path.exists(backup_dir):
                    os.makedirs(backup_dir)
                if verbose is True:
                    logf.write(">> os.renames('%s','%s')... " % (old_images_dir,backup_dir,))
                os.renames(old_images_dir,backup_dir)
                if verbose is True:
                    logf.write("done\n")


    """ 2, extra disk image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the extra disk image directory...\n")
    if os.path.exists(old_disk_dir):
        if verbose is True:
            logf.write(">> old_disk_dir '%s' is found.\n" % (old_disk_dir,))

        for domain_name in os.listdir(old_disk_dir):
            new_disk_dir = "%s/%s/disk" % (domain_data_dir,domain_name,)
            old_domain_disk_dir = "%s/%s" % (old_disk_dir,domain_name,)

            if not os.path.exists(new_disk_dir):
                if test is True:
                    logf.write(">>>[test] os.makedirs('%s')\n" % (new_disk_dir,))
                else:
                    if verbose is True:
                        logf.write(">>> os.makedirs('%s')... " % (new_disk_dir,))
                    os.makedirs(new_disk_dir)
                    if verbose is True:
                        logf.write("done\n")

            for file in os.listdir(old_domain_disk_dir):
                if file[-4:] == ".img":
                    new_disk_path = "%s/%s/%s" % (new_disk_dir,domain_name,file)
                    old_disk_path = "%s/%s" % (old_domain_disk_dir,file,)

                    if os.path.exists(new_disk_path):
                        if verbose is True:
                            logf.write(">>> new disk file '%s' is found.\n" % (new_disk_path,))
                    else:
                        if test is True:
                            logf.write(">>>[test] os.renames('%s','%s')\n" % (old_disk_path,new_disk_path,))
                        else:
                            if verbose is True:
                                logf.write(">>> os.renames('%s','%s')... " % (old_disk_path,new_disk_path,))
                            os.renames(old_disk_path,new_disk_path)
                            if verbose is True:
                                logf.write("done\n")

        """ 古いディレクトリはバックアップへ移動する """
        if os.path.exists(old_disk_dir):
            if test is True:
                logf.write(">>[test] os.renames('%s','%s')\n" % (old_disk_dir,backup_dir,))
            else:
                if not os.path.exists(backup_dir):
                    os.makedirs(backup_dir)
                if verbose is True:
                    logf.write(">> os.renames('%s','%s')... " % (old_disk_dir,backup_dir,))
                os.renames(old_disk_dir,backup_dir)
                if verbose is True:
                    logf.write("done\n")


    """ 3, snapshot image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the snapshot disk image directory...\n")
    if os.path.exists(old_snapshot_dir):
        if verbose is True:
            logf.write(">> old_snapshot_dir '%s' is found.\n" % (old_snapshot_dir,))

        for domain_name in os.listdir(old_snapshot_dir):
            new_snapshot_dir = "%s/%s/snapshot" % (domain_data_dir,domain_name,)
            old_domain_snapshot_dir = "%s/%s" % (old_snapshot_dir,domain_name,)

            if not os.path.exists(new_snapshot_dir):
                if test is True:
                    logf.write(">>>[test] os.makedirs('%s')\n" % (new_snapshot_dir,))
                else:
                    if verbose is True:
                        logf.write(">>> os.makedirs('%s')... " % (new_snapshot_dir,))
                    os.makedirs(new_snapshot_dir)
                    if verbose is True:
                        logf.write("done\n")

            for file in os.listdir(old_domain_snapshot_dir):
                new_snapshot_path = "%s/%s/%s" % (new_snapshot_dir,domain_name,file)
                old_snapshot_path = "%s/%s" % (old_domain_snapshot_dir,file,)

                if os.path.exists(new_snapshot_path):
                    if verbose is True:
                        logf.write(">>> new snapshot file '%s' is found.\n" % (new_snapshot_path,))
                else:
                    if test is True:
                        logf.write(">>>[test] os.renames('%s','%s')\n" % (old_snapshot_path,new_snapshot_path,))
                    else:
                        if verbose is True:
                            logf.write(">>> os.renames('%s','%s')... " % (old_snapshot_path,new_snapshot_path,))
                        os.renames(old_snapshot_path,new_snapshot_path)
                        if verbose is True:
                            logf.write("done\n")

        """ 古いディレクトリはバックアップへ移動する """
        if os.path.exists(old_snapshot_dir):
            if test is True:
                logf.write(">>[test] os.renames('%s','%s')\n" % (old_snapshot_dir,backup_dir,))
            else:
                if not os.path.exists(backup_dir):
                    os.makedirs(backup_dir)
                if verbose is True:
                    logf.write(">> os.renames('%s','%s')... " % (old_snapshot_dir,backup_dir,))
                os.renames(old_snapshot_dir,backup_dir)
                if verbose is True:
                    logf.write("done\n")


    """ 4, boot image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the boot image directory...\n")
    if os.path.exists(old_boot_dir) and os.path.exists(old_xml_config_dir):
        if verbose is True:
            logf.write(">> old_boot_dir '%s' is found.\n" % (old_boot_dir,))

        for file in os.listdir(old_xml_config_dir):
            if file[-4:] == ".xml":
                domain_name = file[0:-4]

                new_boot_dir = "%s/%s/boot" % (domain_data_dir,domain_name,)
                old_domain_xml_config_path = "%s/%s" % (old_xml_config_dir,file,)
                if not os.path.exists(new_boot_dir):
                    if test is True:
                        logf.write(">>>[test] os.makedirs('%s')\n" % (new_boot_dir,))
                    else:
                        if verbose is True:
                            logf.write(">>> os.makedirs('%s')... " % (new_boot_dir,))
                        os.makedirs(new_boot_dir)
                        if verbose is True:
                            logf.write("done\n")

                if os.path.exists(old_domain_xml_config_path):
                    if verbose is True:
                        logf.write(">>> old_domain_xml_config_path '%s' is found.\n" % (old_domain_xml_config_path,))

                    doc = get_xml_parse(old_domain_xml_config_path)

                    kernel = get_xml_xpath(doc,'/domain/os/kernel/text()')
                    if kernel is not None:
                        if os.path.exists(kernel):
                            if verbose is True:
                                logf.write(">>>> old kernel '%s' is found.\n" % (kernel,))

                            b_kernel = os.path.basename(kernel)
                            new_boot_kernel_path = "%s/%s" % (new_boot_dir,b_kernel,)
                            if os.path.exists(new_boot_kernel_path):
                                if verbose is True:
                                    logf.write(">>>> new_boot_kernel_path '%s' is found.\n" % (new_boot_kernel_path,))
                            else:
                                if test is True:
                                    logf.write(">>>>[test] os.renames('%s','%s')\n" % (kernel,new_boot_kernel_path,))
                                else:
                                    if verbose is True:
                                        logf.write(">>>> os.renames('%s','%s')... " % (kernel,new_boot_kernel_path,))
                                    os.renames(kernel,new_boot_kernel_path)
                                    if verbose is True:
                                        logf.write("done\n")

                    initrd = get_xml_xpath(doc,'/domain/os/initrd/text()')
                    if initrd is not None:
                        if os.path.exists(initrd):
                            if verbose is True:
                                logf.write(">>>> old initrd '%s' is found.\n" % (initrd,))

                            b_initrd = os.path.basename(initrd)
                            new_boot_initrd_path = "%s/%s" % (new_boot_dir,b_initrd,)
                            if os.path.exists(new_boot_initrd_path):
                                if verbose is True:
                                    logf.write(">>>> new_boot_initrd_path '%s' is found.\n" % (new_boot_initrd_path,))
                            else:
                                if test is True:
                                    logf.write(">>>>[test] os.renames('%s','%s')\n" % (initrd,new_boot_initrd_path,))
                                else:
                                    if verbose is True:
                                        logf.write(">>>> os.renames('%s','%s')... " % (initrd,new_boot_initrd_path,))
                                    os.renames(initrd,new_boot_initrd_path)
                                    if verbose is True:
                                        logf.write("done\n")

        """ 古いディレクトリはバックアップへ移動する """
        if os.path.exists(old_boot_dir):
            if test is True:
                logf.write(">>[test] os.renames('%s','%s')\n" % (old_boot_dir,backup_dir,))
            else:
                if not os.path.exists(backup_dir):
                    os.makedirs(backup_dir)
                if verbose is True:
                    logf.write(">> os.renames('%s','%s')... " % (old_boot_dir,backup_dir,))
                os.renames(old_boot_dir,backup_dir)
                if verbose is True:
                    logf.write("done\n")


    """ 5, xml config """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the xml config directory...\n")
    if os.path.exists(old_xml_config_dir):
        if verbose is True:
            logf.write(">> old_xml_config_dir '%s' is found.\n" % (old_xml_config_dir,))

        for file in os.listdir(old_xml_config_dir):
            if file[-4:] == ".xml":
                domain_name = file[0:-4]

                new_images_dir = "%s/%s/images" % (domain_data_dir,domain_name,)
                new_disk_dir   = "%s/%s/disk"   % (domain_data_dir,domain_name,)
                new_boot_dir   = "%s/%s/boot"   % (domain_data_dir,domain_name,)
                new_snapshot_dir = "%s/%s/snapshot" % (domain_data_dir,domain_name,)

                old_domain_xml_config_path = "%s/%s" % (old_xml_config_dir,file,)
                new_domain_xml_config_path = "%s/%s" % (new_xml_config_dir,file,)
                if os.path.exists(new_domain_xml_config_path):
                    if verbose is True:
                        logf.write(">>>> new_domain_xml_config_path '%s' is found.\n" % (new_domain_xml_config_path,))
                else:
                    doc = get_xml_parse(old_domain_xml_config_path)

                    old_patterns = []
                    new_patterns = []
                    kernel = get_xml_xpath(doc,'/domain/os/kernel/text()')
                    if kernel is not None:
                        b_kernel = os.path.basename(kernel)
                        new_boot_kernel_path = "%s/%s" % (new_boot_dir,b_kernel,)

                        old_line = "<kernel>%s</kernel>" % (kernel,)
                        new_line = "<kernel>%s</kernel>" % (new_boot_kernel_path,)
                        old_patterns.append(old_line)
                        new_patterns.append(new_line)

                    initrd = get_xml_xpath(doc,'/domain/os/initrd/text()')
                    if initrd is not None:
                        b_initrd = os.path.basename(initrd)
                        new_boot_initrd_path = "%s/%s" % (new_boot_dir,b_initrd,)

                        old_line = "<initrd>%s</initrd>" % (initrd,)
                        new_line = "<initrd>%s</initrd>" % (new_boot_initrd_path,)
                        old_patterns.append(old_line)
                        new_patterns.append(new_line)


                    old_line = "<source file='%s/(.*\.img)'/>" % (old_images_dir,)
                    new_line = "<source file='%s/\\1'/>" % (new_images_dir,)
                    old_patterns.append(old_line)
                    new_patterns.append(new_line)


                    old_line = "<source file='%s/%s/(.*\.img)'/>" % (old_disk_dir,domain_name,)
                    new_line = "<source file='%s/\\1'/>" % (new_disk_dir,)
                    old_patterns.append(old_line)
                    new_patterns.append(new_line)


                    old_line = "<currentSnapshot>%s/%s/(.*)</currentSnapshot>" % (old_snapshot_dir,domain_name,)
                    new_line = "<currentSnapshot>%s/\\1</currentSnapshot>" % (new_snapshot_dir,)
                    old_patterns.append(old_line)
                    new_patterns.append(new_line)

                    if verbose is True:
                        logf.write(">>> old_patterns:\n  "+'\n  '.join(old_patterns))
                        logf.write(">>> replacements:\n  "+'\n  '.join(new_patterns))

                    __TEST_MODE = True
                    if test is True:
                        try:
                            logf.write(">>>[test] xmlstr = ")
                            lines = []
                            fp = open(old_domain_xml_config_path,"r")
                            fcntl.lockf(fp.fileno(), fcntl.LOCK_SH)
                            for line in fp.readlines():
                                lines.append(line)
                            fcntl.lockf(fp.fileno(), fcntl.LOCK_UN)
                            fp.close()
                            lines = array_replace(lines, old_patterns, new_patterns)
                            for line in lines:
                                logf.write("%s",line)
                            __TEST_MODE = True
                        except:
                            pass

                    else:
                        if verbose is True:
                            logf.write(">>> file_contents_replace('%s','%s', ...)... " % (old_domain_xml_config_path, new_domain_xml_config_path,))

                        file_contents_replace(old_domain_xml_config_path, new_domain_xml_config_path, old_patterns, new_patterns)

                        if verbose is True:
                            logf.write("done\n")

                    if os.path.exists(new_domain_xml_config_path) or __TEST_MODE is True:
                        autostart_dir = "%s/autostart" % (new_xml_config_dir,)
                        autostart_file = "%s/%s.xml" % (autostart_dir,domain_name,)

                        if not os.path.exists(autostart_dir):
                            if test is True:
                                logf.write(">>>[test] os.makedirs('%s')\n" % (autostart_dir,))
                            else:
                                if verbose is True:
                                    logf.write(">>> os.makedirs('%s')... " % (autostart_dir,))
                                os.makedirs(autostart_dir)
                                if verbose is True:
                                    logf.write("done\n")

                        if os.path.lexists(autostart_file):
                            if test is True:
                                if verbose is True:
                                    logf.write(">>>>[test] os.unlink('%s')\n" % (autostart_file,))
                                    logf.write(">>>>[test] os.symlink('%s','%s')\n" % (new_domain_xml_config_path,autostart_file,))
                            else:
                                if verbose is True:
                                    logf.write(">>>> os.unlink('%s')... " % (autostart_file,))
                                os.unlink(autostart_file)
                                if verbose is True:
                                    logf.write("done\n")
                                    logf.write(">>>> os.symlink('%s','%s')... " % (new_domain_xml_config_path,autostart_file,))
                                os.symlink(new_domain_xml_config_path,autostart_file)
                                if verbose is True:
                                    logf.write("done\n")

                        command_args = [VENDOR_PREFIX+"/bin/virsh" ,"define",new_domain_xml_config_path]
                        if test is True:
                            if verbose is True:
                                logf.write(">>>>[test] execute_command('%s')\n" % (' '.join(command_args),))
                        else:
                            if verbose is True:
                                logf.write(">>>> execute_command('%s')... " % (' '.join(command_args),))
                            ret = execute_command(command_args)
                            if verbose is True:
                                logf.write("done\n")

    """ 6, karesansui config file """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check Karesansui configuration file...\n")
    if os.path.exists(kss_config_file):
        if verbose is True:
            logf.write(">> kss_config_file '%s' is found.\n" % (kss_config_file,))

        """ new parameter """
        new_params = {
         "application.updater.yum.cachedir"    :"/var/opt/karesansui/cache/yum",
         "application.updater.yum.target.repos":"karesansui-base",
         "application.updater.yum.log.file"    :"/var/log/karesansui/yum.log",
        }
        new_lines = []
        for k,v in new_params.iteritems():
            try:
                sh_config_read(kss_config_file)[k]
            except:
                new_lines.append("%s=%s" % (k,v,))

        if len(new_lines) > 0:
            if test is True:
                if verbose is True:
                    logf.write(">>>>[test] append lines: %s\n" % (new_lines,))
            else:
                if verbose is True:
                    logf.write(">>>> append lines: %s ... " % (new_lines,))
                try:
                    fp = open(kss_config_file,"a")
                    fcntl.lockf(fp.fileno(), fcntl.LOCK_EX)
                    fp.write("%s\n" % ("\n".join(new_lines),))
                    fcntl.lockf(fp.fileno(), fcntl.LOCK_UN)
                    fp.close()
                    if verbose is True:
                        logf.write("done\n")
                except TypeError, e:
                    pass
                except IOError, e:
                    if verbose is True:
                        logf.write("failed. [%s]\n" % e)
                    pass

    """ 7, whitelist config file """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check pysilhouette whitelist configuration file...\n")
    if os.path.exists(whitelist_file):
        if verbose is True:
            logf.write(">> whitelist_file '%s' is found.\n" % (whitelist_file,))

        """ new command """
        new_commands = [
         "/opt/karesansui/bin/update_software.py",   
        ]    
        new_lines = []
        try:
            fp = open(whitelist_file,"r")
            fcntl.lockf(fp.fileno(), fcntl.LOCK_SH)
            commands = []
            for line in fp.readlines():
                line = line.strip()
                commands.append(line)
            for line in new_commands:
                if not line in commands:
                    new_lines.append(line)
            fcntl.lockf(fp.fileno(), fcntl.LOCK_UN)
            fp.close()
        except TypeError, e:
            pass
        except IOError, e:
            pass

        if len(new_lines) > 0:
            if test is True:
                if verbose is True:
                    logf.write(">>>>[test] append lines: %s\n" % (new_lines,))
            else:
                if verbose is True:
                    logf.write(">>>> append lines: %s ... " % (new_lines,))
                try:
                    fp = open(whitelist_file,"a")
                    fcntl.lockf(fp.fileno(), fcntl.LOCK_EX)
                    fp.write("%s\n" % ("\n".join(new_lines),))
                    fcntl.lockf(fp.fileno(), fcntl.LOCK_UN)
                    fp.close()
                    if verbose is True:
                        logf.write("done\n")
                except TypeError, e:
                    pass
                except IOError, e:
                    if verbose is True:
                        logf.write("failed. [%s]\n" % e)
                    pass


    """ 新しいディレクトリ配下のパーミッションを変更する """
    if os.path.exists(domain_data_dir):
        if test is True:
            logf.write(">>[test] r_chgrp('%s','%s')\n" % (domain_data_dir,karesansui_group,))
            logf.write(">>[test] r_chmod('%s','g+rw')\n"  % (domain_data_dir,))
            logf.write(">>[test] r_chmod('%s','o-rwx')\n" % (domain_data_dir,))
        else:
            if verbose is True:
                logf.write(">> r_chgrp('%s','%s')... " % (domain_data_dir,karesansui_group,))
            r_chgrp(domain_data_dir,karesansui_group)
            if verbose is True:
                logf.write("done\n")

            if verbose is True:
                logf.write(">> r_chmod('%s','o-rwx')... " % (domain_data_dir,))
            r_chmod(domain_data_dir,"g+rw")
            r_chmod(domain_data_dir,"o-rwx")
            if verbose is True:
                logf.write("done\n")

    """ XML設定ディレクトリ配下のパーミッションを変更する """
    if os.path.exists(new_xml_config_dir):
        if test is True:
            logf.write(">>[test] r_chgrp('%s','%s')\n" % (new_xml_config_dir,karesansui_group,))
            logf.write(">>[test] r_chmod('%s','g+rw')\n"  % (new_xml_config_dir,))
            logf.write(">>[test] r_chmod('%s','o-rwx')\n" % (new_xml_config_dir,))
        else:
            if verbose is True:
                logf.write(">> r_chgrp('%s','%s')... " % (new_xml_config_dir,karesansui_group,))
            r_chgrp(new_xml_config_dir,karesansui_group)
            if verbose is True:
                logf.write("done\n")

            if verbose is True:
                logf.write(">> r_chmod('%s','o-rwx')... " % (new_xml_config_dir,))
            r_chmod(new_xml_config_dir,"g+rw")
            r_chmod(new_xml_config_dir,"o-rwx")
            if verbose is True:
                logf.write("done\n")


    logf.close()

    return retval

def migrate_11_20(opts,mode=0,log=None,test=False,verbose=False):
    retval = True

    if log is None:
        if verbose is True:
            log = "/dev/stdout"
        else:
            log = "/dev/null"

    logf = open(log,"a")

    domain_data_dir = get_domain_data_dir()
    if domain_data_dir is None:
        retval = False
        if verbose is True:
            sys.stderr.write("Error: VIRT_DOMAINS_DIR is undefined.")
        return retval

    # まず1.0から1.1への変換を試行する
    migrate_10_11(opts,mode=mode,log=log,test=test,verbose=verbose)

    """ pre: define variables """
    data_dir = os.path.dirname(domain_data_dir)
    backup_dir         = "%s/v1.1_save"% (data_dir,)
    kss_config_file    = DEFAULT_KARESANSUI_CONF
    sil_config_file    = DEFAULT_SILHOUETTE_CONF
    whitelist_file     = DEFAULT_SILHOUETTE_WHITELIST

    if os.path.exists(backup_dir):
        if verbose is True:
            logf.write("Notice: the backup directory for v1.1 old data exists. [%s]\n" % backup_dir)
        return retval

    if verbose is True:
        logf.write("kss_config_file :%s\n" % (kss_config_file,))
        logf.write("whitelist_file  :%s\n" % (whitelist_file,))


    """ 1, database """
    try:
        import karesansui
        import karesansui.lib.file.k2v
    except ImportError, e:
        traceback.format_exc()
        raise InstallError(_("ERROR: There are not enough libraries. - %s") % ''.join(e.args))
    karesansui.config = karesansui.lib.file.k2v.K2V(os.path.abspath(opts.config)).read()
    if not karesansui.config:
        traceback.format_exc()
        raise InstallError(_("ERROR: Failed to load configuration file. - %s") % opts.config)

    import installer.database
    dbtype, value = karesansui.config['database.bind'].split(':',1)
    installer.database.import_database_module(dbtype)
    installer.database.initialize(opts,drop=False)

    """ 2, os disk image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the old disk image directory...\n")
    for file in glob.glob("%s/*/images/*.img" % domain_data_dir):
        if verbose is True:
            logf.write(">> image file '%s' is found.\n" % (file,))

        domain_name = os.path.basename(file)[0:-4]

        symlink_exists = False
        for vol in glob.glob("%s/[0-9a-z]*-[0-9a-z]*-[0-9a-z]*-*" % domain_data_dir):
            if os.path.islink(vol) is True and os.path.realpath(vol) == file:
                symlink_exists = True

        if symlink_exists is False:
            new_vol = "%s/%s" % (domain_data_dir,string_from_uuid(generate_uuid()),)
            if test is True:
                logf.write(">>>[test] os.symlink('%s','%s')\n" % (file,new_vol))
            else:
                if verbose is True:
                    logf.write(">>> os.symlink('%s','%s')... " % (file,new_vol))
                os.symlink(file,new_vol) 
                if verbose is True:
                    logf.write("done\n")

    """ 3, extra disk image """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check the extra disk image directory...\n")
    for file in glob.glob("%s/*/disk/*.img" % domain_data_dir):
        if verbose is True:
            logf.write(">> image file '%s' is found.\n" % (file,))

        symlink_exists = False
        for vol in glob.glob("%s/[0-9a-z]*-[0-9a-z]*-[0-9a-z]*-*" % domain_data_dir):
            if os.path.islink(vol) is True and os.path.realpath(vol) == file:
                symlink_exists = True

        if symlink_exists is False:
            new_vol = "%s/%s" % (domain_data_dir,string_from_uuid(generate_uuid()),)
            if test is True:
                logf.write(">>>[test] os.symlink('%s','%s')\n" % (file,new_vol))
            else:
                if verbose is True:
                    logf.write(">>> os.symlink('%s','%s')... " % (file,new_vol))
                os.symlink(file,new_vol) 
                if verbose is True:
                    logf.write("done\n")

    """ 4, karesansui config file """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check Karesansui configuration file...\n")

    if os.path.exists(kss_config_file):
        if verbose is True:
            logf.write(">> kss_config_file '%s' is found.\n" % (kss_config_file,))

        new_kss_config_file = "%s.rpmnew" % kss_config_file

        """ set parameter """
        params = [
                   "application.default.locale",
                   "application.mail.server",
                   "application.mail.port",
                   "application.proxy.status",
                   "application.proxy.server",
                   "application.proxy.port",
                   "application.uniqkey",
                   "database.bind",
                 ]
        for _param in params:
            if test is True:
                logf.write(">>>[test] set param '%s' in '%s'\n" % (_param,new_kss_config_file))
            else:
                if verbose is True:
                    logf.write(">>> set param '%s' in '%s'... " % (_param,new_kss_config_file))
                write_ini_samevalue(_param,kss_config_file,new_kss_config_file)
                if verbose is True:
                    logf.write("done\n")


    """ 5, pysilhouette config file """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check pysilhouette configuration file...\n")

    if os.path.exists(sil_config_file):
        if verbose is True:
            logf.write(">> sil_config_file '%s' is found.\n" % (sil_config_file,))

        new_sil_config_file = "%s.rpmnew" % sil_config_file

        """ set parameter """
        params = [
                   "asynscheduler.interval",
                   "env.uniqkey",
                   "database.url",
                 ]
        for _param in params:
            if test is True:
                logf.write(">>>[test] set param '%s' in '%s'\n" % (_param,new_sil_config_file))
            else:
                if verbose is True:
                    logf.write(">>> set param '%s' in '%s'... " % (_param,new_sil_config_file))
                write_ini_samevalue(_param,sil_config_file,new_sil_config_file)
                if verbose is True:
                    logf.write("done\n")


    """ 6, whitelist config file """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check pysilhouette whitelist configuration file...\n")
    if os.path.exists(whitelist_file):
        if verbose is True:
            logf.write(">> whitelist_file '%s' is found.\n" % (whitelist_file,))

        kss_whitelist_file = "%s/whitelist.conf.example" % KARESANSUI_SYSCONFDIR
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (kss_whitelist_file,whitelist_file))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (kss_whitelist_file,whitelist_file))
            copy_file(kss_whitelist_file,whitelist_file)
            if verbose is True:
                logf.write("done\n")

    """ 7, do what is needed for pysilhouette """
    if verbose is True:
        logf.write("\n")
        logf.write("> Do what is needed for pysilhouette...\n")

    silhouette_py  = "%s/pysilhouette/silhouette.py" % SILHOUETTE_PYTHONPATH
    silhouette_sym = "%s/bin/silhouette.py"          % SILHOUETTE_PREFIX
    if os.path.exists(silhouette_sym) is False:
        if test is True:
            logf.write(">>>[test] os.symlink('%s','%s')\n" % (silhouette_py,silhouette_sym))
        else:
            if verbose is True:
                logf.write(">>> os.symlink('%s','%s')... " % (silhouette_py,silhouette_sym))
            os.symlink(silhouette_py,silhouette_sym) 
            if verbose is True:
                logf.write("done\n")

    """ 8, copy new configurations """
    if verbose is True:
        logf.write("\n")
        logf.write("> Copying the new configurations...\n")

    if not os.path.exists(backup_dir):
        if test is True:
            logf.write(">>>[test] os.makedirs('%s')\n" % (backup_dir))
        else:
            if verbose is True:
                logf.write(">>> os.makedirs('%s')... " % (backup_dir))
            os.makedirs(backup_dir)
            if verbose is True:
                logf.write("done\n")

    new_kss_config_file = "%s.rpmnew" % kss_config_file
    new_sil_config_file = "%s.rpmnew" % sil_config_file
    bak_kss_config_file = "%s/%s" % (backup_dir,os.path.basename(kss_config_file),)
    bak_sil_config_file = "%s/%s" % (backup_dir,os.path.basename(sil_config_file),)

    if os.path.exists(kss_config_file):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (kss_config_file,bak_kss_config_file))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (kss_config_file,bak_kss_config_file))
            copy_file(kss_config_file,bak_kss_config_file)
            if verbose is True:
                logf.write("done\n")

    if os.path.exists(new_kss_config_file):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (new_kss_config_file,kss_config_file))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (new_kss_config_file,kss_config_file))
            copy_file(new_kss_config_file,kss_config_file)
            if verbose is True:
                logf.write("done\n")

    if os.path.exists(sil_config_file):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (sil_config_file,bak_sil_config_file))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (sil_config_file,bak_sil_config_file))
            copy_file(sil_config_file,bak_sil_config_file)
            if verbose is True:
                logf.write("done\n")

    if os.path.exists(new_sil_config_file):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (new_sil_config_file,sil_config_file))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (new_sil_config_file,sil_config_file))
            copy_file(new_sil_config_file,sil_config_file)
            if verbose is True:
                logf.write("done\n")

    """ 9, Initialize pysilhouette database """
    if verbose is True:
        logf.write("\n")
        logf.write("> Initializing pysilhouette database...\n")

    sil_database = "%s/pysilhouette.db" % (SILHOUETTE_DATADIR)
    bak_sil_database = "%s/pysilhouette.db" % (backup_dir,)

    if os.path.exists(sil_database):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (sil_database,bak_sil_database))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (sil_database,bak_sil_database))
            copy_file(sil_database,bak_sil_database)
            if verbose is True:
                logf.write("done\n")

    try:
        __psil_cleandb = glob.glob("%s/share/doc/pysilhouette*/tool/psil-cleandb" % SILHOUETTE_PREFIX)[0]
        cmd = "%s --config=%s >/dev/null" % (__psil_cleandb,sil_config_file,)
        env = {}
        env['PYTHONPATH'] = '%s:%s' % (VENDOR_PYTHONPATH,SILHOUETTE_PYTHONPATH,)
        if test is True:
            logf.write(">>>[test] execute_command('%s')\n" % (cmd))
        else:
            if verbose is True:
                logf.write(">>> execute_command('%s')... " % (cmd))
            (ret,res) = execute_command(cmd.split(' '),env=env)
            if verbose is True:
                logf.write("done\n")
    except Exception, e:
        if verbose is True:
            logf.write(">>> error %s" % str(e.args))
        pass


    """ 10, set lighttpd configurations """
    if verbose is True:
        logf.write("\n")
        logf.write("> Check lighttpd configuration file...\n")

    new_kss_lighty_conf = "%s/lighttpd.example/karesansui.conf" % (KARESANSUI_SYSCONFDIR,)
    kss_lighty_conf     = "%s/lighttpd/conf.d/karesansui.conf"  % (VENDOR_SYSCONFDIR,)

    if os.path.exists(new_kss_lighty_conf):
        if test is True:
            logf.write(">>>[test] copy_file('%s','%s')\n" % (new_kss_lighty_conf,kss_lighty_conf))
        else:
            if verbose is True:
                logf.write(">>> copy_file('%s','%s')... " % (new_kss_lighty_conf,kss_lighty_conf))
            copy_file(new_kss_lighty_conf,kss_lighty_conf)
            if verbose is True:
                logf.write("done\n")


    """ 新しいディレクトリ配下のパーミッションを変更する """
    if os.path.exists(domain_data_dir):
        if test is True:
            logf.write(">>[test] r_chgrp('%s','%s')\n" % (domain_data_dir,karesansui_group,))
            logf.write(">>[test] r_chmod('%s','g+rw')\n"  % (domain_data_dir,))
            logf.write(">>[test] r_chmod('%s','o-rwx')\n" % (domain_data_dir,))
        else:
            if verbose is True:
                logf.write(">> r_chgrp('%s','%s')... " % (domain_data_dir,karesansui_group,))
            r_chgrp(domain_data_dir,karesansui_group)
            if verbose is True:
                logf.write("done\n")

            if verbose is True:
                logf.write(">> r_chmod('%s','o-rwx')... " % (domain_data_dir,))
            r_chmod(domain_data_dir,"g+rw")
            r_chmod(domain_data_dir,"o-rwx")
            if verbose is True:
                logf.write("done\n")



    logf.close()

    return retval

