# Kanbanara Authenticate Component
# Written by Rebecca Shalfield between 2013 and 2017
# Copyright (c) Rebecca Shalfield 2013-2017
# Released under the GNU AGPL v3

'''Kanbanara's Authenticate Component'''

import datetime
from email.mime.text import MIMEText
import os
import random
import re
import smtplib
import socket
import sys

import bson
from bson import ObjectId
import cherrypy
from pymongo import MongoClient


class Authenticate:
    '''Kanbanara's Authenticate Component'''

    def __init__(self):
        self.current_dir = os.path.dirname(os.path.abspath(__file__))

        self.read_mongodb_ini_file()

        # Connect to MongoDB on given host and port
        connection = MongoClient(self.mongodbHost, self.mongodbPort)

        # Connect to 'vault' database, creating if not already exists
        vault_db = connection['vault']

        # Connect to 'members' collection
        self.members_collection = vault_db['members']

        self.kanbanara_settings = {'instance': "Untitled"}
        self.read_administrator_ini_file()
        self.read_kanbanara_ini_file()
        self.instance = self.kanbanara_settings['instance']

        self.major, self.minor, self.revision, self.build, self.date = self.read_version_ini_file()

    @staticmethod
    def getstatusoutput(cmd):
        """Return (status, output) of executing cmd in a shell."""
        mswindows = (sys.platform == "win32")
        if not mswindows:
            cmd = '{ ' + cmd + '; }'

        pipe = os.popen(cmd + ' 2>&1', 'r')
        text = pipe.read()
        sts = pipe.close()
        if sts is None:
            sts = 0

        if text[-1:] == '\n':
            text = text[:-1]

        return sts, text

    @cherrypy.expose
    def changepassword(self, returnurl="", username="", existingpassword="", newpassword="",
                       newpasswordconfirm=""):
        warning = ""
        if username and existingpassword and newpassword and newpasswordconfirm:
            if self.members_collection.find({'username': username,
                                             'password': existingpassword
                                            }).count():
                if newpassword == newpasswordconfirm:
                    member_document = self.members_collection.find_one({'username': username,
                                                                        'password': existingpassword
                                                                       })
                    member_document['password'] = newpassword
                    self.members_collection.save(member_document)
                    message = "Your password has now been changed!"
                    raise cherrypy.HTTPRedirect('/authenticate/confirmation?message='+message+'&returnurl='+returnurl, 302)
                else:
                    warning = '<div class="warning"><p class="warning">New passwords do not match!</p></div>'

            else:
                warning = '<div class="warning"><p class="warning">Access denied!</p></div>'

        content = []
        content.append(self.header(returnurl, title='Change Password'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Change Password</h3>')
        if warning:
            content.append(warning)

        content.append('<form action="/authenticate/changepassword" method="post">')
        content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')
        content.append('<table border="0"><tr><td align="right">Username (Email Address)</td><td>')
        if username:
            content.append('<input type="text" size="40" name="username" value="'+username+'" placeholder="Email Address">')
        else:
            content.append('<input type="text" size="40" name="username" placeholder="Email Address">')

        content.append('</td></tr>')
        content.append('<tr><td align="right">Existing Password</td><td>')
        if existingpassword:
            content.append('<input type="password" size="40" name="existingpassword" value="'+existingpassword+'" placeholder="Existing Password">')
        else:
            content.append('<input type="password" size="40" name="existingpassword" placeholder="Existing Password">')

        content.append('</td></tr><tr><td align="right">New Password</td>')
        content.append('<td><input type="password" size="40" name="newpassword" placeholder="New Password"></td></tr>')
        content.append('<tr><td align="right">New Password Confirmation</td><td><input type="password" size="40" name="newpasswordconfirm" placeholder="New Password"></td></tr>')
        content.append('<tr><td align="center" colspan="2"><input class="button" type="submit" value="Change Password"></td></tr></table></form>')
        if returnurl and returnurl != '/kanban':
            content.append('<p class="returnurl">You will be returned to <a href="'+returnurl+'">'+returnurl+'</a> following a successful logon!</p>')

        content.append(self.insert_links('changepassword', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def insert_links(self, page, username, returnurl):
        content = []
        content.append('<hr>')
        socket_hostname = socket.gethostname()
        if (socket_hostname == 'www.kanbanara.com' or
                os.path.exists(os.path.join(self.current_dir, '..', 'website'))):
            content.append('<form class="inline" action="/" method="post"><button type="submit" value="Website"><span class="fa fa-globe fa-lg"></span>&nbsp;Website</button></form> ')

        if page != 'index':
            content.append('<form class="inline" action="/authenticate/index" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<button type="submit" value="Logon"><span class="fa fa-sign-in fa-lg"></span>&nbsp;Logon</button></form> ')

        if page != 'register':
            content.append('<form class="inline" action="/authenticate/register" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<button type="submit" value="Register"><span class="fa fa-registered fa-lg"></span>&nbsp;Register</button></form> ')

        if page != 'changepassword':
            content.append('<form class="inline" action="/authenticate/changepassword" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            if username:
                content.append('<input type="hidden" name="username" value="'+username+'">')

            content.append('<button type="submit" value="Change Password"><span class="fa fa-unlock-alt fa-lg"></span>&nbsp;Change Password</button></form> ')

        if page != 'resetpassword':
            content.append('<form class="inline" action="/authenticate/resetpassword" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            if username:
                content.append('<input type="hidden" name="username" value="'+username+'">')

            content.append('<button type="submit" value="Reset Password"><span class="fa fa-unlock-alt fa-lg"></span>&nbsp;Reset Password</button></form>')

        return "".join(content)

    @cherrypy.expose
    def confirmation(self, message, returnurl=""):
        content = []
        content.append(self.header('', title='Confirmation'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Confirmation</h3>')
        content.append('<p>'+message+'</p>')
        if returnurl:
            if returnurl != '/kanban':
                content.append('<p class="returnurl">You\'ll be redirected to '+returnurl+' in a few moments...</p>')

        else:
            content.append('<table><tr><td>')
            content.append('<p>Where do you want to go now?</p>')
            content.append('<p>')
            socket_hostname = socket.gethostname()
            if (socket_hostname == 'www.kanbanara.com' or
                    os.path.exists(os.path.join(self.current_dir, '..', 'website'))):
                content.append('<form class="inline" action="/" method="post"><input class="button" type="submit" value="Website"></form> ')

            content.append('<form class="inline" action="/authenticate/register" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<input class="button" type="submit" value="Register"></form> <form class="inline" action="/authenticate" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<input class="button" type="submit" value="Logon"></form>  <form class="inline" action="/authenticate/changepassword" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<input class="button" type="submit" value="Change Password"></form> <form class="inline" action="/authenticate/resetpassword" method="post">')
            if returnurl:
                content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

            content.append('<input class="button" type="submit" value="Reset Password"></form></p>')

            content.append('</td></tr></table>')

        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def deletemember(self, username, password):
        content = []
        content.append(self.header('', title='Delete Member'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Delete Member</h3>')
        if self.members_collection.find({'username': username, 'password': password}).count():
            member_document = self.members_collection.find_one({'username': username,
                                                                'password': password
                                                               })
            self.members_collection.remove({'_id': ObjectId(member_document['_id'])})
            content.append("<p>Member '"+username+"' has now been deleted!</p>")
        else:
            content.append('<p class="warning">Username and/or password are incorrect!</p>')

        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def footer(self):
        content = []
        content.append('<div id="footer_container">')
        content.append('<div id="footer_contents"><h3>Copyright &copy; Rebecca Shalfield 2013-2017</h3></div>')
        content.append('</div>')
        content.append('</body></html>')
        return "".join(content)

    def header(self, returnurl, title):
        content = []
        content.append('<html><head><title>Kanbanara - '+title+'</title>')
        content.append('<link rel="stylesheet" type="text/css" href="/authenticate/css/authenticate.css">')
        content.append('<link rel="stylesheet" type="text/css" href="/css/rootmenuwebapps.css">')
        content.append('<link rel="stylesheet" href="/font-awesome/css/font-awesome.min.css">')
        if title == 'Confirmation' and returnurl:
            content.append('<meta http-equiv="refresh" content="5;URL='+returnurl+'">')

        if title == 'Registration':
            content.append('<link rel="stylesheet" href="/jquery-ui-1.12.1.custom/jquery-ui.css">')
            content.append('<link rel="stylesheet" href="/jquery-ui-1.12.1.custom/jquery-ui.structure.css">')
            content.append('<link rel="stylesheet" href="/jquery-ui-1.12.1.custom/jquery-ui.theme.css">')
            content.append('<script type="text/javascript" src="/jquery-ui-1.12.1.custom/external/jquery/jquery.js"></script>')
            content.append('<script type="text/javascript" src="/jquery-ui-1.12.1.custom/jquery-ui.min.js"></script>')
            content.append('<script type="text/javascript" src="/authenticate/scripts/authenticate.js"></script>')

        slides = os.listdir(self.current_dir+os.sep+'images'+os.sep+'slideshow')
        if slides:
            random_index = random.randrange(0, len(slides))
            slide = slides[random_index]
            content.append("<style>body {background-image : url('/authenticate/images/slideshow/"+slide+"')}</style>")

        content.append('</head><body>')
        content.append('<h1 class="header">Kanbanara</h1>')
        content.append('<h2>A Kanban-Based Project Management System</h2>')
        content.append(f'<h2>Version {self.major}.{self.minor}.{self.revision}.{self.build} ({self.date})</h2>')
        return "".join(content)

    def read_kanbanara_ini_file(self):
        """Reads the kanbanara.ini file and extracts instance information"""
        with open(os.path.join(self.current_dir, '..', 'kanbanara.ini'), 'r') as handle:
            data = handle.read()
            for (setting, regex) in [('instance', r'(?i)instance\s?=\s?([\w\. ]+)')]:
                pattern = re.compile(regex)
                results = pattern.findall(data)
                if results:
                    self.kanbanara_settings[setting] = results[0]

    @cherrypy.expose
    def index(self, returnurl="", username="", password=""):
        if username and password:
            if self.members_collection.find({'username': username, 'password': password}).count():
                member_document = self.members_collection.find_one({'username': username})
                cherrypy.session['username']  = username
                cherrypy.session['firstname'] = member_document['firstname']
                cherrypy.session['lastname']  = member_document['lastname']
                if returnurl:
                    raise cherrypy.HTTPRedirect(returnurl, 302)
                else:
                    raise cherrypy.HTTPRedirect('/kanban', 302)

            else:
                raise cherrypy.HTTPError(401, 'Unauthorised')

        content = []
        content.append(self.header(returnurl, title='Logon'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Logon</h3>')
        content.append('<table class="outer"><tr><td class="outer">')
        content.append('<p>If you\'re already registered, you may logon here<br>by entering your username and password</p>')
        content.append('<form action="/authenticate/index" method="post">')
        if returnurl:
            content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')

        content.append('<table class="logon">')
        content.append('<tr><td><p>'+self.instance+'</p></td></tr>')
        content.append('<tr><td><p><span class="fa fa-user fa-lg"></span> ')
        if username:
            content.append('<input type="text" size="40" name="username" value="'+username+'" placeholder="Email Address">')
        else:
            content.append('<input type="text" size="40" name="username" placeholder="Username">')

        content.append('</p></td></tr><tr><td><p><span class="fa fa-lock fa-lg"></span> <input type="password" size="40" name="password" placeholder="Password"></p></td></tr><tr><td align="center"><p><input class="button" type="submit" value="Logon"></p></td></tr></table></form>')
        if returnurl and returnurl != '/kanban':
            content.append('<p class="returnurl">You will be returned to <a href="'+returnurl+'">'+returnurl+'</a> following a successful logon!</p>')

        content.append('</td></tr></table>')
        content.append(self.insert_links('index', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def validateusername(self, username):
        warning = ""
        if self.members_collection.find({'username': username}).count():
            warning = '<p class="warning">Username already in use!</p>'

        if not re.search(r'^[\w_\-\.]+@\S+\.\w+$',username):
            warning = '<p class="warning">Not a valid email address!</p>'

        return warning

    def read_version_ini_file(self):
        """Reads the version.ini file and extracts versioning information"""
        major = 0
        minor = 0
        revision = 0
        build = 0
        date = ""
        
        with open(os.path.join(self.current_dir, '..', 'version.ini'), 'r') as ip:
            settings = ip.read()
            
            major_pattern = re.compile(r'(?i)major\s?=\s?(\d+)')
            results = major_pattern.findall(settings)
            if results:
                major = int(results[0])        
            
            minor_pattern = re.compile(r'(?i)minor\s?=\s?(\d+)')
            results = minor_pattern.findall(settings)
            if results:
                minor = int(results[0])

            revision_pattern = re.compile(r'(?i)revision\s?=\s?(\d+)')
            results = revision_pattern.findall(settings)
            if results:
                revision = int(results[0])

            build_pattern = re.compile(r'(?i)build\s?=\s?(\d+)')
            results = build_pattern.findall(settings)
            if results:
                build = int(results[0])
                                
            date_pattern = re.compile(r'(?i)date\s?=\s?(\d{4}\-\d{2}\-\d{2})')
            results = date_pattern.findall(settings)
            if results:
                date = results[0]

        return major, minor, revision, build, date

    @cherrypy.expose
    def register(self, returnurl="", username="", firstname="", lastname="", password="",
                 passwordconfirm=""):
        warning = ""
        lowercase_username = username.lower()
        if username and firstname and lastname and password and passwordconfirm:
            if password == passwordconfirm:
                if not self.members_collection.find({'username': lowercase_username}).count():
                    member_document = {'username':  lowercase_username,
                                       'firstname': firstname,
                                       'lastname':  lastname,
                                       'password':  password
                                      }
                    self.members_collection.insert(member_document)
                    message = "You are now registered!"
                    raise cherrypy.HTTPRedirect('/authenticate/confirmation?message='+message+'&returnurl='+returnurl, 302)
                else:
                    warning = '<div class="warning"><p class="warning">Username has already been registered!</p><form action="/authenticate/index" method="post"><input type="hidden" name="returnurl" value="'+returnurl+'"><input type="hidden" name="username" value="'+lowercase_username+'"><input class="button" type="submit" value="Logon"></form></div>'

            else:
                warning = '<div class="warning"><p class="warning">Passwords do not match!</p></div>'

        content = []
        content.append(self.header(returnurl, title='Registration'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Registration</h3>')
        content.append('<p>Should you need to register, you may do so here by entering<br>your email address, first name, last name and a password</p>')

        socket_hostname = socket.gethostname()
        if socket_hostname == 'www.kanbanara.com' or os.path.exists(os.path.join(self.current_dir, '..', 'website')):
            content.append('<p>By registering, you will automatically become<br>a member of the <a href="/foundation">Kanbanara Software Foundation</a> and<br>agree to be bound by its terms and conditions</p>')

        if warning:
            content.append(warning)

        content.append('<form action="/authenticate/register" method="post">')
        content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')
        content.append('<table border="0"><tr><td align="right">Username (Email Address)</td><td>')
        if username:
            content.append('<input id="username" type="text" size="40" name="username" value="'+lowercase_username+'" placeholder="Email Address">')
        else:
            content.append('<input id="username" type="text" size="40" name="username" placeholder="Email Address">')

        content.append('</td><td>')
        content.append('<div id="validateusername">')
        content.append('<!-- This section is automatically populated by JQuery and Ajax -->')
        content.append('</div>')
        content.append('</td></tr><tr><td align="right">First Name</td><td>')
        if firstname:
            content.append('<input type="text" size="40" name="firstname" value="'+firstname+'" placeholder="First Name">')
        else:
            content.append('<input type="text" size="40" name="firstname" placeholder="First Name">')

        content.append('</td><td></td></tr><tr><td align="right">Last Name</td><td>')
        if lastname:
            content.append('<input type="text" size="40" name="lastname" value="'+lastname+'" placeholder="Last Name">')
        else:
            content.append('<input type="text" size="40" name="lastname" placeholder="Last Name">')

        content.append('</td><td></td></tr><tr><td align="right">Password</td><td><input type="password" size="40" name="password" placeholder="Password"></td><td></td></tr><tr><td align="right">Password Confirmation</td><td><input type="password" size="40" name="passwordconfirm" placeholder="Password"></td><td></td></tr><tr><td align="center" colspan="3"><input class="button" type="submit" value="Register"></td></tr></table></form>')

        if returnurl and returnurl != '/kanban':
            content.append('<p class="returnurl">You will be returned to <a href="'+returnurl+'">'+returnurl+'</a> following a successful logon!</p>')

        content.append(self.insert_links('register', lowercase_username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def resetpassword(self, returnurl="", username="", resetpassword="No"):
        content = []
        content.append(self.header(returnurl, title='Reset Password'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Reset Password</h3>')
        if username and resetpassword == "Yes":
            if self.members_collection.find({'username': username}).count():
                new_password = "".join([chr(random.randint(97, 122)) for i in range(8)])
                msg = MIMEText(f"Your password has been reset to '{new_password}'.")
                msg['Subject'] = "Kanbanara - Reset Password"
                msg['From'] = self.kanbanara_settings['email_address']
                msg['To'] = username
                SERVER = self.kanbanara_settings['smtp_server_host']
                PORT = int(self.kanbanara_settings['smtp_server_port'])
                USERNAME = self.kanbanara_settings['smtp_server_username']
                PASSWORD = self.kanbanara_settings['smtp_server_password']
                server = smtplib.SMTP(host=SERVER, port=PORT)
                server.set_debuglevel(True) # show communication with the server
                # identify ourselves to smtp client
                server.ehlo()
                if PORT in [465, 587]:
                    # secure our email with TLS encryption
                    try:
                        server.starttls()
                    except:
                        True
                    
                    # re-identify ourselves as an encrypted connection
                    server.ehlo()

                server.login(USERNAME, PASSWORD)
                server.send_message(msg)
                server.quit()
                member_document = self.members_collection.find_one({'username': username})
                member_document['password'] = new_password
                self.members_collection.save(member_document)
                message = f"An email has been sent to {username}!"
                raise cherrypy.HTTPRedirect(f'/authenticate/confirmation?message={message}&returnurl={returnurl}', 302)
            else:
                content.append('<p class="warning">Username does not exist!</p>')

        content.append('<p>If you\'ve forgotten your password,<br>you may reset it by sending us your username</p>')
        content.append('<form action="/authenticate/resetpassword" method="post">')
        content.append('<input type="hidden" name="returnurl" value="'+returnurl+'">')
        content.append('<input type="hidden" name="resetpassword" value="Yes">')
        content.append('<table border="0"><tr><td align="right">Username</td><td>')
        if username:
            content.append('<input type="text" size="40" name="username" value="'+username+'" placeholder="Email Address">')
        else:
            content.append('<input type="text" size="40" name="username" placeholder="Email Address">')

        content.append('</td></tr><tr><td align="center" colspan="2"><input class="button" type="submit" value="Reset Password"></td></tr></table></form>')
        if returnurl and returnurl != '/kanban':
            content.append('<p class="returnurl">You will be returned to <a href="'+returnurl+'">'+returnurl+'</a> following a successful logon!</p>')

        content.append(self.insert_links('resetpassword', username, returnurl))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def logoff(self):
        cherrypy.session['username']=None
        cherrypy.session.delete()
        cherrypy.lib.sessions.expire()
        content = []
        content.append(self.header('', title='Logoff'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Logoff</h3>')
        content.append('<p>You are now logged off!</p>')
        content.append(self.insert_links('logoff', '', ''))
        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    @cherrypy.expose
    def members(self, username="", password=""):
        content = []
        content.append(self.header('', title='Members'))
        content.append('<div id="maincontent">')
        content.append('<h3 class="subheading">Members</h3>')
        if username == self.kanbanara_settings['admin_username'] and password == self.kanbanara_settings['admin_password']:
            content.append('<table class="members"><tr><th>Last&nbsp;Name</th><th>First&nbsp;Name</th><th>Username&nbsp;(Email&nbsp;Address)</th><th>Password</th><th>Logon</th><th>Change Password</th><th>Reset Password</th><th>Delete Member</th></tr>')
            for document in self.members_collection.find({}):
                content.append('<tr><td>'+document['lastname']+'</td><td>'+document['firstname']+'</td><td>'+document['username']+'</td><td>'+document['password']+'</td><td><form action="/authenticate" method="post"><input type="hidden" name="username" value="'+document['username']+'"><input type="hidden" name="password" value="'+document['password']+'"><input class="button" type="submit" value="Logon"></form></td><td><form action="/authenticate/changepassword" method="post"><input type="hidden" name="username" value="'+document['username']+'"><input type="hidden" name="existingpassword" value="'+document['password']+'"><input class="button" type="submit" value="Change Password"></form></td><td><form action="/authenticate/resetpassword" method="post"><input type="hidden" name="username" value="'+document['username']+'"><input class="button" type="submit" value="Reset Password"></form></td><td><form action="/authenticate/deletemember" method="post"><input type="hidden" name="username" value="'+document['username']+'"><input type="hidden" name="password" value="'+document['password']+'"><input class="button" type="submit" value="Delete Member"></form></td></tr>')

            content.append('</table>')
        else:
            content.append('<form action="/authenticate/members" method="post">')
            content.append('<table border="0"><tr><td align="right">Administrator Username</td><td>')
            content.append('<input type="password" size="40" name="username" placeholder="Username">')
            content.append('</td></tr><tr><td align="right">Administrator Password</td><td><input type="password" size="40" name="password" placeholder="Password"></td></tr><tr><td align="center" colspan="2"><input class="button" type="submit" value="Enter"></td></tr></table></form>')

        content.append('</div>')
        content.append(self.footer())
        return "".join(content)

    def read_administrator_ini_file(self):
        """ Reads the administrator.ini file and extracts username and password information """
        with open(os.path.join(self.current_dir, '..', 'administrator.ini'), 'r') as handle:
            data = handle.read()
            for (setting, regex) in [
                    ('admin_username', r'(?i)username\s?=\s?(\S+)'),
                    ('admin_password', r'(?i)password\s?=\s?(\S+)'),
                    ('smtp_server_host', r'(?i)smtp_server_host\s?=\s?(\S+)'),
                    ('smtp_server_password', r'(?i)smtp_server_password\s?=\s?(\S+)'),
                    ('smtp_server_port', r'(?i)smtp_server_port\s?=\s?(\d+)'),
                    ('smtp_server_username', r'(?i)smtp_server_username\s?=\s?(\S+)'),
                    ('email_address', r'(?i)email_address\s?=\s?(\S+)')]:
                pattern = re.compile(regex)
                results = pattern.findall(data)
                if results:
                    self.kanbanara_settings[setting] = results[0]
                else:
                    self.kanbanara_settings[setting] = ""

    def read_mongodb_ini_file(self):
        """ Reads the common mongodb.ini file and extracts host and port information """
        self.mongodbHost = 'localhost'
        self.mongodbPort = 27017
        ip = open(self.current_dir+os.sep+'..'+os.sep+"mongodb.ini","r")
        settings = ip.read()
        ip.close()
        mongodb_host_pattern = re.compile(r'(?i)host\s?=\s?(\S+)')
        results = mongodb_host_pattern.findall(settings)
        if results != []:
            self.mongodbHost = results[0]

        mongodb_port_pattern = re.compile('(?i)port\s?=\s?(\d+)')
        results = mongodb_port_pattern.findall(settings)
        if results != []:
            self.mongodbPort = int(results[0])

CURRENTDIR = os.path.dirname(os.path.abspath(__file__))
conf = {}
conf['/'] = {'tools.staticdir.root': CURRENTDIR,
             'tools.sessions.on':    True}
for directory in ['css', 'images', 'scripts']:
    if os.path.exists(CURRENTDIR+os.sep+directory):
        conf['/'+directory] = {'tools.staticdir.on':  True,
                               'tools.staticdir.dir': directory}

cherrypy.tree.mount(Authenticate(), '/authenticate', config=conf)
