#!/usr/local/bin/ruby 

require 'uri'
require 'misen/style'
require 'misen/util'

require 'PageParser.rb'
require 'ToWikiHtml.rb'
require 'FilePageDataSource.rb'
require 'PStorePageDataSource.rb'

class KakiRequestHandler
	TEMPLATE_DIR = "./template"

	def initialize(page_data_source, app_url)
		@app_url = app_url
		@page_data_source = page_data_source
		@wiki2html = Wiki::ToWikiHtml.new(page_data_source, '')
		@wiki2html.link_base_uri = "#{@app_url}?p="
		@wiki2html.xhtml = true;

		self.site_property = @page_data_source.site_property()
		self.theme_dir = './theme'
		self.theme_base_url = './theme'
	end

	attr 'theme_dir', true
	attr 'theme_base_url', true

	def main(request_variables, request_parameters)
		command = request_command(request_parameters)

		data = {}
		begin
			# template file
			tmpl_file = "#{command}.html"

			# command 
			case command
			when 'create' then
				create(request_variables, request_parameters, data)
			when 'property' then
				property(request_variables, request_parameters, data)
			when 'edit' then
				edit(request_variables, request_parameters, data)
			when 'preview' then
				preview(request_variables, request_parameters, data)
			when 'save' then
				save(request_variables, request_parameters, data)
			when 'change_passwd' then
				tmpl_file = "save.html"
				change_passwd(request_variables, request_parameters, data)
			when 'snapshot' then
				tmpl_file = "save.html"
				snapshot(request_variables, request_parameters, data)
			when 'index' then
				index(request_variables, request_parameters, data)
			when 'recent' then
				recent(request_variables, request_parameters, data)
			when 'admin' then
				admin(request_variables, request_parameters, data)
			when 'delete' then
				delete(request_variables, request_parameters, data)
			else 
				# default command is 'view'
				command = 'view'
				tmpl_file = 'view.html'
				view(request_variables, request_parameters, data)
			end


			# side menu
			request_lang_list = self.reuqest_language_list(request_variables)
			sub_page_id = @page_data_source.suitable_page_id('SideMenu', request_lang_list)
			if sub_page_id then
				sub_page_name, sub_page_lang, = @page_data_source.parse_page_id(sub_page_id)
				sub_page_doc = @page_data_source.load_page_document(sub_page_name, sub_page_lang);
				if sub_page_doc then
					@wiki2html.output = ''
					@wiki2html.default_lang_list = request_lang_list
					sub_page_doc.accept(@wiki2html)
					data[:sub_page] = @wiki2html.output
				end
			end

		rescue Racc::ParseError
			data[:parse_error] = true
			unless command == 'preview' then
				if command != 'edit' then
					command = 'edit'
					retry
				else
					data[:exception] = $!.to_html if $! 
				end
			end
		rescue 
			case $!.message
			when "NoPageID" then
				if command == 'edit' then
					command = 'create'
					data[:no_page_id] = true
					retry
				end
			when "NoSuchPage" then
				if command == 'view' then
					command = 'edit'
					request_parameters['p'] = ['FrontPage']
					retry
				else
					data[:exception] = $!.to_html if $! 
				end
			when "PageAuthorizationFail" then
				case command
				when 'save' then
					command = 'preview'
					retry
				when 'snapshot', 'change_passwd' then
					command = 'edit'
					retry
				else
					data[:exception] = $!.to_html if $! 
				end
			else
				data[:exception] = $!.to_html if $! 
			end
		else 
			data[:exception] = $!.to_html if $! 
		end

		# common data 
		data.update(site_common_data())

		# template file path
		request_lang_list = [] unless request_lang_list
		tmpl_path, lang = template_path(request_lang_list + ['en'], tmpl_file)
		tmpl =  File.readlines(tmpl_path).to_s

		# http response header
		response_header = { }
		response_header['Content-Type'] = "text/html; charset=UTF-8"
		response_header['Language'] = lang if lang

=begin 
		if command == 'view' then
			last_modified_time = self.site_property()['last_modified_time']
			if data[:source_time] then 
				if data[:source_time] > last_modified_time then
					last_modified_time = data[:source_time]
				end
			end
			response_header['Last-Modified'] = CGI.rfc1123_date(last_modified_time) 
		end
=end

#		data[:exception] = CGI.escapeHTML(request_variables.inspect)

		# http response body
		body = Misen.expand_text(Misen::STYLE_SGML, tmpl, data)
		return [response_header, body];
	end

protected

	attr_reader "site_property"
	attr_writer "site_property"

	def view(request_variables, request_parameters, data)
		request_page_id = request_page_id(request_parameters, 'FrontPage')
		page_name, lang, version = @page_data_source.parse_page_id(request_page_id)

		request_lang_list = self.reuqest_language_list(request_variables)

		if lang then
			if @page_data_source.exist?(page_name, lang, version) then
				page_id = request_page_id
			end
		else
			page_id = @page_data_source.suitable_page_id(page_name, request_lang_list, version)
		end

		raise "NoSuchPage" unless page_id 

		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)
		wiki_doc = @page_data_source.load_page_document(page_name, page_lang, page_version)
		if wiki_doc then
			@wiki2html.output = ''
#			@wiki2html.default_lang_list = request_lang_list
			@wiki2html.default_lang_list = nil
			wiki_doc.accept(@wiki2html)
			data[:content] = @wiki2html.output 
			@page_data_source.page_property(page_name, page_lang, page_version){ |property|
				passwd = property['passwd']
				data[:is_frozen] = self.site_property['is_frozen']
				data[:is_frozen] = true if (passwd and passwd.length > 0)
				data[:is_not_frozen] = (not data[:is_frozen])
				data[:counter] = { :count => property['count_int']}
				data[:source_time] = property['source_time']
				data[:page_title] = CGI.escapeHTML(property['page_title'])
				data[:modified_time] = property['source_time'].to_data_hash
				append_referer(request_variables, property)
				property['count_int'] += 1
			}
			
			data[:page_id_uri] = URI.escape(page_id)
			data[:page_id] = CGI.escapeHTML(page_id)
			data[:page_name] = CGI.escapeHTML(page_name)
			data[:page_lang] = CGI.escapeHTML(page_lang)
			data[:page_name_and_lang] = URI.escape(@page_data_source.make_page_id(page_name, page_lang))
			if page_version then
				data[:page_version] = { :version_int => page_version.to_i } 
			end

			data[:history] = page_history_data(page_name, page_lang)
			data[:lang_selector] = page_language_selector_data(page_name, page_lang)
		end
	end

	def create(request_variables, request_parameters, data)
		data[:language_list] = make_language_list_data(supported_language_list())
	end

	def property(request_variables, request_parameters, data)
		request_page_id = request_page_id(request_parameters, 'FrontPage')
		page_name, lang, version = @page_data_source.parse_page_id(request_page_id)

		request_lang_list = self.reuqest_language_list(request_variables)

		if lang then
			if @page_data_source.exist?(page_name, lang, version) then
				page_id = request_page_id
			end
		else
			page_id = @page_data_source.suitable_page_id(page_name, request_lang_list, version)
		end

		raise "NoSuchPage" unless page_id 

		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)
		property = @page_data_source.page_property(page_name, page_lang, page_version)
		passwd = property['passwd']
		data[:is_frozen] = self.site_property['is_frozen']
		data[:is_frozen] = true if (passwd and passwd.length > 0)
		data[:is_not_frozen] = (not data[:is_frozen])
		data[:counter] = { :count => property['count_int']}
		data[:source_time] = property['source_time']
		data[:page_title] = CGI.escapeHTML(property['page_title'])
		data[:modified_time] = property['source_time'].to_data_hash
		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_name_and_lang] = URI.escape(@page_data_source.make_page_id(page_name, page_lang))
		if page_version then
			data[:page_version] = { :version_int => page_version.to_i } 
		end

		data[:history] = page_history_data(page_name, page_lang)
		data[:lang_selector] = page_language_selector_data(page_name, page_lang)
		data[:referer] = page_referer_data(property)
	end


	def edit(request_variables, request_parameters, data)
		page_id = self.request_page_id(request_parameters)
		raise 'NoPageID' unless page_id
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		unless page_lang then
			l = request_parameters['l'].to_s.untaint
			if l.length > 0 then
				page_lang = l
			else
				page_lang = self.site_property['default_language']
			end
		end

		page_id = @page_data_source.make_page_id(page_name, page_lang)
		page_title = page_name

		content_source = ''
		source_time_str = '0'
		if @page_data_source.exist?(page_name, page_lang) then
			content_source = @page_data_source.load_page_source(page_name, page_lang);
			@page_data_source.page_property(page_name, page_lang) { |property|
				page_title = property['page_title']
				source_time_str = property['source_time'].to_i.to_s.reverse
				passwd = property['passwd']
				data[:is_frozen] = self.site_property['is_frozen']
				data[:is_frozen] = true if (passwd and passwd.length > 0)
			}
		else
			original_language = request_parameters['original_language'].to_s.untaint
			if original_language.length > 0 and @page_data_source.exist?(page_name, original_language) then
				content_source = @page_data_source.load_page_source(page_name, original_language);
			end
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(page_title)
		data[:is_not_frozen] = (not data[:is_frozen])

		data[:content_source] = CGI.escapeHTML(content_source)
		salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
		data[:edit_key] = source_time_str.crypt(salt)
	end

	def preview(request_variables, request_parameters, data)
		page_id = request_page_id(request_parameters)
		content_source = request_content_source(request_parameters)
		edit_key = request_edit_key(request_parameters)
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)
		unless page_lang then
			page_lang = self.site_property['default_language']
		end
		page_id = @page_data_source.make_page_id(page_name, page_lang)
		page_title = request_page_title(request_parameters)

		is_frozen = self.site_property['is_frozen']
		if @page_data_source.exist?(page_name, page_lang) then
			@page_data_source.page_property(page_name, page_lang, page_version){ |property|
				if property['passwd'] and property['passwd'].length > 0 then
					is_frozen = true
				end
			}
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(page_title)

		data[:is_frozen] = is_frozen
		data[:is_not_frozen] = (not is_frozen)
		data[:content_source] = CGI.escapeHTML(content_source)
		data[:edit_key] = edit_key

		wiki_doc =  Wiki::PageParser.new().parse(content_source);
		if wiki_doc then
			@wiki2html.output = ''
#			@wiki2html.default_lang_list = [page_lang] + self.reuqest_language_list(request_variables)
#			@wiki2html.default_lang_list = self.reuqest_language_list(request_variables)
			@wiki2html.default_lang_list = nil
			wiki_doc.accept(@wiki2html)
			data[:content] = @wiki2html.output 
		else
			data[:content] = ''
		end
	end

	def save(request_variables, request_parameters, data)
		page_id = request_page_id(request_parameters)
		content_source = request_content_source(request_parameters)
		edit_key = request_edit_key(request_parameters)
		passwd = request_passwd(request_parameters)
		page_title = request_page_title(request_parameters)

		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		property = {}
		if @page_data_source.exist?(page_name, page_lang) then
			property = @page_data_source.page_property(page_name, page_lang)
		end

		if self.page_login_ok?(property, passwd) then
			# check editing conflict 
			source_time_str = property['source_time'].to_i.to_s.reverse if property 
			if edit_key != source_time_str.crypt(edit_key) then
				data[:save_conflict_msg] = true
			else
				@page_data_source.save_page_source(page_name, page_lang, nil, content_source)
				property = @page_data_source.page_property(page_name, page_lang){ |prop|
					prop['source_time'] = Time.now
					prop['page_title'] = page_title
				}
				data[:save_ok_msg] = { :page_id => page_id, :page_name => page_name }
			end
		else
			data[:page_save_auth_fail] = true
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(property['page_title'])

		if data[:page_save_auth_fail] then
			raise "PageAuthorizationFail"
		end
	end

	def change_passwd(request_variables, request_parameters, data)
		passwd = request_parameters['passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_passwd1'].to_s.untaint
		page_id = request_page_id(request_parameters)
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		@page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(property, passwd) then
				# if non passwd or authorized
				if new_passwd1.length > 0 then
					if new_passwd1 == new_passwd2 then
						if property['passwd'].to_s.length > 0 then
							data[:passwd_change_ok] = true
						else
							data[:freeze_ok] = { :page_id => CGI.escapeHTML(page_id) }
						end

						salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
						property['passwd'] = new_passwd1.crypt(salt)
					else
						data[:new_passwd_miss_match] = true
					end
				else
					property['passwd'] = nil
					data[:thaw_ok] = { :page_id => CGI.escapeHTML(page_id) }
				end
			else
				data[:auth_fail] = true
			end
			data[:page_id_uri] = URI.escape(page_id)
			data[:page_id] = CGI.escapeHTML(page_id)
			data[:page_name] = CGI.escapeHTML(page_name)
			data[:page_lang] = CGI.escapeHTML(page_lang)
			data[:page_title] = CGI.escapeHTML(property['page_title'])
		}

		if data[:auth_fail] then
			raise "PageAuthorizationFail"
		end
	end

	def snapshot(request_variables, request_parameters, data)
		page_id = request_page_id(request_parameters)
		passwd = request_passwd(request_parameters)
		page_name, page_lang, = @page_data_source.parse_page_id(page_id)

		@page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(property, passwd) then
				version_str = sprintf("%08d", property['version_int'])
				source = @page_data_source.load_page_source(page_name, page_lang)
				@page_data_source.save_page_source(page_name, page_lang, version_str, source)
				@page_data_source.page_property(page_name, page_lang, version_str){ |snap_prop|
					property.each(){ |key, value|
						snap_prop[key] = value
					}
					snap_prop['passwd'] = nil
				}
				property['version_int'] += 1
				data[:snapshot_ok_msg] = { :page_id => CGI.escapeHTML(page_id) }
				data[:page_id_uri] = URI.escape(page_id)
				data[:page_id] = CGI.escapeHTML(page_id)
				data[:page_name] = CGI.escapeHTML(page_name)
				data[:page_lang] = CGI.escapeHTML(page_lang)
				data[:page_title] = CGI.escapeHTML(property['page_title'])
			else
				data[:auth_fail] = true
			end
		}

		if data[:auth_fail] then
			raise "PageAuthorizationFail"
		end

	end

	def index(request_variables, request_parameters, data)
		pages = @page_data_source.page_list
		page_data_by_page_name = {}
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]
			page_name, lang, version = @page_data_source.parse_page_id(page_id)

			page_data = page_data_by_page_name[page_name]
			unless page_data then
				page_data = {
					:page_id_uri => URI.escape(page_name),
					:page_id => CGI.escapeHTML(page_name),
					:app_url => @app_url,
					:page_name => CGI.escapeHTML(page_name),
					:lang_list => [],
					'lang_hash' => {},
				}
				page_data_by_page_name[page_name] = page_data
			end

			lang_data = page_data['lang_hash'][lang]
			unless lang_data then
				is_frozen = self.site_property['is_frozen']
				is_frozen = true if (page_property['passwd'].to_s.length > 0)
				lang_data = {
					:app_url => @app_url,
					:page_id_uri => URI.escape("#{page_name}.#{lang}"),
					:page_id => CGI.escapeHTML("#{page_name}.#{lang}"),
					:page_lang => CGI.escapeHTML(lang),
					:page_title => CGI.escapeHTML(page_property['page_title']),
					:is_frozen => is_frozen, 
					:is_not_frozen => (not is_frozen), 
					:count => page_property['count_int'],
					:version_list => [],
				}
				lang_data.update(page_property['source_time'].to_data_hash)
				page_data['lang_hash'][lang] = lang_data
				page_data[:lang_list] << lang_data
			end

			if version then
				version_data = {
					:page_id_uri => URI.escape(page_id),
					:page_id => CGI.escapeHTML(page_id),
					:app_url => @app_url,
					:version_int => page_property['version_int'],
				}
				version_data.update(page_property['source_time'].to_data_hash)
				lang_data[:version_list] << version_data
			end
		}
		page_list = page_data_by_page_name.values.sort(){ |a, b|
			a[:page_name].upcase <=> b[:page_name].upcase 
		}
		data[:index] ={ :page_list => page_list}
	end

	def recent(request_variables, request_parameters, data)
		pages = @page_data_source.page_list
		pages.sort!(){ |a, b|
			b[1]['source_time'] <=> a[1]['source_time']
		}

		date_hash = {} # key: date string "%Y-%m-%d", value: hash
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]

			date_str = page_property['source_time'].strftime("%Y-%m-%d") 
			date_data = date_hash[date_str]
			unless date_data then
				date_data = page_property['source_time'].to_data_hash
				date_data[:page_list] = []
				date_hash[date_str] = date_data
			end

			page_name, lang, version = @page_data_source.parse_page_id(page_id)
			page_data = page_property['source_time'].to_data_hash
			page_data[:app_url] = @app_url
			page_data[:page_id_uri] = URI.escape(page_id)
			page_data[:page_id] = CGI.escapeHTML(page_id)
			page_data[:page_name] = CGI.escapeHTML(page_name)
			page_data[:page_lang] = CGI.escapeHTML(lang)
			page_data[:is_frozen] = self.site_property['is_frozen']
			page_data[:is_frozen] = true if (page_property['passwd'].to_s.length > 0)
			page_data[:is_not_frozen] = (not page_data[:is_frozen])
			if version then
				page_data[:version] = { :version_int => page_property['version_int'] }
			else
				page_data[:counter] = { :count_int => page_property['count_int'] }
			end
			date_data[:page_list] << page_data
		}
		date_list = date_hash.to_a
		date_list.sort!()
		date_list.reverse!()
		date_list.collect!{ |date_pair|
			date_pair[1]
		}
		data[:recent] ={ :date_list => date_list }
	end

	def admin_save_property(request_variables, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		site_name = request_parameters['site_name'].to_s.untaint
		admin_name = request_parameters['admin_name'].to_s.untaint
		admin_email = request_parameters['admin_email'].to_s.untaint
		theme = request_parameters['theme'].to_s.untaint
		sidebar_control = request_parameters['sidebar_control'].to_s.untaint
		default_language = request_parameters['default_language'].to_s.untaint
		default_language = 'en' if default_language.length <= 0
		languages = request_parameters['languages'].to_s.untaint

		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				property['site_name'] = site_name 
				property['admin_name'] = admin_name 
				property['admin_email'] = admin_email 
				property['theme'] = theme 
				property['sidebar_control'] = sidebar_control 
				property['default_language'] = default_language 
				property['languages'] = languages 
				self.site_property = property
				data[:site_property_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin_change_passwd(request_variables, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_admin_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_admin_passwd2'].to_s.untaint
		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				if new_passwd1.length > 0 then
					if new_passwd1 == new_passwd2 then
						salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
						property['admin_passwd'] = new_passwd1.crypt(salt)
					else
						data[:new_passwd_miss_match] = true
					end
				else
					property['admin_passwd'] = nil
				end
				self.site_property = property
				data[:admin_passwd_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin_site_freeze(request_variables, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		site_freeze = request_parameters['site_freeze'].to_s.untaint
		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				case site_freeze.downcase 
				when 'true', 'yes' then
					property['is_frozen'] = true
					data[:site_freeze_ok] = true
				when 'false', 'no' then
					property['is_frozen'] = false
					data[:site_thaw_ok] = true
				end
				self.site_property = property
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin(request_variables, request_parameters, data)
		save_property = request_parameters['save_property'].to_s.untaint
		change_passwd = request_parameters['change_passwd'].to_s.untaint
		site_freeze = request_parameters['site_freeze'].to_s.untaint
		if save_property.length > 0 then
			admin_save_property(request_variables, request_parameters, data)
		elsif change_passwd.length > 0 then
			admin_change_passwd(request_variables, request_parameters, data)
		elsif site_freeze.length > 0 then
			admin_site_freeze(request_variables, request_parameters, data)
		end

		theme_list = Dir.glob("#{self.theme_dir}/*/").sort().collect(){ |theme_dir|
			theme_name = File.basename(File.dirname(theme_dir))
			theme_data = { :theme_name => theme_name }
			if theme_name == self.site_property['theme'] then
				theme_data[:is_selected] = true
			end
			theme_data
		}
		data[:theme_list] = theme_list

		case self.site_property['sidebar_control']
		when "no_sidebar" then
			data[:no_sidebar_selected] = true 
		when "left_sidebar" then
			data[:left_sidebar_selected] = true 
		when "left_sidebar_fixed" then
			data[:left_sidebar_fixed_selected] = true 
		when "right_sidebar" then
			data[:right_sidebar_selected] = true 
		when "right_sidebar_fixed" then
			data[:right_sidebar_fixed_selected] = true 
		else
			data[:none_selected] = true 
		end

		data[:languages] = self.site_property['languages']
	end

	def delete_page(request_variables, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				page_list = request_parameters['page_list']
				page_list.each(){ |page_id|
					name, lang, version = @page_data_source.parse_page_id(page_id)
					if @page_data_source.exist?(name, lang, version) then
						@page_data_source.delete_page(name, lang, version)
					end
				}
				data[:delete_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def delete(request_variables, request_parameters, data)
		delete_page = request_parameters['delete_page'].to_s.untaint
		if delete_page.length > 0 then
			delete_page(request_variables, request_parameters, data)
		end

		page_list = []
		@page_data_source.page_list.each(){ |page_info|
			page_id = page_info[0] 
			property = page_info[1]
			page_data = {
				:app_url => @app_url, 
				:page_id_uri => URI.escape(page_id), 
				:page_id => CGI.escapeHTML(page_id), 
				:page_title => CGI.escapeHTML(property['page_title']),
			}
			page_list << page_data
		}
		data[:page_list] = page_list
	end

	def request_command(request_parameters)
		result = request_parameters['c'].to_s
		result = 'view' if result.length <= 0 
		result.untaint
		return result;
	end

	def request_page_id(request_parameters, default_page_id = nil)
		page_id = request_parameters['p'].to_s
		page_id = default_page_id if page_id.length <= 0 
		page_id.untaint
		return page_id;
	end

	def request_content_source(request_parameters)
		content_source = request_parameters['content_source'].to_s
		content_source.gsub!(/\r\n/, "\n");
		content_source = "\n" if content_source.length <= 0 
		content_source.untaint
		return content_source;
	end

	def request_edit_key(request_parameters)
		edit_key = request_parameters['e'].to_s
		edit_key.untaint
		return edit_key;
	end

	def request_passwd(request_parameters)
		k = request_parameters['k'].to_s
		passwd = nil
		passwd = k if k.length > 0 
		passwd.untaint
		return passwd;
	end

	def request_page_title(request_parameters)
		result = request_parameters['page_title'].to_s
		if result.length <= 0 then
			result = request_page_id(request_parameters, 'FrontPage') 
			result, = @page_data_source.parse_page_id(result)
		end
		result.untaint
		return result;
	end

	def reuqest_language_list(request_variables)
		result = parse_accept_language(request_variables['HTTP_ACCEPT_LANGUAGE'].to_s.untaint)
		result.push(self.site_property["default_language"].untaint)
		return result;
	end

	def parse_accept_language(accept_language_str)
		result = accept_language_str.split(',')
		result.collect!{ |lang_prop|
			lang, q = lang_prop.split(';q=')
			lang.strip!()
			if q then
				[q.to_f, lang]
			else
				[1.0, lang_prop]
			end
		}
		result.sort!()
		result.reverse!()
		result.collect!{ |lang| lang[1] }
		result
	end

	def site_common_data()
		result ={}
		result[:app_url] = @app_url

		result[:theme] = self.site_property['theme']
		result[:sidebar_control] = self.site_property['sidebar_control'] 
		result[:site_name] = self.site_property['site_name']

		result[:admin_name] = self.site_property['admin_name']
		result[:admin_email] = self.site_property['admin_email']
		if self.site_property['admin_email'] and self.site_property['admin_email'].length > 0 then
			result[:admin] = {
				:admin_name => self.site_property['admin_name'],
				:admin_email => self.site_property['admin_email'],
			}
		end

		result[:default_language] = self.site_property['default_language']
		result[:language_list] = make_language_list_data(supported_language_list())
		result[:theme_base_url] = self.theme_base_url
		result[:site_is_frozen] = self.site_property['is_frozen']
		result[:site_is_not_frozen] = (not result[:site_is_frozen])

		return result
	end

	def page_history_data(page_name, page_lang)
		result = nil
		version_list = []
		@page_data_source.page_list().each(){ |page|
			id = page[0]
			property = page[1]
			n, l, v= @page_data_source.parse_page_id(id)
			if n == page_name and l == page_lang then
				is_snapshot = nil
				if v then
					is_snapshot = { :version_int => property['version_int'] }
				end
				version_list << {
					:app_url => @app_url,
					:page_id_uri => URI.escape(id),
					:is_recent=> (is_snapshot.nil?),
					:is_snapshot => is_snapshot,
					:modified_time => property['source_time'].to_data_hash,
				}
			end
		}
		result = { :version_list => version_list } if version_list.length > 1 
		return result;
	end

	def page_language_selector_data(page_name, page_lang)
		result = nil
		lang_list = []
		@page_data_source.page_list().each(){ |page|
			id = page[0]
			prop = page[1]
			n, l, v= @page_data_source.parse_page_id(page[0])
			if n == page_name and v == nil then
				is_selected = false
				is_selected = true if page_lang == l 
				lang_list << {
					:page_id_uri => URI.escape("#{n}.#{l}"),
					:page_id => CGI.escapeHTML("#{n}.#{l}"),
					:page_title => CGI.escapeHTML(prop['page_title']),
					:is_selected => is_selected,
					:lang => l,
				}
			end
		}
		if lang_list.length > 1 then
			result = { 
				:lang_list => lang_list,
				:app_url => @app_url, 
			} 
		end
		return result;
	end

	def page_referer_data(page_property)
		result = nil
		referers = page_property['referers'].to_s.split(", ")
		referer_list = referers.collect(){ |referer|
			{ :referer_url => referer }
		}

		if referer_list.length > 0 then
			result = { :referer_list => referer_list }
		end
		result
	end

	def make_language_list_data(language_list)
		result = []
		default_language = self.site_property['default_language'] 
		language_list.each(){ |lang|
			is_selected = false
			is_selected = true if lang == default_language 
			result << { 
				:lang => lang,
				:is_selected => is_selected,
			}
		}
		result
	end

	def supported_language_list()
		languages = self.site_property['languages']
		language_list = languages.split(',')
		language_list.each(){ |lang|
			lang.strip!()
		}
		language_list.delete_if(){ |lang|
			(/[A-Za-z][A-Za-z](-[A-Za-z][A-Za-z])?/ =~ lang) == nil
		}
		language_list
	end

	def template_path(request_lang_list, tmpl_file)
		result = nil
		lang = nil
		request_lang_list.each(){ |l|
			path = "#{TEMPLATE_DIR}/#{l}/#{tmpl_file}"
			if File.file?(path) and File.readable?(path) then
				result = path
				lang = l
				break;
			end
		}
		[result, lang]
	end

	def append_referer(request_variables, page_property)
		referer = request_variables['HTTP_REFERER']
		
		if referer then
			request_uri = URI.parse(request_variables['REQUEST_URI'])
			if request_uri.host then
				server_uri = request_uri.host
				server_uri += ":#{request_uri.port}" if request_uri.port != 80
			else
				server_uri = request_variables['HTTP_HOST']
			end
			server_uri += request_uri.path

			url_str = referer.to_s 
			uri = URI.parse(url_str)
			referer_uri = uri.host
			referer_uri += ":#{uri.port}" if uri.port != 80
			referer_uri += uri.path
			if server_uri != referer_uri then
				referers = page_property['referers'].to_s.split(", ")
				referers.delete(url_str)
				referers.delete_if() { |ref|
					ref =~ /#{Regexp.escape(server_uri)}/ 
				}
				referers.unshift(url_str)
				referers = referers[0,10]
				page_property['referers'] = referers.join(", ")
			end
		end
	end

	def page_login_ok?(page_property, passwd)
		result = false

		unless self.site_property['is_frozen'] then
			result = page_owner_login_ok?(page_property, passwd)
		end

		unless result then
			if admin_passwd_exist?(self.site_property) then
				result = admin_login_ok?(self.site_property, passwd)
			end
		end
		
		return result;
	end

	def page_owner_login_ok?(page_property, passwd)
		result = false
		if page_property['passwd'] and page_property['passwd'].length > 0 then 
			# page is frozen 
			if passwd and page_property['passwd'] == passwd.crypt(page_property['passwd']) then
				result = true;
			end
		else
			result =  true;
		end
		return result;
	end

	def admin_passwd_exist?(site_property)
		result = false
		if site_property['admin_passwd'] and site_property['admin_passwd'].length > 0 then
			result = true
		end
		result
	end

	def admin_login_ok?(site_property, admin_passwd)
		result = false
		if admin_passwd_exist?(site_property) then
			if  admin_passwd and admin_passwd.length > 0 and 
				(site_property['admin_passwd'] == admin_passwd.crypt(site_property['admin_passwd'])) then
				result = true
			end
		else
			result = true
		end
		return result
	end
end

class Exception
	def to_html
		result = self.inspect + "\n" 
		result += self.to_s + "\n"
		result += self.backtrace.join("\n") 
		result = CGI.escapeHTML(result)
		result = "<pre>#{result}</pre>"
	end
end

class Time
	def to_data_hash
		{
			:yyyy => self.strftime("%Y"),
			:yy => self.strftime("%y"),
			:mm => self.strftime("%m"),
			:dd => self.strftime("%d"),
			:HH => self.strftime("%H"),
			:MM => self.strftime("%M"),
			:SS => self.strftime("%S"),
			:AA => self.strftime("%A"),
			:aa => self.strftime("%a"),
			:BB => self.strftime("%B"),
			:bb => self.strftime("%b"),
		}
	end
end
