
module PropertyFile
	CR = "\015"
	LF = "\012"
	EOL = CR + LF

	def property(file_name, default_property = {}, &block)
		property = default_property

		unless File.file?(file_name) then
			File.open(file_name, "w") { |prop_file|
				prop_file.flock(File::LOCK_EX) 
				write_property(prop_file, default_property)
				prop_file.flush()
				prop_file.flock(File::LOCK_UN) 
			}
		end
			
		if File.file?(file_name) and File.readable?(file_name) and File.writable?(file_name) then
			File.open(file_name, "r+") { |prop_file|
				prop_file.flock(File::LOCK_EX) 
				property = read_property(prop_file, default_property)
				original_property = prop_file.dup
				if block_given? then
					yield(property)

					unless property == original_property then
						property['last_modified_time'] = Time.now()
						prop_file.rewind()
						prop_str = write_property('', property)
						prop_file.write(prop_str)
						prop_file.truncate(prop_str.length)
						prop_file.flush()
					end
				end
				prop_file.flock(File::LOCK_UN) 
			}
		end

		property
	end

	def read_property(input, default_property = {})
		property = default_property.dup
		input.each(){ |line|
			line.chomp!(EOL)
			line.chomp!()
			key, value = line.split('=', 2)
			case key
			when /_time$/ then
				value = str2time(value)
			when /_int$/ then
				value = value.to_i
			when /^is_/ then
				case value.downcase
				when 'true', 'yes' then
					value = true
				else
					value = false
				end
			when nil then
				value = ''
			end
			property[key] = value
		}
		return property;
	end

	def write_property(output, property)
		property.each(){ |key, value|
			case key
			when /_time$/ then
				value = time2str(value)
			when /_int$/ then
				value = value.to_s
			when /^is_/ then
				value = value ? 'true' : 'false'
			else
				value = '' unless value
				value = value.chomp(EOL)
				value = value.chomp()
			end
			output << key + "=" + value + "\n"
		}
		output
	end

	def str2time(str)
		time = nil
		if str and str =~ /(\d+)-(\d\d?)-(\d\d?) (\d\d?):(\d\d?):(\d\d?)/ then
			time = Time.local($1, $2, $3, $4, $5, $6)
		end
		return time;
	end

	def time2str(time)
		time.strftime("%Y-%m-%d %H:%M:%S")
	end

	module_function :property
	module_function :read_property
	module_function :write_property
	module_function :str2time
	module_function :time2str
end
