#!/usr/local/bin/ruby 

require 'uri'
require 'misen/style'
require 'misen/util'

require 'PageParser.rb'
require 'ToWikiHtml.rb'
require 'FilePageDataSource.rb'
require 'PStorePageDataSource.rb'

class KakiRequestHandler
	TEMPLATE_DIR = "./template"

	def initialize(page_data_source, app_url)
		@app_url = app_url
		@page_data_source = page_data_source
		@wiki2html = Wiki::ToWikiHtml.new(page_data_source, '')
		@wiki2html.link_base_uri = "#{@app_url}?p="
		@wiki2html.xhtml = true;

		self.site_property = @page_data_source.site_property()
		self.theme_dir = './theme'
		self.theme_base_url = './theme'
	end

	attr 'theme_dir', true
	attr 'theme_base_url', true

	def main(request_header, request_parameters)
		command = request_command(request_parameters)

		data = {}
		begin

			# template file
			tmpl_file = "#{command}.html"

			# command 
			case command
			when 'create' then
				create(request_header, request_parameters, data)
			when 'edit' then
				edit(request_header, request_parameters, data)
			when 'preview' then
				preview(request_header, request_parameters, data)
			when 'save' then
				save(request_header, request_parameters, data)
			when 'change_passwd' then
				change_passwd(request_header, request_parameters, data)
				tmpl_file = "save.html"
			when 'snapshot' then
				snapshot(request_header, request_parameters, data)
				tmpl_file = "save.html"
			when 'index' then
				index(request_header, request_parameters, data)
			when 'recent' then
				recent(request_header, request_parameters, data)
			when 'admin' then
				admin(request_header, request_parameters, data)
			else 
				# default command is 'view'
				command = 'view'
				tmpl_file = 'view.html'
				view(request_header, request_parameters, data)
			end

			# common data 
			data.update(site_common_data())

			# side menu
			request_lang_list = self.reuqest_language_list(request_header)
			sub_page_id = @page_data_source.suitable_page_id('SideMenu', request_lang_list)
			if sub_page_id then
				sub_page_name, sub_page_lang, = @page_data_source.parse_page_id(sub_page_id)
				sub_page_doc = @page_data_source.load_page_document(sub_page_name, sub_page_lang);
				if sub_page_doc then
					@wiki2html.output = ''
					@wiki2html.default_lang_list = request_lang_list
					sub_page_doc.accept(@wiki2html)
					data[:sub_page] = @wiki2html.output
				end
			end

		rescue Racc::ParseError
			data[:parse_error] = true
			unless command == 'preview' then
				if command != 'edit' then
					command = 'edit'
					retry
				else
					data[:exception] = exception2html($!) if $! 
				end
			end
		rescue 
			case $!.message
			when "NoPageID" then
				if command == 'edit' then
					command = 'create'
					data[:no_page_id] = true
					retry
				end
			when "NoSuchPage" then
				if command == 'view' then
					command = 'edit'
					retry
				else
					data[:exception] = exception2html($!) if $! 
				end
			when "PageAuthorizationFail" then
				if command == 'save' then
					command = 'preview'
					retry
				elsif command == 'snapshot' then
					command = 'edit'
					retry
				else
					data[:exception] = exception2html($!) if $! 
				end
			else
				data[:exception] = exception2html($!) if $! 
			end
		else 
			data[:exception] = exception2html($!) if $! 
		end

		# template file path
		tmpl_path, lang = template_path(request_lang_list + ['en'], tmpl_file)
		tmpl =  File.readlines(tmpl_path).to_s

		# http response header
		response_header = { }
		response_header['Content-Type'] = "text/html; charset=UTF-8"
		response_header['Language'] = lang if lang

=begin 
		if command == 'view' then
			last_modified_time = self.site_property()['last_modified_time']
			if data[:source_time] then 
				if data[:source_time] > last_modified_time then
					last_modified_time = data[:source_time]
				end
			end
			response_header['Last-Modified'] = CGI.rfc1123_date(last_modified_time) 
		end
=end

		# http response body
		body = Misen.expand_text(Misen::STYLE_SGML, tmpl, data)
		return [response_header, body];
	end

protected

	attr_reader "site_property"
	attr_writer "site_property"

	def view(request_header, request_parameters, data)
		request_page_id = request_page_id(request_parameters, 'FrontPage')
		page_name, lang, version = @page_data_source.parse_page_id(request_page_id)

		request_lang_list = self.reuqest_language_list(request_header)

		if lang then
			if @page_data_source.exist?(page_name, lang, version) then
				page_id = request_page_id
			end
		else
			page_id = @page_data_source.suitable_page_id(page_name, request_lang_list, version)
		end

		raise "NoSuchPage" unless page_id 

		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)
		wiki_doc = @page_data_source.load_page_document(page_name, page_lang, page_version)
		if wiki_doc then
			@wiki2html.output = ''
#			@wiki2html.default_lang_list = request_lang_list
			@wiki2html.default_lang_list = nil
			wiki_doc.accept(@wiki2html)
			data[:content] = @wiki2html.output 
			@page_data_source.page_property(page_name, page_lang, page_version){ |property|
				passwd = property['passwd']
				data[:is_freezed] = true if (passwd and passwd.length > 0)
				data[:counter] = { :count => property['count_int']}
				data[:source_time] = property['source_time']
				data[:page_title] = property['page_title']
				data[:modified_time] = time2hash(property['source_time'])
				property['count_int'] += 1
			}
			
			data[:page_id] = page_id
			data[:page_name] = page_name
			data[:page_lang] = page_lang
			data[:page_name_and_lang] = @page_data_source.make_page_id(page_name, page_lang)
			if page_version then
				data[:page_version] = { :version_int => page_version.to_i } 
			end

			data[:history] = page_history_data(page_name, page_lang)
			data[:lang_selector] = page_language_selector_data(page_name, page_lang)
		end
	end

	def create(request_header, request_parameters, data)
		data[:language_list] = make_language_list_data(supported_language_list())
	end

	def edit(request_header, request_parameters, data)
		page_id = self.request_page_id(request_parameters)
		raise 'NoPageID' unless page_id
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		unless page_lang then
			l = request_parameters['l'].to_s.untaint
			if l.length > 0 then
				page_lang = l
			else
				page_lang = self.site_property['default_language']
			end
		end

		page_id = @page_data_source.make_page_id(page_name, page_lang)
		page_title = page_id

		content_source = ''
		source_time_str = '0'
		if @page_data_source.exist?(page_name, page_lang) then
			content_source = @page_data_source.load_page_source(page_name, page_lang);
			content_source = CGI.escapeHTML(content_source);
			@page_data_source.page_property(page_name, page_lang) { |property|
				page_title = property['page_title']
				source_time_str = property['source_time'].to_i.to_s.reverse
				passwd = property['passwd']
				data[:is_freezed] = true if (passwd and passwd.length > 0)
			}
		else
			original_language = request_parameters['original_language'].to_s.untaint
			if original_language.length > 0 and @page_data_source.exist?(page_name, original_language) then
				content_source = @page_data_source.load_page_source(page_name, original_language);
				content_source = CGI.escapeHTML(content_source);
			end
		end

		data[:page_id] = page_id
		data[:page_name] = page_name
		data[:page_lang] = page_lang
		data[:page_title] = page_title
		data[:is_not_freezed] = (not data[:is_freezed])

		data[:content_source] = content_source
		salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
		data[:edit_key] = source_time_str.crypt(salt)
	end

	def preview(request_header, request_parameters, data)
		page_id = request_page_id(request_parameters)
		content_source = request_content_source(request_parameters)
		edit_key = request_edit_key(request_parameters)
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)
		unless page_lang then
			page_lang = self.site_property['default_language']
		end
		page_id = @page_data_source.make_page_id(page_name, page_lang)
		page_title = request_page_title(request_parameters)

		is_freezed = false
		if @page_data_source.exist?(page_name, page_lang) then
			@page_data_source.page_property(page_name, page_lang, page_version){ |property|
				if property['passwd'] and property['passwd'].length > 0 then
					is_freezed = true
				end
			}
		end

		data[:page_id] = page_id
		data[:page_name] = page_name
		data[:page_lang] = page_lang
		data[:page_title] = page_title

		data[:is_freezed] = is_freezed
		data[:content_source] = content_source
		data[:edit_key] = edit_key

		wiki_doc =  Wiki::PageParser.new().parse(content_source);
		if wiki_doc then
			@wiki2html.output = ''
#			@wiki2html.default_lang_list = [page_lang] + self.reuqest_language_list(request_header)
#			@wiki2html.default_lang_list = self.reuqest_language_list(request_header)
			@wiki2html.default_lang_list = nil
			wiki_doc.accept(@wiki2html)
			data[:content] = @wiki2html.output 
		else
			data[:content] = ''
		end
	end

	def save(request_header, request_parameters, data)
		page_id = request_page_id(request_parameters)
		content_source = request_content_source(request_parameters)
		edit_key = request_edit_key(request_parameters)
		passwd = request_passwd(request_parameters)
		page_title = request_page_title(request_parameters)

		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		property = {}
		if @page_data_source.exist?(page_name, page_lang) then
			property = @page_data_source.page_property(page_name, page_lang)
		end

		if self.page_login_ok?(property, passwd) or
			(admin_passwd_exist?(@page_data_source.site_property()) and 
			admin_login_ok?(@page_data_source.site_property(), passwd)) then

			# check editing conflict 
			source_time_str = property['source_time'].to_i.to_s.reverse if property 
			if edit_key != source_time_str.crypt(edit_key) then
				data[:save_conflict_msg] = true
			else
				@page_data_source.save_page_source(page_name, page_lang, nil, content_source)
				property = @page_data_source.page_property(page_name, page_lang){ |prop|
					prop['source_time'] = Time.now
					prop['page_title'] = page_title
				}
				data[:save_ok_msg] = { :page_id => page_id, :page_name => page_name }
			end
		else
			data[:page_save_auth_fail] = true
		end
		data[:page_id] = page_id
		data[:page_name] = page_name
		data[:page_lang] = page_lang
		data[:page_title] = property['page_title']

		if data[:page_save_auth_fail] then
			raise "PageAuthorizationFail"
		end
	end

	def change_passwd(request_header, request_parameters, data)
		passwd = request_parameters['passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_passwd1'].to_s.untaint
		page_id = request_page_id(request_parameters)
		page_name, page_lang, page_version = @page_data_source.parse_page_id(page_id)

		@page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(property, passwd) then
					# if non passwd or authorized
					if new_passwd1.length > 0 then
						if new_passwd1 == new_passwd2 then
							if property['passwd'].to_s.length > 0 then
								data[:passwd_change_ok] = { :page_id => page_id, :page_name => page_name }
							else
								data[:freeze_ok] = { :page_id => page_id, :page_name => page_name }
							end

							salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
							property['passwd'] = new_passwd1.crypt(salt)
						else
							data[:new_passwd_miss_match] = true
						end
					else
						property['passwd'] = nil
						data[:thaw_ok] = { :page_id => page_id, :page_name => page_name }
					end
			else
				data[:auth_fail_msg] = true
				raise "PageAuthorizationFail"
			end
			data[:page_id] = page_id
			data[:page_name] = page_name
			data[:page_lang] = page_lang
			data[:page_title] = property['page_title']
		}

	end

	def snapshot(request_header, request_parameters, data)
		page_id = request_page_id(request_parameters)
		passwd = request_passwd(request_parameters)
		page_name, page_lang, = @page_data_source.parse_page_id(page_id)

		@page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(page_id, passwd) then
				version_str = sprintf("%08d", property['version_int'])
				source = @page_data_source.load_page_source(page_name, page_lang)
				@page_data_source.save_page_source(page_name, page_lang, version_str, source)
				@page_data_source.page_property(page_name, page_lang, version_str){ |snap_prop|
					property.each(){ |key, value|
						snap_prop[key] = value
					}
				}
				property['version_int'] += 1
				data[:snapshot_ok_msg] = { :page_id => page_id, :page_name => page_name }
				data[:page_id] = page_id
				data[:page_name] = page_name
				data[:page_lang] = page_lang
				data[:page_title] = property['page_title']
			else
				data[:page_snapshot_auth_fail] = true
				raise "PageAuthorizationFail"
			end
		}

	end

	def index(request_header, request_parameters, data)
		pages = @page_data_source.page_list
		page_data_by_page_name = {}
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]
			page_name, lang, version = @page_data_source.parse_page_id(page_id)

			page_data = page_data_by_page_name[page_name]
			unless page_data then
				page_data = {
					:page_id => URI.escape(page_name),
					:app_url => @app_url,
					:page_name => page_name,
					:lang_list => [],
					'lang_hash' => {},
				}
				page_data_by_page_name[page_name] = page_data
			end

			lang_data = page_data['lang_hash'][lang]
			unless lang_data then
				lang_data = {
					:app_url => @app_url,
					:page_id => URI.escape("#{page_name}.#{lang}"),
					:page_lang => lang,
					:page_title => page_property['page_title'],
					:count => page_property['count_int'],
					:version_list => [],
				}
				lang_data.update(time2hash(page_property['source_time']))
				page_data['lang_hash'][lang] = lang_data
				page_data[:lang_list] << lang_data
			end

			if version then
				version_data = {
					:page_id => URI.escape(page_id),
					:app_url => @app_url,
					:version_int => page_property['version_int'],
				}
				version_data.update(time2hash(page_property['source_time']))
				lang_data[:version_list] << version_data
			end
		}
		page_list = page_data_by_page_name.values.sort(){ |a, b|
			a[:page_name].upcase <=> b[:page_name].upcase 
		}
		data[:index] ={ :page_list => page_list}
	end

	def recent(request_header, request_parameters, data)
		pages = @page_data_source.page_list
		pages.sort!(){ |a, b|
			b[1]['source_time'] <=> a[1]['source_time']
		}

		date_hash = {} # key: date string "%Y-%m-%d", value: hash
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]

			date_str = page_property['source_time'].strftime("%Y-%m-%d") 
			date_data = date_hash[date_str]
			unless date_data then
				date_data = time2hash(page_property['source_time'])
				date_data[:page_list] = []
				date_hash[date_str] = date_data
			end

			page_name, lang, version = @page_data_source.parse_page_id(page_id)
			page_data = time2hash(page_property['source_time'])
			page_data[:app_url] = @app_url
			page_data[:page_id] = page_id
			page_data[:page_name] = page_name
			page_data[:page_lang] = lang
			if version then
				page_data[:version] = { :version_int => page_property['version_int'] }
			else
				page_data[:counter] = { :count_int => page_property['count_int'] }
			end
			date_data[:page_list] << page_data
		}
		date_list = date_hash.to_a
		date_list.sort!()
		date_list.reverse!()
		date_list.collect!{ |date_pair|
			date_pair[1]
		}
		data[:recent] ={ :date_list => date_list }
	end

	def admin_save_property(request_header, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		site_name = request_parameters['site_name'].to_s.untaint
		admin_name = request_parameters['admin_name'].to_s.untaint
		admin_email = request_parameters['admin_email'].to_s.untaint
		theme = request_parameters['theme'].to_s.untaint
		sidebar_control = request_parameters['sidebar_control'].to_s.untaint
		default_language = request_parameters['default_language'].to_s.untaint
		default_language = 'en' if default_language.length <= 0
		languages = request_parameters['languages'].to_s.untaint


		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				property['site_name'] = site_name 
				property['admin_name'] = admin_name 
				property['admin_email'] = admin_email 
				property['theme'] = theme 
				property['sidebar_control'] = sidebar_control 
				property['default_language'] = default_language 
				property['languages'] = languages 
				self.site_property = property
				data[:site_property_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin_change_passwd(request_header, request_parameters, data)
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_admin_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_admin_passwd2'].to_s.untaint
		@page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				if new_passwd1.length > 0 then
					if new_passwd1 == new_passwd2 then
						salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
						property['admin_passwd'] = new_passwd1.crypt(salt)
					else
						data[:new_passwd_miss_match] = true
					end
				else
					property['admin_passwd'] = nil
				end
				data[:admin_passwd_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin(request_header, request_parameters, data)
		save_property = request_parameters['save_property'].to_s.untaint
		change_passwd = request_parameters['change_passwd'].to_s.untaint
		if save_property.length > 0 then
			admin_save_property(request_header, request_parameters, data)
		elsif change_passwd.length > 0 then
			admin_change_passwd(request_header, request_parameters, data)
		end

		theme_list = Dir.glob("#{self.theme_dir}/*/").sort().collect(){ |theme_dir|
			theme_name = File.basename(File.dirname(theme_dir))
			theme_data = { :theme_name => theme_name }
			if theme_name == self.site_property['theme'] then
				theme_data[:is_selected] = true
			end
			theme_data
		}
		data[:theme_list] = theme_list

		case self.site_property['sidebar_control']
		when "no_sidebar" then
			data[:no_sidebar_selected] = true 
		when "left_sidebar" then
			data[:left_sidebar_selected] = true 
		when "left_sidebar_fixed" then
			data[:left_sidebar_fixed_selected] = true 
		when "right_sidebar" then
			data[:right_sidebar_selected] = true 
		when "right_sidebar_fixed" then
			data[:right_sidebar_fixed_selected] = true 
		else
			data[:none_selected] = true 
		end

		data[:languages] = self.site_property['languages']


	end

	def request_command(request_parameters)
		result = request_parameters['c'].to_s
		result = 'view' if result.length <= 0 
		result.untaint
		return result;
	end

	def request_page_id(request_parameters, default_page_id = nil)
		page_id = request_parameters['p'].to_s
		page_id = default_page_id if page_id.length <= 0 
		page_id.untaint
		return page_id;
	end

	def request_content_source(request_parameters)
		content_source = request_parameters['content_source'].to_s
		content_source.gsub!(/\r\n/, "\n");
		content_source = "\n" if content_source.length <= 0 
		content_source.untaint
		return content_source;
	end

	def request_edit_key(request_parameters)
		edit_key = request_parameters['e'].to_s
		edit_key.untaint
		return edit_key;
	end

	def request_passwd(request_parameters)
		k = request_parameters['k'].to_s
		passwd = nil
		passwd = k if k.length > 0 
		passwd.untaint
		return passwd;
	end

	def request_page_title(request_parameters)
		result = request_parameters['page_title'].to_s
		if result.length <= 0 then
			result = request_page_id(request_parameters, 'FrontPage') 
			result, = @page_data_source.parse_page_id(result)
		end
		result.untaint
		return result;
	end

	def reuqest_language_list(request_header)
		result = parse_accept_language(request_header['accept-language'].to_s.untaint)
		result.push(self.site_property["default_language"].untaint)
		return result;
	end

	def parse_accept_language(accept_language_str)
		result = accept_language_str.split(',')
		result.collect!{ |lang_prop|
			lang, q = lang_prop.split(';q=')
			lang.strip!()
			if q then
				[q.to_f, lang]
			else
				[1.0, lang_prop]
			end
		}
		result.sort!()
		result.reverse!()
		result.collect!{ |lang| lang[1] }
		result
	end

	def site_common_data()
		result ={}
		result[:app_url] = @app_url

		result[:theme] = self.site_property['theme']
		result[:sidebar_control] = self.site_property['sidebar_control'] 
		result[:site_name] = self.site_property['site_name']

		result[:admin_name] = self.site_property['admin_name']
		result[:admin_email] = self.site_property['admin_email']
		if self.site_property['admin_email'] and self.site_property['admin_email'].length > 0 then
			result[:admin] = {
				:admin_name => self.site_property['admin_name'],
				:admin_email => self.site_property['admin_email'],
			}
		end

		result[:default_language] = self.site_property['default_language']
		result[:language_list] = make_language_list_data(supported_language_list())
		result[:theme_base_url] = self.theme_base_url

		return result
	end

	def page_history_data(page_name, page_lang)
		result = nil
		version_list = []
		@page_data_source.page_list().each(){ |page|
			id = page[0]
			property = page[1]
			n, l, v= @page_data_source.parse_page_id(id)
			if n == page_name and l == page_lang then
				version_name = nil
				if v then
					version_name = [true, property['version_int'].to_s]
				end
				version_list << {
					:app_url => @app_url,
					:page_id => id,
					:recent_name=> (version_name.nil?),
					:version_name => version_name,
					:modified_time => time2hash(property['source_time']),
				}
			end
		}
		result = { :version_list => version_list } if version_list.length > 1 
		return result;
	end

	def page_language_selector_data(page_name, page_lang)
		result = nil
		lang_list = []
		@page_data_source.page_list().each(){ |page|
			id = page[0]
			n, l, v= @page_data_source.parse_page_id(page[0])
			if n == page_name and v == nil then
				is_selected = false
				is_selected = true if page_lang == l 
				lang_list << {
					:page_name=> "#{n}.#{l}",
					:is_selected => is_selected,
					:lang => l,
				}
			end
		}
		if lang_list.length > 1 then
			result = { 
				:lang_list => lang_list,
				:app_url => @app_url, 
			} 
		end
		return result;
	end

	def make_language_list_data(language_list)
		result = []
		default_language = self.site_property['default_language'] 
		language_list.each(){ |lang|
			is_selected = false
			is_selected = true if lang == default_language 
			result << { 
				:lang => lang,
				:is_selected => is_selected,
			}
		}
		result
	end

	def supported_language_list()
		languages = self.site_property['languages']
		language_list = languages.split(',')
		language_list.each(){ |lang|
			lang.strip!()
		}
		language_list.delete_if(){ |lang|
			(/[A-Za-z][A-Za-z](-[A-Za-z][A-Za-z])?/ =~ lang) == nil
		}
		language_list
	end

	def template_path(request_lang_list, tmpl_file)
		result = nil
		lang = nil
		request_lang_list.each(){ |l|
			path = "#{TEMPLATE_DIR}/#{l}/#{tmpl_file}"
			if File.file?(path) and File.readable?(path) then
				result = path
				lang = l
				break;
			end
		}
		[result, lang]
	end

	def page_login_ok?(page_property, passwd)
		if page_property['passwd'] and page_property['passwd'].length > 0 then 
			# page is freezed 
			if passwd and page_property['passwd'] == passwd.crypt(page_property['passwd']) then
				return true;
			else
				return false;
			end
		else
			return true;
		end
	end

	def admin_passwd_exist?(site_property)
		result = false
		if site_property['admin_passwd'] and site_property['admin_passwd'].length > 0 then
			result = true
		end
		result
	end

	def admin_login_ok?(site_property, admin_passwd)
		result = false
		if admin_passwd_exist?(site_property) then
			if  admin_passwd and admin_passwd.length > 0 and 
				(site_property['admin_passwd'] == admin_passwd.crypt(site_property['admin_passwd'])) then
				result = true
			end
		else
			result = true
		end
		result
	end

	def exception2html(e)
		result = ''
		result += '<pre>' 
		result += CGI.escapeHTML($!.inspect) + "\n"
		result += CGI.escapeHTML($!.backtrace.join("\n") )
		result += '</pre>'
	end

	def time2hash(time)
		{
			:yyyy => time.strftime("%Y"),
			:yy => time.strftime("%y"),
			:mm => time.strftime("%m"),
			:dd => time.strftime("%d"),
			:HH => time.strftime("%H"),
			:MM => time.strftime("%M"),
			:SS => time.strftime("%S"),
			:AA => time.strftime("%A"),
			:aa => time.strftime("%a"),
			:BB => time.strftime("%B"),
			:bb => time.strftime("%b"),
		}
	end
end

