require 'uri.rb'

module Wiki

class Element
	def children_accept(visitor)
		result = []
#		if defined?(self.children) then
		if self.is_a?(Container) then
			self.children.each(){ |child|
				result << child.accept(visitor)
			}
		end
		return result;
	end

	def accept(visitor)
	end

end

module Container

	def initialize(child = nil)
		if child then 
			self.append_child(child);
		end
	end

	def children
		unless @children then
			@children = []
		end
		return @children;
	end

	def can_append?(child)
		child.is_a?(Element) 
	end

	def append_child(child)
		if child.is_a?(Array) then
			child.each(){ |c|
				self.append_child(c)
			}
		else
			unless child.is_a?(Element) 
				child = Text.new(child.to_s())
			end
			if self.can_append?(child) then
				self.children << child
			else
				puts "Illegal child : self = #{self.inspect} child = #{child.inspect}"
			end
		end
	end

	def remove_child(child)
		self.children.delete(child);
	end

	def ==(other)
		(self.class == other.class) and (self.children == other.children)
	end

	def to_s
		str = ''
		children.each(){ |child|
			str << child.to_s
		}
		str
	end
end

module InLineOnlyContainer 
	include Container
	def can_append?(child)
		child.is_a?(InLineElement) 
	end
end

class InLineElement < Element
end

class Emphasis < InLineElement
	include InLineOnlyContainer
	def accept(visitor)
		visitor.visit_emphasis(self);
	end
end

class Strong < InLineElement
	include InLineOnlyContainer
	def accept(visitor)
		visitor.visit_strong(self);
	end
end

class Strike < InLineElement
	include InLineOnlyContainer
	def accept(visitor)
		visitor.visit_strike(self);
	end
end

class Link < InLineElement
	def initialize(label)
		self.label = label;
	end

	attr :label, true

	def accept(visitor)
		visitor.visit_link(self);
	end

	def ==(other)
		(self.class == other.class) and (self.label == other.label)
	end

	def to_s
		label
	end
end

class URILink < Link
	def initialize(label, uri)
		super(label);
		self.uri = uri;
	end

	attr :uri, true

	def accept(visitor)
		visitor.visit_uri_link(self);
	end

	def ==(other)
		super(other) and (self.uri == other.uri)
	end
end

class InterLink < Link
	def initialize(label, prefix)
		super(label);
		self.prefix = prefix;
	end

	attr :prefix, true

	def accept(visitor)
		visitor.visit_inter_link(self);
	end

	def ==(other)
		(self.class == other.class) and 
		(self.label == other.label) and 
		(self.prefix == other.prefix)
	end

	def to_s
		self.prefix + ':' + self.label
	end
end

class Text < InLineElement
	def initialize(text = '')
		self.text = text
	end

	attr :text, false

	def accept(visitor)
		visitor.visit_text(self);
	end

	def ==(other)
		(self.class == other.class) and (self.text == other.text)
	end

	def to_s()
		text
	end

protected
	attr_writer :text
end

class ListElement < Element
	include Container
	def can_append?(child)
		child.is_a?(ListItem) 
	end
end

class UnorderedList < ListElement
	def accept(visitor)
		visitor.visit_unordered_list(self);
	end
end

class OrderedList < ListElement
	def accept(visitor)
		visitor.visit_ordered_list(self);
	end
end

class ListItem < Element
	include Container

	def accept(visitor)
		visitor.visit_list_item(self);
	end
end

class DefinitionList < Element
	include Container
	def can_append?(child)
		child.is_a?(DefinitionItem) 
	end

	def accept(visitor)
		visitor.visit_definition_list(self);
	end
end

class DefinitionItem < Element
	include InLineOnlyContainer

	def initialize(term, description)
		if term then 
			self.term = term
			self.append_child(description);
		end
	end
	attr :term, true

	def accept(visitor)
		visitor.visit_definition_item(self);
	end

	def ==(other)
		(self.class == other.class) and (self.term == other.term)
	end

	def to_s
		term
	end
end

class Table < Element
	include Container

	def can_append?(child)
		child.is_a?(TableRow) 
	end

	def accept(visitor)
		visitor.visit_table(self);
	end
end

class TableRow < Element
	include Container

	def can_append?(child)
		child.is_a?(TableColumn) 
	end

	def accept(visitor)
		visitor.visit_table_row(self);
	end
end

class TableColumn < Element
	include Container

	def can_append?(child)
		child.is_a?(InLineElement) 
	end

	def accept(visitor)
		visitor.visit_table_column(self);
	end
end

class Preformat < Element
	include Container

	def can_append?(child)
		child.is_a?(Text) 
	end

	def accept(visitor)
		visitor.visit_preformat(self);
	end
end

class Quotation < Element
	include Container

	def can_append?(child)
		child.is_a?(Text) 
	end

	def accept(visitor)
		visitor.visit_quotation(self);
	end
end


class Paragraph < Element
	include InLineOnlyContainer

	def accept(visitor)
		visitor.visit_paragraph(self);
	end
end

class HorizontalLine < Element
	def accept(visitor)
		visitor.visit_horizontal_line(self);
	end

	def ==(other)
		self.class == self.class
	end
end

class Headline < Element
	include InLineOnlyContainer
	def initialize(level, child = nil)
		super(child)
		@level = level
	end
	attr :level, false

	def accept(visitor)
		visitor.visit_headline(self);
	end
end

class Page < Element
	include Container
	def accept(visitor)
		visitor.visit_page(self);
	end
end

end

