module Wiki

require 'cgi'
require 'uri'
require 'base64'

require 'InterWikiNameAnalyzer.rb'


class ToHtml
	attr :xhtml, true;
	attr :output, true;
	attr :inter_wiki_name_document, true;

	def initialize(output = STDOUT)
		self.output = output;
		self.xhtml = false;
		self.inter_wiki_name_document = nil;
		@inter_wiki_name_table = nil
	end

	def visit_emphasis(element)
		output << "<em>"
		element.children_accept(self)
		output << "</em>"
	end

	def visit_strong(element)
		output << "<strong>"
		element.children_accept(self)
		output << "</strong>"
	end

	def visit_strike(element)
		output << "<s>"
		element.children_accept(self)
		output << "</s>"
	end

	def visit_link(element)
		output << %Q!<a href="##{URI.escape(element.label)}">#{CGI.escapeHTML(element.label)}</a>!
	end

	def visit_uri_link(element)
		if element.uri =~ /\.(png|gif|jpg|jpeg|)$/ then
			if self.xhtml then
				str = %Q!<img src="#{element.uri}" alt="#{CGI.escapeHTML(element.label)}" />!
			else
				str = %Q!<img src="#{element.uri}" alt="#{CGI.escapeHTML(element.label)}" >!
			end
		else
			if element.uri =~ /^#(.*)/ then
				# link to same page
				anchor_name = header2name($1)
				str = %Q!<a href="##{anchor_name}" >#{CGI.escapeHTML(element.label)}</a>!
			else
				str = %Q!<a href="#{element.uri}" >#{CGI.escapeHTML(element.label)}</a>!
			end
		end
		output << str
	end

	def visit_inter_link(element)
		prefix_uri = nil
		if self.inter_wiki_name_table[element.prefix] then
			prefix_uri = self.inter_wiki_name_table[element.prefix]
		end
		if prefix_uri then
			output << %Q!<a href="#{prefix_uri}#{URI::escape(element.label)}">#{CGI.escapeHTML(element.to_s)}</a>!
		else
			self.visit_link(element);
		end
	end

	def visit_text(element)
		output << CGI.escapeHTML(element.text)
	end

	def visit_unordered_list(element)
		output << "<ul>\n"
		element.children_accept(self)
		output << "</ul>\n"
	end

	def visit_ordered_list(element)
		output << "<ol>\n"
		element.children_accept(self)
		output << "</ol>\n"
	end

	def visit_list_item(element)
		output << "<li>"
		element.children_accept(self)
		output << "</li>\n"
	end

	def visit_definition_list(element);
		output << "<dl>\n"
		element.children_accept(self)
		output << "</dl>\n"
	end

	def visit_definition_item(element);
		output << "<dt>"
		anchor_name = header2name(element.term.to_s)
		if self.xhtml then
			output << %Q!<a id="#{anchor_name}" name="#{anchor_name}" href="##{anchor_name}" >#{CGI.escapeHTML(element.term)}</a>!
		else
			output << %Q!<a name="#{anchor_name}" >#{CGI.escapeHTML(element.term)}</a>!
		end
		output << "</dt><dd>"
		element.children_accept(self)
		output << "</dd>\n"
	end

	def visit_table(element)
		output << %Q!<table frame="box" rules="all" >\n<tbody>!
		element.children_accept(self)
		output << "</tbody>\n</table>\n"
	end

	def visit_table_row(element)
		output << "<tr>\n"
		element.children_accept(self)
		output << "</tr>\n"
	end

	def visit_table_column(element)
		output << "<td>"
		element.children_accept(self)
		output << "</td>\n"
	end

	def visit_preformat(element)
		output << "<pre>"
		element.children_accept(self)
		output << "</pre>\n"
	end

	def visit_quotation(element)
		output << "<blockquote>\n<p>\n"
		element.children_accept(self)
		output << "</p>\n</blockquote>\n"
	end

	def visit_paragraph(element)
		output << "<p>\n"
		element.children_accept(self)
		output << "</p>\n"
	end

	def visit_horizontal_line(element)
		if self.xhtml then
			output << "<hr />\n"
		else
			output << "<hr>\n"
		end
	end

	def visit_headline(element)
		level = element.level + 1
		anchor_name = header2name(element.to_s)
		output << "<h#{level}>"
		if self.xhtml then
			output << %Q!<a id="#{anchor_name}" name="#{anchor_name}" href="##{anchor_name}" ><span class="no_display" >&sect;</span></a>. !
		else
			output << %Q!<a name="#{anchor_name}" href="##{anchor_name}" ><span class="no_display" >&sect;</span></a>. !
		end
		element.children_accept(self)
		output << "</h#{level}>\n"
	end

	def visit_page(element)
		output << %Q!<div class="WikiPage" >\n!
		element.children_accept(self)
		output << "</div>\n"
	end

protected
	def header2name(str)
		'label' + encode64(str.strip).chomp().tr('+=', '__');
	end

	def inter_wiki_name_table
		unless @inter_wiki_name_table then
			@inter_wiki_name_table  = {}
			if self.inter_wiki_name_document then
				analyzer = InterWikiNameAnalyzer::new(@inter_wiki_name_table);
				self.inter_wiki_name_document.accept(analyzer);
			end
		end
		@inter_wiki_name_table
	end

end

end

if $0 == __FILE__ then
	require 'PageParser.rb'
	begin
		parser = Wiki::PageParser.new
		text = ARGF.readlines().join()
		doc =  parser.parse(text);
		formatter = Wiki::ToHtml.new
		doc.accept(formatter)
	rescue Racc::ParseError
		p $!
	end
end
