#!/usr/local/bin/ruby 

require 'uri'
require 'misen/style'
require 'misen/util'

require 'PageParser.rb'
require 'ToWikiHtml.rb'
require 'FilePageDataSource.rb'
require 'PStorePageDataSource.rb'
require 'TemplateDataSource.rb'

class KakiRequestHandler
	DEFAULT_THEME_DIR = './theme'
	DEFAULT_THEME_BASE_URL = './theme'

	def initialize(page_data_source, template_data_source)
		self.reset();
		self.page_data_source = page_data_source
		self.template_data_source = template_data_source
	end

	def reset
		@data = nil
		@exception = nil
		@site_property = nil
		@page_data_source = nil
		@template_data_source = nil
		@wiki2html = nil

		@request_variables = nil
		@request_parameters = nil

		@response_body = nil
		@response_header = nil

		@request_command = nil
		@request_max_number = nil
		@request_content_source = nil
		@request_edit_key = nil
		@request_language_list = nil
		@request_page_id = nil
		@request_page_title = nil
		@request_passwd = nil

		@script_path = nil
		@server_uri = nil
		@remote_address = nil
		@site_property = nil

		@theme_base_url = nil
		@theme_dir = nil
	end

	attr 'request_variables', true
	attr 'request_parameters', true

	attr 'theme_dir', true
	def theme_dir
		unless @theme_dir then
			@theme_dir = DEFAULT_THEME_DIR
		end
		@theme_dir
	end

	attr 'theme_base_url', true
	def theme_base_url
		unless @theme_base_url then
			@theme_base_url = DEFAULT_THEME_BASE_URL
		end
		@theme_base_url
	end

	def main()
		command = request_command()

		begin
			# command 
			case command
			when 'create' then
				create()
			when 'property' then
				property()
			when 'edit' then
				edit()
			when 'preview' then
				preview()
			when 'save' then
				save()
			when 'change_passwd' then
				change_passwd()
			when 'snapshot' then
				snapshot()
			when 'index' then
				index()
			when 'recent' then
				recent()
			when 'rss' then
				rss()
			when 'admin' then
				admin()
			when 'delete' then
				delete()
			when 'error' then
				error()
			else 
				# default command is 'view'
				command = 'view'
				view()
			end

		rescue Racc::ParseError
			data[:parse_error] = true
			unless command == 'preview' then
				if command != 'edit' then
					command = 'edit'
					retry
				else
					command = 'error'
					@exception = $!
					retry
				end
			end
		rescue NoPageIDError
			if command == 'edit' then
				command = 'create'
				data[:no_page_id] = true
				retry
			else
				command = 'error'
				@exception = $!
				retry
			end
		rescue NoSuchPageError
			case command
			when 'view', 'property' then
				command = 'edit'
				retry
			else
				command = 'error'
				@exception = $!
				retry
			end
		rescue PageAuthorizationError
			case command
			when 'save' then
				command = 'preview'
				retry
			when 'snapshot', 'change_passwd' then
				command = 'edit'
				retry
			else
				command = 'error'
				@exception = $!
				retry
			end
		rescue 
			if command != 'error' then
				command = 'error'
				@exception = $!
				retry
			else # if error in error page generating...
				self.response_header['Language'] = 'en';
				self.response_header['Content-Type'] = "text/plain"
				self.response_header['Charset'] = "UTF-8"
				bodyText = $!.inspect + "\n"
				bodyText += $!.to_s + "\n"
				bodyText += $!.backtrace.join("\n")  + "\n"
				self.response_body = bodyText;
			end
		end
	end

	def response_header
		unless @response_header then
			@response_header = { }
			@response_header['Content-Type'] = "text/html"
			@response_header['Charset'] = "UTF-8"
		end
		@response_header
	end

	def response_body
		unless @response_body then
			@response_body = ''
		end
		@response_body
	end

protected

	attr 'page_data_source', true
	attr 'template_data_source', true

	attr_writer 'response_header'
	attr_writer 'response_body'

	attr_writer 'site_property'
	def site_property
		unless @site_property then
			@site_property = self.page_data_source.site_property()
		end
		@site_property
	end

	def script_path
		unless @script_path then
			@script_path = self.request_variables['SCRIPT_NAME']
			if @script_path == nil || @script_path.length <= 0 then
				@script_path = '/';
			end
		end
		@script_path
	end

	def server_uri
		unless @server_uri then
			if request_variables['HTTPS'] then
				scheme = "https" 
			else
				scheme = "http" 
			end
			host, port = request_variables['HTTP_HOST'].to_s.split(":", 2);
			if port then
				port = ":#{port}" 
			else
				port = "" 
			end

			@server_uri = "#{scheme}://#{host}#{port}"
		end
		@server_uri
	end

	def remote_address
		unless @remote_address then
			@remote_address = request_variables['REMOTE_ADDR'].to_s
		end
		@remote_address
	end

	def ignore_site_access?()
		result = false;
		ignore_list = self.site_property['ignore_sites'].split(",")
		ignore_list.each(){ |ignore_ip|
			begin
				if Regexp.new(ignore_ip) =~ self.remote_address then
					result = true;
				end
			rescue RegexpError
			end
		}
		result
	end

	def data
		unless @data then
			@data = {}
		end
		@data
	end

	def wiki2html
		unless @wiki2html then
			@wiki2html = Wiki::ToWikiHtml.new(self.page_data_source, '')
			@wiki2html.xhtml = true;
		end
		@wiki2html
	end

	def view()
		request_page_id = request_page_id('FrontPage')
		page_name, lang, version = self.page_data_source.parse_page_id(request_page_id)

		if lang then
			if self.page_data_source.exist?(page_name, lang, version) then
				page_id = request_page_id
			end
		else
			page_id = self.page_data_source.suitable_page_id(page_name, self.request_language_list, version)
		end

		raise NoSuchPageError.new unless page_id 

		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)
		wiki_doc = self.page_data_source.load_page_document(page_name, page_lang, page_version)
		if wiki_doc then
			self.wiki2html.output = ''
			self.wiki2html.link_base_uri = "#{self.script_path}?p="
#			self.wiki2html.default_lang_list = self.request_language_list
			self.wiki2html.default_lang_list = nil
			wiki_doc.accept(self.wiki2html)
			data[:content] = self.wiki2html.output 
			self.page_data_source.page_property(page_name, page_lang, page_version){ |property|
				passwd = property['passwd']
				data[:is_frozen] = self.site_property['is_frozen']
				data[:is_frozen] = true if (passwd and passwd.length > 0)
				data[:is_not_frozen] = (not data[:is_frozen])
				data[:counter] = { :count => property['count_int']}
				data[:source_time] = property['source_time']
				data[:page_title] = CGI.escapeHTML(property['page_title'])
				data[:modified_time] = property['source_time'].to_data_hash
				append_referer(property)
				unless self.ignore_site_access?() then
					property['count_int'] += 1
				end
			}
			
			data[:page_id_uri] = URI.escape(page_id)
			data[:page_id] = CGI.escapeHTML(page_id)
			data[:page_name] = CGI.escapeHTML(page_name)
			data[:page_lang] = CGI.escapeHTML(page_lang)
			data[:page_name_and_lang] = URI.escape(self.page_data_source.make_page_id(page_name, page_lang))
			if page_version then
				data[:page_version] = { :version_int => page_version.to_i } 
			end

			data[:history] = page_history_data(page_name, page_lang)
			data[:lang_selector] = page_language_selector_data(page_name, page_lang)
		end

=begin 
		last_modified_time = self.site_property()['last_modified_time']
		if data[:source_time] and data[:source_time] > last_modified_time then
			last_modified_time = data[:source_time]
		end
		response_header['Last-Modified'] = CGI.rfc1123_date(last_modified_time) 
=end

		make_response_from_template('view.html')
	end

	def create()
		make_response_from_template('create.html')
	end

	def property()
		request_page_id = request_page_id('FrontPage')
		page_name, lang, version = self.page_data_source.parse_page_id(request_page_id)

		if lang and self.page_data_source.exist?(page_name, lang, version) then
			page_id = request_page_id
		else
			page_id = self.page_data_source.suitable_page_id(page_name, self.request_language_list, version)
		end

		raise NoSuchPageError.new unless page_id 

		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)
		property = self.page_data_source.page_property(page_name, page_lang, page_version)
		passwd = property['passwd']
		data[:is_frozen] = self.site_property['is_frozen']
		data[:is_frozen] = true if (passwd and passwd.length > 0)
		data[:is_not_frozen] = (not data[:is_frozen])
		data[:counter] = { :count => property['count_int']}
		data[:source_time] = property['source_time']
		data[:page_title] = CGI.escapeHTML(property['page_title'])
		data[:modified_time] = property['source_time'].to_data_hash
		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_name_and_lang] = URI.escape(self.page_data_source.make_page_id(page_name, page_lang))
		if page_version then
			data[:page_version] = { :version_int => page_version.to_i } 
		end

		data[:history] = page_history_data(page_name, page_lang)
		data[:lang_selector] = page_language_selector_data(page_name, page_lang)
		data[:referers] = page_referer_data(property)

		make_response_from_template('property.html')
	end


	def edit()
		page_id = self.request_page_id()
		raise NoPageIDError.new unless page_id
		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)

		unless page_lang then
			l = request_parameters['l'].to_s.untaint
			if l.length > 0 then
				page_lang = l
			else
				page_lang = self.site_property['default_language']
			end
		end

		page_id = self.page_data_source.make_page_id(page_name, page_lang)
		page_title = page_name

		content_source = ''
		source_time_str = '0'
		if self.page_data_source.exist?(page_name, page_lang) then
			content_source = self.page_data_source.load_page_source(page_name, page_lang);
			self.page_data_source.page_property(page_name, page_lang) { |property|
				page_title = property['page_title']
				source_time_str = property['source_time'].to_i.to_s.reverse
				passwd = property['passwd']
				data[:is_frozen] = self.site_property['is_frozen']
				data[:is_frozen] = true if (passwd and passwd.length > 0)
			}
		else
			original_language = request_parameters['original_language'].to_s.untaint
			if original_language.length > 0 and self.page_data_source.exist?(page_name, original_language) then
				content_source = self.page_data_source.load_page_source(page_name, original_language);
			end
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(page_title)
		data[:is_not_frozen] = (not data[:is_frozen])

		data[:content_source] = CGI.escapeHTML(content_source)
		salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
		data[:edit_key] = source_time_str.crypt(salt)

		make_response_from_template('edit.html')
	end

	def preview()
		page_id = request_page_id()
		content_source = request_content_source()
		edit_key = request_edit_key()
		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)
		unless page_lang then
			page_lang = self.site_property['default_language']
		end
		page_id = self.page_data_source.make_page_id(page_name, page_lang)
		page_title = request_page_title()

		is_frozen = self.site_property['is_frozen']
		if self.page_data_source.exist?(page_name, page_lang) then
			self.page_data_source.page_property(page_name, page_lang, page_version){ |property|
				if property['passwd'] and property['passwd'].length > 0 then
					is_frozen = true
				end
			}
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(page_title)

		data[:is_frozen] = is_frozen
		data[:is_not_frozen] = (not is_frozen)
		data[:content_source] = CGI.escapeHTML(content_source)
		data[:edit_key] = edit_key

		wiki_doc =  Wiki::PageParser.new().parse(content_source);
		if wiki_doc then
			self.wiki2html.output = ''
			self.wiki2html.link_base_uri = "#{self.script_path}?p="
#			self.wiki2html.default_lang_list = [page_lang] + self.request_language_list
#			self.wiki2html.default_lang_list = self.request_language_list
			self.wiki2html.default_lang_list = nil
			wiki_doc.accept(self.wiki2html)
			data[:content] = self.wiki2html.output 
		else
			data[:content] = ''
		end

		make_response_from_template('preview.html')
	end

	def save()
		page_id = request_page_id()
		content_source = request_content_source()
		edit_key = request_edit_key()
		passwd = request_passwd()
		page_title = request_page_title()

		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)

		property = {}
		if self.page_data_source.exist?(page_name, page_lang) then
			property = self.page_data_source.page_property(page_name, page_lang)
		end

		if self.page_login_ok?(property, passwd) then
			# check editing conflict 
			source_time_str = property['source_time'].to_i.to_s.reverse if property 
			if edit_key != source_time_str.crypt(edit_key) then
				data[:save_conflict_msg] = true
			else
				self.page_data_source.save_page_source(page_name, page_lang, nil, content_source)
				property = self.page_data_source.page_property(page_name, page_lang){ |prop|
					prop['source_time'] = Time.now
					prop['page_title'] = page_title
				}
				self.site_property = self.page_data_source.site_property(){ |site_prop|
					site_prop['last_modified_time'] = Time.now
				}
				
				data[:save_ok_msg] = { :page_id => page_id, :page_name => page_name }
			end
		else
			data[:page_save_auth_fail] = true
		end

		data[:page_id_uri] = URI.escape(page_id)
		data[:page_id] = CGI.escapeHTML(page_id)
		data[:page_name] = CGI.escapeHTML(page_name)
		data[:page_lang] = CGI.escapeHTML(page_lang)
		data[:page_title] = CGI.escapeHTML(property['page_title'])

		if data[:page_save_auth_fail] then
			raise PageAuthorizationError.new
		end

		make_response_from_template('save.html')
	end

	def change_passwd()
		passwd = request_parameters['passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_passwd1'].to_s.untaint
		page_id = request_page_id(request_parameters)
		page_name, page_lang, page_version = self.page_data_source.parse_page_id(page_id)

		self.page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(property, passwd) then
				# if non passwd or authorized
				if new_passwd1.length > 0 then
					if new_passwd1 == new_passwd2 then
						if property['passwd'].to_s.length > 0 then
							data[:passwd_change_ok] = true
						else
							data[:freeze_ok] = { :page_id => CGI.escapeHTML(page_id) }
						end

						salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
						property['passwd'] = new_passwd1.crypt(salt)
					else
						data[:new_passwd_miss_match] = true
					end
				else
					property['passwd'] = nil
					data[:thaw_ok] = { :page_id => CGI.escapeHTML(page_id) }
				end
			else
				data[:auth_fail] = true
			end
			data[:page_id_uri] = URI.escape(page_id)
			data[:page_id] = CGI.escapeHTML(page_id)
			data[:page_name] = CGI.escapeHTML(page_name)
			data[:page_lang] = CGI.escapeHTML(page_lang)
			data[:page_title] = CGI.escapeHTML(property['page_title'])
		}

		if data[:auth_fail] then
			raise PageAuthorizationError.new
		end

		make_response_from_template('save.html')
	end

	def snapshot()
		page_id = request_page_id()
		passwd = request_passwd()
		page_name, page_lang, = self.page_data_source.parse_page_id(page_id)

		self.page_data_source.page_property(page_name, page_lang){ |property|
			if self.page_login_ok?(property, passwd) then
				version_str = sprintf("%08d", property['version_int'])
				source = self.page_data_source.load_page_source(page_name, page_lang)
				self.page_data_source.save_page_source(page_name, page_lang, version_str, source)
				self.page_data_source.page_property(page_name, page_lang, version_str){ |snap_prop|
					property.each(){ |key, value|
						snap_prop[key] = value
					}
					snap_prop['passwd'] = nil
				}
				property['version_int'] += 1
				data[:snapshot_ok_msg] = { :page_id => CGI.escapeHTML(page_id) }
				data[:page_id_uri] = URI.escape(page_id)
				data[:page_id] = CGI.escapeHTML(page_id)
				data[:page_name] = CGI.escapeHTML(page_name)
				data[:page_lang] = CGI.escapeHTML(page_lang)
				data[:page_title] = CGI.escapeHTML(property['page_title'])
			else
				data[:auth_fail] = true
			end
		}

		if data[:auth_fail] then
			raise PageAuthorizationError.new
		end

		make_response_from_template('save.html')
	end

	def index()
		pages = self.page_data_source.page_list
		page_data_by_page_name = {}
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]
			page_name, lang, version = self.page_data_source.parse_page_id(page_id)

			page_data = page_data_by_page_name[page_name]
			unless page_data then
				page_data = {
					:page_id_uri => URI.escape(page_name),
					:page_id => CGI.escapeHTML(page_name),
					:script_path => self.script_path,
					:page_name => CGI.escapeHTML(page_name),
					:lang_list => [],
					'lang_hash' => {},
				}
				page_data_by_page_name[page_name] = page_data
			end

			lang_data = page_data['lang_hash'][lang]
			unless lang_data then
				is_frozen = self.site_property['is_frozen']
				is_frozen = true if (page_property['passwd'].to_s.length > 0)
				lang_data = {
					:script_path => self.script_path,
					:page_id_uri => URI.escape("#{page_name}.#{lang}"),
					:page_id => CGI.escapeHTML("#{page_name}.#{lang}"),
					:page_lang => CGI.escapeHTML(lang),
					:page_title => CGI.escapeHTML(page_property['page_title']),
					:is_frozen => is_frozen, 
					:is_not_frozen => (not is_frozen), 
					:count => page_property['count_int'],
					:version_list => [],
				}
				lang_data.update(page_property['source_time'].to_data_hash)
				page_data['lang_hash'][lang] = lang_data
				page_data[:lang_list] << lang_data
			end

			if version then
				version_data = {
					:page_id_uri => URI.escape(page_id),
					:page_id => CGI.escapeHTML(page_id),
					:script_path => self.script_path,
					:version_int => page_property['version_int'],
					:count => page_property['count_int'],
				}
				version_data.update(page_property['source_time'].to_data_hash)
				lang_data[:version_list] << version_data
			end
		}
		page_list = page_data_by_page_name.values.sort(){ |a, b|
			a[:page_name].upcase <=> b[:page_name].upcase 
		}
		data[:index] ={ :page_list => page_list}

		make_response_from_template('index.html')
	end

	def recent()
#		raise "TestException"
		pages = self.page_data_source.page_list
		pages.sort!(){ |a, b|
			b[1]['source_time'] <=> a[1]['source_time']
		}

		date_hash = {} # key: date string "%Y-%m-%d", value: hash
		pages.each{ |page_info|
			page_id = page_info[0]
			page_property = page_info[1]

			date_str = page_property['source_time'].strftime("%Y-%m-%d") 
			date_data = date_hash[date_str]
			unless date_data then
				date_data = page_property['source_time'].to_data_hash
				date_data[:page_list] = []
				date_hash[date_str] = date_data
			end

			page_name, lang, version = self.page_data_source.parse_page_id(page_id)
			page_data = page_property['source_time'].to_data_hash
			page_data[:script_path] = self.script_path
			page_data[:page_id_uri] = URI.escape(page_id)
			page_data[:page_id] = CGI.escapeHTML(page_id)
			page_data[:page_name] = CGI.escapeHTML(page_name)
			page_data[:page_lang] = CGI.escapeHTML(lang)
			page_data[:is_frozen] = self.site_property['is_frozen']
			page_data[:is_frozen] = true if (page_property['passwd'].to_s.length > 0)
			page_data[:is_not_frozen] = (not page_data[:is_frozen])
			if version then
				page_data[:version] = { :version_int => page_property['version_int'] }
			end
			page_data[:count] = page_property['count_int']
			date_data[:page_list] << page_data
		}
		date_list = date_hash.to_a
		date_list.sort!()
		date_list.reverse!()
		date_list.collect!{ |date_pair|
			date_pair[1]
		}
		if self.request_max_number > 0 then
			date_list = date_list[0...self.request_max_number];
		end
		data[:recent] ={ :date_list => date_list }
		make_response_from_template('recent.html')
	end

	def rss()
		pages = self.page_data_source.page_list
		pages.sort!(){ |a, b|
			b[1]['source_time'] <=> a[1]['source_time']
		}

		last_modified_time = self.site_property()['last_modified_time']
		page_list = pages.collect(){ |page_info|
			page_data = nil
			page_id = page_info[0]
			page_property = page_info[1]

			page_name, lang, version = self.page_data_source.parse_page_id(page_id)
			unless version then
				page_modified_time = page_property['source_time']
				if page_modified_time > last_modified_time then
					last_modified_time = page_modified_time 
				end
				page_data = page_modified_time.to_data_hash
				page_data[:page_id_uri] = URI.escape(page_id)
				page_data[:page_id] = CGI.escapeHTML(page_id)
				page_data[:page_name] = CGI.escapeHTML(page_name)
				page_data[:page_title] = CGI.escapeHTML(page_property['page_title'])
				page_data[:page_lang] = CGI.escapeHTML(lang)
				page_data[:script_path] = self.script_path
				page_data[:server_uri] = self.server_uri
			end
			page_data
		}
		page_list.delete(nil)
		if self.request_max_number > 0 then
			page_list = page_list[0...self.request_max_number];
		end
		data[:page_list] = page_list
		data[:server_uri] = self.server_uri

		data.update(last_modified_time.to_data_hash)

		make_response_from_template('kaki.rss')
		self.response_header['Content-Type'] = "text/xml"
		self.response_header['Last-Modified'] = CGI.rfc1123_date(last_modified_time) 
	end

	def admin_save_property()
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		site_name = request_parameters['site_name'].to_s.untaint
		admin_name = request_parameters['admin_name'].to_s.untaint
		admin_email = request_parameters['admin_email'].to_s.untaint
		theme = request_parameters['theme'].to_s.untaint
		sidebar_control = request_parameters['sidebar_control'].to_s.untaint
		default_language = request_parameters['default_language'].to_s.untaint
		default_language = 'en' if default_language.length <= 0
		languages = request_parameters['languages'].to_s.untaint
		ignore_sites = request_parameters['ignore_sites'].to_s.untaint

		self.page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				property['site_name'] = site_name 
				property['admin_name'] = admin_name 
				property['admin_email'] = admin_email 
				property['theme'] = theme 
				property['sidebar_control'] = sidebar_control 
				property['default_language'] = default_language 
				property['languages'] = languages 
				property['ignore_sites'] = ignore_sites 
				self.site_property = property
				data[:site_property_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}

	end

	def admin_change_passwd()
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		new_passwd1 = request_parameters['new_admin_passwd1'].to_s.untaint
		new_passwd2 = request_parameters['new_admin_passwd2'].to_s.untaint
		self.page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				# if non passwd or authorized
				if new_passwd1.length > 0 then
					if new_passwd1 == new_passwd2 then
						salt = [[rand(4096)].pack("v")].pack("m").tr("+", ".")
						property['admin_passwd'] = new_passwd1.crypt(salt)
					else
						data[:new_passwd_miss_match] = true
					end
				else
					property['admin_passwd'] = nil
				end
				self.site_property = property
				data[:admin_passwd_change_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin_site_freeze()
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		site_freeze = request_parameters['site_freeze'].to_s.untaint
		self.page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				case site_freeze.downcase 
				when 'true', 'yes' then
					property['is_frozen'] = true
					data[:site_freeze_ok] = true
				when 'false', 'no' then
					property['is_frozen'] = false
					data[:site_thaw_ok] = true
				end
				self.site_property = property
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def admin()
		save_property = request_parameters['save_property'].to_s.untaint
		change_passwd = request_parameters['change_passwd'].to_s.untaint
		site_freeze = request_parameters['site_freeze'].to_s.untaint
		if save_property.length > 0 then
			admin_save_property()
		elsif change_passwd.length > 0 then
			admin_change_passwd()
		elsif site_freeze.length > 0 then
			admin_site_freeze()
		end

		theme_list = Dir.glob("#{self.theme_dir}/*/*.css").sort().collect(){ |theme_css_file|
			theme_name = File.basename(File.dirname(theme_css_file))
			theme_data = { :theme_name => theme_name }
			if theme_name == self.site_property['theme'] then
				theme_data[:is_selected] = true
			end
			theme_data
		}
		data[:theme_list] = theme_list

		case self.site_property['sidebar_control']
		when "no_sidebar" then
			data[:no_sidebar_selected] = true 
		when "left_sidebar" then
			data[:left_sidebar_selected] = true 
		when "left_sidebar_fixed" then
			data[:left_sidebar_fixed_selected] = true 
		when "right_sidebar" then
			data[:right_sidebar_selected] = true 
		when "right_sidebar_fixed" then
			data[:right_sidebar_fixed_selected] = true 
		else
			data[:none_selected] = true 
		end

		data[:languages] = self.site_property['languages']
		data[:ignore_sites] = self.site_property['ignore_sites']
		make_response_from_template('admin.html')
	end

	def delete_page()
		admin_passwd = request_parameters['admin_passwd'].to_s.untaint
		self.page_data_source.site_property(){ |property|
			if admin_login_ok?(property, admin_passwd) then
				page_list = request_parameters['page_list']
				page_list.each(){ |page_id|
					page_id.untaint
					name, lang, version = self.page_data_source.parse_page_id(page_id)
					if self.page_data_source.exist?(name, lang, version) then
						self.page_data_source.delete_page(name, lang, version)
					end
				}
				data[:delete_ok] = true
			else
				data[:admin_auth_fail_msg] = true
			end
		}
	end

	def delete()
		delete_page = request_parameters['delete_page'].to_s.untaint
		if delete_page.length > 0 then
			delete_page()
		end

		page_list = []
		self.page_data_source.page_list.each(){ |page_info|
			page_id = page_info[0] 
			property = page_info[1]
			page_data = {
				:script_path => self.script_path, 
				:page_id_uri => URI.escape(page_id), 
				:page_id => CGI.escapeHTML(page_id), 
				:page_title => CGI.escapeHTML(property['page_title']),
			}
			page_list << page_data
		}
		data[:page_list] = page_list
		make_response_from_template('delete.html')
	end

	def error()
		if @exception then
			if @exception.is_a?(Exception) then
				data[:exception] = @exception.to_data_hash 
			else
				data[:exception] = { :message => @exception.to_s }
			end
		end
		make_response_from_template('error.html')
	end


	def request_command()
		unless @request_command then
			@request_command = request_parameters['c'].to_s
			@request_command = 'view' if @request_command.length <= 0 
			@request_command.untaint
		end
		@request_command 
	end

	def request_max_number()
		unless @request_max_number then
			@request_max_number = request_parameters['n'].to_s
			@request_max_number = 0 if @request_max_number.length <= 0 
			@request_max_number = @request_max_number.to_i
			@request_max_number.untaint
		end
		@request_max_number
	end


	def request_page_id(default_page_id = nil)
		unless @request_page_id then
			@request_page_id = request_parameters['p'].to_s
			@request_page_id = default_page_id if @request_page_id.length <= 0 
			@request_page_id.untaint
		end
		@request_page_id
	end


	def request_content_source()
		unless @request_content_source then
			@request_content_source = request_parameters['content_source'].to_s
			@request_content_source.gsub!(/\r\n/, "\n");
			@request_content_source = "\n" if @request_content_source.length <= 0 
			@request_content_source.untaint
		end
		@request_content_source
	end

	def request_edit_key()
		unless @request_edit_key then
			@request_edit_key = request_parameters['e'].to_s
			@request_edit_key.untaint
		end
		@request_edit_key
	end

	def request_passwd()
		unless @request_passwd then
			k = request_parameters['k'].to_s
			@request_passwd = k if k.length > 0 
			@request_passwd.untaint
		end
		@request_passwd;
	end

	def request_page_title()
		unless @request_page_title then
			@request_page_title = request_parameters['page_title'].to_s
			if @request_page_title.length <= 0 then
				id = request_page_id('FrontPage') 
				@request_page_title, = self.page_data_source.parse_page_id(id)
			end
			@request_page_title.untaint
		end
		@request_page_title
	end

	def request_language_list()
		unless @request_language_list then
#			accept_language = request_variables['HTTP_ACCEPT_LANGUAGE'].to_s.untaint
			if request_variables['HTTP_ACCEPT_LANGUAGE'] then
				accept_language = request_variables['HTTP_ACCEPT_LANGUAGE'].dup.untaint
			else
				accept_language = '';
			end
			@request_language_list = accept_language.split(',')
			@request_language_list.collect!{ |lang_prop|
				lang, q = lang_prop.split(';q=')
				lang.strip!()
				if q then
					[q.to_f, lang]
				else
					[1.0, lang_prop]
				end
			}
			@request_language_list.sort!()
			@request_language_list.reverse!()
			@request_language_list.collect!{ |lang| lang[1] }
			@request_language_list.push(self.site_property["default_language"].untaint)
			@request_language_list.push('en')
		end
		@request_language_list
	end


	def site_common_data()
		result ={}
		result[:script_path] = self.script_path

		result[:theme] = self.site_property['theme']
		result[:sidebar_control] = self.site_property['sidebar_control'] 
		result[:site_name] = self.site_property['site_name']

		result[:admin_name] = self.site_property['admin_name']
		result[:admin_email] = self.site_property['admin_email']
		if self.site_property['admin_email'] and self.site_property['admin_email'].length > 0 then
			result[:admin] = {
				:admin_name => self.site_property['admin_name'],
				:admin_email => self.site_property['admin_email'],
			}
		end

		result[:default_language] = self.site_property['default_language']
		result[:language_list] = self.language_list_data
		result[:theme_base_url] = self.theme_base_url
		result[:site_is_frozen] = self.site_property['is_frozen']
		result[:site_is_not_frozen] = (not result[:site_is_frozen])

		return result
	end


	def page_history_data(page_name, page_lang)
		result = nil
		version_list = []
		self.page_data_source.page_list().each(){ |page|
			id = page[0]
			property = page[1]
			n, l, v= self.page_data_source.parse_page_id(id)
			if n == page_name and l == page_lang then
				is_snapshot = nil
				if v then
					is_snapshot = { :version_int => property['version_int'] }
				end
				version_list << {
					:script_path => self.script_path,
					:page_id_uri => URI.escape(id),
					:is_recent=> (is_snapshot.nil?),
					:is_snapshot => is_snapshot,
					:modified_time => property['source_time'].to_data_hash,
				}
			end
		}
		result = { :version_list => version_list } if version_list.length > 1 
		return result;
	end

	def page_language_selector_data(page_name, page_lang)
		result = nil
		lang_list = []
		self.page_data_source.page_list().each(){ |page|
			id = page[0]
			prop = page[1]
			n, l, v= self.page_data_source.parse_page_id(page[0])
			if n == page_name and v == nil then
				is_selected = false
				is_selected = true if page_lang == l 
				lang_list << {
					:page_id_uri => URI.escape("#{n}.#{l}"),
					:page_id => CGI.escapeHTML("#{n}.#{l}"),
					:page_title => CGI.escapeHTML(prop['page_title']),
					:is_selected => is_selected,
					:lang => l,
				}
			end
		}
		if lang_list.length > 1 then
			result = { 
				:lang_list => lang_list,
				:script_path => self.script_path, 
			} 
		end
		result
	end

	def page_referer_data(page_property)
		result = nil
		referers = page_property['referers'].to_s.split(", ")
		referer_list = referers.collect(){ |referer|
			{ 
				:referer_uri => referer,
				:referer => CGI.escapeHTML(referer),
			}
		}

		if referer_list.length > 0 then
			result = { :referer_list => referer_list }
		end
		result
	end

	def language_list_data()
		result = []
		languages = self.site_property['languages']
		supported_language_list = languages.split(',')
		supported_language_list.each(){ |lang|
			lang.strip!()
		}
		supported_language_list.delete_if(){ |lang|
			(/[A-Za-z][A-Za-z](-[A-Za-z][A-Za-z])?/ =~ lang) == nil
		}
		default_language = self.site_property['default_language'] 
		supported_language_list.each(){ |lang|
			is_selected = false
			is_selected = true if lang == default_language 
			result << { 
				:lang => lang,
				:is_selected => is_selected,
			}
		}
		result
	end

	def make_response_from_template(tmpl_file)
		data[:sub_page] = self.sub_page()
		data.update(self.site_common_data())
		data[:server_time] = Time.now.to_data_hash

		tmpl, tmpl_lang = self.template_data_source.template(tmpl_file, self.request_language_list)

		self.response_header['Language'] = tmpl_lang if tmpl_lang
		self.response_body = Misen.expand_text(Misen::STYLE_SGML, tmpl, self.data)
	end

	def append_referer(page_property)
		referer = request_variables['HTTP_REFERER']
		
		if referer then
			script_uri = server_uri + script_path
			script_uri = URI.escape(URI.unescape(script_uri));

			referer_uri = URI.parse(URI.escape(URI.unescape(referer.to_s)))
			referer_uri.fragment = nil
			referer_uri_str = referer_uri.to_s
			referer_uri.query = nil
			referer_script_uri = referer_uri.to_s

			if script_uri != referer_script_uri then
				referers = page_property['referers'].to_s.split(", ")
				referers.delete(referer_uri_str)
				referers.unshift(referer_uri_str)
				referers = referers[0,10]
				page_property['referers'] = referers.join(", ")
			end
		end
	end

	def page_login_ok?(page_property, passwd)
		result = false

		unless self.site_property['is_frozen'] then
			result = page_owner_login_ok?(page_property, passwd)
		end

		unless result then
			if admin_passwd_exist?(self.site_property) then
				result = admin_login_ok?(self.site_property, passwd)
			end
		end
		
		return result;
	end

	def page_owner_login_ok?(page_property, passwd)
		result = false
		if page_property['passwd'] and page_property['passwd'].length > 0 then 
			# page is frozen 
			if passwd and page_property['passwd'] == passwd.crypt(page_property['passwd']) then
				result = true;
			end
		else
			result =  true;
		end
		return result;
	end

	def admin_passwd_exist?(site_property)
		result = false
		if site_property['admin_passwd'] and site_property['admin_passwd'].length > 0 then
			result = true
		end
		result
	end

	def admin_login_ok?(site_property, admin_passwd)
		result = false
		if admin_passwd_exist?(site_property) then
			if  admin_passwd and admin_passwd.length > 0 and 
				(site_property['admin_passwd'] == admin_passwd.crypt(site_property['admin_passwd'])) then
				result = true
			end
		else
			result = true
		end
		return result
	end

	def sub_page()
		result = nil;
		sub_page_id = self.page_data_source.suitable_page_id('SideMenu', self.request_language_list)
		if sub_page_id then
			sub_page_name, sub_page_lang, = self.page_data_source.parse_page_id(sub_page_id)
			sub_page_doc = self.page_data_source.load_page_document(sub_page_name, sub_page_lang);
			if sub_page_doc then
				self.wiki2html.output = ''
				self.wiki2html.link_base_uri = "#{self.script_path}?p="
				self.wiki2html.default_lang_list = self.request_language_list
				sub_page_doc.accept(self.wiki2html)
				result = self.wiki2html.output
			end
		end
		return result;
	end
end


class Exception
	def to_html
		result = self.inspect + "\n" 
		result += self.to_s + "\n"
		result += self.backtrace.join("\n") 
		result = CGI.escapeHTML(result)
	end

	def to_data_hash
		inspect = CGI.escapeHTML(self.inspect)
		message = CGI.escapeHTML(self.to_s)
		backtrace = CGI.escapeHTML(self.backtrace.join("\n").to_s)
		{
			:inspect => inspect,
			:message => message,
			:backtrace => backtrace,
		}
	end
end

class Time
	def to_data_hash
		offset = self.utc_offset
		flag = offset < 0 ? "-" : "+"
		offset = offset.abs()
		minutes = offset / 60 
		hours = minutes / 60
		minutes = minutes % 60
		zone = sprintf("%s%d:%02d", flag, hours, minutes)
		{
			:yyyy => self.strftime("%Y"),
			:yy => self.strftime("%y"),
			:mm => self.strftime("%m"),
			:dd => self.strftime("%d"),
			:HH => self.strftime("%H"),
			:MM => self.strftime("%M"),
			:SS => self.strftime("%S"),
			:AA => self.strftime("%A"),
			:aa => self.strftime("%a"),
			:BB => self.strftime("%B"),
			:bb => self.strftime("%b"),
			:ZZ => zone,
		}
	end
end

class NoPageIDError < StandardError; end
class NoSuchPageError < StandardError; end
class PageAuthorizationError < StandardError; end

