/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.ImageButton;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.Toolbar;
import net.hizlab.kagetaka.awt.event.StateEvent;
import net.hizlab.kagetaka.awt.event.StateListener;
import net.hizlab.kagetaka.bookmarks.Bookmark;
import net.hizlab.kagetaka.bookmarks.BookmarkManager;
import net.hizlab.kagetaka.theme.Theme;
import net.hizlab.kagetaka.util.URLHistory;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Font;
import java.awt.Image;
import java.awt.Insets;
import java.awt.MenuItem;
import java.awt.Toolkit;
import java.awt.Panel;
import java.awt.Polygon;
import java.awt.PopupMenu;
import java.awt.SystemColor;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.Vector;

/**
 * ӥ塼ѥġС󶡤ޤ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
class ViewerToolbar
	extends Toolbar
	implements ContextListener
{
	private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resource";
	
	static final String BACK     = "back"    ;
	static final String FORWARD  = "forward" ;
	static final String RELOAD   = "reload"  ;
	static final String STOP     = "stop"    ;
	static final String GO       = "go"      ;
	static final String HOME     = "home"    ;
	static final String BOOKMARK = "bookmark";
	static final String PTF      = "ptf"     ;
	static final String ADDRESS  = "address" ;
	
	static final int TYPE_NAVI = 1;
	static final int TYPE_LINK = 2;
	
	private static final int BUTTON_BACK       =  0;
	private static final int BUTTON_FORWARD    =  1;
	private static final int BUTTON_RELOAD     =  2;
	private static final int BUTTON_STOP       =  3;
	private static final int BUTTON_GO         =  4;
	private static final int BUTTON_HOME       =  5;
	private static final int BUTTON_BOOKMARK   =  6;
	private static final int BUTTONS_NUM       =  7;
	
	private static final int TEXTFIELD_ADDRESS =  0;
	private static final int TEXTFIELD_SEARCH  =  1;
	private static final int TEXTFIELDS_NUM    =  2;
	
	private static final int PTF_BOOKMARK      = -1;
	
	private static final int MENU_SIZE         = 20;
	
	private static final int LOAD_BEFORE  = 0;
	private static final int LOAD_LOADING = 1;
	private static final int LOAD_LOADED  = 2;
	
	private HawkViewer    viewer;
	private int           type;
	private ViewerOption  option;
	private Theme         theme;
	
	private Font  toolbarFont;
	private Color toolbarFore;
	private Color toolbarBack;
	private Image toolbarImage;
	
	private ImageButton[] buttons;
	private TextField  [] textFields;
	private Panel         ptPanel;
	private GridBagLayout ptPanelLayout;
	private PopupMenu     popupMenu;
	private boolean       showPopupMenu;
	private Bookmark[]    bookmarks;
	private Image         fileImage;
	private Image         folderImage;
	
	private ViewerContext activeContext;
	
	/**
	 * ӥ塼ѥġССޤ
	 * 
	 * @param     viewer ӥ塼
	 */
	ViewerToolbar(HawkViewer viewer, int type)
	{
		this.viewer = viewer;
		this.type   = type;
		this.option = viewer.getOption();
		this.theme  = option.getThemeManager().getTheme();
		
		Toolkit tk = getToolkit();
		
		toolbarFont  = theme.getFont (Theme.BROWSER_TOOLBAR_FONT      );
		toolbarFore  = theme.getColor(Theme.BROWSER_TOOLBAR_FOREGROUND);
		toolbarBack  = theme.getColor(Theme.BROWSER_TOOLBAR_BACKGROUND);
		toolbarImage = theme.getImage(Theme.BROWSER_TOOLBAR_BACKGROUND_IMAGE, tk);
		
		buttons    = new ImageButton[BUTTONS_NUM   ];
		textFields = new TextField  [TEXTFIELDS_NUM];
		
		int index = 0;
		String key = null;
		String[] list = null;
		
		switch (type) {
		case TYPE_NAVI: list = option.getShowNavibarButtons(); break;
		case TYPE_LINK: list = option.getShowLinkbarButtons(); break;
		}
		
		for (int i = 0; i < list.length; i++) {
			if (list[i].compareTo(BACK    ) == 0) { index = BUTTON_BACK    ; key = Theme.BROWSER_TOOLBAR_BACK    ; } else
			if (list[i].compareTo(FORWARD ) == 0) { index = BUTTON_FORWARD ; key = Theme.BROWSER_TOOLBAR_FORWARD ; } else
			if (list[i].compareTo(RELOAD  ) == 0) { index = BUTTON_RELOAD  ; key = Theme.BROWSER_TOOLBAR_RELOAD  ; } else
			if (list[i].compareTo(STOP    ) == 0) { index = BUTTON_STOP    ; key = Theme.BROWSER_TOOLBAR_STOP    ; } else
			if (list[i].compareTo(GO      ) == 0) { index = BUTTON_GO      ; key = Theme.BROWSER_TOOLBAR_GO      ; } else
			if (list[i].compareTo(HOME    ) == 0) { index = BUTTON_HOME    ; key = Theme.BROWSER_TOOLBAR_HOME    ; } else
			if (list[i].compareTo(BOOKMARK) == 0) { index = BUTTON_BOOKMARK; key = Theme.BROWSER_TOOLBAR_BOOKMARK; } else
			if (list[i].compareTo(PTF     ) == 0) {
				if (ptPanel != null)
					continue;
				
				ptPanel       = new Panel        ();
				ptPanelLayout = new GridBagLayout();
				ptPanel.setLayout(ptPanelLayout);
				add(ptPanel);
				continue;
			} else
			if (list[i].compareTo(ADDRESS ) == 0) {
				if (textFields[TEXTFIELD_ADDRESS] != null)
					continue;
				
				textFields[TEXTFIELD_ADDRESS] = new TextField();
				textFields[TEXTFIELD_ADDRESS].setFont      (theme.getFont (Theme.BROWSER_TOOLBAR_ADDRESSBAR_FONT      ));
				textFields[TEXTFIELD_ADDRESS].setForeground(theme.getColor(Theme.BROWSER_TOOLBAR_ADDRESSBAR_FOREGROUND));
				textFields[TEXTFIELD_ADDRESS].setBackground(theme.getColor(Theme.BROWSER_TOOLBAR_ADDRESSBAR_BACKGROUND));
				textFields[TEXTFIELD_ADDRESS].addKeyListener  (new AddressBarKeyListener  ());
				textFields[TEXTFIELD_ADDRESS].addFocusListener(new AddressBarFocusListener());
				add(textFields[TEXTFIELD_ADDRESS], MAXIMUM);
				continue;
			} else
				continue;
			
			if (buttons[index] != null)
				continue;
			
			if ((buttons[index] = theme.getImageButton(key, tk)) == null)
				continue;
			
			if (toolbarFont  != null) buttons[index].setFont      (toolbarFont );
			if (toolbarFore  != null) buttons[index].setForeground(toolbarFore );
			if (toolbarBack  != null) buttons[index].setBackground(toolbarBack );
			if (toolbarImage != null) buttons[index].setBackImage (toolbarImage);
			buttons[index].addMouseListener(new ButtonMouseListener(index));
			
			if (theme.getBoolean(key + Theme.EXT_LABEL))
				buttons[index].setLabel(Resource.getMessage(RESOURCE, "toolbar." + list[i] + ".label", null));
			
			add(buttons[index]);
		}
		
		if (buttons[BUTTON_BACK    ] != null)
			buttons[BUTTON_BACK    ].addStateListener(new ButtonStateListener(BUTTON_BACK    ));
		if (buttons[BUTTON_FORWARD ] != null)
			buttons[BUTTON_FORWARD ].addStateListener(new ButtonStateListener(BUTTON_FORWARD ));
		if (buttons[BUTTON_BOOKMARK] != null)
			buttons[BUTTON_BOOKMARK].addStateListener(new ButtonStateListener(BUTTON_BOOKMARK));
		
		setForeground(SystemColor.controlText);
		setBackground(SystemColor.control    );
		
		if (toolbarFont  != null) setFont      (toolbarFont );
		if (toolbarFore  != null) setForeground(toolbarFore );
		if (toolbarBack  != null) setBackground(toolbarBack );
		if (toolbarImage != null) setBackImage (toolbarImage);
		
		bookmarkChanged();
	}
	
	/**
	 * ɥ쥹ޤ
	 * 
	 * @return    ɥ쥹
	 */
	public String getAddress()
	{
		if (textFields[TEXTFIELD_ADDRESS] == null)
			return null;
		
		return textFields[TEXTFIELD_ADDRESS].getText();
	}
	
	/**
	 * ɥ쥹ꤷޤ
	 * 
	 * @param     address ɥ쥹
	 */
	public void setAddress(String address)
	{
		if (textFields[TEXTFIELD_ADDRESS] != null)
			textFields[TEXTFIELD_ADDRESS].setText(address);
	}
	
//### Bookmark
	/** ֥åޡɲä줿 */
	void addToBookmark(Bookmark parent, Bookmark bookmark)
	{
		bookmarkChanged();
	}
	
	/** ֥åޡѹ줿 */
	void bookmarkChanged()
	{
		if (ptPanel == null)
			return;
		
		BookmarkManager bm = option.getBookmarkManager();
		synchronized (bm) {
			Bookmark root = bm.getPersonalToolbarFolder();
			Vector   v    = null;
			if (root != null)
				v = root.getBookmarks();
			
			if (v == null || v.size() == 0) {
				if (this.bookmarks != null) {
					ptPanel.removeAll();
					this.bookmarks = null;
				}
				return;
			}
			
			Bookmark[] bookmarks = new Bookmark[v.size()];
			v.copyInto(bookmarks);
			
			// ֥åޡѹʤɤå
			if (this.bookmarks != null && this.bookmarks.length == bookmarks.length) {
				boolean same = true;
				for (int i = 0; i < bookmarks.length; i++) {
					if (this.bookmarks[i] != bookmarks[i]) {
						same = false;
						break;
					}
				}
				if (same)
					return;
			}
			
			ImageButton      ib       = null;
			Insets           insets   = new Insets(0, 0, 0, 0);
			BookmarkListener listener = null;
			ptPanel.removeAll();
			for (int i = 0; i < bookmarks.length; i++) {
				switch (bookmarks[i].getType()) {
				case Bookmark.FOLDER:
					if (folderImage == null)
						folderImage = theme.getImage(Theme.BOOKMARK_FOLDER, getToolkit());
					ib = new ImageButton(bookmarks[i].getText(), folderImage, null);
					break;
				case Bookmark.BOOKMARK:
					if (fileImage   == null)
						fileImage   = theme.getImage(Theme.BOOKMARK_FILE  , getToolkit());
					ib = new ImageButton(bookmarks[i].getText(), fileImage  , null);
					break;
				default:
					continue;
				}
				
				ib.setFont      (toolbarFont );
				ib.setForeground(toolbarFore );
				ib.setBackground(toolbarBack );
				ib.setBackImage (toolbarImage);
				listener = new BookmarkListener(bookmarks[i]);
				ib.addMouseListener(listener);
				ib.addStateListener(listener);
				LayoutUtils.addGridBag(ptPanel, ib, ptPanelLayout, i, 0, 1, 1, 0, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
			}
			
			validate();
			this.bookmarks = bookmarks;
		}
	}
	
//### ContextListener
	/** ƥ֤ʥƥȤѹ줿 */
	public void activeChanged(ViewerContext context)
	{
		this.activeContext = context;
		
		historyChanged(context);
		
		if (context.currentURL == null)
			setLoaded(LOAD_BEFORE );
		else if (context.loading)
			setLoaded(LOAD_LOADING);
		else
			setLoaded(LOAD_LOADED );
	}
	
	/** ɥȤΥɤϤ줿 */
	public void documentLoading(ViewerContext context)
	{
		if (activeContext != context)
			return;
		
		setLoaded(LOAD_LOADING);
	}
	
	/** ɥȤɤ줿 */
	public void documentLoaded(ViewerContext context)
	{
		if (activeContext != context)
			return;
		
		setLoaded(LOAD_LOADED);
	}
	
	/** ɥ쥹ѹ줿 */
	public void addressChanged(ViewerContext context, String address)
	{
		if (activeContext != context)
			return;
		
		if (textFields[TEXTFIELD_ADDRESS] != null)
			textFields[TEXTFIELD_ADDRESS].setText(address);
	}
	
	/** ȥ뤬ѹ줿 */
	public void titleChanged(ViewerContext context, String title)
	{
	}
	
	/** ơѹ줿 */
	public void statusChanged(ViewerContext context, String status)
	{
	}
	
	/** ѹ줿 */
	public void historyChanged(ViewerContext context)
	{
		if (activeContext != context)
			return;
		
		// URL ꥹȤ
		URLHistory topURL     = null;
		URLHistory currentURL = null;
		URLHistory endURL     = null;
		
		synchronized (context.historyLock) {
			topURL     = context.topURL;
			currentURL = context.currentURL;
			endURL     = context.endURL;
		}
		
		if (buttons[BUTTON_BACK    ] != null) buttons[BUTTON_BACK    ].setEnabled(topURL != currentURL);
		if (buttons[BUTTON_FORWARD ] != null) buttons[BUTTON_FORWARD ].setEnabled(endURL != currentURL);
	}
	
//### private
	/** ɤ߹ޤƤ뤫ɤ */
	private void setLoaded(int load)
	{
		if (buttons[BUTTON_RELOAD] != null) buttons[BUTTON_RELOAD].setEnabled(load != LOAD_BEFORE );
		if (buttons[BUTTON_STOP  ] != null) buttons[BUTTON_STOP  ].setEnabled(load == LOAD_LOADING);
	}
	
	/** ݥåץåץ˥塼ɽ */
	private void showPopupMenu(ViewerContext context, Component comp, int index, Bookmark bookmark)
	{
		synchronized (popupMenu) {
			if (showPopupMenu)
				return;
			
			popupMenu.removeAll();
			
			switch (index) {
			case BUTTON_BACK    :
			case BUTTON_FORWARD :
				{
					// URL ꥹȤ
					URLHistory url;
					URLHistory[] allURL = new URLHistory[MENU_SIZE];
					int num = 0;
					
					synchronized (context.historyLock) {
						url = context.currentURL;
						if (url == null)
							return;
						
						do {
							switch (index) {
							case BUTTON_BACK    : url = url.getPreviousURLHistory(); break;
							case BUTTON_FORWARD : url = url.getNextURLHistory    (); break;
							}
							
							if (url == null)
								break;
							
							allURL[num++] = url;
							
						} while (num < MENU_SIZE);
					}
					
					MenuItem mi;
					String   title;
					for (int i = 0; i < num; i++) {
						url   = allURL[i];
						title = url.getTitle();
						if (title == null)
							title = "";
						
						mi = new MenuItem(title);
						mi.addActionListener(new MoveActionListener(context, url));
						
						popupMenu.add(mi);
					}
				}
				break;
			case BUTTON_BOOKMARK:
				viewer.createBookmarkMenu(popupMenu, true);
				break;
			case PTF_BOOKMARK:
				viewer.createBookmarkMenu(popupMenu, bookmark.getBookmarks());
				break;
			}
			
			showPopupMenu = true;
			popupMenu.show(comp, 0, comp.getSize().height);
		}
	}
	
	/** ϤƤ륢ɥ쥹˰ư */
	private void goAddress()
	{
		if (textFields[TEXTFIELD_ADDRESS] == null)
			return;
		
		String path = textFields[TEXTFIELD_ADDRESS].getText();
		if (path != null && path.length() > 0) {
			String surl = option.getBookmarkManager().resolveShortcut(path);
			if (surl != null)
				viewer.open(surl, null);
			else
				viewer.open(path, null);
		}
	}
	
//### ButtonMouseListener
	private class ButtonMouseListener
		extends MouseAdapter
	{
		private int index;
		
		/** 󥹥󥹤 */
		private ButtonMouseListener(int index)
		{
			this.index = index;
		}
		
		/** å줿 */
		public void mouseClicked(MouseEvent e)
		{
			if (!e.getComponent().isEnabled())
				return;
			
			// å
			if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK) {
				// ۥåȥݥå
				switch (index) {
				case BUTTON_BACK    :
				case BUTTON_FORWARD :
					Polygon spot = ((ImageButton)e.getComponent()).getHotspotArea();
					if (spot != null && spot.contains(e.getX(), e.getY()))
						return;
				}
				
				boolean isShift = ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0);
				
				switch (index) {
				case BUTTON_BACK    : viewer.moveHistory(HawkViewer.CURRENT_TAB,      -1); break;
				case BUTTON_FORWARD : viewer.moveHistory(HawkViewer.CURRENT_TAB,       1); break;
				case BUTTON_RELOAD  : viewer.reload     (HawkViewer.CURRENT_TAB, isShift); break;
				case BUTTON_STOP    : viewer.stop       (HawkViewer.CURRENT_TAB         ); break;
				case BUTTON_GO      : goAddress()                                        ; break;
				case BUTTON_HOME    : viewer.moveHome   (HawkViewer.CURRENT_TAB         ); break;
				case BUTTON_BOOKMARK: break;
				}
			}
		}
	}
	
//### ButtonStateListener
	private class ButtonStateListener
		implements StateListener
	{
		private int index;
		private int downState;
		
		/** 󥹥󥹤 */
		private ButtonStateListener(int index)
		{
			this.index = index;
			
			if (ViewerToolbar.this.popupMenu == null) {
				ViewerToolbar.this.popupMenu = new PopupMenu();
				ViewerToolbar.this.add(popupMenu);
			}
			
			switch (index) {
			case BUTTON_BACK    :
			case BUTTON_FORWARD :
				downState = ImageButton.HOTSPOT_DOWN;
				break;
			case BUTTON_BOOKMARK:
				downState = ImageButton.DOWN;
				break;
			}
		}
		
		/** ֤ѹ줿 */
		public void stateChanged(StateEvent e)
		{
			if (((ImageButton)e.getComponent()).getState() == downState)
				showPopupMenu(activeContext, e.getComponent(), index, null);
			else
				showPopupMenu = false;
		}
	}
	
//### MoveActionListener
	/** ưѤΥƥꥹ */
	private class MoveActionListener
		implements ActionListener
	{
		private ViewerContext context;
		private URLHistory    urlHistory;
		
		/** ˥塼ꥹ  */
		private MoveActionListener(ViewerContext context, URLHistory urlHistory)
		{
			this.context    = context;
			this.urlHistory = urlHistory;
		}
		
		/** ϥɥ */
		public void actionPerformed(ActionEvent e)
		{
			context.load(null, null, urlHistory, false, false);
		}
	}
	
//### AddressBarKeyListener
	private class AddressBarKeyListener
		extends KeyAdapter
	{
		/** 󥹥󥹤 */
		private AddressBarKeyListener()
		{
		}
		
		/** Υ줿Ȥ */
		public void keyPressed(KeyEvent e)
		{
			// keyTyped Ǥ ENTER 򥭥åǤʤĶ
			if (e.getKeyCode() == KeyEvent.VK_ENTER) {
				e.consume();
				goAddress();
			}
		}
	}
	
//### AddressBarFocusListener
	/** եϥɥ */
	private class AddressBarFocusListener
		implements FocusListener
	{
		/** 󥹥󥹤 */
		private AddressBarFocusListener()
		{
		}
		
		/** եμ */
		public void focusGained(FocusEvent e)
		{
			TextField c = (TextField)e.getSource();
			c.setCaretPosition(c.getText().length());
			c.selectAll();
		}
		
		/** եΥ */
		public void focusLost(FocusEvent e)
		{
			TextField c = (TextField)e.getSource();
			c.select(0, 0);
		}
	}
	
//### BookmarkListener
	private class BookmarkListener
		extends MouseAdapter
		implements StateListener
	{
		private Bookmark bookmark;
		private boolean  folder;
		
		/** 󥹥󥹤 */
		private BookmarkListener(Bookmark bookmark)
		{
			this.bookmark = bookmark;
			this.folder   = (bookmark.getType() == Bookmark.FOLDER);
			
			if (ViewerToolbar.this.popupMenu == null) {
				ViewerToolbar.this.popupMenu = new PopupMenu();
				ViewerToolbar.this.add(popupMenu);
			}
		}
		
		/** å줿 */
		public void mouseClicked(MouseEvent e)
		{
			if (folder || !e.getComponent().isEnabled())
				return;
			
			// å
			if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK) {
				URL url = bookmark.getURL();
				if (url != null)
					viewer.open(url, null);
			}
		}
		
		/** ֤ѹ줿 */
		public void stateChanged(StateEvent e)
		{
			if (!folder)
				return;
			
			if (((ImageButton)e.getComponent()).getState() == ImageButton.DOWN)
				showPopupMenu(activeContext, e.getComponent(), PTF_BOOKMARK, bookmark);
			else
				showPopupMenu = false;
		}
	}
}
