/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Reporter;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.download.Download;
import net.hizlab.kagetaka.history.History;
import net.hizlab.kagetaka.history.HistoryManager;
import net.hizlab.kagetaka.java2.WheelListener;
import net.hizlab.kagetaka.java2.WheelWrapper;
import net.hizlab.kagetaka.rendering.Document;
import net.hizlab.kagetaka.rendering.EngineListener;
import net.hizlab.kagetaka.rendering.HawkContext;
import net.hizlab.kagetaka.rendering.HawkEngine;
import net.hizlab.kagetaka.rendering.HawkPanel;
import net.hizlab.kagetaka.rendering.ItemMap;
import net.hizlab.kagetaka.rendering.Option;
import net.hizlab.kagetaka.rendering.PanelListener;
import net.hizlab.kagetaka.rendering.PostData;
import net.hizlab.kagetaka.token.FormItem;
import net.hizlab.kagetaka.util.URLHistory;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Menu;
import java.awt.MenuItem;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.ImageProducer;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * ӥ塼ѤΡ륳ƥȤμǤ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
class ViewerContext
	implements HawkContext,PanelListener,EngineListener
{
	private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resource";
	
	/** @serial ݡȥץȥ */
	private static Hashtable supportProtocol = new Hashtable();
	{
		StringTokenizer st = new StringTokenizer(Resource.getMessage("config.support.protocol", null), ",; ");
		while (st.hasMoreTokens())
			supportProtocol.put(st.nextToken(), "");
	}
	
	private static int tagIndex;
	
	private ViewerOption    option;
	private Reporter        reporter;
	private HistoryManager  historyManager;
	private Connector       connector;
	
	private HawkEngine      hawkEngine;
	private HawkPanel       hawkPanel;
	private ContextListener contextListener;
	private int             loadIndex;
	
	private HawkViewer      hawkViewer;
	
	private boolean         forceLoad;
	
	// ƥȤξ
	private Object  activeLock = new Object();
	private boolean active;
	private String  tag;
	private String  statusText;
	private long    tabMode;
	
	// ɽΥɥȤξ
	private ItemMap itemMap;
	
	// 
	History        currentHistory;
	Object         historyLock = new Object();
	URLHistory     topURL;
	URLHistory     currentURL;
	URLHistory     endURL;
	boolean        loading;
	
	//  URL
	private URL            newURL;          // ɤ߹߻ URL
	private URLHistory     moveURL;         // ư (,)  URL
	
	/** 󥹥󥹤 */
	ViewerContext(ViewerOption option)
	{
		this.option         = option;
		this.reporter       = option.getViewerConsole ().getReporter();
		this.historyManager = option.getHistoryManager();
		this.tag            = String.valueOf(++tagIndex);
		
		hawkEngine = new HawkEngine(this);
		hawkPanel  = new HawkPanel (this);
		connector  = new Connector (this);
		
		hawkEngine.setEngineListener(this);
		hawkPanel .setPanelListener (this);
		
		// ޥۥ
		// 1.4.2  Sun μǤưʤ
//		WheelWrapper wheelWrapper = WheelWrapper.getInstance();
//		if (wheelWrapper != null) {
//			wheelWrapper.addWheelListener(hawkPanel, new WheelListener()
//				{
//					/** ޥۥ뤬ž줿 */
//					public void spin(int type, int value, int modifiers)
//					{
//						hawkPanel.scroll(HawkPanel.SCROLL_HORIZONTAL, type, value);
//					}
//				}
//			);
//		}
		
		// ˥塼ѹ
		initContextMenu();
	}
	
	/** ѥͥΥƥȥ˥塼 */
	private void initContextMenu()
	{
		String         label   = null;
		Vector         items   = null;
		Menu           menu    = null;
		MenuItem       mi      = null;
		ActionListener listener= null;
		
		// ǳ
		items    = hawkPanel.getLinkMenuItems();
		listener = new NewActionListener();
		mi = new MenuItem(getMessage("contextmenu.opennewwindow", null));
		mi.setActionCommand (NewActionListener.WINDOW);
		mi.addActionListener(listener);
		items.insertElementAt(mi, 0);
		mi = new MenuItem(getMessage("contextmenu.opennewtab", null));
		mi.setActionCommand (NewActionListener.TAB   );
		mi.addActionListener(listener);
		items.insertElementAt(mi, 1);
		items.insertElementAt(new MenuItem("-"), 2);
		
		// ץꥱǳ˥塼
		ViewerOption.Application[] apps = option.getApplications();
		label = getMessage("contextmenu.openapp", null);
		if (apps != null && apps.length > 0) {
			for (int i = 0; i < 2; i++) {
				switch (i) {
				case 0:
					listener = new ApplicationActionListener(ApplicationActionListener.MAIN, option);
					items    = hawkPanel.getMainMenuItems();
					break;
				case 1:
					listener = new ApplicationActionListener(ApplicationActionListener.LINK, option);
					items    = hawkPanel.getLinkMenuItems();
					break;
				}
				
				menu = new Menu(label);
				for (int j = 0; j < apps.length; j++)
					createApplicationMenu(apps[j], menu, listener);
				
				items.addElement(new MenuItem("-"));
				items.addElement(menu);
			}
		}
	}
	
	/** ץꥱ˥塼 */
	private void createApplicationMenu(ViewerOption.Application app,
	                                   Menu parent,
	                                   ActionListener listener)
	{
		MenuItem mi = new MenuItem(app.getName());
		
		mi.setActionCommand (app.getId());
		mi.addActionListener(listener   );
		
		parent.add(mi);
	}
	
	/** ѥ֤ͥ */
	HawkPanel attachViewer(HawkViewer hawkViewer)
	{
		this.hawkViewer = hawkViewer;
		connector.setOwner(hawkViewer);
		return hawkPanel;
	}
	
	/** ѥͥΥХ֤ */
	Component getCanvas()
	{
		return hawkPanel.getCanvas();
	}
	
	/** ƥȥꥹʤɲ */
	void addContextListener(ContextListener l)
	{
		this.contextListener = ContextEventMulticaster.add(contextListener, l);
	}
	
	/** ƥȥꥹʤ */
	void removeContextListener(ContextListener l)
	{
		this.contextListener = ContextEventMulticaster.remove(contextListener, l);
	}
	
	/** ƥ֤ */
	void setActived(boolean active)
	{
		synchronized (activeLock) {
			this.active = active;
			
			if (active) {
				// ӥ塼ɽ򹹿
				URLHistory url = currentURL;
				if (url != null) {
					setViewerTitle  (url.getTitle()             );
					setViewerAddress(url.getRealURL().toString());
				} else {
					setViewerTitle  (null);
					setViewerAddress(null);
				}
				setViewerStatus(statusText);
				
				contextListener.activeChanged(this);
			}
		}
	}
	
	/** URL  */
	void load(URL url, PostData pd, URLHistory move, boolean reload, boolean force)
	{
		//### BUGS
		// POST ξ϶Ū˺ɤ߹ߡ֤ͭޤǤλ
		if (pd != null)
			reload = true;
		
		// ɹǤϤʤϡư褬ɽƤեƱå
		if (!reload) {
			synchronized (historyLock) {
				if (currentURL != null) {
					if (url != null) {
						// URL ưξ
						if (currentURL.getURL().sameFile(url)) {
							addToHistory(url, pd, url);
							moveToRef(true);
							return;
						}
					} else if (move != null) {
						// ưξ
						if (currentURL.getURL().sameFile(move.getURL())) {
							moveHistory(move);
							movePosition();
							return;
						}
					}
				}
			}
		}
		
		// ̤Υ󥯤ξ
		forceLoad = force;
		
		if (url != null) {
			newURL  = url;
			moveURL = null;
		} else if (move != null) {
			url = move.getURL     ();
			pd  = move.getPostData();
			if (pd != null) {
				if (MessageBox.show(hawkViewer,
				                    getMessage("message.dorepost.text" , null),
				                    getMessage("message.dorepost.title", null),
				                    MessageBox.BUTTON_OKCANCEL | MessageBox.ICON_QUESTION) != MessageBox.BUTTON_OK)
					return;
			}
			newURL  = null;
			moveURL = move;
		} else {
			return;
		}
		
		hawkEngine.stop();
		
//System.out.println("URL=["+url+"]");
		loading = true;
		contextListener.documentLoading(this);
		
		hawkEngine.load(++loadIndex, url, pd);
		hawkPanel.requestFocus();
		hawkPanel.requestFocus();             // ߥ󥰤ˤꣲƤӽФʤȥᡩ
	}
	
	/** ȥ */
	String getTitle()
	{
		synchronized (historyLock) {
			if (currentURL == null)
				return "";
			
			String title = currentURL.getTitle();
			if (title == null)
				title = "";
			return title;
		}
	}
	
	/**  */
	String getTag()
	{
		return tag;
	}
	
	/** ⡼ɤ */
	long getTabMode()
	{
		return tabMode;
	}
	
	/** ⡼ɤɲ */
	void addTabMode(long mode)
	{
		tabMode |= mode;
	}
	
	/** ⡼ɤ */
	void removeTabMode(long mode)
	{
		tabMode &= ~mode;
	}
	
	/** ˴ */
	void dispose()
	{
		hawkEngine.dispose();
	}
	
	//### HawkContext
	/** 륦ɥѤα륳ƥȤ */
	public Reporter getReporter()
	{
		return reporter;
	}
	
	/** ġ륭åȤ֤ */
	public Toolkit getToolkit()
	{
		return hawkPanel.getToolkit();
	}
	
	/** ꤷ URL ɽ */
	public void openHawk(URL url, PostData pd, boolean newWindow, boolean extension)
	{
		String protocol = url.getProtocol();
		if (supportProtocol.containsKey(protocol)) {
			if (newWindow)
				hawkViewer.createWindow().open(url, pd);
			else if (extension)
				hawkViewer.open(hawkViewer.createTab(HawkViewer.CURRENT_TAB, false), url, pd);
			else
				load(url, pd, null, false, false);
		} else {
			MessageBox.show(hawkViewer,
			                getMessage("message.unknownprotocol.text" , new String[]{protocol}),
			                getMessage("message.unknownprotocol.title", null),
			                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
		}
	}
	
	/** ꤵ줿 URL Υ֤ͥ */
	public URLConnection getURLConnection(URL url, PostData pd)
		throws IOException
	{
		return connector.getURLConnection(url, pd, forceLoad);
	}
	
	/**  ꤵ줿ɥȤ */
	public void download(Document document, URLConnection connection)
	{
		Download.show(hawkViewer, option, connection, null);
		
		loading = false;
		contextListener.documentLoaded(this);
	}
	
	/** ᡼֤ޤ */
	public Image createImage(ImageProducer producer)
	{
		return hawkPanel.createImage(producer);
	}
	
	/** ᡼֤ޤ */
	public Image createImage(int width, int height)
	{
		return hawkPanel.createImage(width, height);
	}
	
	/** ǥȥå֤ */
	public MediaTracker getMediaTracker()
	{
		return hawkPanel.getMediaTracker();
	}
	
	/** ꡼󥤥᡼ɲ */
	public void addImage(Image image, Dimension size)
	{
		hawkPanel.addImage(image, size);
	}
	
	/** եॢƥɲ */
	public void addFormItem(FormItem item)
	{
		hawkPanel.addFormItem(item);
	}
	
	/** ѥ򥯥ꥢ */
	public void cleanCanvas()
	{
		setTitle(null);
		hawkPanel.cleanCanvas();
	}
	
	/** ѥϰϤꤷƺɽ */
	public void repaint(int x, int y, int width, int height)
	{
		hawkPanel.repaintCanvas(x, y, width, height);
	}
	
	/** ѤιԤȡʸι⤵ */
	public void setScale(Dimension scale)
	{
		hawkPanel.setScale(scale);
	}
	
	/** طʤ */
	public void setBackground(Color color)
	{
		hawkPanel.setBackground(color);
	}
	
	/** طʲ */
	public void setBackground(Image image)
	{
		hawkPanel.setBackground(image);
	}
	
	/** ƥޥåפ */
	public void setItemMap(ItemMap map)
	{
		itemMap = map;
		hawkPanel.setItemMap(map);
	}
	
	/** ȥ */
	public void setTitle(String title)
	{
		currentURL     .setTitle(title);
		currentHistory .setTitle(title);
		contextListener.titleChanged  (this, title);
		contextListener.historyChanged(this);
	}
	
	/** ơå */
	public void setStatus(String status)
	{
		statusText = status;
		
		setViewerStatus(status);
	}
	
	/** Ūʥơå */
	public void setTemporaryStatus(String status)
	{
		setViewerStatus((status != null ? status: statusText));
	}
	
	/**  */
	public void setCursor(int cursor)
	{
		hawkPanel.setCursor(cursor);
	}
	
	/** ӥ塼ݡȤθߤΥ֤ */
	public Dimension getViewportSize()
	{
		return hawkPanel.getViewportSize();
	}
	
	/** ץ֤ */
	public Option getOption()
	{
		return option;
	}
	
	/** ꤷ URL ˬ䤷Ȥ뤫ɤ֤ */
	public boolean haveEverVisited(URL url)
	{
		return historyManager.contains(url.toString());
	}
	
	/** å */
	public void setCookie(String value, URL url)
	{
		connector.setCookie(value, url);
	}
	
	//### PanelListener
	/**  */
	public void back()
	{
		synchronized (historyLock) {
			if (canBack())
				load(null, null,
				     (moveURL == null ? currentURL : moveURL).getPreviousURLHistory(),
				     false, false);
		}
	}
	
	/** 뤫ɤ */
	public boolean canBack()
	{
		synchronized (historyLock) {
			if (topURL == null)
				return false;
			
			return ((moveURL == null ? currentURL : moveURL).getPreviousURLHistory() != null);
		}
	}
	
	/** ؿʤ */
	public void forward()
	{
		synchronized (historyLock) {
			if (canForward())
				load(null, null,
				     (moveURL == null ? currentURL : moveURL).getNextURLHistory(),
				     false, false);
		}
	}
	
	/** ؿʤ뤫ɤ */
	public boolean canForward()
	{
		synchronized (historyLock) {
			if (endURL == null)
				return false;
			
			return ((moveURL == null ? currentURL : moveURL).getNextURLHistory() != null);
		}
	}
	
	/** ɹ */
	public void reload(boolean force)
	{
		synchronized (historyLock) {
			if (canReload())
				load(null, null, currentURL, true, force);
		}
	}
	
	/** ɹǤ뤫ɤ */
	public boolean canReload()
	{
		synchronized (historyLock) {
			return (currentURL != null);
		}
	}
	
	/**  */
	public void stop()
	{
		hawkEngine.stop();
	}
	
	/** ߤǤ뤫ɤ */
	public boolean canStop()
	{
		return hawkEngine.isLoading();
	}
	
	//### EngineListener
	/** ³Ϥ˸ƤӽФ */
	public void connecting(int index)
	{
		if (loadIndex != index)
			return;
	}
	
	/** ³λȸƤӽФ */
	public void connected(int index, boolean noerror)
	{
		if (loadIndex != index)
			return;
		
		if (!noerror) {
			loading = false;
			contextListener.documentLoaded(this);
		}
	}
	
	/** 褬Ϥ˸ƤӽФ */
	public void renderingStarted(int index, URL openUrl, PostData pd, URL realUrl)
	{
		if (loadIndex != index)
			return;
		
		synchronized (historyLock) {
			if (newURL != null) {
				// ɤξ
				addToHistory(newURL, pd, realUrl);
				newURL = null;
			} else if (moveURL != null) {
				// ΰưξ
				moveURL.setRealURL(realUrl);
				moveHistory(moveURL);
				moveURL = null;
			}
		}
	}
	
	/** 褬λȸƤӽФ */
	public void renderingStopped(int index, boolean noerror)
	{
		if (loadIndex != index)
			return;
		
		// ȤդƤ硢뤹
		synchronized (historyLock) {
			if (currentURL != null)
				movePosition();
		}
		
		loading = false;
		contextListener.documentLoaded(this);
	}
	
	/** ꤷ URL ɽ */
	public void refresh(int index, URL url)
	{
		load(url, null, null, true, false);
	}
	
	//### private
	/**  URL ɲ */
	private void addToHistory(URL openUrl, PostData pd, URL realUrl)
	{
		setViewerAddress(openUrl.toString());
		
		// ߤƱ URL ̵뤹
		if (currentURL != null && openUrl.equals(currentURL.getURL()))
			return;
		
		URLHistory h = new URLHistory(openUrl, pd, realUrl);
		
		if (currentURL != null) {
			currentURL.setPosition(hawkPanel.getScrollPosition());
			currentURL.removeAfter();
			currentURL.append(h);
		}
		
		if (topURL == null)
			topURL = h;
		
		endURL = currentURL = h;
		
		// 򹹿
		currentHistory = historyManager.visited(openUrl.toString());
	}
	
	/** Ȥưư */
	private void moveHistory(URLHistory move)
	{
		if (currentURL != null)
			currentURL.setPosition(hawkPanel.getScrollPosition());
		currentURL = move;
		
		// 򹹿
		currentHistory = historyManager.visited(currentURL.getRealURL().toString());
		
		setViewerAddress(currentURL.getRealURL().toString());
	}
	
	/** ݥư */
	private void movePosition()
	{
		Point position = currentURL.getPosition();
		if (position != null)
			hawkPanel.setScrollPosition(position.x, position.y);
		else
			moveToRef(false);
	}
	
	/** Ȱ֤إ */
	private void moveToRef(boolean force)
	{
		// ˥桼ˤ̤뤵Ƥ뤫å
		if (!force) {
			Point position = hawkPanel.getScrollPosition();
			if (position.x != 0 || position.y != 0)
				return;
		}
		
		String ref = currentURL.getRealURL().getRef();
		
		if (ref == null) {
			hawkPanel.setScrollPosition(0, 0);
			return;
		}
		
		if (itemMap == null)
			return;
		
		Point p = itemMap.getIdPoint(ref);
		if (p == null)
			return;
		
		hawkPanel.setScrollPosition(p.x, p.y);
	}
	
	/** ꥽ʸ */
	private String getMessage(String key, String[] args)
	{
		return Resource.getMessage(RESOURCE, key, args);
	}
	
//### Viewer Ф
	/** ȥ */
	private void setViewerTitle(String title)
	{
		contextListener.titleChanged(this, title);
	}
	
	/** ɥ쥹 */
	private void setViewerAddress(String address)
	{
		contextListener.addressChanged(this, address);
		contextListener.historyChanged(this);
	}
	
	/** ơС */
	private void setViewerStatus(String status)
	{
		contextListener.statusChanged(this, status);
	}
	
//### NewActionListener
	/** ǳѤΥꥹ */
	private class NewActionListener
		implements ActionListener
	{
		private static final String WINDOW = "open.newwindow";
		private static final String TAB    = "open.newtab";
		
		/** ꥹʤ */
		private NewActionListener()
		{
		}
		
		/** ϥɥ */
		public void actionPerformed(ActionEvent e)
		{
			String command = e.getActionCommand();
			
			URL url = hawkPanel.getSelectedLink();
			if (url == null)
				return;
			
			if (WINDOW.compareTo(command) == 0) {
				openHawk(url, null, true, false);
			} else
			if (TAB   .compareTo(command) == 0) {
				openHawk(url, null, false, true);
			}
		}
	}
	
//### ApplicationActionListener
	/** ץꥱ󤫤鳫ѤΥꥹ */
	private class ApplicationActionListener
		implements ActionListener
	{
		private static final int MAIN = 0;
		private static final int LINK = 1;
		
		private int          type;
		private ViewerOption option;
		
		/** ꥹʤ */
		private ApplicationActionListener(int type, ViewerOption option)
		{
			this.type   = type;
			this.option = option;
		}
		
		/** ϥɥ */
		public void actionPerformed(ActionEvent e)
		{
			String command = e.getActionCommand();
			ViewerOption.Application[] apps = option.getApplications();
			if (apps == null)
				return;
			
			for (int i = 0; i < apps.length; i++) {
				ViewerOption.Application app = apps[i];
				if (command.compareTo(app.getId()) != 0)
					continue;
				
				URL url = (type == MAIN ? (currentURL != null ? currentURL.getURL() : null)
				                        : hawkPanel.getSelectedLink());
				if (url == null)
					return;
				
				StringBuffer path = new StringBuffer();
				
				try {
					app.exec(url, path);
				} catch (IOException ex) {
					MessageBox.show(hawkViewer,
					                getMessage("message.appnotopen.text" , new String[]{path.toString(), url.toString(), ex.toString()}),
					                getMessage("message.appnotopen.title", null),
					                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
				}
			}
		}
	}
}
