/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.ColorConverter;
import net.hizlab.kagetaka.awt.ImageButton;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.awt.StatusBar;
import net.hizlab.kagetaka.awt.TabbedContainer;
import net.hizlab.kagetaka.awt.Toolbar;
import net.hizlab.kagetaka.awt.event.TabEvent;
import net.hizlab.kagetaka.awt.event.TabListener;
import net.hizlab.kagetaka.bookmarks.Bookmark;
import net.hizlab.kagetaka.bookmarks.BookmarkAs;
import net.hizlab.kagetaka.bookmarks.BookmarkManager;
import net.hizlab.kagetaka.history.HistoryManager;
import net.hizlab.kagetaka.java2.DnDListener;
import net.hizlab.kagetaka.java2.DnDWrapper;
import net.hizlab.kagetaka.protocol.Protocol;
import net.hizlab.kagetaka.protocol.about.About;
import net.hizlab.kagetaka.rendering.PostData;
import net.hizlab.kagetaka.theme.Theme;
import net.hizlab.kagetaka.util.Environment;
import net.hizlab.kagetaka.util.StringUtils;
import net.hizlab.kagetaka.util.URLHistory;
import net.hizlab.kagetaka.util.windows.Shortcut;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Menu;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.TextField;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * 褹뤿ΡΩѤΥɥǤ
 * ΥɥˤϡѤΥ˥塼ʤɤ֤졢
 * Τ٤ƤεǽѤ뤳Ȥޤ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.5 $
 */
public class HawkViewer
	extends Frame
	implements HawkWindow
{
	private static final String    RESOURCE          = "net.hizlab.kagetaka.viewer.Resource";
	private static final Dimension MINIMUM_SIZE      = new Dimension(100, 70);
	private static final String    URL_FILE_PROTOCOL = "file";
	private static final String    URL_FILE_SCHEME   = URL_FILE_PROTOCOL + ":/";
	
	/** ȥ֤ɽǥå */
	public static final int CURRENT_TAB  = -1;
	/** ٤ƤΥ֤ɽǥå */
	public static final int ALL_TAB      = -2;
	/** Ȥ꺸¦ɽǥå */
	public static final int LEFTALL_TAB  = -3;
	/** Ȥ걦¦ɽǥå */
	public static final int RIGHTALL_TAB = -4;
	/** ȰʳΤ٤ƤΥ֤ɽǥå */
	public static final int OTHERALL_TAB = -5;
	/** ֺΥ֤ɽǥå */
	public static final int TOP_TAB      = -6;
	/** ֱΥ֤ɽǥå */
	public static final int LAST_TAB     = -7;
	
	/** ʥӥС */
	public static final int COMPONENT_NAVIBAR    =  1;
	/** С */
	public static final int COMPONENT_SEARCHBAR  =  2;
	/** 󥯥С */
	public static final int COMPONENT_LINKBAR    =  3;
	/** ơС */
	public static final int COMPONENT_STATUSBAR  =  4;
	
	/** ֥å */
	public static final long TAB_MODE_LOCK = 1;
	
	/** @serial ǥեȥɥꥹ */
	private static Vector windowListeners = new Vector();
	
	/** @serial ӥ塼ץ */
	private ViewerOption    option;
	/** @serial ɥޥ͡ */
	private WindowManager   windowManager;
	/** @serial ֥åޡޥ͡ */
	private BookmarkManager bookmarkManager;
	/** @serial ҥȥޥ͡ */
	private HistoryManager  historyManager;
	/** @serial ơ */
	private Theme           theme;
	
	/** @serial ˥塼 */
	private ViewerMenu      viewerMenu;
	/** @serial ᥤ쥤ȥޥ͡ */
	private GridBagLayout   gbl;
	/** @serial ᥤޡ */
	private Insets          insets;
	/** @serial ʥӥС */
	private ViewerToolbar   naviBar;
	/** @serial 󥯥С */
	private ViewerToolbar   linkBar;
	/** @serial ᥤѥͥ */
	private TabbedContainer mainPanel;
	/** @serial ơС */
	private StatusBar       statusBar;
	/** @serial ƥȥꥹ */
	private ViewerContextListener contextListener;
	
	/** @serial ɥåɥɥåץޥ͡ */
	private DnDWrapper      dndWrapper;
	/** @serial ɥåɥɥåץꥹ */
	private DnDListener     dndListener;
	
	/** @serial ᥤ󥿥ȥ */
	private String     titleMain        = null;
	/** @serial ȥղäʸ */
	private String     titleSuffix      = null;
	/** @serial ơå */
	private String     statusText       = null;
	
	/** @serial ȱ륳ƥ */
	private Vector        contexts    = new Vector();
	/** @serial ȱ륳ƥ */
	private Hashtable     contextHash = new Hashtable();
	/** @serial ȱ륳ƥ */
	private ViewerContext context;
	
	/** @serial ֤Υƥȥ˥塼 */
	private int    tabContextIndex = -1;
	
	/** ٤¹ */
	static {
		Protocol.initialize();
		HttpURLConnection.setFollowRedirects(false);
	}
	
	/**
	 * Υɥޤ
	 * 
	 * @param     option ץ
	 */
	public HawkViewer(ViewerOption option)
	{
		this.option          = option;
		this.windowManager   = WindowManager.getInstance();
		this.bookmarkManager = option.getBookmarkManager();
		this.historyManager  = option.getHistoryManager ();
		this.theme           = option.getThemeManager   ().getTheme();
		
		this.viewerMenu      = new ViewerMenu(this);
		this.mainPanel       = new TabbedContainer      ();
		this.contextListener = new ViewerContextListener();
		
		setTitleSuffix(getMessage("title", null));
		Image image = Resource.getImageResource(RESOURCE, "icon", getToolkit());
		if (image != null)
			setIconImage(image);
		
		gbl    = new GridBagLayout();
		insets = new Insets(0, 0, 0, 0);
		setLayout(gbl);
		LayoutUtils.addGridBag(this, new MenuPanel(), gbl, 0, 0, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
		LayoutUtils.addGridBag(this, mainPanel      , gbl, 0, 3, 1, 1, 1, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
		
		// ɲåݡͥȤ
		if (option.getShowNavibar  ())
			setVisibleComponent(COMPONENT_NAVIBAR  , true);
		if (option.getShowLinkbar  ())
			setVisibleComponent(COMPONENT_LINKBAR  , true);
		if (option.getShowStatusbar())
			setVisibleComponent(COMPONENT_STATUSBAR, true);
		
		validate();
		
		// ֤
		{
			setSize(option.getWindowSize());
			
			Point position = option.getWindowPosition();
			if (position != null) {
				setLocation(position);
			} else {
				Dimension size        = getSize();
				Dimension desktopSize = getToolkit().getScreenSize();
				// ǥȥåפ
				setLocation((desktopSize.width  - size.width ) / 2,
				            (desktopSize.height - size.height) / 2);
			}
		}
		
		// ꥹʡϿ
		addWindowListener(new MainWindowListener());
		for (int i = 0; i < windowListeners.size(); i++)
			addWindowListener((WindowListener)windowListeners.elementAt(i));
		mainPanel.addTabListener     (new MainPanelTabListener     ());
		mainPanel.addTabMouseListener(new MainPanelTabMouseListener());
		
		// ɥåɥɥå
		if ((dndWrapper = DnDWrapper.getInstance()) != null) {
			dndListener = new DnDListener()
			{
				/** ɥåפ */
				public boolean isActive()
				{
					return true;
				}
				
				/** ɥåפ줿 */
				public void drop(String data)
				{
					open(data.trim(), null);
				}
			};
			dndWrapper.addDnDListener(this, dndListener);
		}
		
		// ơޤŬ
		applyTheme();
		
		// ֤ĺ
		createTab(TOP_TAB, true);
		
//### DEBUG
//enableEvents(-1);
	}
	
//### DEBUG
/*
protected void processEvent(java.awt.AWTEvent e)
{
	System.out.println(e);
	super.processEvent(e);
}
*/
	
//### Original
	/**
	 * Υɥ򳫤ȤɬϿ륦ɥꥹʤɲäޤ
	 * 
	 * @param     listener ɥꥹ
	 */
	public static void addDefaultWindowListener(WindowListener listener)
	{
		windowListeners.addElement(listener);
	}
	
	/**
	 * Υɥ򳫤ȤɬϿ륦ɥꥹʤޤ
	 * 
	 * @param     listener ɥꥹ
	 */
	public static void removeDefaultWindowListener(WindowListener listener)
	{
		windowListeners.removeElement(listener);
	}
	
	/**
	 * ΥɥΥȥղäʸꤷޤ
	 * ⤷ɬפʤ <code>null</code> ꤷޤ
	 * 
	 * @param     suffix ȥղäʸ
	 */
	public synchronized void setTitleSuffix(String suffix)
	{
		this.titleSuffix = suffix;
		setTitle(titleMain);
	}
	
	/**
	 * ץ֤ޤ
	 * 
	 * @return    ץ
	 */
	public ViewerOption getOption()
	{
		return option;
	}
	
	/**
	 * ˻ꤷ URL ɤ߹ߡ̤ɽޤ
	 * Υ᥽åɤϡ̥åɤǽԤᡢ褬
	 * ᥽åɤλޤ
	 * 
	 * @param     url ɤ߹ URL
	 * @param     pd  POST Ϥǡ
	 *                POST ǤϤʤ <code>null</code>
	 */
	public void open(URL url, PostData pd)
	{
		open(CURRENT_TAB, url, pd);
	}
	
	/**
	 * ˻ꤷ URL ɤ߹ߡ̤ɽޤ
	 * Υ᥽åɤϡ̥åɤǽԤᡢ褬
	 * ᥽åɤλޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     url   ɤ߹ URL
	 * @param     pd    POST Ϥǡ
	 *                  POST ǤϤʤ <code>null</code>
	 */
	public void open(int index, URL url, PostData pd)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++)
			contextList[index].load(url, pd, null, true, false);
	}
	
	/**
	 * ˻ꤷѥɤ߹ߡ̤ɽޤ
	 * Υ᥽åɤϡ̥åɤǽԤᡢ褬
	 * ᥽åɤλޤ
	 * 
	 * @param     path ɤ߹ URL
	 * @param     pd   POST Ϥǡ
	 *                 POST ǤϤʤ <code>null</code>
	 */
	public void open(String path, PostData pd)
	{
		open(CURRENT_TAB, path, pd);
	}
	
	/**
	 * ˻ꤷѥɤ߹ߡ̤ɽޤ
	 * Υ᥽åɤϡ̥åɤǽԤᡢ褬
	 * ᥽åɤλޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     path  ɤ߹ URL
	 * @param     pd    POST Ϥǡ
	 *                  POST ǤϤʤ <code>null</code>
	 */
	public void open(int index, String path, PostData pd)
	{
		if (path == null)
			return;
		
		URL url = null;
		try {
			url = StringUtils.toURL(path);
			
			// file:/  Windows ξϥ硼ȥåȥեβ򤹤
			if (url.getProtocol().compareTo(URL_FILE_PROTOCOL) == 0) {
				if (Environment.isWindows) {
					String filePath = url.toString().substring(URL_FILE_SCHEME.length());
					// ?  # ͭϡ硼ȥåȤβ򤷤ʤ
					if (filePath.indexOf('?') == -1 && filePath.indexOf('#') == -1) {
						String realPath = Shortcut.resolve(filePath);
						if (realPath != null)
							url = StringUtils.toURL(realPath);
					}
				}
			}
		} catch (MalformedURLException e) {
			MessageBox.show(HawkViewer.this,
			                getMessage("message.invalidurl.text" , new String[]{path, e.toString()}),
			                getMessage("message.invalidurl.title", null),
			                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
			return;
		}
		
		open(index, url, pd);
	}
	
	/**
	 * ץ¸ޤ
	 */
	public void saveOption()
	{
		option.save();
	}
	
	/**
	 * ɽƤ륢ɥ쥹֤ޤ
	 * 
	 * @return    ɽƤ륢ɥ쥹
	 *            ɽƤʤ <code>null</code>
	 */
	public String getAddress()
	{
		if (naviBar == null)
			return null;
		
		return naviBar.getAddress();
	}
	
	/**
	 * ɥ쥹ɽޤ
	 * 
	 * @param     address ɥ쥹Ȥɽʸ
	 */
	public void showAddress(String address)
	{
		if (naviBar != null)
			naviBar.setAddress(address);
	}
	
	/**
	 * ˻ꤷʸ򡢥ơСɽ褦׵ᤷޤ
	 * ơСʸõϡ<code>null</code> ꤷޤ
	 * 
	 * @param     status ơСɽʸ
	 */
	public void showStatus(String status)
	{
		if (statusBar != null)
			statusBar.setText(status);
	}
	
	/**
	 * ꤷݡͥȤɽƤ뤫֤ޤ
	 * 
	 * @param     target оݤΥݡͥ
	 * 
	 * @return    ɽƤ <code>true</code>
	 *            ʳξ <code>false</code>
	 */
	public boolean isVisibleComponent(int target)
	{
		switch (target) {
		case COMPONENT_NAVIBAR  : return (naviBar   != null);
		case COMPONENT_SEARCHBAR: return false;
		case COMPONENT_LINKBAR  : return (linkBar   != null);
		case COMPONENT_STATUSBAR: return (statusBar != null);
		}
		return false;
	}
	
	/**
	 * ꤷݡͥȤɽ걣ꤷޤ
	 * 
	 * @param     target оݤΥݡͥ
	 * @param     b ɽ <code>true</code>
	 *              ʳξ <code>false</code>
	 */
	public void setVisibleComponent(int target, boolean b)
	{
		if (isVisibleComponent(target) == b)
			return;
		
		if (b) {
			// ɽ
			switch (target) {
			case COMPONENT_NAVIBAR  :
				this.naviBar = new ViewerToolbar(this, ViewerToolbar.TYPE_NAVI);
				synchronized (contexts) {
					for (int i = 0; i < contexts.size(); i++)
						((ViewerContext)contexts.elementAt(i)).addContextListener(naviBar);
					if (context != null)
						naviBar.activeChanged(context);
				}
				LayoutUtils.addGridBag(this, naviBar, gbl, 0, 1, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
				break;
			case COMPONENT_SEARCHBAR:
				return;
			case COMPONENT_LINKBAR  :
				this.linkBar = new ViewerToolbar(this, ViewerToolbar.TYPE_LINK);
				synchronized (contexts) {
					for (int i = 0; i < contexts.size(); i++)
						((ViewerContext)contexts.elementAt(i)).addContextListener(linkBar);
					if (context != null)
						linkBar.activeChanged(context);
				}
				LayoutUtils.addGridBag(this, linkBar, gbl, 0, 2, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
				break;
			case COMPONENT_STATUSBAR:
				this.statusBar = new StatusBar();
				LayoutUtils.addGridBag(this, statusBar , gbl, 0, 4, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
				break;
			}
		} else {
			// ɽ
			switch (target) {
			case COMPONENT_NAVIBAR   :
				synchronized (contexts) {
					for (int i = 0; i < contexts.size(); i++)
						((ViewerContext)contexts.elementAt(i)).removeContextListener(naviBar);
				}
				remove(naviBar);
				naviBar = null;
				break;
			case COMPONENT_SEARCHBAR :
				return;
			case COMPONENT_LINKBAR   :
				synchronized (contexts) {
					for (int i = 0; i < contexts.size(); i++)
						((ViewerContext)contexts.elementAt(i)).removeContextListener(linkBar);
				}
				remove(linkBar);
				linkBar = null;
				break;
			case COMPONENT_STATUSBAR :
				remove(statusBar);
				statusBar = null;
				break;
			}
		}
		
		validate();
		viewerMenu.changeVisibleComponent(target, b);
		
		// ¸
		switch (target) {
		case COMPONENT_NAVIBAR   : option.setShowNavibar  (b); break;
		case COMPONENT_SEARCHBAR : option.setShowSearchbar(b); break;
		case COMPONENT_LINKBAR   : option.setShowLinkbar  (b); break;
		case COMPONENT_STATUSBAR : option.setShowStatusbar(b); break;
		}
	}
	
	/**
	 * ƥ֤ʥ֤Υǥå֤ޤ
	 * 
	 * @return    ֥ǥå
	 *            ƥ֤ʥ֤¸ߤʤ <code>-1</code>
	 */
	public int getActiveTab()
	{
		synchronized (contexts) {
			if (context == null)
				return -1;
			
			return contexts.indexOf(context);
		}
	}
	
	/**
	 * ꤷ֤򥢥ƥ֤ˤޤ
	 * 
	 * @param     index ֥ǥå
	 * 
	 * @exception IndexOutOfBoundsException ϰϳΥǥåꤷ
	 */
	public void setActiveTab(int index)
	{
		synchronized (contexts) {
			ViewerContext c = (ViewerContext)contexts.elementAt(index);
			
			if (c == context)
				return;
			
			try {
				mainPanel.setActivePanel(c.getTag());
			} catch (IllegalArgumentException e) {
				return;
			}
			
			if (context != null)
				context.setActived(false);
			
			context = c;
			context.setActived(true);
		}
	}
	
	/**
	 * ֤Υ⡼ɤɲäޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     mode  ɲä⡼
	 */
	public void addTabMode(int index, long mode)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++)
			contextList[index].addTabMode(mode);
		
		viewerMenu.changeTabMode(context);
	}
	
	/**
	 * ֤Υ⡼ɤޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     mode  ɲä⡼
	 */
	public void removeTabMode(int index, long mode)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++)
			contextList[index].removeTabMode(mode);
		
		viewerMenu.changeTabMode(context);
	}
	
	/** ֤Υƥȥ˥塼򳫤ǥå֤ */
	int getTabContextIndex()
	{
		return tabContextIndex;
	}
	
	/**
	 * ֥åޡ˥塼ޤ
	 * 
	 * @param     menu ֥åޡۤ˥塼
	 * @param     tool ֥åޡΥ˥塼ɲä
	 *                 <code>true</code>ʤ <code>false</code>
	 */
	public void createBookmarkMenu(Menu menu, boolean tool)
	{
		viewerMenu.createBookmarkMenu(menu, tool);
	}
	
	/**
	 * ֥åޡ˥塼ޤ
	 * 
	 * @param     menu ֥åޡۤ˥塼
	 * @param     bookmarks ˥塼ɽ֥åޡ
	 */
	public void createBookmarkMenu(Menu menu, Vector bookmarks)
	{
		viewerMenu.createBookmarkMenu(menu, bookmarks);
	}
	
//### Override
	/**
	 *  ɥɽޤ
	 *  ɥǤ˸ƤФ˽Фޤ
	 */
	public void show()
	{
		Dimension size1 = getSize();
		super.show();
		Dimension size2 = getSize();
		
		if (!size1.equals(size2)) {
			Environment.windowSizeOffset = new Dimension(size2.width  - size1.width ,
			                                             size2.height - size1.height);
		}
	}
	
	/**
	 * ΥɥκǾ֤ޤ
	 * 
	 * @return    Ǿ
	 */
	public Dimension getMinimumSize()
	{
		return MINIMUM_SIZE;
	}
	
	/**
	 * ΥɥΥȥ򡢻ꤵ줿ͤꤷޤ
	 * ºݤˤϡǻꤷͤθˡ{@link #setTitleSuffix(String)}
	 * ˤäꤵ줿ʸղäޤ
	 * 
	 * @param     title ΥɥΥȥ
	 */
	public synchronized void setTitle(String title)
	{
		this.titleMain = title;
		
		if (titleMain == null && titleSuffix == null) {
			super.setTitle(null);
		} else if (titleMain   == null) {
			super.setTitle(titleSuffix);
		} else if (titleSuffix == null) {
			super.setTitle(titleMain);
		} else {
			super.setTitle(titleMain + " - " + titleSuffix);
		}
		
		windowManager.changeWindow(this);
	}
	
	/**
	 * ̤򥢥åץǡȤޤ
	 * 
	 * @param     g եå
	 */
	public void update(Graphics g)
	{
		paint(g);
	}
	
	/**
	 * ̤褷ޤ
	 * 
	 * @param     g եå
	 */
	public void paint(Graphics g)
	{
		super.paint(g);
	}
	
	/**
	 * ɥõޤ
	 * ɥѤƤ꥽뤿
	 * Υ᥽åɤƤФʤФʤޤ
	 */
	public void dispose()
	{
		//### TODO ٤ƤΥƥȤ˴
		
		super.dispose();
	}
	
	/**
	 * ΥɥΥѥ᡼ʸ֤ޤ
	 * 
	 * @return    ѥ᡼ʸ
	 */
	protected String paramString()
	{
		String str = super.paramString();
		if (titleSuffix != null)
		    str += ",suffix=" + titleSuffix;
		return str;
	}
	
//### Action
	/**
	 * ֿɥ򳫤פ¹Ԥޤ
	 */
	public HawkViewer createWindow()
	{
		Point location = getLocationOnScreen();
		location.x += 32;
		location.y += 32;
		
		Dimension desktopSize = getToolkit().getScreenSize();
		if (desktopSize.width  <= location.x ||
		    desktopSize.height <= location.y)
			location.x = location.y = 0;
		
		Dimension size = getSize();
		size.width  -= Environment.windowSizeOffset.width;
		size.height -= Environment.windowSizeOffset.height;
		
		HawkViewer hawkWindow = new HawkViewer((ViewerOption)option.clone());
//		hawkWindow.setSize    (size    );
		hawkWindow.setLocation(location);
		hawkWindow.show();
		return hawkWindow;
	}
	
	/**
	 * ֿ֤򳫤פ¹Ԥޤ
	 * 
	 * @param     index  ֥ǥå
	 * @param     active ֤򥢥ƥ֤ˤ <code>true</code>
	 *                   ʳξ <code>false</code>
	 * 
	 * @return    ɲä֤Υǥå
	 */
	public int createTab(int index, boolean active)
	{
		ViewerContext c = new ViewerContext(option);
		c.addContextListener(contextListener);
		c.addContextListener(viewerMenu     );
		c.addContextListener(naviBar        );
		c.addContextListener(linkBar        );
		
		if (dndWrapper != null)
			dndWrapper.addDnDListener(c.getCanvas(), dndListener);
		
		String tag = c.getTag();
		
		synchronized (contexts) {
			switch (index) {
			case CURRENT_TAB: index = contexts.indexOf(context) + 1; break;
			case TOP_TAB    : index =  0; break;
			case LAST_TAB   : index = -1; break;
			}
			
			if (index < 0 || contexts.size() < index)
				index = contexts.size();
			
			contexts.insertElementAt(c, index);
			contextHash.put(tag, c);
			
			if (index <= tabContextIndex)
				tabContextIndex++;
			
			mainPanel.addPanel(tag,
			                   c.getTitle(),
			                   theme.getImage(Theme.BROWSER_TAB_ICON_NORMAL, getToolkit()),
			                   c.attachViewer(this), active, index);
			
			if (active || contexts.size() == 1) {
				if (context != null)
					context.setActived(false);
				
				context = c;
				context.setActived(true);
			}
		}
		
		return index;
	}
	
	/**
	 * Web 򳫤פ¹Ԥޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void openURL(int index)
	{
		String     path   = null;
		OpenDialog dialog = new OpenDialog(this);
		dialog.show();
		
		if (dialog.getResult() == OpenDialog.RESULT_OPEN) {
			path = dialog.getPath();
			if (path != null && path.length() == 0)
				path = null;
		}
		dialog.dispose();
		
		if (path != null)
			open(index, path, null);
	}
	
	/**
	 * ֥ե򳫤פ¹Ԥޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void openFile(int index)
	{
		FileDialog fd = new FileDialog(this,
		                               getMessage("dialog.file.open", null),
		                               FileDialog.LOAD);
		fd.show();
		
		String path = fd.getFile();
		if (path == null) {
			fd.dispose();
			return;
		}
		
		path = (fd.getDirectory() != null ? fd.getDirectory() : "") + path;
		fd.dispose();
		
		open(index, URL_FILE_SCHEME + path, null);
	}
	
	/**
	 * ֥ɥĤפ¹Ԥޤ
	 * <p>
	 * Υ᥽åɤˤꡢɥõ٤ƤΥ꥽ޤ
	 * äơΥ᥽åɤƤǥɥõϡ
	 * {@link #dispose()} ƤӽФɬפϤޤ
	 */
	public void closeWindow()
	{
		Point position = getLocation();
		
		if (position.x >= 0 || position.y >= 0) {
			option.setWindowPosition(position);
			Dimension size = getSize();
			size.width  -= Environment.windowSizeOffset.width;
			size.height -= Environment.windowSizeOffset.height;
			option.setWindowSize(size);
		}
		
		dispose();
	}
	
	/**
	 * ֥֤Ĥפ¹Ԥޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void closeTab(int index)
	{
		boolean needCreate = false;
		
		synchronized (contexts) {
			ViewerContext[] contextList = getContexts(index);
			if (contextList.length == 0)
				return;
			
			ViewerContext c;
			String        key;
			int           i;
			int           activeIndex = 0;
			if (context != null)
				activeIndex = Math.max(contexts.indexOf(context), 0);
			
			for (index = 0; index < contextList.length; index++) {
				c   = contextList[index];
				
				if ((c.getTabMode() & TAB_MODE_LOCK) != 0)
					continue;
				
				key = c.getTag();
				
				// Ǥ˺Ѥߤɤå
				if (!contextHash.containsKey(key))
					continue;
				
				// ֤
				mainPanel.removePanel(key);
				
				i = contexts.indexOf(c);
				
				if (i < tabContextIndex)
					tabContextIndex--;
				else if (i == tabContextIndex)
					tabContextIndex = -1;
				
				if (i <= activeIndex)
					activeIndex--;
				
				// ƥȤ
				c.dispose();
				contexts.removeElementAt(i);
				contextHash.remove(key);
			}
			
			// ƥ֤ʥ֤
			if (contexts.size() > 0)
				setActiveTab(Math.max(activeIndex, 0));
			else
				needCreate = true;
		}
		
		// Ĥ⥿̵֤ʤäȤĺƤ
		if (needCreate)
			createTab(TOP_TAB, true);
	}
	
	/**
	 * ֽλפ¹Ԥޤ
	 */
	public void exitViewer()
	{
		windowManager.exit();
	}
	
	/**
	 * ߡפ¹Ԥޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void stop(int index)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++)
			contextList[index].stop();
	}
	
	/**
	 * ֺɤ߹ߡפ¹Ԥޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     force Ū˺ɹ <code>true</code>
	 *                  ʳξ <code>false</code>
	 */
	public void reload(int index, boolean force)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++)
			contextList[index].reload(force);
	}
	
	/**
	 * ưޤ
	 * 
	 * @param     index ֥ǥå
	 * @param     num ư̤ꤷޤ
	 *                οʤ鼡ءοʤذưޤ
	 */
	public void moveHistory(int index, int num)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++) {
			URLHistory url = contextList[index].currentURL, newUrl = null;
			if (num == 0 || url == null)
				continue;
			
			synchronized (contextList[index].historyLock) {
				while (url != null) {
					if (num > 0) {
						newUrl = url.getNextURLHistory();
						num--;
					} else {
						newUrl = url.getPreviousURLHistory();
						num++;
					}
					
					if (newUrl == null)
						break;
					if (num == 0) {
						if (newUrl != null)
							url = newUrl;
						break;
					}
				}
			}
			
			contextList[index].load(null, null, url, false, false);
		}
	}
	
	/**
	 * ۡɽޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void moveHome(int index)
	{
		String home = option.getHomeURL();
		if (home == null || home.length() == 0)
			home = About.BLANK;
		open(index, home, null);
	}
	
	/**
	 * 򥨥ǥɽޤ
	 */
	public void showHistoryEditor()
	{
		historyManager.showEditor();
	}
	
	/**
	 * ɽΥڡ֥åޡɲäޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void addBookmark(int index)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++) {
			URLHistory url = contextList[index].currentURL;
			if (url == null)
				continue;
			
			Bookmark bookmark = new Bookmark(url.getTitle(), url.getURL());
			Bookmark parent   = bookmarkManager.getRootBookmark();
			parent.addBookmark(bookmark);
			
			// ɥ
			windowManager.addToBookmark(parent, bookmark);
		}
	}
	
	/**
	 * ɽΥڡ֥åޡ˾ܺ٤ꤷɲäޤ
	 * 
	 * @param     index ֥ǥå
	 */
	public void addBookmarkAs(int index)
	{
		ViewerContext[] contextList = getContexts(index);
		for (index = 0; index < contextList.length; index++) {
			URLHistory url = contextList[index].currentURL;
			if (url == null)
				continue;
			
			BookmarkAs.Entry[] entries = BookmarkAs.show(this, bookmarkManager, url.getTitle(), url.getURL().toString());
			
			if (entries == null)
				continue;
			
			// ɥ
			synchronized (windowManager) {
				for (int i = 0; i < entries.length; i++) {
					entries[i].parent.addBookmark(entries[i].bookmark);
					windowManager.addToBookmark(entries[i].parent, entries[i].bookmark);
				}
			}
		}
	}
	
	/**
	 * ֥åޡǥ򳫤ޤ
	 */
	public void showBookmarkEditor()
	{
		bookmarkManager.showEditor();
	}
	
	/**
	 * ץɽޤ
	 */
	public void showOptionManager()
	{
		OptionDialog.show(this, option);
	}
	
	/**
	 * About ɽޤ
	 */
	public void showAbout()
	{
		createWindow().open(About.ABOUT, null);
	}
	
//### HawkWindow
	/**
	 * Υ᥽åɤľܸƤӽФƤϹԤޤ
	 * ˡ{@link WindowManager#addWindow(HawkWindow)}
	 * ƤӽФɬפޤ
	 */
	public void addWindowMenu(HawkWindow window)
	{
		viewerMenu.addWindowMenu(window);
	}
	
	/**
	 * Υ᥽åɤľܸƤӽФƤϹԤޤ
	 * ˡ{@link WindowManager#removeWindow(HawkWindow)}
	 * ƤӽФɬפޤ
	 */
	public void removeWindowMenu(int index)
	{
		viewerMenu.removeWindowMenu(index);
	}
	
	/**
	 * Υ᥽åɤľܸƤӽФƤϹԤޤ
	 * ˡ{@link WindowManager#changeWindow(HawkWindow)}
	 * ƤӽФɬפޤ
	 */
	public void changeWindowMenu(int index, HawkWindow window)
	{
		viewerMenu.changeWindowMenu(index, window);
	}
	
	/**
	 * Υ᥽åɤľܸƤӽФƤϹԤޤ
	 * ˡ{@link WindowManager#addToBookmark(Bookmark, Bookmark)}
	 * ƤӽФɬפޤ
	 */
	public void addToBookmark(Bookmark parent, Bookmark bookmark)
	{
		viewerMenu.addToBookmark(parent, bookmark);
		if (naviBar != null) naviBar.addToBookmark(parent, bookmark);
		if (linkBar != null) linkBar.addToBookmark(parent, bookmark);
	}
	
	/**
	 * Υ᥽åɤľܸƤӽФƤϹԤޤ
	 * ˡ{@link WindowManager#bookmarkChanged()}
	 * ƤӽФɬפޤ
	 */
	public void bookmarkChanged()
	{
		viewerMenu.bookmarkChanged();
		if (naviBar != null) naviBar.bookmarkChanged();
		if (linkBar != null) linkBar.bookmarkChanged();
	}
	
//### private
	/** ꥽ʸ */
	private String getMessage(String key, String[] args)
	{
		return Resource.getMessage(RESOURCE, key, args);
	}
	
	/** ꤵ줿֤ΰ */
	private ViewerContext[] getContexts(int index)
	{
		Vector v = new Vector();
		
		try {
			if (index >= 0) {
				v.addElement(contexts.elementAt(index));
			} else {
				switch (index) {
				case CURRENT_TAB:
					v.addElement(context);
					break;
				case ALL_TAB:
					for (int i = 0; i < contexts.size(); i++)
						v.addElement(contexts.elementAt(i));
					break;
				case LEFTALL_TAB:
					for (int i = 0; i < contexts.size(); i++) {
						if (contexts.elementAt(i) == context)
							break;
						v.addElement(contexts.elementAt(i));
					}
					break;
				case RIGHTALL_TAB:
					{
						int current = 0;
						while (current < contexts.size() &&
						       contexts.elementAt(current++) != context)
							;
						for (int i = current; i < contexts.size(); i++)
							v.addElement(contexts.elementAt(i));
					}
					break;
				case OTHERALL_TAB:
					for (int i = 0; i < contexts.size(); i++)
						if (contexts.elementAt(i) != context)
							v.addElement(contexts.elementAt(i));
					break;
				case TOP_TAB:
					v.addElement(contexts.elementAt(0));
					break;
				case LAST_TAB:
					v.addElement(contexts.elementAt(contexts.size() - 1));
					break;
				}
			}
		} catch (ArrayIndexOutOfBoundsException e) {}
		
		ViewerContext[] cs = new ViewerContext[v.size()];
		v.copyInto(cs);
		
		return cs;
	}
	
	/** ݡͥȤ顢֤Υǥåꤹ */
	private int getTabContextIndex(Component c)
	{
		String key = mainPanel.getKey(c);
		if (key == null)
			return -1;
		
		synchronized (contexts) {
			for (int i = 0; i < contexts.size(); i++)
				if (((ViewerContext)contexts.elementAt(i)).getTag().compareTo(key) == 0)
					return i;
		}
		
		return -1;
	}
	
	/** ƥݡͥȤ˥ơޤŬ */
	private void applyTheme()
	{
		Color c;
		Font  f;
		ImageButton b, b1, b2;
		Toolkit tk = getToolkit();
		
		// ֥С
		if ((c = theme.getColor(Theme.BROWSER_TAB_FOREGROUND         )) != null) mainPanel.setForeground             (c);
		if ((c = theme.getColor(Theme.BROWSER_TAB_BACKGROUND         )) != null) mainPanel.setBackground             (c);
		if ((c = theme.getColor(Theme.BROWSER_TAB_FOCUS              )) != null) mainPanel.setFocusFrameColor        (c);
		if ((f = theme.getFont (Theme.BROWSER_TAB_ACTIVE_FONT        )) != null) mainPanel.setActiveTitleFont        (f);
		if ((c = theme.getColor(Theme.BROWSER_TAB_ACTIVE_FOREGROUND  )) != null) mainPanel.setActiveTitleForeground  (c);
		if ((c = theme.getColor(Theme.BROWSER_TAB_ACTIVE_BACKGROUND  )) != null) mainPanel.setActiveTitleBackground  (c);
		if ((f = theme.getFont (Theme.BROWSER_TAB_UNACTIVE_FONT      )) != null) mainPanel.setUnActiveTitleFont      (f);
		if ((c = theme.getColor(Theme.BROWSER_TAB_UNACTIVE_FOREGROUND)) != null) mainPanel.setUnActiveTitleForeground(c);
		if ((c = theme.getColor(Theme.BROWSER_TAB_UNACTIVE_BACKGROUND)) != null) mainPanel.setUnActiveTitleBackground(c);
		if ((b1 = theme.getImageButton(Theme.BROWSER_TAB_SCROLLBAR_LEFT , tk)) != null &&
		    (b2 = theme.getImageButton(Theme.BROWSER_TAB_SCROLLBAR_RIGHT, tk)) != null)
			mainPanel.setTabScrollButton(b1, b2);
	}
	
//### MainWindowListener
	/** ɥϥɥ */
	private class MainWindowListener
		extends WindowAdapter
	{
		/** 󥹥󥹤 */
		private MainWindowListener()
		{
		}
		
		/** ɥƥ֤ˤʤäȤ */
		public void windowActivated(WindowEvent e)
		{
			windowManager.changeActiveViewer(HawkViewer.this);
		}
		
		/** ɥ줿Ȥ */
		public void windowOpened(WindowEvent e)
		{
			windowManager.addWindow(HawkViewer.this);
		}
		
		/** ɥĤ褦ȤȤ */
		public void windowClosing(WindowEvent e)
		{
			closeWindow();
		}
		
		/** ɥĤ줿Ȥ */
		public void windowClosed(WindowEvent e)
		{
			windowManager.removeWindow(HawkViewer.this);
		}
	}
	
//### ContextListener
	private class ViewerContextListener
		implements ContextListener
	{
		/** 󥹥󥹤 */
		private ViewerContextListener()
		{
		}
		
		/** ƥ֤ʥƥȤѹ줿 */
		public void activeChanged(ViewerContext context)
		{
		}
		
		/** ɥȤΥɤϤ줿 */
		public void documentLoading(ViewerContext context)
		{
			mainPanel.setImage(context.getTag(), theme.getImage(Theme.BROWSER_TAB_ICON_LOADING, getToolkit()));
		}
		
		/** ɥȤɤ줿 */
		public void documentLoaded(ViewerContext context)
		{
			mainPanel.setImage(context.getTag(), theme.getImage(Theme.BROWSER_TAB_ICON_NORMAL, getToolkit()));
		}
		
		/** ɥ쥹ѹ줿 */
		public void addressChanged(ViewerContext context, String address)
		{
			synchronized (contexts) {
				if (HawkViewer.this.context == context)
					showAddress(address);
			}
		}
		
		/** ȥ뤬ѹ줿 */
		public void titleChanged(ViewerContext context, String title)
		{
			mainPanel.setTitle(context.getTag(), title);
			synchronized (contexts) {
				if (HawkViewer.this.context == context)
					setTitle(title);
			}
		}
		
		/** ơѹ줿 */
		public void statusChanged(ViewerContext context, String status)
		{
			synchronized (contexts) {
				if (HawkViewer.this.context == context)
					showStatus(status);
			}
		}
		
		/** ѹ줿 */
		public void historyChanged(ViewerContext context)
		{
		}
	}
	
//### MainPanelTabListener
	private class MainPanelTabListener
		implements TabListener
	{
		/** 󥹥󥹤 */
		private MainPanelTabListener()
		{
		}
		
		/** ƥ֤ʥ֤ѹ줿 */
		public void activeChanged(TabEvent e)
		{
			synchronized (contexts) {
				if (context != null)
					context.setActived(false);
				
				context = (ViewerContext)contextHash.get(e.getKey());
				context.setActived(true);
			}
		}
		
		/** ֤ɲä줿 */
		public void tabAdded(TabEvent e)
		{
		}
		
		/** ֤줿 */
		public void tabRemoved(TabEvent e)
		{
		}
		
		/** ֤ѹ줿 */
		public void tabChanged(TabEvent e)
		{
		}
	}
	
//### MainPanelTabMouseListener
	private class MainPanelTabMouseListener
		extends MouseAdapter
	{
		/** 󥹥󥹤 */
		private MainPanelTabMouseListener()
		{
		}
		
		/** å줿 */
		public void mouseClicked(MouseEvent e)
		{
			// å
			if (e.isPopupTrigger() ||
			    ((e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK)) {
				
				synchronized (contexts) {
					if ((tabContextIndex = getTabContextIndex(e.getComponent())) == -1)
						return;
					
					viewerMenu.showTabPopupMenu(e.getComponent(), e.getX(), e.getY());
				}
			}
			
			// 楯å
			if ((e.getModifiers() & InputEvent.BUTTON2_MASK) == InputEvent.BUTTON2_MASK) {
				synchronized (contexts) {
					int index = getTabContextIndex(e.getComponent());
					if (index == -1)
						return;
					
					closeTab(index);
				}
				e.consume();
			}
		}
	}
	
//### MenuPanel
	private class MenuPanel
		extends Component
	{
		/** 󥹥󥹤 */
		private MenuPanel()
		{
		}
		
		/**  */
		public void update(Graphics g)
		{
			paint(g);
		}
		
		/**  */
		public void paint(Graphics g)
		{
			Dimension size = getSize();
			
			g.setColor(ColorConverter.getDarker(SystemColor.menu));
			g.drawLine(0, size.height - 1, size.width - 1, size.height - 1);
		}
		
		/** 侩֤ */
		public Dimension getPreferredSize()
		{
			return new Dimension(0, 1);
		}
		
		/** Ǿ֤ */
		public Dimension getMinimumSize()
		{
			return new Dimension(1, 1);
		}
	}
}
