/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.fclabs.util.Base64;
import net.hizlab.kagetaka.Reporter;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.cookie.CookieManager;
import net.hizlab.kagetaka.rendering.PostData;

import java.awt.Frame;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.MalformedURLException;
import java.util.Hashtable;

/**
 * ץȥ³򥵥ݡȤ륯饹Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.5 $
 */
class Connector
{
	private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resource";
	
	private static Hashtable authCache  = new Hashtable();
	
	private ViewerContext context;
	private ViewerOption  option;
	private Reporter      reporter;
	private Frame         owner;
	private CookieManager cookieManager;
	
	/**
	 * ͥΥ󥹥󥹤ޤ
	 */
	Connector(ViewerContext context)
	{
		this.context       = context;
		this.reporter      = context.getReporter();
		this.option        = (ViewerOption)context.getOption();
		this.owner         = owner;
		this.cookieManager = option.getCookieManager();
	}
	
	/** ʡ */
	void setOwner(Frame owner)
	{
		this.owner = owner;
	}
	
	/** ꤵ줿 URL Υ֤ͥ */
	URLConnection getURLConnection(URL url, PostData pd, boolean forceLoad)
		throws IOException
	{
		URLConnection connection = null;
		URL           baseUrl    = url;
		boolean       debug      = (reporter != null && reporter.getLevel() <= Reporter.DEBUG);
		
		if (reporter != null)
			reporter.report(Reporter.INFO, getMessage("report.connect.open", new String[]{url.toString()}), 0, 0);
		
		String  authorization = null;
		boolean newAuth       = false;
		
		CONNECT:
		for (;;) {
			connection = url.openConnection();
			
			if (!newAuth)
				authorization = getAuthorization(url);
			
			// ꥯȥإå
			setHTTPHeader(url, connection, forceLoad, authorization);
			
			// POST ǡ³
			if (pd != null)
				pd.send((HttpURLConnection)connection);
			
			// ³
			connection.connect();
			if (debug)
				dumpConnection(connection);
			
			// 쥹ݥ󥹥إå
			getHTTPHeader(url, connection);
			
			if (!(connection instanceof HttpURLConnection))
				break;
			
			HttpURLConnection httpConnection = (HttpURLConnection)connection;
			int               code           = httpConnection.getResponseCode();
			
			if (code == HttpURLConnection.HTTP_UNAUTHORIZED) {
				// ǧڽ
				if ((authorization = getWWWAuthenticate(url, connection)) == null)
					break CONNECT;
				newAuth = true;
				
				// ɹԤ
				if (reporter != null)
					reporter.report(Reporter.INFO, getMessage("report.connect.reopen", new String[]{url.toString()}), 0, 0);
			} else {
				// ǧھϤ줬ϥåɲ
				if (newAuth) {
					authCache.put(cutPath(url.toString()), authorization);
					authorization = null;
					newAuth       = false;
				}
				
				switch (code) {
				case HttpURLConnection.HTTP_MULT_CHOICE:         // 300
				case HttpURLConnection.HTTP_MOVED_PERM:          // 301
				case 307:                                        // 307 Temporary Redirect
					break;
				case HttpURLConnection.HTTP_MOVED_TEMP:          // 302
					if (pd != null && !option.getHttpStrict())
						pd = null;
					break;
				case HttpURLConnection.HTTP_SEE_OTHER :          // 303
					if (pd != null)
						pd = null;
					break;
				default:
					break CONNECT;
				}
				
				// 쥯Ƚ
				if ((url = getRedirectURL(url, connection, pd, baseUrl)) == null)
					break CONNECT;
				
				if (reporter != null)
					reporter.report(Reporter.INFO, getMessage("report.connect.redirect", new String[]{url.toString()}), 0, 0);
			}
			
			httpConnection.disconnect();
		}
		
		return connection;
	}
	
	/** å */
	void setCookie(String value, URL url)
	{
		cookieManager.setCookie(value, url);
	}
	
	/** HTTP ꥯȥإå */
	private void setHTTPHeader(URL url, URLConnection connection, boolean forceLoad, String authorization)
		throws IOException
	{
		connection.setRequestProperty("User-Agent"     , Resource.getMessage("kagetaka.useragent", null));
		connection.setRequestProperty("Accept-Language", getMessage("http.request.acceptlanguage", null));
		if (forceLoad) {
			connection.setRequestProperty("Pragma"       , "no-cache");
			connection.setRequestProperty("Cache-Control", "no-cache");
		}
		if (authorization != null)
			connection.setRequestProperty("Authorization", authorization);
		
		String cookie = cookieManager.getCookie(url);
		if (cookie != null)
			connection.setRequestProperty("Cookie", cookie);
	}
	
	/** HTTP ꥯȥإå */
	private void getHTTPHeader(URL url, URLConnection connection)
		throws IOException
	{
		String key;
		int    index = 1;
		
		for (;;) {
			if ((key = connection.getHeaderFieldKey(index)) == null)
				break;
			
			key = key.toLowerCase();
			if (key.compareTo("set-cookie") == 0)
				setCookie(connection.getHeaderField(index), url);
			
			index++;
		}
	}
	
	/** ǧھ桼 */
	private String getWWWAuthenticate(URL url, URLConnection connection)
	{
		String realm = connection.getHeaderField("WWW-Authenticate");
		if (realm == null)
			return null;
		
		// ǧھʬ (WWW-Authenticate: Basic realm=dokoka)
		String type = realm;
		int p = realm.indexOf(' ');
		if (p != -1) {
			type  = realm.substring(0, p);
			realm = realm.substring(p + 1);
			if (realm.toLowerCase().startsWith("realm")) {
				for (p = 5; p < realm.length(); p++) {
					if (realm.charAt(p) != ' ') {
						if (realm.charAt(p) == '=')
							realm = realm.substring(p + 1).trim();
						break;
					}
				}
			}
		} else {
			realm = null;
		}
		
		// Basic ʳǧڤϥݡȳ
		if (type.toLowerCase().compareTo("basic") != 0) {
			MessageBox.show(owner,
			                getMessage("message.unknown_authtype.text" , new String[]{type, url.toString()}),
			                getMessage("message.unknown_authtype.title", null),
			                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
			return null;
		}
		
		// 桼˼
		String[] data = AuthorizationDialog.show(owner, url.getHost(), realm);
		if (data == null)
			return null;
		
		return "Basic " + new String(Base64.encode((data[0] + ":" + data[1]).getBytes()));
	}
	
	/** 쥯ȤԤ֤ */
	private URL getRedirectURL(URL url, URLConnection connection, PostData pd, URL baseUrl)
		throws IOException
	{
		String location = connection.getHeaderField("Location");
		if (location == null)
			return null;
		
		try {
			url = new URL(location);
			if (pd != null &&
			    MessageBox.show(owner,
			                    getMessage("message.redirect.text" , new String[]{baseUrl.toString(), location}),
			                    getMessage("message.redirect.title", null),
			                    MessageBox.BUTTON_OKCANCEL | MessageBox.ICON_QUESTION) != MessageBox.BUTTON_OK)
				return null;
		} catch (MalformedURLException e) {
			MessageBox.show(owner,
			                getMessage("message.invalidurl.text" , new String[]{location, e.toString()}),
			                getMessage("message.invalidurl.title", null),
			                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
			return null;
		}
		
		return url;
	}
	
	/** ꤵ줿 URL ǧھʸ */
	private String getAuthorization(URL url)
	{
		String authorization = null;
		String path          = url.toString();
		
		// Υå夫ѥɤ򸡺
		for (;;) {
			if ((path = cutPath(path)) == null)
				return null;
			
			if ((authorization = (String)authCache.get(path)) != null)
				return authorization;
		}
	}
	
	/** ѥû */
	private String cutPath(String path)
	{
		int p = path.indexOf('?');
		
		if (p != -1)
			path = path.substring(0, p);
		else
			p = path.length();
		
		// http://
		if ((p = path.lastIndexOf('/', p - 1)) < 7)
			return null;
		
		return path.substring(0, p);
	}
	
	/** ꥽ʸ */
	private String getMessage(String key, String[] args)
	{
		return Resource.getMessage(RESOURCE, key, args);
	}
	
	/** ͥ */
	private void dumpConnection(URLConnection connection)
		throws IOException
	{
		if (connection instanceof HttpURLConnection) {
			HttpURLConnection httpConnection = (HttpURLConnection)connection;
			reporter.report(Reporter.DEBUG, "HTTP Response: " + httpConnection.getResponseCode()
			                                            + " " + httpConnection.getResponseMessage(), 0, 0);
		}
		
		int index = 0;
		String key;
		
		while ((key = connection.getHeaderFieldKey(++index)) != null)
			reporter.report(Reporter.DEBUG, "HTTP Response: [" + key + "]:[" + connection.getHeaderField(index) + "]", 0, 0);
	}
}
