/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.theme;

import net.hizlab.kagetaka.awt.ImageButton;
import net.hizlab.kagetaka.token.AttributeConverter;

import java.awt.Color;
import java.awt.Font;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.Polygon;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilterInputStream;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * ơޤɽ饹Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.3 $
 */
public class Theme
{
	private static final int BUFFER_SIZE = 0x4000;
	
	private static final int RESOURCE  = 0;
	private static final int DIRECTORY = 1;
	private static final int ARCHIVE   = 2;
	
	private static final String THEME_INI  = "theme.properties";
	
	public static final String THEME_NAME    = "theme.name";
	public static final String THEME_AUTHOR  = "theme.author";
	public static final String THEME_VERSION = "theme.version";
	public static final String THEME_DATE    = "theme.date";
	public static final String THEME_MEMO    = "theme.memo";
	
	public static final String EXT_HOTSPOT = ".hs";
	public static final String EXT_BASE    = ".base";
	public static final String EXT_OVER    = ".over";
	public static final String EXT_DOWN    = ".down";
	public static final String EXT_DISABLE = ".disable";
	public static final String EXT_AREA    = ".area";
	public static final String EXT_LABEL   = ".label";
	
	public static final String BROWSER_TOOLBAR_FONT             = "browser.toolbar.font";
	public static final String BROWSER_TOOLBAR_FOREGROUND       = "browser.toolbar.foreground";
	public static final String BROWSER_TOOLBAR_BACKGROUND       = "browser.toolbar.background";
	public static final String BROWSER_TOOLBAR_BACKGROUND_IMAGE = "browser.toolbar.background.image";
	
	public static final String BROWSER_TOOLBAR_BACK                  = "browser.toolbar.back";
	public static final String BROWSER_TOOLBAR_FORWARD               = "browser.toolbar.forward";
	public static final String BROWSER_TOOLBAR_RELOAD                = "browser.toolbar.reload";
	public static final String BROWSER_TOOLBAR_STOP                  = "browser.toolbar.stop";
	public static final String BROWSER_TOOLBAR_GO                    = "browser.toolbar.go";
	public static final String BROWSER_TOOLBAR_HOME                  = "browser.toolbar.home";
	public static final String BROWSER_TOOLBAR_BOOKMARK              = "browser.toolbar.bookmark";
	public static final String BROWSER_TOOLBAR_ADDRESSBAR_FONT       = "browser.toolbar.addressbar.font";
	public static final String BROWSER_TOOLBAR_ADDRESSBAR_FOREGROUND = "browser.toolbar.addressbar.foreground";
	public static final String BROWSER_TOOLBAR_ADDRESSBAR_BACKGROUND = "browser.toolbar.addressbar.background";
	
	public static final String BROWSER_TAB_ICON_NORMAL         = "browser.tab.icon.normal";
	public static final String BROWSER_TAB_ICON_LOADING        = "browser.tab.icon.loading";
	public static final String BROWSER_TAB_FOREGROUND          = "browser.tab.foreground";
	public static final String BROWSER_TAB_BACKGROUND          = "browser.tab.background";
	public static final String BROWSER_TAB_FOCUS               = "browser.tab.focus";
	public static final String BROWSER_TAB_ACTIVE_FONT         = "browser.tab.active.font";
	public static final String BROWSER_TAB_ACTIVE_FOREGROUND   = "browser.tab.active.foreground";
	public static final String BROWSER_TAB_ACTIVE_BACKGROUND   = "browser.tab.active.background";
	public static final String BROWSER_TAB_UNACTIVE_FONT       = "browser.tab.unactive.font";
	public static final String BROWSER_TAB_UNACTIVE_FOREGROUND = "browser.tab.unactive.foreground";
	public static final String BROWSER_TAB_UNACTIVE_BACKGROUND = "browser.tab.unactive.background";
	public static final String BROWSER_TAB_SCROLLBAR_LEFT      = "browser.tab.scrollbar.left";
	public static final String BROWSER_TAB_SCROLLBAR_RIGHT     = "browser.tab.scrollbar.right";
	
	public static final String BOOKMARK_FILE   = "bookmark.file";
	public static final String BOOKMARK_FOLDER = "bookmark.folder";
	
	private String      path;
	private int         type;
	private Properties  p;
	private ClassLoader cl;
	private String      name;
	private String      key;
	
	private Hashtable cache = new Hashtable();
	
	/**
	 * ơޤޤ
	 * 
	 * @param     path ѥ
	 */
	public Theme(File path)
		throws InvalidThemeException
	{
		this.path = path.getPath();
		this.type = (path.isDirectory() ? DIRECTORY : ARCHIVE);
		this.p    = new Properties();
		
		loadProperties();
		
		name = p.getProperty(THEME_NAME);
		if (name == null)
			new InvalidThemeException("Theme `" + path + "' has no name");
		key = path.getName() + "*" + name;
	}
	
	/**
	 * ơޤޤ
	 * 
	 * @param     path ѥ
	 */
	Theme(String path)
		throws InvalidThemeException
	{
		this.path = path;
		this.type = RESOURCE;
		this.p    = new Properties();
		this.cl   = getClass().getClassLoader();
		
		loadProperties();
		
		key = name = p.getProperty(THEME_NAME);
	}
	
	/** ץѥƥ */
	private void loadProperties()
		throws InvalidThemeException
	{
		InputStream is = null;
		try {
			is = getInputStream(THEME_INI);
			if (is == null)
				throw new InvalidThemeException("`" + path + "' is invalid");
			p.load(is);
		} catch (IOException e) {
			new InvalidThemeException(e.toString());
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {}
			}
		}
	}
	
	/** ץåȥȥ꡼ޤ */
	private InputStream getInputStream(String file)
		throws InvalidThemeException
	{
		try {
			switch (type) {
			case RESOURCE:
				if (cl != null)
					return cl.getResourceAsStream(path + "/" + file);
				else
					return ClassLoader.getSystemResourceAsStream(path + "/" + file);
			case DIRECTORY:
				return new FileInputStream(new File(path, file));
			case ARCHIVE:
				return new ZipInputStream(path, file);
			}
		} catch (IOException e) {
			new InvalidThemeException(e.toString());
		}
		return null;
	}
	
	/** Хޤ */
	private byte[] getBytes(String file)
	{
		if (file == null)
			return null;
		
		try {
			InputStream is = getInputStream(file);
			if (is == null)
				return null;
			
			ByteArrayOutputStream baos = new ByteArrayOutputStream(BUFFER_SIZE);
			byte[] buffer = new byte[BUFFER_SIZE];
			int    length = 0;
			
			while ((length = is.read(buffer, 0, BUFFER_SIZE)) != -1)
				baos.write(buffer, 0, length);
			
			baos.flush();
			
			byte[] data = baos.toByteArray();
			
			is  .close();
			baos.close();
			
			return data;
		} catch (IOException e) {
System.out.println(e);
		} catch (InvalidThemeException e) {
System.out.println(e);
		}
		return null;
	}
	
	/** ƥ */
	void load()
		throws InvalidThemeException
	{
		//### TODO
	}
	
	/**
	 * ơ޼ΤΥ֤ޤ
	 * Υϡơ˰äͤˤʤޤ
	 * ƱơޤǤɬƱͤˤʤޤ
	 * ΥѤơơޤ̤뤳ȤǽǤ
	 * 
	 * @return    
	 */
	public String getKey()
	{
		return key;
	}
	
	/**
	 * ơޤͤ򿿵ͤȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * 
	 * @return    <code>true</code> ꤵ줿 <code>true</code>
	 *            ʳξ <code>false</code>
	 */
	public boolean getBoolean(String key)
	{
		String value = p.getProperty(key);
		if (value == null)
			return false;
		
		return Boolean.valueOf(value).booleanValue();
	}
	
	/**
	 * ơޤͤ򿧤ȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * 
	 * @return    
	 */
	public Color getColor(String key)
	{
		String value = p.getProperty(key);
		if (value == null)
			return null;
		
		return AttributeConverter.convertToColorSilent(value);
	}
	
	/**
	 * ơޤͤեȤȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * 
	 * @return    ե
	 */
	public Font getFont(String key)
	{
		String value = p.getProperty(key);
		if (value == null)
			return null;
		
		return Font.decode(value);
	}
	
	/**
	 * ơޤͤݥꥴȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * 
	 * @return    ݥꥴ
	 */
	public Polygon getPolygon(String key)
	{
		String value = p.getProperty(key);
		if (value == null)
			return null;
		
		StringTokenizer st = new StringTokenizer(value, ",");
		Vector          v  = new Vector();
		while (st.hasMoreTokens())
			v.addElement(st.nextToken().trim());
		
		int   npoints = v.size() / 2;
		int[] xpoints = new int[npoints];
		int[] ypoints = new int[npoints];
		try {
			for (int i = 0; i < npoints; i++) {
				xpoints[i] = Integer.valueOf((String)v.elementAt(i * 2    )).intValue();
				ypoints[i] = Integer.valueOf((String)v.elementAt(i * 2 + 1)).intValue();
			}
		} catch (NumberFormatException e) {
			return null;
		}
		
		return new Polygon(xpoints, ypoints, npoints);
	}
	
	/**
	 * ơޤͤ򥤥᡼ȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * @param     tk  ġ륭å
	 * 
	 * @return    ᡼
	 */
	public Image getImage(String key, Toolkit tk)
	{
		Image image = null;
		
		try {
			image = (Image)cache.get(key);
		} catch (ClassCastException e) {}
		
		if (image == null) {
			byte[] data = getBytes(p.getProperty(key));
			if (data == null)
				return null;
			
			image = tk.createImage(data);
			if (image != null)
				cache.put(key, image);
		}
		
		return image;
	}
	
	/**
	 * ơޤͤ򸵤ˡ᡼ܥȤƼޤ
	 * 
	 * @param     key ơޤιܥ
	 * @param     tk  ġ륭å
	 * 
	 * @return    ᡼ܥ
	 */
	public ImageButton getImageButton(String key, Toolkit tk)
	{
		ImageButton ib    = null;
		Image       image = getImage(key, tk);
		if (image != null) {
			// 
			ib = new ImageButton(null, image, getImage(key + EXT_HOTSPOT, tk));
		} else if ((image = getImage(key + EXT_BASE, tk)) != null) {
			// 
			Image   over    = getImage  (key + EXT_OVER   , tk);
			Image   down    = getImage  (key + EXT_DOWN   , tk);
			Image   disable = getImage  (key + EXT_DISABLE, tk);
			Polygon sarea   = getPolygon(key + EXT_HOTSPOT + EXT_AREA);
			Image   sover   = getImage  (key + EXT_HOTSPOT + EXT_OVER, tk);
			Image   sdown   = getImage  (key + EXT_HOTSPOT + EXT_DOWN, tk);
			ib = new ImageButton(image, down, over, disable, sarea, sover, sdown);
		} else {
			return null;
		}
		
		ib.setName(key);
		
		return ib;
	}
	
	/**
	 * ʸѴޤ
	 * 
	 * @return    ʸ
	 */
	public String toString()
	{
		return getClass().getName() + "[" + key + "]";
	}
	
//### ZipInputStream
	/** ZIP ץåȥȥ꡼ */
	private class ZipInputStream
		extends FilterInputStream
	{
		private ZipFile zipFile;
		
		/** 󥹥󥹤 */
		private ZipInputStream(String zip, String path)
			throws IOException
		{
			super(null);
			
			zipFile = new ZipFile(zip);
			ZipEntry zipEntry = zipFile.getEntry(path);
			if (zipEntry == null)
				throw new FileNotFoundException("`" + path + "' in `" + zip + "'");
			
			in = zipFile.getInputStream(zipEntry);
		}
		
		/** Ĥ */
		public void close()
			throws IOException
		{
			super.close();
			zipFile.close();
		}
	}
}
