/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.style;

import net.hizlab.kagetaka.Reporter;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.util.Charset;

import net.fclabs.util.Queue;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * CSS αǡեϤ뤿Υ饹Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class CSSParser
{
	private static final int INPUT_BUFFER = 4096;
	private static final int TT_NOTHING   = 0;
	private static final int TT_EOF       = 1;
	private static final int TT_STRINGS   = 2;
	private static final int TT_KEYWORD   = 3;
	private static final int TT_SYNTAX    = 4;
	private static final String[] START_BLOCK  = {"{"};
	private static final String[] END_BLOCK    = {"}"};
	private static final String[] END_PROPERTY = {":"};
	private static final String[] END_LINE     = {";"};
	private static final String[] END_STYLE    = {"}", ";"};
	
	private InputStream         is       = null;   // ץåȥȥ꡼
	private BufferedInputStream bis      = null;   // Хåեץåȥȥ꡼
	private String              encoding = null;   // 󥳡ǥ
	private CSSReader           reader   = null;   // CSS ꡼
	private Reporter            reporter = null;   // 顼ݡ
	
	private boolean  commitEncoding = false;  // 󥳡ǥ󥰤ꤷ
	private int      media          = 0;      // ǥ
	
	// 
	private Style        style        = null;
	private String       token        = null;
	private String       tokenLC      = null;
	private StringBuffer sb           = new StringBuffer();
	private boolean      beforeSyntax = false;
	private boolean      nowSyntax    = false;
	private Selector     selector     = null;
	
	/**
	 * CSS ΥǡϤѡޤ
	 * 
	 * @param     reporter 顼ݡ
	 * @param     data     Ϥǡ
	 */
	public CSSParser(Reporter reporter, String data)
	{
		this.reporter = reporter;
		this.reader   = new CSSReader(reporter, new BufferedReader(new StringReader(data)));
		this.commitEncoding = true;
	}
	
	/**
	 * CSS ΥǡեϤѡޤ
	 * 
	 * @param     reporter 顼ݡ
	 * @param     is       ϤǡΥץåȥȥ꡼
	 * @param     encoding ϤǡΥ󥳡ǥ󥰡
	 *                     Τʤ <code>null</code>
	 */
	public CSSParser(Reporter reporter, InputStream is, String encoding)
	{
		this.reporter = reporter;
		this.is       = is;
		this.encoding = encoding;
		
		BufferedReader br = null;
		
		if (encoding != null && (br = createBufferedReader(is, encoding)) != null) {
			this.commitEncoding = true;
		} else {
			// 󥳡ǥ󥰤ǤƤʤ
			this.bis = new BufferedInputStream(is, INPUT_BUFFER);
			this.bis.mark(INPUT_BUFFER * 4);
			br = createBufferedReader(bis, Charset.getDefaultEncoding());
		}
		
		this.reader = new CSSReader(reporter, br);
	}
	
	/** Хåե꡼ */
	private BufferedReader createBufferedReader(InputStream is, String encoding)
	{
		if (encoding != null) {
			try {
				return new BufferedReader(new InputStreamReader(is, encoding), INPUT_BUFFER);
			} catch (UnsupportedEncodingException e) {
				addWarning("charset.warning.invalid",
				           new String[]{encoding}, 0);
			}
			return null;
		} else {
			return new BufferedReader(new InputStreamReader(is), INPUT_BUFFER);
		}
	}
	
	/**
	 * style °ͤϤޤ
	 * 
	 * @param     reporter 顼ݡ
	 * @param     s        Ϥ륹ʸ
	 * 
	 * @return    롢ϤǤʤ <code>null</code>
	 */
	public static Style valueOf(Reporter reporter, String s)
	{
		s = "{" + s + "}";
		CSSReader reader = new CSSReader(reporter, new BufferedReader(new StringReader(s)));
		try {
			return reader.readStyle(0, null, false);
		} catch (EOFException e) {
		} catch (IOException e) {
			//### ERROR
		}
		return null;
	}
	
	/**
	 * ǡեϤϰ֤ƬΥ֤ޤ
	 * 
	 * @return    Ϥ̤ƬΥ롢
	 *            뤬¸ߤʤ <code>null</code>
	 * 
	 * @exception IOException IO 顼ȯ
	 */
	public synchronized Style next()
		throws IOException
	{
		try {
			for (;;) {
				// ƬФԤ
				while ((token = reader.readToken(true, true)) == null)
					;
				tokenLC = token.toLowerCase();
				
				// @charset §
				if (tokenLC.compareTo("@charset") == 0) {
					if ((token = reader.readToken(false, true)) != null && reader.isSyntax(END_LINE)) {
						if (!commitEncoding)
							resetEncoding(token);
					}
					continue;
				}
				// @charset Ƭʳ̵
				if (!commitEncoding)
					commitEncoding = true;
				
				// @media §
				if (tokenLC.compareTo("@media") == 0) {
					if (media != 0)
						reader.skipNextBlock("cssparser.warning.nestedmedia", null);
					else if ((media = Style.convertMedia(reader.getList(true, true))) == 0)
						reader.skipNextBlock("cssparser.warning.invalidmedia", null);
					else if ((token = reader.readToken(true, true)) == null)
						media = 0;
					else if (token.compareTo("{") != 0) {
						reader.skipNextBlock("cssparser.warning.invalidposition", reader.lastToken);
						media = 0;
					}
					continue;
				}
				
				// @import §
				if (tokenLC.compareTo("@import") == 0) {
					if ((token = reader.readToken(false, true)) != null && reader.isSyntax(END_LINE))
						importStyle(token);
					continue;
				}
				
				// @page §
				if (tokenLC.compareTo("@page") == 0) {
					if ((style = reader.readStyle(media, null, true)) != null)
						return style;
					continue;
				}
				
				// ¾ @ §
				if (tokenLC.startsWith("@")) {
					reader.skipNextBlock("cssparser.warning.unknown", token);
					continue;
				}
				
				// Ĥ "}" (@media Ĥ̤ʤ)
				if (tokenLC.compareTo("}") == 0) {
					if (media == 0)
						reader.moveNextLine("cssparser.warning.invalidposition", token);
					else
						media = 0;
					continue;
				}
				
				// 󥿥å
				if (reader.lastTokenType == TT_SYNTAX) {
					reader.pushBack();
					reader.skipNextBlock("cssparser.warning.invalidposition", reader.lastToken);
					continue;
				}
				
				// 쥯줿
				reader.pushBack();
				beforeSyntax = true;
				for (;;) {
					token = reader.readToken(true, false);
					if (token != null && token.compareTo("{") == 0) {
						reader.pushBack();
						break;
					}
					nowSyntax = (reader.lastTokenType == TT_SYNTAX);
					if (!beforeSyntax && !nowSyntax)
						sb.append(' ');
					sb.append(reader.lastToken);
					beforeSyntax = nowSyntax;
				}
				if (sb.length() > 0) {
					token = sb.toString();
					sb.setLength(0);
					try {
						selector = new Selector(token);
					} catch (java.text.ParseException e) {
						if (reporter != null)
							reporter.report(Reporter.WARNING, e.getMessage(), reader.lineNumber, 0);
						reader.skipNextBlock(null, null);
						continue;
					}
				} else
					selector = null;
				if ((style = reader.readStyle(media, selector, false)) != null)
					return style;
			}
		} catch (EOFException e) {}
		
		return null;
	}
	
	/** 󥳡ǥ󥰤ѹ */
	private void resetEncoding(String charset)
		throws IOException
	{
		if (charset == null)
			return;
		
		int lineno = reader.lineno();
		
		String encoding = Charset.toEncoding(charset);
		if (encoding == null) {
			addWarning("charset.warning.invalid",
			           new String[]{charset}, lineno);
			return;
		}
		
		try {
			bis.reset();
			reader = new CSSReader(reporter, createBufferedReader(bis, encoding));
		} catch (IOException e) {
			// ꥻåȤ˼ԤǤ⡢󥳡ǥ󥰤ꤷ³
			addWarning("charset.warning.reset",
			           new String[]{charset}, lineno);
		}
		
		this.encoding       = encoding;
		this.commitEncoding = true;
		
		reporter.report(Reporter.INFO,
		                Resource.getMessage("charset.info.encoding", new String[]{encoding}),
		                lineno, 0);
	}
	
	/** ¾Υ륷Ȥɤ߹ */
	private void importStyle(String url)
	{
		// next() ƵƤӽФˤ٤
		//### TODO 6.3
	}
	
	/**
	 * ߤɤ߹ΥǡΥ󥳡ǥ֤̾ޤ
	 * 
	 * @return    󥳡ǥ̤̾ξ <code>null</code>
	 */
	public String getEncoding()
	{
		return encoding;
	}
	
	/**
	 * ѡλ꥽ޤ
	 */
	public void close()
	{
		try {
			if (bis != null)
				bis.close();
		} catch (IOException e) {
			addWarning("cssparser.warning.stream.close",
			           new String[]{e.toString()}, 0);
		}
		try {
			if (is != null)
				is.close();
		} catch (IOException e) {
			addWarning("cssparser.warning.stream.close",
			           new String[]{e.toString()}, 0);
		}
	}
	
	/** ٹɲä */
	private void addWarning(String key, String[] args, int lineNumber)
	{
		reporter.report(Reporter.WARNING,
		                Resource.getMessage(key, args),
		                lineNumber, 0);
	}
	
//### CSSReader
	private static class CSSReader
	{
		private Reporter        reporter      = null;
		private StreamTokenizer tokenizer     = null;
		private boolean         pushBack      = false;
		private String          lastToken     = null;
		private int             lastTokenType = TT_NOTHING;
		private int             lineNumber    = 0;
		
		private String token = null;
		
		/** 󥹥󥹤 */
		private CSSReader(Reporter reporter, BufferedReader br)
		{
			this.reporter  = reporter;
			this.tokenizer = new StreamTokenizer(br);
			
			// ȡʥ
			tokenizer.resetSyntax();
			
			tokenizer.wordChars   (0x00, 0xFFFF);
			tokenizer.ordinaryChar('"');
			tokenizer.ordinaryChar(':');
			initTokenizerEndText  ();
			
			tokenizer.eolIsSignificant  (false);
			tokenizer.slashSlashComments(false);
		}
		
		/** ʸ¦Ѥ˥ȡʥ */
		private void initTokenizerStartText()
		{
			tokenizer.wordChars(';', ';');
			tokenizer.wordChars('{', '{');
			tokenizer.wordChars('}', '}');
			tokenizer.wordChars(',', ',');
			tokenizer.wordChars('/', '/');          // 
			tokenizer.wordChars('(', '(');          // url "("
			tokenizer.wordChars('>', '>');          // 쥯
			tokenizer.wordChars('+', '+');          // 쥯
			
			tokenizer.wordChars(0x09, 0x10);
			tokenizer.wordChars(0x12, 0x13);
			tokenizer.wordChars(0x20, 0x20);
			
			tokenizer.slashStarComments(false);
		}
		
		/** ʸγ¦Ѥ˥ȡʥ */
		private void initTokenizerEndText()
		{
			tokenizer.ordinaryChar(';');
			tokenizer.ordinaryChar('{');
			tokenizer.ordinaryChar('}');
			tokenizer.ordinaryChar(',');
			tokenizer.ordinaryChar('/');
			tokenizer.ordinaryChar('(');
			tokenizer.ordinaryChar('>');
			tokenizer.ordinaryChar('+');
			
			tokenizer.whitespaceChars(0x09, 0x10);
			tokenizer.whitespaceChars(0x12, 0x13);
			tokenizer.whitespaceChars(0x20, 0x20);
			
			tokenizer.slashStarComments(true);
		}
		
		/** Υ֥å򥹥ȤƲϤ֤ޤ { key : value; ... }  */
		private Style readStyle(int media, Selector selector, boolean page)
			throws IOException, EOFException
		{
			int toplineno = tokenizer.lineno();
			
			tokenizer.ordinaryChar(':');
			
			// Ƭ "{" ǻϤޤ뤫å
			if (!isSyntax(START_BLOCK))
				return null;
			
			String   name   = null;
			String[] values = null;
			Style    style = new Style(reporter, toplineno, 0,
			                           (media != 0 ? media : Style.MEDIA_ALL), selector, page);
			
			for (;;) {
				// ƬФԤ
				while ((token = readToken(true, true)) == null)
					;
				
				if (token.compareTo("}") == 0)
					break;
				if (token.compareTo(";") == 0)
					continue;
				if (lastTokenType == TT_SYNTAX) {
					moveNextLine("cssparser.warning.invalidposition", token);
					continue;
				}
				
				// ץѥƥ
				name = token;
				if (!isSyntax(END_PROPERTY))
					continue;
				
				// 
				values = getList(false, false);
				if (!isSyntax(END_STYLE))
					continue;
				
				if (values != null)
					style.initProperties(name, values);
				
				// ";" ǤϤʤ "}" ǽäƤ
				if (token.compareTo("}") == 0)
					break;
			}
			
			tokenizer.wordChars(':', ':');
			
//System.out.println("S["+style+"]");
			return style;
		}
		
		/**
		 * Υȡ֤
		 * (keyword ? ʸλ null : ɤλ null)
		 * (error ? null ξϡιԤ˰ư)
		 */
		private String readToken(boolean keyword, boolean error)
			throws IOException, EOFException
		{
			if (pushBack) {
				pushBack = false;
//System.out.println("@["+lastToken+"]");
				return lastToken;
			}
			
			String returnToken = null;
			
			READ:
			switch (tokenizer.nextToken()) {
			case StreamTokenizer.TT_EOF:
				lastToken     = null;
				lastTokenType = TT_EOF;
				lineNumber    = tokenizer.lineno();
				throw new EOFException();
			case StreamTokenizer.TT_WORD:
				returnToken   = lastToken = tokenizer.sval;
				lastTokenType = TT_KEYWORD;
				lineNumber    = tokenizer.lineno();
				break READ;
			case '"':
			case '(':
				{
					lineNumber = tokenizer.lineno();
					
					// ʸνõ
					StringBuffer sb = new StringBuffer();
					int     start  = tokenizer.ttype;
					boolean escape = false;
					int     length = 0, i;
					String  value;
					
					initTokenizerStartText();
					if (start == '(') {
						sb.append('(');
						tokenizer.wordChars   ('"', '"');
						tokenizer.ordinaryChar(')');
					}
					for (;;) {
						switch (tokenizer.nextToken()) {
						case StreamTokenizer.TT_EOF:
							lastToken     = null;
							lastTokenType = TT_EOF;
							lineNumber    = tokenizer.lineno();
							throw new EOFException();
						case '"':
						case ')':
							if (escape) {
								sb.append((char)tokenizer.ttype);
								escape = true;
								break;
							}
							if (start == '(') {
								sb.append(')');
								tokenizer.ordinaryChar('"');
								tokenizer.wordChars   (')', ')');
							}
							initTokenizerEndText();
							lastToken     = sb.toString();
							lastTokenType = TT_STRINGS;
							if (!keyword)
								returnToken = lastToken;
							else if (error)
								moveNextLine("cssparser.warning.invalidstrings", lastToken);
							break READ;
						default:
							{
								value  = tokenizer.sval;
								if (value == null)
									value = String.valueOf(tokenizer.ttype);
								length = value.length();
								for (i = 0; i < length; i++) {
									if (escape)
										escape = false;
									else if (value.charAt(i) == '\\')
										escape = true;
								}
								sb.append(value);
							}
						}
					}
				}
			default:
				lastToken     = String.valueOf((char)tokenizer.ttype);
				lastTokenType = TT_SYNTAX;
				lineNumber    = tokenizer.lineno();
				if (keyword)
					returnToken = lastToken;
				else if (error)
					moveNextLine("cssparser.warning.invalidposition", lastToken);
			}
			
//System.out.println("@["+lastToken+"],["+returnToken+"]");
			return returnToken;
		}
		
		/** ǸΤ⤦֤褦ˤ */
		private void pushBack()
		{
//System.out.println("@pushBack");
			pushBack = true;
		}
		
		/** ֹ֤ */
		private int lineno()
		{
			return lineNumber;
		}
		
		/** Υȡ󤬻ꤵ줿󥿥åΤɤ줫 */
		private boolean isSyntax(String[] syntax)
			throws IOException, EOFException
		{
			if ((token = readToken(true, true)) == null)
				return false;
			
			if (lastTokenType == TT_SYNTAX)
				for (int i = 0; i < syntax.length; i++)
					if (syntax[i].compareTo(token) == 0)
						return true;
			
			moveNextLine("cssparser.warning.invalidposition", token);
			return false;
		}
		
		/** ˸ ";" ľ夫 "}" ľ"{" б "}" ľޤǰư */
		private void moveNextLine(String key, String args)
			throws IOException, EOFException
		{
			if (key != null)
				addWarning(key, (args != null ? new String[]{args} : null), tokenizer.lineno());
			
			for (;;) {
				// ƬФԤ
				while ((token = readToken(true, false)) == null)
					;
				
				// ";" ľذư
				if (token.compareTo(";") == 0)
					return;
				
				// "}" ľذư
				if (token.compareTo("}") == 0) {
					pushBack();
					return;
				}
				
				// "{" б "}" ľذư
				if (token.compareTo("{") == 0) {
					int level = 1;
					for (;;) {
						while ((token = readToken(true, false)) == null)
							;
						if (token.compareTo("{") == 0) {
							level++;
						} else if (token.compareTo("}") == 0) {
							if (--level == 0)
								return;
						}
					}
				}
			}
		}
		
		/** Υ֥åΤ̵ ... { ... }  */
		private void skipNextBlock(String key, String args)
			throws IOException, EOFException
		{
			if (key != null)
				addWarning(key, (args != null ? new String[]{args} : null), tokenizer.lineno());
			
			//  "{" ޤɤФ
			while ((token = readToken(true, false)) == null ||
			       token.compareTo("{") != 0)
				;
			
			pushBack();
			
			// Ƭ "{" ʤΤǡ"}" ޤǳμ¤ɤФƤ
			moveNextLine(null, null);
		}
		
		/** ꥹȤ(ꥹʬʤޤʤ) */
		private String[] getList(boolean comma, boolean keyword)
			throws IOException, EOFException
		{
			Vector list = new Vector();
			boolean nowcomma = false;
			
			for (;;) {
				token = readToken((keyword ? true : comma && nowcomma), false);
				
				// ȡ null ξǤ⡢̵ʸꥹȤϥޤоݤˤ
				if (token == null) {
					if (!comma && !keyword)
						token = lastToken;
					else
						break;
				}
				
				if (comma) {
					if (nowcomma) {
						// ް֤ξ
						if (token.compareTo(",") == 0) {
							nowcomma = false;
							continue;
						}
						// ްʳƤ齪λ
						break;
					} else
						nowcomma = true;
				} else {
					// ޤʤξϡްʳΥ󥿥åǥꥹȤνȤ
					if (lastTokenType == TT_SYNTAX && token.compareTo(",") != 0)
						break;
				}
				
				list.addElement(token);
			}
			
			pushBack();
			
			if (list.size() == 0)
				return null;
			
			String[] values = new String[list.size()];
			list.copyInto(values);
			return values;
		}
		
		/** ٹɲä */
		private void addWarning(String key, String[] args, int lineNumber)
		{
//(new Exception()).printStackTrace(System.out);
			if (reporter != null)
				reporter.report(Reporter.WARNING,
				                Resource.getMessage(key, args),
				                lineNumber, 0);
		}
	}
	
//### EOFException
	private static class EOFException
		extends Exception
	{
		private EOFException()
		{
		}
	}
}
