/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.fclabs.util.Queue;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.token.FormItem;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.CharList;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;

/**
 * ʣʸפθǤޤ (ʸʬ/ʬ)
 */
class Wadge
{
	static final int NORMAL = 0;    // ̵̤ʸ
	static final int BASE   = 1;    // դΥ١ʸ
	static final int RUBY   = 2;    // ʸ
	
	private Caddy     caddy;
	private int       type;
	private Wadge     baseWadge;
	
	private Drawkit   drawkit;
	private Paragraph paragraph;
	private int       preferredHeight;
	
	// üʸꥹ
	private CharList  charsSpinRight;
	private CharList  charsSpinLtrb;
	private CharList  charsKinsokuHead;
	private CharList  charsKinsokuTail;
	private int       TD_UNDERLINE;
	private int       TD_OVERLINE;
	
	// ơ
	private Apply  topApply;
	private Apply  lastApply;
	private Status status;
	private int    offsetRight;
	private int    offsetLeft;
	private int    linePitch;
	private int    letterSpacing;
	
	private boolean lastHankakuEisu = false;  // ǸʸȾѱѿ
	
	// ѤߥХåե
	private StringBuffer committedBuffer = new StringBuffer();
	private StringBuffer committedSpin   = new StringBuffer();
	private int          committedWidth  = 0;
	private int          committedHeight = 0;
	private int          committedOffset = 0;
	private int          committedPitch  = 0;
	
	// ̤Хåե
	private StringBuffer pendingBuffer   = new StringBuffer();
	private StringBuffer pendingSpin     = new StringBuffer();
	private int          pendingWidth    = 0;
	private int          pendingHeight   = 0;
	private int          pendingOffset   = 0;
	private int          pendingPitch    = 0;
	
	/** 󥹥󥹤 */
	Wadge(Caddy caddy, int type, Wadge baseWadge)
	{
		this.caddy     = caddy;
		this.type      = type;
		this.baseWadge = baseWadge;
		
		paragraph       = caddy.paragraph;
		drawkit         = paragraph.box.drawkit;
		preferredHeight = paragraph.preferredHeight;
		
		charsSpinRight   = drawkit.charsSpinRight;
		charsSpinLtrb    = drawkit.charsSpinLtrb;
		charsKinsokuHead = drawkit.charsKinsokuHead;
		charsKinsokuTail = drawkit.charsKinsokuTail;
		if (drawkit.swapDecoration) {
			TD_UNDERLINE = Value.TD_OVERLINE;
			TD_OVERLINE  = Value.TD_UNDERLINE;
		} else {
			TD_UNDERLINE = Value.TD_UNDERLINE;
			TD_OVERLINE  = Value.TD_OVERLINE;
		}
		
		status   = drawkit.status;
		topApply = lastApply = new Apply(status);
		cacheStatus();
	}
	
	/** ʸɲ */
	void append(String text, int begin, int end)
	{
		if (!drawkit.idQueue.isEmpty())
			appendId();
		
		lastApply.length += end - begin;
		
		appendInner(text, begin, end);
	}
	
	/** ʸɲ */
	private void appendInner(String text, int begin, int end)
	{
		// ѿ
		char c, s;
		
		for (int i = begin; i < end; i++) {
			c = text.charAt(i);
			
			// Ԥξ
			if (c == 0x0A) {
				commitPending();
				committedBuffer.append(c);
				committedSpin  .append(Constant.SPIN_BR);
				if (committedWidth == 0) {
					committedWidth = status.fdFontSize.width;
					committedPitch = linePitch;
				}
				newLine(true, 0);
				continue;
			}
			
			// Ⱦѱѿʸ
			if ((0x30 <= c && c <= 0x39) ||
			    (0x41 <= c && c <= 0x5A) ||
			    (0x61 <= c && c <= 0x7A)) {
				if (!lastHankakuEisu)
					commitPending();
				
				appendPending(c, Constant.SPIN_RIGHT);
				lastHankakuEisu = true;
				continue;
			}
			
			lastHankakuEisu = false;
			s = Constant.SPIN_NONE;
			
			// žη
			if ((c <= 0xFF) ||
			    (0xFF61 <= c && c <= 0xFF9F) ||
			    charsSpinRight.contains(c))
				s = Constant.SPIN_RIGHT;
			else if (charsSpinLtrb.contains(c))
				s = Constant.SPIN_LTRB;
			
			// Ƭ§ʸξϡ̤Хåե
			if (charsKinsokuHead.contains(c)) {
				appendPending(c, s);
				continue;
			}
			
			// §ʸξϡߥåȤƳХåե
			if (charsKinsokuTail.contains(c)) {
				commitPending();
				appendCommit(c, s);
				continue;
			}
			
			// ̤ʸξ硢ߥåȤ̤Хåե
			commitPending();
			appendPending(c, s);
		}
	}
	
	/** ߤιԤɲäǤ뤫å */
	private boolean canAppend(int h)
	{
		if (status.whiteSpace != Value.WHITESPACE_NORMAL)
			return true;
		
		return (paragraph.lineHeight + paragraph.floatHeight +
		        committedHeight + pendingHeight + h <= preferredHeight);
	}
	
	/** ԤԤ */
	private void newLine(boolean br, int newHeight)
	{
		if (type == NORMAL) {
			if (committedWidth == 0) {
				paragraph.appendLastLine(br, newHeight);
				return;
			}
			
			paragraph.lineWidth  =  Math.max(paragraph.lineWidth , committedWidth );
			paragraph.lineHeight += committedHeight;
			paragraph.lineOffset =  Math.max(paragraph.lineOffset, committedOffset);
			paragraph.linePitch  =  Math.max(paragraph.linePitch , committedPitch );
			paragraph.appendLastLine(br, newHeight);
			
			committedWidth  = 0;
			committedHeight = 0;
			committedOffset = 0;
			committedPitch  = 0;
		} else {
			// ƬϤޤäƤƤʤΤߡʸǲ
			if (paragraph.lineHeight != 0)
				paragraph.appendLastLine(br, newHeight);
		}
	}
	
	/** ѤߥХåեɲ */
	private void appendCommit(char c, char s)
	{
		int h = calculateHeight(c, s);
		
		if (!canAppend(h))
			newLine(false, h);
//System.out.println("["+c+"],["+(paragraph.lineHeight + pendingHeight + h)+"]");
		
		committedBuffer.append(c);
		committedSpin  .append(s);
		committedWidth  =  calculateWidth(c, s, committedWidth);
		committedHeight += h;
		committedOffset =  Math.max(committedOffset, offsetLeft);
		committedPitch  =  Math.max(committedPitch , linePitch );
	}
	
	/** ̤Хåեɲ */
	private void appendPending(char c, char s)
	{
		int h = calculateHeight(c, s);
		
		if (!canAppend(h))
			newLine(false, pendingHeight + h);
//System.out.println("["+c+"],["+(paragraph.lineHeight + pendingHeight + h)+"]");
		
		pendingBuffer.append(c);
		pendingSpin  .append(s);
		pendingWidth  =  calculateWidth(c, s, pendingWidth);
		pendingHeight += h;
		pendingOffset =  Math.max(pendingOffset, offsetLeft);
		pendingPitch  =  Math.max(pendingPitch , linePitch );
	}
	
	/** եȤ֤ޤ */
	private int calculateWidth(char c, char s, int maxWidth)
	{
		switch (s) {
		case Constant.SPIN_RIGHT:
		case Constant.SPIN_LTRB :
			return Math.max(maxWidth,
			                ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F)
			                 ? status.fontHeightHalf
			                 : status.fdFontSize.width) + offsetRight);
		default:
			return Math.max(maxWidth, status.fdFontSize.width + offsetRight);
		}
	}
	
	/** եȤι⤵֤ޤ */
	private int calculateHeight(char c, char s)
	{
		switch (s) {
		case Constant.SPIN_RIGHT:
		case Constant.SPIN_LTRB :
			return status.fm.charWidth(c) + letterSpacing;
		default:
			return status.fdFontSize.height + letterSpacing;
		}
	}
	
	/** ɲ */
	void append(String src, String alt, Value width, Value height, int border, int floatType)
	{
		if (!drawkit.idQueue.isEmpty())
			appendId();
		
		ImageHolder ih = new ImageHolder(drawkit, status, src, alt, width, height, border, floatType);
		
		char index = 0;
		synchronized (drawkit.images) {
			index = (char)drawkit.images.size();
			drawkit.images.addElement(ih);
		}
		
		// ɲý
		lastApply.length++;
		
		appendInner(ih, index);
	}
	
	/** ɲ */
	private void appendInner(ImageHolder ih, char index)
	{
		commitPending();
		
		if (ih.floatType != Value.FLOAT_NONE) {
			paragraph.appendFloatBlock(ih.position.width  + ih.border * 2,
			                           ih.position.height + ih.border * 2,
			                           ih.floatType, ih.floatOffset,
			                           committedHeight);
			committedBuffer.append(index);
			committedSpin  .append(Constant.SPIN_IMAGE);
			return;
		}
		
		int h = ih.position.height + ih.border * 2 + letterSpacing;
		
		if (!canAppend(h))
			newLine(false, h);
		
		committedBuffer.append(index);
		committedSpin  .append(Constant.SPIN_IMAGE);
		committedWidth  =  Math.max(committedWidth , ih.position.width + ih.border * 2 + offsetRight);
		committedHeight += h;
		committedOffset =  Math.max(committedOffset, offsetLeft - (((status.decoration & TD_UNDERLINE) != 0) ? Constant.LINE_GAP: 0));
		committedPitch  =  Math.max(committedPitch , linePitch );
	}
	
	/** եȤɲ */
	void append(Box box)
	{
		char index = 0;
		synchronized (drawkit.floats) {
			index = (char)drawkit.floats.size();
			drawkit.floats.addElement(box);
		}
		
		// ɲý
		lastApply.length++;
		
		appendInner(box, index);
	}
	
	/** եȤɲ */
	void appendInner(Box box, char index)
	{
		commitPending();
		
		paragraph.appendFloatBlock(box.drawWidth, box.drawHeight,
		                           box.status.floatType, box.floatOffset,
		                           committedHeight);
		
		committedBuffer.append(index);
		committedSpin  .append(Constant.SPIN_FLOAT);
	}
	
	/** եॢƥɲ */
	void append(FormItem item)
	{
		char index = 0;
		synchronized (drawkit.formItems) {
			index = (char)drawkit.formItems.size();
			drawkit.formItems.addElement(item);
		}
		
		// ɲý
		lastApply.length++;
		
		appendInner(item, index);
	}
	
	/** եॢƥɲ */
	void appendInner(FormItem item, char index)
	{
		commitPending();
		
		Dimension size = item.getSize();
		
		if (!canAppend(size.height))
			newLine(false, size.height);
		
		committedBuffer.append(index);
		committedSpin  .append(Constant.SPIN_FORM);
		committedWidth  =  Math.max(committedWidth , size.width + offsetRight);
		committedHeight += size.height;
		committedOffset =  Math.max(committedOffset, offsetLeft);
		committedPitch  =  Math.max(committedPitch , linePitch );
	}
	
	/** ID ֤ɲ */
	private void appendId()
	{
		String id;
		while ((id = (String)drawkit.idQueue.get()) != null) {
			appendInner((char)drawkit.ids.size());
			drawkit.ids.addElement(id);
			lastApply.length++;
		}
	}
	
	/** ID ֤ɲ */
	private void appendInner(char index)
	{
		pendingBuffer.append(index);
		pendingSpin  .append(Constant.SPIN_ID);
	}
	
	/** ơѹ줿 */
	void statusChanged()
	{
		status    = drawkit.status;
		lastApply = lastApply.next = new Apply(status);
		cacheStatus();
	}
	
	/** ơͤ׻ݻ */
	private void cacheStatus()
	{
		offsetRight = Math.max(status.offsetX, 0);
		if ((status.decoration & TD_OVERLINE ) != 0)
			offsetRight += Constant.LINE_GAP;
		
		offsetLeft  = Math.max(-status.offsetX, 0);
		if ((status.decoration & TD_UNDERLINE) != 0)
			offsetLeft  += Constant.LINE_GAP;
		
		linePitch = 0;
		if (type != RUBY && status.lineHeight != null)
			linePitch = status.lineHeight.getValue(status.fd, status.fdFontSize.width, Value.DATA_HORIZONTAL) - status.fdFontSize.width;
		
		letterSpacing = 0;
		if (type != RUBY && status.letterSpacing != null)
			letterSpacing = status.letterSpacing.getValue(status.fd, status.fdFontSize.width, Value.DATA_VERTICAL);
		
		lastApply.offsetRight   = offsetRight;
		lastApply.offsetLeft    = offsetLeft;
		lastApply.linePitch     = linePitch;
		lastApply.letterSpacing = letterSpacing;
	}
	
	/** ̤ʬ */
	void commitPending()
	{
		if (pendingBuffer.length() == 0)
			return;
		
		committedBuffer.append(pendingBuffer.toString());
		committedSpin  .append(pendingSpin  .toString());
		committedWidth  =  Math.max(committedWidth , pendingWidth );
		committedHeight += pendingHeight;
		committedOffset =  Math.max(committedOffset, pendingOffset);
		committedPitch  =  Math.max(committedPitch , pendingPitch );
		
		pendingBuffer.setLength(0);
		pendingSpin  .setLength(0);
		pendingWidth  = 0;
		pendingHeight = 0;
		pendingOffset = 0;
		pendingPitch  = 0;
	}
	
	/** ٤Ƴ */
	void commit()
	{
		commitPending();
		
		switch (type) {
		case NORMAL:
		case BASE:
			caddy.width  = committedWidth;
			caddy.height = committedHeight;
			caddy.offset = committedOffset;
			caddy.pitch  = committedPitch;
			break;
		case RUBY:
			caddy.width  += committedWidth;
			caddy.width  += (int)(baseWadge.committedWidth * Constant.RUBY_PITCH);
			caddy.width  += committedOffset;
			caddy.height =  Math.max(caddy.height, committedHeight);
			break;
		}
//System.out.println("Wadge.commit["+committedBuffer+"]");
	}
	
	/** Ԥ */
	void rearrange(int newHeight)
	{
		preferredHeight = newHeight;
		
		int length = committedBuffer.length();
		if (length == 0)
			return;
		
		String buffer = committedBuffer.toString();
		String spin   = committedSpin  .toString();
		
		committedBuffer.setLength(0);
		committedSpin  .setLength(0);
		committedWidth  = 0;
		committedHeight = 0;
		committedOffset = 0;
		committedPitch  = 0;
		pendingBuffer  .setLength(0);
		pendingSpin    .setLength(0);
		pendingWidth    = 0;
		pendingHeight   = 0;
		pendingOffset   = 0;
		pendingPitch    = 0;
		
		Apply apply = topApply;
		int   applyLength = apply.length;
		char  index, s;
		int   end, num = 0;
		Box   box;
		
		for (int i = 0; i < length;) {
			// Apply ƬФԤ
			while ((applyLength = apply.length) == 0)
				apply = apply.next;
			status        = apply.status;
			offsetRight   = apply.offsetRight;
			offsetLeft    = apply.offsetLeft;
			linePitch     = apply.linePitch;
			letterSpacing = apply.letterSpacing;
			
			// ID ľǤɬ statusChanged ƤΤǡ
			// ID ֤ϡɬ
			
			end = i + applyLength;
//System.out.println("@["+spin.substring(i, end)+"]");
			s   = spin.charAt(end - 1);
			switch (s) {
			case Constant.SPIN_IMAGE:
			case Constant.SPIN_FLOAT:
			case Constant.SPIN_FORM :
				end--;
				break;
			case Constant.SPIN_ID:
				do {
					end--;
					num++;
				} while ((s = spin.charAt(end - 1)) == Constant.SPIN_ID);
				break;
			}
			
			// ʸɲ
			if (i < end)
				appendInner(buffer, i, end);
			
			// ID ɲ
			switch (s) {
			case Constant.SPIN_IMAGE:
				index = buffer.charAt(end);
				appendInner((ImageHolder)drawkit.images.elementAt(index), index);
				break;
			case Constant.SPIN_ID:
				do {
					appendInner(buffer.charAt(end));
					end++;
					num--;
				} while (num > 0);
				break;
			case Constant.SPIN_FLOAT:
				index = buffer.charAt(end);
				box   = (Box)drawkit.floats.elementAt(index);
				box.rearrange(box.resolvePercent(newHeight), newHeight);
				appendInner(box, index);
				break;
			case Constant.SPIN_FORM:
				index = buffer.charAt(end);
				appendInner((FormItem)drawkit.formItems.elementAt(index), index);
				break;
			}
			i += applyLength;
			applyLength = 0;
			apply = apply.next;
		}
		
		commit();
	}
	
	/**  */
	void draw(Position p, int maxHeight)
	{
//System.out.println("Wadge.draw["+committedBuffer+"]");
		int length = committedBuffer.length();
		if (length == 0)
			return;
		
		// 
		Graphics g = drawkit.g;
		status = null;
		
		int begin;                              // Хåեγϰ
		int x;                                  // Ѥ x 
		int y1, y2;                             // ʸϰ
		int w, h;                               // Ʊʸ⤵
		char c, s;                              // ʸʸ
		int    startx = p.x;                    // Ϥ x 
		double space  = 0;                      // ʸ
		
		if (committedHeight < maxHeight)
			space = (double)(maxHeight - committedHeight) / length / 2;
		
		// Ӥξ硢y 򳫻ϰ֤᤹
		if (type == RUBY) {
			p.x += baseWadge.committedWidth;
			p.x += (int)(baseWadge.committedWidth * Constant.RUBY_PITCH);
			p.y -= maxHeight;
		}
		
		// 
		ImageHolder ih = null;
		
//System.out.println("@["+committedBuffer+"]");
//System.out.println("@["+committedSpin  +"]");
		for (int i = 0; i < length;) {
if (p.y > p.height) {
System.out.println("### Wadge.draw.newLine??? ["+p.y+","+p.height+"]");
System.out.println("### ["+(i-Math.max(i - 10, 0))+"],["+committedBuffer.toString().substring(Math.max(i - 10, 0), Math.min(i + 10, length))+"]");
}
			// Apply ƬФԤ
			while (topApply.length == 0)
				topApply = topApply.next;
			if (status != topApply.status) {
				status        = topApply.status;
				letterSpacing = topApply.letterSpacing;
				g.setColor(status.foreColor);
				g.setFont (status.font     );
			}
			
			begin = i;
			c  = committedBuffer.charAt(i);
			s  = committedSpin  .charAt(i);
			if (s == Constant.SPIN_IMAGE)
				ih = (ImageHolder)drawkit.images.elementAt(c);
			
			if (p.y >= p.height &&
			    s != Constant.SPIN_ID    &&
			    s != Constant.SPIN_FLOAT &&
			    !(s == Constant.SPIN_IMAGE && ih.floatType != Value.FLOAT_NONE))
				p.newLine();
			
			y1 = p.y;
			x  = p.x + status.offsetX;
//System.out.println("["+c+"]");
			
			switch (s) {
			case Constant.SPIN_RIGHT:
			case Constant.SPIN_LTRB :
				// žʸ
				h = 0;
				w = status.fontHeightMax;
				do {
					h +=  status.fm.charWidth(committedBuffer.charAt(i));
					h += (int)(((i * 2) + 2) * space) - (int)((i * 2) * space);
					h += letterSpacing;
					topApply.length--; i++;
					if (p.y + h >= p.height)
						break;
				} while (i < length && topApply.length > 0 && committedSpin.charAt(i) == s);
				drawkit.drawTextWithSpin(committedBuffer, begin, i, x, p.y, h, w,
				                         status, space, letterSpacing, s);
				p.y += h;
				break;
			case Constant.SPIN_IMAGE:
				// 
				if (ih.floatType == Value.FLOAT_NONE) {
					p.y += (int)(((i * 2) + 1) * space) - (int)((i * 2) * space);
					ih.draw(x, p.y);
					p.y += ih.position.height + ih.border * 2;
					p.y += (int)(((i * 2) + 2) * space) - (int)(((i * 2) + 1) * space);
					p.y += letterSpacing;
				} else {
					ih.floatDelayX = p.startX - ih.floatOffset.x - ih.position.width - ih.border * 2;
					ih.floatDelayY = p.startY + ih.floatOffset.y;
					if (drawkit.floatQueue == null)
						drawkit.floatQueue = new Queue();
					drawkit.floatQueue.put(ih);
				}
				topApply.length--; i++;
				continue;
			case Constant.SPIN_BR:
				topApply.length--; i++;
				continue;
			case Constant.SPIN_ID:
				drawkit.itemMap.addId((String)drawkit.ids.elementAt(c),
				                      drawkit.size.width - (x + p.width + Constant.ID_GAP), 0);
				topApply.length--; i++;
				continue;
			case Constant.SPIN_FLOAT:
				{
					Box box = (Box)drawkit.floats.elementAt(c);
					box.floatDelayX = p.startX - box.floatOffset.x;
					box.floatDelayY = p.startY + box.floatOffset.y;
					if (drawkit.floatQueue == null)
						drawkit.floatQueue = new Queue();
					drawkit.floatQueue.put(box);
				}
				topApply.length--; i++;
				continue;
			case Constant.SPIN_FORM:
				{
					FormItem item  = (FormItem)drawkit.formItems.elementAt(c);
					Dimension size = item.getSize();
					
					if (item.isNeedBackImage()) {
						Image    image = drawkit.context.createImage(size.width, size.height);
						Graphics og    = image.getGraphics();
						drawkit.drawImageSync(og, drawkit.panel,
						                      0  , 0  , size.width      , size.height      ,
						                      p.x, p.y, size.width + p.x, size.height + p.y);
						og.dispose();
						item.setBackImage(image);
					}
					
					item.show(drawkit.size.width - (x + item.getSize().width + 1), y1);
					p.y += size.height;
				}
				topApply.length--; i++;
				continue;
			default:
				// ̾νĽʸ
				h = 0;
				w = status.fdFontSize.width;
				do {
					h +=  status.fdFontSize.height;
					h += (int)(((i * 2) + 2) * space) - (int)((i * 2) * space);
					h += letterSpacing;
					topApply.length--; i++;
					if (p.y + h >= p.height)
						break;
				} while (i < length && topApply.length > 0 && committedSpin.charAt(i) == s);
				drawkit.drawTextTate(committedBuffer, begin, i, x, p.y, status, space, letterSpacing);
				p.y += h;
			}
			
			y2 = p.y;
			// 
			if (status.href != null)
				drawkit.itemMap.addHref(drawkit.size.width - (x + w + 1), y1,
				                        drawkit.size.width - x          , y2,
				                        status.href);
			
			// TIP 
			if (status.tip != null)
				drawkit.itemMap.addTip(drawkit.size.width - (x + w + 1), y1,
				                       drawkit.size.width - x          , y2,
				                       status.tip);
			
			// ʸ
			if (status.decoration != 0 && type != RUBY) {
				int right = x;
				
				// 
				if ((status.decoration & TD_UNDERLINE) != 0) {
					right = x - Constant.LINE_GAP;
					g.drawLine(right, y1, right, y2);
				}
				// 
				if ((status.decoration & TD_OVERLINE ) != 0) {
					right = x + status.fdFontSize.width + Constant.LINE_GAP - 1;
					g.drawLine(right, y1, right, y2);
				}
				// Ǥä
				if ((status.decoration & Value.TD_STRIKE   ) != 0) {
					right = x + (int)(status.fdFontSize.width * Constant.STRIKE_GAP);
					g.drawLine(right, y1, right, y2);
				}
			}
		}
		
		// Ӥξϡx ưƤΤǸ᤹
		if (type == RUBY)
			p.x = startx;
	}
	
	/**
	 * ʸɽ֤ޤ
	 * 
	 * @return    ʸɽ
	 */
	public String toString()
	{
		return committedBuffer.toString();
	}
}
