/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.Reporter;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.image.RotateFilter;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.Environment;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.FilteredImageSource;
import java.awt.image.ImageObserver;
import java.net.URL;

/**  */
class ImageHolder
	implements DelayRectangle
{
	private static final int IMAGE_WAIT     = 60000;         // Ԥ
	private static final int IMG_WIDTH      =    28;         // 
	private static final int IMG_HEIGHT     =    28;         // ι⤵
	private static final int D_IMG_PITCH    =     4;         // ߡ
	
	private static final int TYPE_UNKNOWN = 0;
	private static final int TYPE_GIF     = 1;
	private static final int TYPE_JPEG    = 2;
	
	// ߡݻƤ
	private static Image     dummyImage     = null;  // ǥեȲ
	private static Object    dummyImageLock = new Object();
	private static Dimension dummyImageSize = null;
	
	private HawkContext context;
	private Drawkit     drawkit;
	private Status      status;
	
	Rectangle position;
	int       border;
	int       floatType;
	Point     floatOffset;
	int       floatDelayX;
	int       floatDelayY;
	
	private URL         url;
	private String      src;
	private String      alt;
	private Observer    observer;
	private Image       original;
	private Image       image;
	private Dimension   imageSize;
	private Graphics    g;
	private Counter     gCounter;
	private int         offset; // ºݤβ̤Ф뱦ΰ
	private boolean     sameSize = false;     // ꥵȼºݤβƱ
	private boolean     hasError = false;
	private boolean     spin     = false;
	
	boolean complete = false;
	
	/** 󥹥󥹤 */
	ImageHolder(Drawkit drawkit, Status status, String src, String alt,
	            Value width, Value height, int border, int floatType)
	{
		this.context   = drawkit.context;
		this.drawkit   = drawkit;
		this.status    = status;
		this.url       = drawkit.createURL(src);
		this.src       = src;
		this.alt       = alt;
		this.observer  = new Observer();
		this.imageSize = new Dimension(0, 0);
		this.position  = new Rectangle(0, 0,
		                               (width  == null || width .getUnit() == Value.UNIT_PERCENT
		                                ? 0 : width .getNumber().intValue()),
		                               (height == null || height.getUnit() == Value.UNIT_PERCENT
		                                ? 0 : height.getNumber().intValue()));
		this.image     = drawkit.getImage(url, src);
		this.border    = border;
		this.floatType = floatType;
		if (floatType != Value.FLOAT_NONE)
			floatOffset = new Point(0, 0);
		
		Option option = context.getOption();
		int    type   = TYPE_UNKNOWN;
		
		// ž뤫ɤå
		if ((option.getSpinGifImage() || option.getSpinJpegImage()) && url != null) {
			String ct = ImageCache.getContentType(url);
			if (ct != null) {
				if (ct.startsWith("image/gif"))
					type = TYPE_GIF;
				else if (ct.startsWith("image/jpeg") || ct.startsWith("image/jpg"))
					type = TYPE_JPEG;
			}
			
			spin = ((option.getSpinGifImage () && type == TYPE_GIF ) ||
			        (option.getSpinJpegImage() && type == TYPE_JPEG));
		}
		
		// ΰ֤
		if (image != null) {
			// ̡˥å򤫤ʤȡimageUpdate ˤޤʤ
			synchronized (position) {
				if (position.width  <= 0) position.width  = image.getWidth (observer);
			}
			synchronized (position) {
				if (position.height <= 0) position.height = image.getHeight(observer);
			}
			
			// ꤵƤʤϡɤ߹
			synchronized (position) {
				if ((!hasError) && (position.width < 0 || position.height < 0)) {
					try {
						position.wait(IMAGE_WAIT);
					} catch (InterruptedException e) {
						throw new StopException("wait to get a position");
					}
				}
			}
		}
		
		// ʬʤä
		if (position.width <= 0 || position.height <= 0) {
			if (position.width  <= 0) position.width  = IMG_WIDTH;
			if (position.height <= 0) position.height = IMG_HEIGHT;
			image = null;
		} else {
			// % ξν
			if (width  != null && width .getUnit() == Value.UNIT_PERCENT)
				position.width  = (position.width  * width .getNumber().intValue()) / 100;
			if (height != null && height.getUnit() == Value.UNIT_PERCENT)
				position.height = (position.height * height.getNumber().intValue()) / 100;
			
			// ΨˤäƲž
			if (spin) {
				int    minWidth    = 0;
				double aspectRatio = 0;
				switch (type) {
				case TYPE_GIF : minWidth = option.getSpinGifMinWidth (); aspectRatio = option.getSpinGifAspectRatio (); break;
				case TYPE_JPEG: minWidth = option.getSpinJpegMinWidth(); aspectRatio = option.getSpinJpegAspectRatio(); break;
				}
				
				if (position.width >= minWidth &&
				    !(position.width / (double)position.height > aspectRatio ||
				      position.height / (double)position.width > aspectRatio) )
					spin = false;
			}
			
			// ž
			if (spin) {
				// ѹ
				int n = position.width;
				position.width  = position.height;
				position.height = n;
				
				observer.enabled = false;
				observer = new Observer();
				
				image = context.createImage(new FilteredImageSource(image.getSource(),
				                                                    new RotateFilter()));
			}
		}
	}
	
	
	/**  */
	void draw(int x, int y)
	{
		g          = drawkit.g;
		gCounter   = drawkit.gCounter;
		position.x = x + border;
		position.y = y + border;
		offset     = drawkit.size.width - position.x - position.width;
		
		gCounter.increment();
		drawkit.imageCounter.increment();
		
		// ܡ
		if (border > 0) {
			g.setColor(status.foreColor);
			for (int i = 1; i <= border; i++)
				g.drawRect(position.x - i,
				           position.y - i,
				           position.width  + i * 2 - 1,
				           position.height + i * 2 - 1);
		}
		
		// 
		if (status.href != null)
			drawkit.itemMap.addHref(offset - border, y,
			                       offset + position.width + border, position.y + position.height + border,
			                       status.href);
		
		// ־
		if (url != null)
			drawkit.itemMap.addImage(offset                 , position.y                  ,
			                        offset + position.width, position.y + position.height,
			                        url);
		
		// TIP 
		if (alt != null || status.tip != null)
			drawkit.itemMap.addInfo(offset                 , position.y                  ,
			                       offset + position.width, position.y + position.height,
			                       (alt != null ? alt : status.tip));
		
		if (image != null) {
			// 󥿡쥹 + ƩβեѤˡߤΰХåå
			original    = context.createImage(position.width, position.height);
			Graphics og = original.getGraphics();
			og.drawImage(drawkit.panel,
			             0, 0, position.width, position.height,
			             position.x                 , position.y                  ,
			             position.x + position.width, position.y + position.height,
			             observer);
			og.dispose();
			if (drawImage(image))
				dispose();
		} else {
			drawDummy();
			dispose();
		}
	}
	
	/** ٱ򳫻 */
	public void drawDelay()
	{
		draw(floatDelayX, floatDelayY);
	}
	
	/**  */
	private boolean drawImage(Image image)
	{
		try {
			return g.drawImage(image, position.x, position.y, position.width, position.height, observer);
		} catch (Exception e) {
			// JVM μˤäƤϡʥե Exception ȯ
			drawkit.reportMessage(Reporter.WARNING, "render.warning.image.draw", new String[]{src, e.toString()});
			image = null;
		}
		return true;
	}
	
	/** ɤνλ */
	private synchronized void loaded()
	{
		if (!complete) {
			synchronized (drawkit.imageCounter) {
				drawkit.imageCounter.decrement();
				drawkit.imageCounter.notify   ();
			}
			complete = true;
		}
	}
	
	/** ꥽ */
	synchronized void dispose()
	{
		loaded();
		if (g != null) {
			if (gCounter.decrement() <= 0)
				g.dispose();
			
			if (original != null)
				original.flush();
			
			g = null;
		}
	}
	
	/** ߡɽ */
	private void overwriteDummy()
	{
		if (g == null)
			return;
		
		// ߤ褵Ƥ뤫⤷ʤΤǡβǾ
		if (original != null)
			drawkit.drawImageSync(g, original, position.x, position.y, position.width, position.height);
		
		drawDummy();
		context.repaint(offset, position.y, position.width, position.height);
	}
	
	/** ߡɽ */
	private synchronized void drawDummy()
	{
		// draw, imageUpdate ƤФǽΤǡǤ dispose Ƥ뤫⤷ʤ
		if (g == null)
			return;
		
		// ߡ򥭥åɤ߹
		synchronized (dummyImageLock) {
			if (dummyImage == null) {
				dummyImage = Resource.getImageResource("render.image.notfound", context.getToolkit());
				if (dummyImage == null)
					return;
				
				drawkit.loadImage(dummyImage, "wait to load a dummy image");
				
				dummyImageSize = new Dimension(dummyImage.getWidth (null),
				                               dummyImage.getHeight(null));
			}
		}
		
		// Ȥ
		g.setColor(Color.black);
		g.drawLine(position.x                     , position.y                      ,
		           position.x + position.width - 1, position.y                      );
		g.drawLine(position.x                     , position.y                      ,
		           position.x                     , position.y + position.height - 1);
		g.setColor(Color.lightGray);
		g.drawLine(position.x + position.width - 1, position.y + 1                  ,
		           position.x + position.width - 1, position.y + position.height - 1);
		g.drawLine(position.x + 1                 , position.y + position.height - 1,
		           position.x + position.width - 1, position.y + position.height - 1);
		
		// ߡ
		int w = Math.min(dummyImageSize.width , position.width  - D_IMG_PITCH);
		int h = Math.min(dummyImageSize.height, position.height - D_IMG_PITCH);
		drawkit.drawImageSync(g, dummyImage,
		                     position.x + D_IMG_PITCH    , position.y + D_IMG_PITCH    ,
		                     position.x + D_IMG_PITCH + w, position.y + D_IMG_PITCH + h,
		                     0, 0, w, h);
	}
	
	/**
	 * ۥʸɽ֤ޤ
	 * 
	 * @return    ʸɽ
	 */
	public String toString()
	{
		if (url != null)
			return url.toString();
		return src;
	}
	
//### Observer
	/** ᡼򹹿 */
	private class Observer
		implements ImageObserver
	{
		private boolean enabled = true;
		
		/** 󥹥󥹤 */
		private Observer()
		{
		}
		
		/** ᡼ */
		public synchronized boolean imageUpdate(Image img, int infoflags,
		                                        int x, int y, int width, int height)
		{
//System.out.println("src="+src+",alt="+alt+",iu="+infoflags+","+x+","+y+","+width+","+height);
			if (!enabled)
				return false;
			
			// 顼λ
			if ((infoflags & (ImageObserver.ABORT | ImageObserver.ERROR)) != 0) {
				synchronized (position) {
					hasError = true;
					if (position.width <= 0 || position.height <= 0)
						position.notify();
				}
				if ((infoflags & ImageObserver.ERROR) != 0) {
					overwriteDummy();
					drawkit.reportMessage(Reporter.WARNING, "render.warning.image.load.error", new String[]{src});
				} else {
					drawkit.reportMessage(Reporter.WARNING, "render.warning.image.load.abort", new String[]{src});
				}
				dispose();
				
				return false;
			}
			
			// 
			if ((infoflags & (ImageObserver.WIDTH | ImageObserver.HEIGHT)) != 0) {
				synchronized (position) {
					if (((infoflags & ImageObserver.WIDTH ) != 0) && (position.width  <= 0))
						position.width  = width;
					if (((infoflags & ImageObserver.HEIGHT) != 0) && (position.height <= 0))
						position.height = height;
					
					if (position.width > 0 && position.height > 0)
						position.notify();
				}
				
				// 󥿡쥹ɽ뤫ɤꤹ뤿ˡºݤβ¸
				if ((infoflags & ImageObserver.WIDTH ) != 0) imageSize.width  = width ;
				if ((infoflags & ImageObserver.HEIGHT) != 0) imageSize.height = height;
				sameSize = (imageSize.width  == position.width &&
				            imageSize.height == position.height);
			}
			
			// ʹߤ
			if (g != null) {
				try {
					// ʬ
					if ((infoflags & ImageObserver.SOMEBITS) != 0) {
						if (sameSize) {
							// 󥿡쥹 + ƩβեѤˡХååפ᤹
							if (original != null)
								g.drawImage(original,
								            position.x + x        , position.y + y         ,
								            position.x + x + width, position.y + y + height,
								            x, y, x + width, y + height,
								            observer);
							g.drawImage(img,
							            position.x + x        , position.y + y         ,
							            position.x + x + width, position.y + y + height,
							            x, y, x + width, y + height,
							            observer);
							context.repaint(offset + (position.width - (x + width)), position.y + y, width, height);
						}
					}
					
					// 
					if ((infoflags & (ImageObserver.FRAMEBITS | ImageObserver.ALLBITS)) != 0) {
						g.drawImage(img, position.x, position.y, position.width, position.height, observer);
						context.repaint(offset, position.y, position.width, position.height);
					}
					
					// ˥᡼ΰ쥳ޤ轪λ
					if ((infoflags & ImageObserver.FRAMEBITS) != 0)
						loaded();
					
					// λ
					if ((infoflags & ImageObserver.ALLBITS) != 0)
						dispose();
				} catch (Exception e) {
					// JVM μˤäƤϡʥե Exception ȯ
					drawkit.reportMessage(Reporter.WARNING, "render.warning.image.draw", new String[]{src, e.toString()});
					overwriteDummy();
					dispose();
					return false;
				}
			}
			
			return ((infoflags & (ImageObserver.ALLBITS | ImageObserver.ABORT | ImageObserver.ERROR)) == 0);
		}
	}
}
