/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.Reporter;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.build.Parser;
import net.hizlab.kagetaka.build.ParserManager;
import net.hizlab.kagetaka.build.TokenCoordinator;
import net.hizlab.kagetaka.io.CounterInputStream;
import net.hizlab.kagetaka.io.CounterListener;
import net.hizlab.kagetaka.token.EndToken;
import net.hizlab.kagetaka.token.StartToken;
import net.hizlab.kagetaka.token.Style;
import net.hizlab.kagetaka.token.TextToken;
import net.hizlab.kagetaka.token.Token;
import net.hizlab.kagetaka.token.TokenTypes;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.Charset;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.util.StringUtils;

import java.awt.Cursor;
import java.awt.Image;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.text.ParseException;

/**
 * 襨󥸥Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.5 $
 */
public class HawkEngine
{
	private static final ContentType DEFAULT_CONTENT_TYPE = new ContentType("text", "html");
	
	private static final int LT_TEXT   = 0;
	private static final int LT_HTML   = 1;
	private static final int LT_IMAGE  = 2;
	private static final int LT_BINARY = 3;
	
	private final HawkContext context;
	private final Reporter    reporter;
	private final Option      option;
	
	private Object         threadLock = new Object();
	private LoadThread     oldThread  = null;
	private LoadThread     loadThread = null;
	private EngineListener listener   = null;
	private Object         renderLock = new Object();
	private Render         render     = null;
	
	/**
	 * 襨󥸥ޤ
	 * 
	 * @param     context 륳ƥ
	 */
	public HawkEngine(HawkContext context)
	{
		this.context  = context;
		this.reporter = context.getReporter();
		this.option   = context.getOption  ();
	}
	
	/**
	 * 󥸥ꥹʤϿޤ
	 * 
	 * @param     listener 󥸥ꥹ
	 */
	public void setEngineListener(EngineListener listener)
	{
		this.listener = listener;
	}
	
	/**
	 * ѤΥǡ򡢻ꤵ줿ѡɤ߹ߡԤޤ
	 * 
	 * @param     index   ƤӽФѥǥå
	 * @param     realUrl ɤ߹Υѥ
	 * @param     pd      POST Ϥǡ
	 *                    POST ǤϤʤ <code>null</code>
	 */
	public synchronized void load(int index, URL realUrl, PostData pd)
	{
		if (isLoading())
			stop();
		
		synchronized (threadLock) {
			if (oldThread == null)
				oldThread = loadThread;
		}
		
		// ɤ߹ߥå
		loadThread = new LoadThread(index, realUrl, pd);
		
		loadThread.start();
	}
	
	/**
	 * ߥνߤޤ
	 * ˤäƤϡߤǤ櫓ǤϤޤ
	 */
	public synchronized void stop()
	{
		if (isLoading())
			loadThread.interrupt();
	}
	
	/**
	 * ߥ椫ɤ֤ޤ
	 * 
	 * @return    ξ <code>true</code>
	 *            ʳξ <code>false</code>
	 */
	public synchronized boolean isLoading()
	{
		return (loadThread != null && loadThread.isAlive());
	}
	
	/**
	 * Τ٤ƤΥ꥽˴ޤ
	 */
	public synchronized void dispose()
	{
		if (isLoading())
			loadThread.interrupt();
		
		synchronized (renderLock) {
			if (render != null)
				render.dispose();
		}
	}
	
//### LoadThread
	/** ɤ߹߽Ԥå */
	private class LoadThread
		extends Thread
	{
		private int      index;
		private URL      openUrl;
		private URL      realUrl;
		private PostData pd;
		private boolean  active = true;
		
		/** 󥹥󥹤 */
		private LoadThread(int index, URL path, PostData pd)
		{
			this.index   = index;
			this.openUrl = path;
			this.realUrl = path;
			this.pd      = pd;
		}
		
		/** ºݤν */
		public void run()
		{
			int           loadType      = LT_HTML;
			boolean       noerror       = false;
			URLConnection connection    = null;
			InputStream   is            = null;
			ContentType   contentType   = DEFAULT_CONTENT_TYPE;
			String        encoding      = null;
			int           contentLength = -1;
			
			if (active)
				context.setCursor(Cursor.WAIT_CURSOR);
			
			RUN:
			{
				// ΥåɤƤߤԤ
				try {
					synchronized (threadLock) {
						if (oldThread != null) {
							oldThread.join();
							oldThread = null;
						}
					}
				} catch (InterruptedException e) {
					setStatus("engine.warning.stop", new String[]{"old thread stop"}, Reporter.WARNING);
					break RUN;
				}
				
				// ե򳫤³Ƽ
				OPENING:
				{
					setStatus("engine.status.load.open", new String[]{realUrl.toString()}, Reporter.INFO);
					noerror = false;
					if (listener != null)
						listener.connecting(index);
					
					// ɤ߹褬åˤʤå */
					try {
						if (ImageCache.getImage(context, realUrl, realUrl.toString(), false) != null) {
							loadType = LT_IMAGE;
							noerror  = true;
							break OPENING;
						}
					} catch (IOException e) {/* ȯʤ */}
					
					// URL 򥪡ץ󤷤Ƥߤ
					try {
						if ((connection = context.getURLConnection(realUrl, pd)) == null) {
							setStatus(null, null, 0);
							break OPENING;
						}
						// 쥯Ȥ줿ǽΤǡURL Ƽ
						realUrl = connection.getURL();
						
						// 饯åȤ
						String ct = connection.getContentType();
						if (ct != null) {
							try {
								contentType = new ContentType(ct);
							} catch (ParseException e) {}
						} else {
							//### ĥҤˤäƽȽ̤٤
						}
						String media   = contentType.getType   ();
						String subtype = contentType.getSubType();
						
						// פȽ
						if (media.compareTo("text") == 0) {
							if (subtype != null && subtype.compareTo("html") == 0)
								loadType = LT_HTML;
							else
								loadType = LT_TEXT;
							is = connection.getInputStream();
							
							String charset = contentType.getParameter("charset");
							if (charset != null && (encoding = Charset.toEncoding(charset)) == null)
								setStatus("charset.warning.invalid", new String[]{charset}, Reporter.WARNING);
						} else if (media.compareTo("image") == 0) {
							// 
							loadType = LT_IMAGE;
						} else {
							loadType = LT_BINARY;
						}
						
						// 
						contentLength = connection.getContentLength();
					} catch (SecurityException e) {
						setStatus("engine.status.load.error.security", new String[]{realUrl.toString()}, Reporter.WARNING);
						break OPENING;
					} catch (FileNotFoundException e) {
						setStatus("engine.status.load.error.nofile", new String[]{realUrl.toString()}, Reporter.WARNING);
						break OPENING;
					} catch (IOException e) {
						setStatus("engine.status.load.error", new String[]{realUrl.toString(), e.getMessage()}, Reporter.WARNING);
						break OPENING;
					}
					
					noerror = true;
				} /* OPENING: END */
				if (listener != null)
					listener.connected(index, noerror);
				
				if (!noerror || isInterrupted())
					break RUN;
				
				// ɥȾ
				Document document = new Document(realUrl, contentType, encoding, contentLength, Style.MEDIA_SCREEN);
				
				// Хʥξϡlistener ƤӽФʤ
				if (loadType == LT_BINARY) {
					doBinary(document, connection);
					setStatus(null, null, 0);
					noerror = true;
					break RUN;
				}
				
				// 󥰳
				RENDERING:
				{
					setStatus("engine.status.load.start", new String[]{realUrl.toString()}, Reporter.INFO);
					noerror = false;
					if (listener != null)
						listener.renderingStarted(index, openUrl, pd, realUrl);
					
					try {
						createRender(document);
						
						switch (loadType) {
						case LT_TEXT:
							doText(document, (is = connection.getInputStream()));
							break;
						case LT_HTML:
							doHtml(document, (is = connection.getInputStream()));
							break;
						case LT_IMAGE:
							doImage(document);
							break;
						}
						
						setStatus("engine.status.load.complete", null, Reporter.INFO);
						noerror = true;
					} catch (StopException e) {
						if (reporter != null)
							reporter.report(Reporter.INFO, Resource.getMessage("engine.warning.stop", new String[]{e.toString()}), 0, 0);
						render.stopImageLoad();
						// ߡפ줿ϡˤ
						noerror = true;
					} catch (OutOfMemoryError e) {
						setStatus("engine.status.load.error.oom", new String[]{realUrl.toString()}, Reporter.WARNING);
						if (reporter != null && reporter.getLevel() <= Reporter.DEBUG)
							e.printStackTrace(System.out);
					} catch (Exception e) {
						setStatus("engine.status.load.error", new String[]{realUrl.toString(), e.toString()}, Reporter.ERROR);
						e.printStackTrace(System.out);
					} finally {
						// λ
						if (is != null) {
							try {
								is.close();
							} catch (IOException e) {}
						}
					}
				} /* RENDERING: END */
				if (listener != null)
					listener.renderingStopped(index, noerror);
				
				if (!noerror || isInterrupted())
					break RUN;
				
			} /* RUN: END */
			
			if (active)
				context.setCursor(Cursor.DEFAULT_CURSOR);
			
			// 顼̵ text/html ξ
			if (noerror && loadType == LT_HTML) {
				ImageCache.clear();
				
				// եåν
				URL refreshUrl = null;
				try {
					refreshUrl = render.getRefreshURL();
				} catch (StopException e) {}
				if (refreshUrl != null && listener != null)
					listener.refresh(index, refreshUrl);
			}
			
			active = false;
		}
		
		/** Ǥ */
		public void interrupt()
		{
			if (active) {
				setStatus("engine.status.load.stop", null, Reporter.INFO);
				context.setCursor(Cursor.DEFAULT_CURSOR);
				active = false;
			}
			super.interrupt();
		}
		
		/**  */
		private void createRender(Document document)
		{
			synchronized (renderLock) {
				if (render != null) {
					render.dispose();
				}
				context.cleanCanvas();
				render = new Render(context, document);
			}
		}
		
		/** ƥȷϤν */
		private void doText(Document document, InputStream is)
		{
			BufferedReader br = null;
			String encoding = document.getEncoding();
			if (encoding != null) {
				try {
					br = new BufferedReader(new InputStreamReader(is, encoding));
					document.setEncoding(encoding);
				} catch (UnsupportedEncodingException e) {}
			}
			if (br == null)
				br = new BufferedReader(new InputStreamReader(is));
			
			String urlString = document.getURL().toString();
			Status status    = render.getStatus();
			
			status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));
			status.lineHeight = new Value(1.2, Value.UNIT_EM);
			status.whiteSpace = Value.WHITESPACE_PRE;
			status.reference  = false;
			status.setFixedFont();
			
			// ȥɽʸ
			render.setTitle(StringUtils.getFile(document.getURL()));
			
			render.startBody();
			
			String line;
			int    num;
			do {
				num = 0;
				render.save(TokenTypes._BLOCK_START);
				render.statusChanged();
				render.startBlock();
				
				try {
					while ((line = br.readLine()) != null) {
						render.drawText(line);
						render.drawText("\n");
						
						// 100 
						if (++num >= 100)
							break;
					}
				} catch (IOException e) {
					line = null;
					setStatus("engine.status.load.error", new String[]{urlString, e.getMessage()}, Reporter.WARNING);
				}
				
				render.endBlock();
				render.reset();
				render.statusChanged();
			} while (line != null);
			
			render.endBody();
			
			try {
				br.close();
			} catch (IOException e) {}
		}
		
		/** HTML Ϥν */
		private void doHtml(Document document, InputStream is)
		{
			// ȽƤϡ𤹤褦ˤ
			int contentLength = document.getContentLength();
			if (is != null && contentLength != -1)
				is = new CounterInputStream(is, new Chatty(this, realUrl.toString(), contentLength));
			
			// ѡѰդ
			String inputParser = option.getInputStreamParser();
			if (inputParser == null)
				inputParser = Resource.getMessage("parser.default", null);
			Parser parser = ParserManager.createInstance(inputParser, document, is, reporter);
			
			// ե륿Ѱդ
			String[] filters = option.getFilterParsers();
			if (filters != null && filters.length > 0) {
				for (int i = 0; i < filters.length; i++) {
					Parser p = ParserManager.createInstance(filters[i], document, parser, reporter);
					if (p != null)
						parser = p;
					else {
						ParserManager.ParserInfo pi = ParserManager.getParser(filters[i]);
						String name = (pi != null ? pi.getName() : "?");
						setStatus("engine.status.load.error.filter", new String[]{name, filters[i]}, Reporter.WARNING);
					}
				}
			}
			
			// ȡʬ
			TokenCoordinator tc       = new TokenCoordinator(document, parser, reporter);
			Token            token    = null;
			int              type     = TokenTypes.UNKNOWN;
			boolean          isBlock  = false;
			int              frameSet = 0;
			boolean          inHead   = false;
			boolean          inBody   = false;
			boolean          debug    = (reporter != null && reporter.getLevel() <= Reporter.DEBUG);
			
			READ:
			while ((token = tc.next()) != null) {
				if (debug)
					reporter.report(Reporter.DEBUG, "["+token+"]", token.getLineNumber(), token.getColumnNumber());
				
				if (Thread.interrupted())
					break READ;
				
				type = token.getType();
				
				// ֥åɤȽ
				if (type == TokenTypes.FORM_START ||
				    type == TokenTypes.FORM_END)
					isBlock = TokenTypes.canHaveBlock(token.getParent().getType());
				else
					isBlock = (TokenTypes.isBlockEx(type) ||
					           type == TokenTypes.TABLE_START ||
					           type == TokenTypes.TABLE_END);
				
				// Υȡ̵뤹
				switch (type) {
				case TokenTypes.UNKNOWN:
				case TokenTypes.DTD    :
				case TokenTypes.COMMENT:
				case TokenTypes.PI     :
				case TokenTypes.MISC   :
					token.render(render);
					continue;
				}
				
				// BODY γξ
				if (!inBody) {
					// ƥȤ̵
					if (type == TokenTypes.TEXT)
						continue;
					
					//### BUGS FRAMESET Ϥ줿
					if (type == TokenTypes.FRAMESET_START) {
						if (++frameSet == 1) {
							token.render(render);
							render.startBody();
							Status status = render.getStatus();
							status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));
							status.lineHeight    = new Value(1.2, Value.UNIT_EM);
							status.letterSpacing = new Value(1  , Value.UNIT_PX);
						}
						render.save(type);
						token.render(render);
						render.statusChanged();
						render.startBlock();
						if (frameSet == 1) {
							// åɽ
							render.save(type);
							render.getStatus().setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(1, Value.UNIT_EM));
							render.statusChanged();
							render.startBlock();
							render.drawText(Resource.getMessage("engine.message.useframe", null));
							render.endBlock();
							render.reset();
							render.statusChanged();
						}
						continue;
					}
					
					//### BUGS FRAMESET λ
					if (frameSet > 0) {
						switch (type) {
						case TokenTypes.FRAME:
							render.save(type);
							token.render(render);
							render.statusChanged();
							render.startBlock();
							render.endBlock();
							render.reset();
							render.statusChanged();
							break;
						case TokenTypes.FRAMESET_END:
							token.render(render);
							render.endBlock();
							render.reset();
							render.statusChanged();
							if (--frameSet == 0) {
								token.render(render);
								render.endBody();
							}
							break;
						}
						continue;
					}
					
					// BODY Ϥ줿
					if (type == TokenTypes.BODY_START) {
						inBody = true;
						token.render(render);
						render.startBody();
						continue;
					}
					
					if (!inHead) {
						if (type == TokenTypes.HEAD_START)
							inHead = true;
					} else {
						if (type == TokenTypes.HEAD_END)
							inHead = false;
					}
					
					token.render(render);
					continue;
				}
				
				// BODY λ
				if (inBody && type == TokenTypes.BODY_END) {
					inBody = false;
					token.render(render);
					render.endBody();
					continue;
				}
				
				///////////////////////
				// BODY ν
				
				// ϥȡ
				if (token instanceof StartToken) {
					render.save(type);
					token.render(render);
					render.statusChanged();
					if (isBlock)
						render.startBlock();
					
					// ξϡΤޤ޽λƤޤ
					if (TokenTypes.isEmpty(token.getType())) {
						if (isBlock)
							render.endBlock();
						render.reset();
						render.statusChanged();
					}
					
					continue;
				}
				
				// λȡ
				if (token instanceof EndToken) {
					token.render(render);
					if (isBlock)
						render.endBlock();
					render.reset();
					render.statusChanged();
					
					continue;
				}
				
				// ƥȥȡ
				if (token instanceof TextToken) {
					token.render(render);
					continue;
				}
			}
			
			render.waitImageLoad();
		}
		
		/** Ϥν */
		private void doImage(Document document)
		{
			String urlString = document.getURL().toString();
			Status status    = render.getStatus();
			
			status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));
			
			// ȥɽʸ
			render.setTitle(StringUtils.getFile(document.getURL()));
			
			render.startBody();
			render.save(TokenTypes._BLOCK_START);
			render.statusChanged();
			render.startBlock();
			render.drawImage(urlString, null, null, null, new Integer(0), Value.FLOAT_NONE);
			render.endBlock();
			render.reset();
			render.statusChanged();
			render.endBody();
		}
		
		/** ХʥϤν */
		private void doBinary(Document document, URLConnection connection)
		{
			context.download(document, connection);
		}
		
		/** ơɽ */
		private void setStatus(String key, Object[] args, int type)
		{
			String message = (key != null ? Resource.getMessage(key, args) : null);
			
			if (message != null && reporter != null)
				reporter.report(type, message, 0, 0);
			
			if (active) {
				if (message != null)
					context.setStatus(message);
				else
					context.setStatus(null);
			}
		}
	}
	
//### Chatty
	/** ɤ߹Ψ𤹤ꥹ */
	private class Chatty
		implements CounterListener
	{
		private LoadThread thread;
		private String     name;
		private int        max;
		
		private int percent = 0;
		
		private int p;
		
		/** 󥹥󥹤 */
		private Chatty(LoadThread thread, String name, int max)
		{
			this.thread = thread;
			this.name   = name;
			this.max    = max;
		}
		
		/** ɤ߹ХȿѤä */
		public void changeLength(long length)
		{
			p = (int)(length * 100 / max);
			
			// 100 % ˤϤʤʲɤ߹ߤ뤫⤷ʤ
			if (p >= 100)
				p = 99;
			
			if (p != percent) {
				// ΥѡȤɽ
				if (percent > 0)
					thread.setStatus("engine.status.load.percent",
					                 new Object[]{name, new Integer(percent)},
					                 Reporter.INFO);
				percent = p;
			}
		}
	}
}
