/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.cookie;

import net.hizlab.kagetaka.Resource;

import java.io.Writer;
import java.io.InvalidObjectException;
import java.io.IOException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * åɽ饹Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class Cookie
{
	private static final String   RESOURCE = "net.hizlab.kagetaka.cookie.Resource";
	private static final String   EXPIRE   = "E, dd MMM yyyy HH:mm:ss z";
	private static final SimpleDateFormat[] EXPIRES =
	{
		new SimpleDateFormat("E, dd MMM yy HH:mm:ss z", Locale.US),
		new SimpleDateFormat("E, dd MMM yy HH:mm z"   , Locale.US),
		new SimpleDateFormat("dd MMM yy HH:mm:ss z"   , Locale.US),
		new SimpleDateFormat("dd MMM yy HH:mm z"      , Locale.US),
		new SimpleDateFormat("E, dd-MM-yy HH:mm:ss z" , Locale.US),
		new SimpleDateFormat("E, dd-MMM-yy HH:mm:ss z", Locale.US),
		new SimpleDateFormat("E MMM dd HH:mm:ss yy"   , Locale.US),
	};
	
	private String  name;                     // NAME
	private String  value;                    // NAME=VALUE
	private long    expires = -1;
	private String  path;
	private String  domain;
	private boolean secure;
	private String  pathDirectory;
	private boolean isDomain;
	
	// ꥹȤǽ۴ĤƤʤ
	Cookie next;
	Cookie prev;
	
	/**
	 * <code>Set-Cookie</code> إåʸ󤫤顢åޤ
	 * 
	 * @param     data <code>Set-Cookie</code> إåʸ
	 * @param     url  åäƤ URL
	 * 
	 * @exception ParseException ʥեޥåȤξ
	 */
	public Cookie(String data, URL url, boolean strict)
		throws ParseException
	{
		int start = 0, end = 0, length = data.length();
		
		if (length == 0)
			throw new ParseException(Resource.getMessage(RESOURCE, "cookie.parse.none", null), 0);
		
		String s;
		int    p;
		
		// NAME=VALUE 
		if ((end = data.indexOf(';')) == -1)
			s = data;
		else
			s = data.substring(start, end);
		
		if ((p = s.indexOf('=')) != -1) {
			this.name  = s.substring(0,  p).trim();
			this.value = s.substring(p + 1).trim();
		} else {
			this.name  = s.trim();
		}
		
		char c;
		String key, value;
		boolean unknown;
		
		PARSE:
		while ((start = end) != -1) {
			// ڡƬФ
			for (;;) {
				if (++start == length)
					break PARSE;
				c = data.charAt(start);
				if (c != ' ' && c != ';')
					break;
			}
			
			// s ɬ 1 ʸʾ
			if ((end = data.indexOf(';', start)) == -1)
				s = data.substring(start);
			else
				s = data.substring(start, end);
			
			unknown = false;
			
			if ((p = s.indexOf('=')) == -1) {
				key = s;
				// 奢
				if (key.trim().toLowerCase().compareTo("secure") == 0) {
					if (!secure) {
						secure = true;
						continue;
					}
				} else
					unknown = true;
			} else {
				key   = s.substring(0,  p).trim().toLowerCase();
				value = s.substring(p + 1).trim();
				
				// ͭ
				if (key.compareTo("expires") == 0) {
					if (expires == -1) {
						Date date = null;
						for (int i = 0; i < EXPIRES.length; i++) {
							try {
								date = EXPIRES[i].parse(value);
								break;
							} catch (ParseException e) {}
						}
						if (date == null)
							throw new ParseException(Resource.getMessage(RESOURCE, "cookie.parse.invalidexpires", new String[]{value}), start);
						this.expires = date.getTime();
						continue;
					}
				} else
				
				// ѥ
				if (key.compareTo("path"   ) == 0) {
					if (path == null) {
						this.path = value;
						if (this.path.length() > 0 && this.path.charAt(this.path.length() - 1) == '/')
							this.pathDirectory = this.path;
						else
							this.pathDirectory = this.path + "/";
						continue;
					}
				} else
				
				// ɥᥤ
				if (key.compareTo("domain" ) == 0) {
					if (domain == null) {
						// Τˤ .  gTLD(3 ʸʾ?)  2 ġccTLD  3 ɬפ
						//  jp ɥᥤߤʤΤΤǡ2 İʾǥå
						if ((p = value.lastIndexOf('.')) != -1) {
							//int need  = (value.length() - p > 3 ? 2 : 3);
							int need  = 2;
							int count = 0;
							for (int i = value.length() - 1; i >= 0; i--)
								if (value.charAt(i) == '.')
									count++;
							if (count >= need) {
								this.domain   = value;
								this.isDomain = (this.domain.charAt(0) == '.');
								continue;
							}
						}
						
						if (strict)
							throw new ParseException(Resource.getMessage(RESOURCE, "cookie.parse.invaliddomain", new String[]{value}), start);
						continue;
					}
				} else {
					unknown = true;
				}
			}
			
			// 
			if (strict)
				throw new ParseException(Resource.getMessage(RESOURCE,
				                                              (unknown ? "cookie.parse.unknownoption"
				                                                       : "cookie.parse.duplicate"), new String[]{key}), 0);
		}
		
		// ɥᥤΥå
		if (this.domain != null && !isTargetHost(url.getHost()))
			throw new ParseException(Resource.getMessage(RESOURCE, "cookie.parse.outofdomain", new String[]{url.getHost(), this.domain}), start);
		
		// ǥեͤ
		if (this.path   == null || this.path  .length() == 0) { this.path = "/"; this.pathDirectory = "/"; }
		if (this.domain == null || this.domain.length() == 0) this.domain = url.getHost();
	}
	
	/**  */
	Cookie(String line)
		throws InvalidObjectException
	{
		int p1 = 0, p2, length = line.length();
		String[] s = new String[7];
		for (int i = 0; i < 7; i++) {
			if ((p2 = line.indexOf('\t', p1)) == -1)
				p2 = length;
			
			s[i] = line.substring(p1, p2);
			if ((p1 = p2 + 1) >= length)
				break;
		}
		
		if (s[5] == null)
			throw new InvalidObjectException("invalid format");
		
		try {
			this.name    = s[5];
			this.value   = (s[6] != null && s[6].length() > 0 ? s[6] : null);
			this.expires = Long.valueOf(s[4]).longValue() * 1000;
			this.path    = s[2];
			this.domain  = s[0];
			this.secure  = Boolean.valueOf(s[3]).booleanValue();
			if (this.path.length() > 0 && this.path.charAt(this.path.length() - 1) == '/')
				this.pathDirectory = this.path;
			else
				this.pathDirectory = this.path + "/";
			this.isDomain = Boolean.valueOf(s[1]).booleanValue();
		} catch (NumberFormatException e) {
			throw new InvalidObjectException("invalid format");
		}
	}
	
	/**
	 * å֤̾ޤ
	 * 
	 * @return    å̾
	 */
	public String getName()
	{
		return name;
	}
	
	/**
	 * å֤ͤޤ
	 * 
	 * @return    å
	 */
	public String getValue()
	{
		return value;
	}
	
	/**
	 * å̾ͤȤ֤ޤ
	 * 
	 * @return    å̾ͤ
	 */
	public String getFullValue()
	{
		if (value == null)
			return name;
		else
			return name + "=" + value;
	}
	
	/**
	 * ͭ¤֤ޤ
	 * 
	 * @return    ͭ
	 */
	public long getExpires()
	{
		return expires;
	}
	
	/**
	 * ѥ֤ޤ
	 * 
	 * @return    ѥ
	 */
	public String getPath()
	{
		return path;
	}
	
	/**
	 * ɥᥤ֤̾ޤ
	 * 
	 * @return    ɥᥤ̾
	 */
	public String getDomain()
	{
		return domain;
	}
	
	/**
	 * 奢ꤵƤ뤫֤ޤ
	 * 
	 * @return    奢ꤵƤ <code>true</code>
	 *            ʳξ <code>false
	 */
	public boolean isSecure()
	{
		return secure;
	}
	
	/**
	 * ꤵ줿ۥȤΥåоݤɤ֤ޤ
	 * 
	 * @param     host Ĵ٤ۥ
	 * 
	 * @return    Υåоݤξ <code>true</code>
	 *            ʳξ <code>false
	 */
	public boolean isTargetHost(String host)
	{
		if (isDomain)
			return host.endsWith(domain);
		else
			return (host.compareTo(domain) == 0);
	}
	
	/**
	 * ꤵ줿ѥΥåоݤɤ֤ޤ
	 * 
	 * @param     path Ĵ٤ѥ
	 * 
	 * @return    Υåоݤξ <code>true</code>
	 *            ʳξ <code>false
	 */
	public boolean isTargetPath(String path)
	{
		if (this.path.compareTo(path) == 0)
			return true;
		
		return path.startsWith(this.pathDirectory);
	}
	
	/**
	 * ʸѴޤ
	 * 
	 * @return    ʸɽ
	 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer();
		
		sb.append(getClass().getName());
		sb.append('[');
		
		sb.append(name);
		if (value != null) {
			sb.append('=');
			sb.append(value);
		}
		if (expires != -1) {
			SimpleDateFormat sdf = new SimpleDateFormat(EXPIRE, Locale.US);
			sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
			sb.append("; expires=");
			sb.append(sdf.format(new Date(expires)));
		}
		sb.append("; path=");
		sb.append(path);
		sb.append("; domain=");
		sb.append(domain);
		if (secure)
			sb.append("; secure");
		
		sb.append(']');
		
		return sb.toString();
	}
	
	/**  */
	void write(Writer w)
		throws IOException
	{
		w.write(domain);
		w.write("\t");
		w.write((isDomain ? "TRUE" : "FALSE"));
		w.write("\t");
		w.write(path);
		w.write("\t");
		w.write((secure ? "TRUE" : "FALSE"));
		w.write("\t");
		w.write(String.valueOf(expires / 1000));
		w.write("\t");
		w.write(name);
		w.write("\t");
		if (value != null)
			w.write(value);
	}
}
