/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.bookmarks;

import net.hizlab.kagetaka.build.Tag;
import net.hizlab.kagetaka.build.TagReader;
import net.hizlab.kagetaka.build.ParseException;
import net.hizlab.kagetaka.util.Charset;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.util.TextFormat;
import net.hizlab.kagetaka.viewer.ViewerOption;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.FileNotFoundException;
import java.io.CharConversionException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;

/**
 * ֥åޡ륯饹Ǥ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.3 $
 */
public class BookmarkManager
{
	private static final String ROOT_NAME = "/";
	private static final int    LF        = 0x0A;
	
	private ViewerOption   option;
	private BookmarkEditor editor;
	
	private Bookmark  rootFolder   = new Bookmark(ROOT_NAME);
	private boolean   commitEncoding;
	private String    encoding;
	private String    charset;
	private Hashtable shortcuts    = new Hashtable();
	private Object    ptFolderLock = new Object   ();
	private Bookmark  ptFolder;
	
	/**
	 * ֥åޡޥ͡ޤ
	 * 
	 * @param     option ץ
	 */
	public BookmarkManager(ViewerOption option)
	{
		this.option = option;
		load(option.getBookmarksPath());
	}
	
	/**
	 * ե뤫ɤޤ
	 * 
	 * @param     file ɤե
	 */
	private synchronized void load(File file)
	{
		commitEncoding = false;
		
		TagReader reader = null;
		
		try {
			OPEN:
			for (;;) {
				Tag tag, target;
				
				String   text      = null;
				String   element   = null;
				int      level     = 0;
				Bookmark parent    = rootFolder;
				Bookmark bookmark  = parent;
				Vector   bookmarks = rootFolder.getBookmarks();
				Stack    stack     = new Stack();
				
				// ɹΤ˽
				if (reader != null) {
					try {
						reader.close();
					} catch (IOException e) {}
				}
				bookmarks.removeAllElements();
				
				reader = createTagReader(file);
				
				READ:
				while ((tag = reader.readTag()) != null) {
					if (tag.getType() != Tag.TAG)
						continue;
					
					if ((element = tag.getElement()) == null)
						continue;
					
					// ʸɤ̤ξ˳ꤵ
					if (element.compareTo("META") == 0 && !commitEncoding && checkCharset(tag)) {
						reader.close();
						continue OPEN;
					}
					
					if (element.compareTo("DL") == 0) {
						if (!tag.isEndTag()) {
							// <DL> ξ
							if (bookmark == null || bookmark.getType() != Bookmark.FOLDER)
								break READ;
							stack.push(parent);
							parent = bookmark;
						} else {
							// </DL> ξ
							parent = (Bookmark)stack.pop();
						}
						bookmarks = parent.getBookmarks();
						bookmark  = null;
						continue;
					}
					
					if (element.compareTo("HR") == 0) {
						if (bookmarks == null)
							continue;
						bookmark = new Bookmark();
						bookmark.parent = parent;
						bookmarks.addElement(bookmark);
					} else
					if (element.compareTo("DT") == 0) {
						// ϥ <a href="">
						if ((target = reader.readTag()) == null)
							break READ;
						
						// ʸ
						if ((tag = reader.readTag()) == null)
							break READ;
						text = TextFormat.convertXhtml(tag.getText(), true, true, true, true);
						
						// λ </a>
						if ((tag = reader.readTag()) == null || !tag.isEndTag())
							break READ;
						
						bookmark = new Bookmark(text, target);
						bookmark.parent = parent;
						bookmarks.addElement(bookmark);
						
						// 硼ȥåȤϿ
						if (bookmark.getShortcutURL() != null)
							shortcuts.put(bookmark.getShortcutURL(), bookmark);
						
						// ѡʥġСե
						if (bookmark.getPersonalToolbarFolder() && ptFolder == null)
							ptFolder = bookmark;
					}
				}
				
				break;
			}
			
			// Ǹޤɤ߹ϡɤꤷƤʤƤ⤤Ȥˤ
			commitEncoding = true;
		} catch (FileNotFoundException e) {
			return;
		} catch (ParseException e) {
			//### ERROR
System.out.println(e);
e.printStackTrace(System.out);
		} catch (IOException e) {
			//### ERROR
System.out.println(e);
e.printStackTrace(System.out);
		}
		
		if (reader != null) {
			try {
				reader.close();
			} catch (IOException e) {}
		}
	}
	
	/**
	 * ե˥֥åޡ¸ޤ
	 */
	public synchronized void save()
	{
		File file = option.getBookmarksPath();
		BufferedWriter bw = null;
		
		try {
			// ʸ̤ʿ¸ξΤߡˤˡUTF-8 ǻƤߤơ
			// ʤ饷ƥɸʸɤ¸
			
			if (!commitEncoding)
				encoding = Charset.toEncoding(charset = "UTF-8");
			
			OPEN:
			for (;;) {
				try {
					{
						FileOutputStream   fos = new FileOutputStream(file);
						OutputStreamWriter osw = (encoding != null
						                         ? new OutputStreamWriter(fos, encoding)
						                         : new OutputStreamWriter(fos));
						bw = new BufferedWriter(osw);
					}
					
					String charset = (this.charset != null ? "; charset=" + this.charset : "");
					
					bw.write("<!DOCTYPE NETSCAPE-Bookmark-file-1>"                                    ); bw.write(LF);
					bw.write("<!-- This is an automatically generated file."                          ); bw.write(LF);
					bw.write("It will be read and overwritten."                                       ); bw.write(LF);
					bw.write("Do Not Edit! -->"                                                       ); bw.write(LF);
					bw.write("<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html" + charset + "\">"); bw.write(LF);
					bw.write("<TITLE>Bookmarks</TITLE>"                                               ); bw.write(LF);
					bw.write("<H1>Bookmarks</H1>"                                                     ); bw.write(LF);
					bw.write(LF);
					bw.write("<DL><p>" ); bw.write(LF);
					Vector bookmarks = getBookmarks();
					if (bookmarks != null)
						for (int i = 0; i < bookmarks.size(); i++)
							((Bookmark)bookmarks.elementAt(i)).write(bw, 1);
					bw.write("</DL><p>"); bw.write(LF);
					bw.flush();
					break;
				} catch (CharConversionException e) {
					if (commitEncoding)
						throw e;
					
					commitEncoding = true;
					encoding = null;
					charset  = null;
				}
			}
		} catch (IOException e) {
			//### ERROR
System.out.println(e);
e.printStackTrace(System.out);
		}
		
		if (bw != null) {
			try {
				bw.close();
			} catch (IOException e) {}
		}
	}
	
	/**
	 * ٤ƤΥ֥åޡ縵ɽüʥ֥åޡե֤ޤ
	 * 
	 * @return    롼ȥ֥åޡե
	 */
	public Bookmark getRootBookmark()
	{
		return rootFolder;
	}
	
	/**
	 * ֥åޡ֤ޤ
	 * 
	 * @return    ֥åޡ
	 */
	public Vector getBookmarks()
	{
		if (rootFolder == null)
			return null;
		
		return rootFolder.getBookmarks();
	}
	
	/**
	 * 򥨥ǥɽޤ
	 */
	public synchronized void showEditor()
	{
		if (editor == null || !editor.isVisible())
			editor = new BookmarkEditor(option);
		
		editor.show();
	}
	
	/**
	 * 硼ȥåȤβ򤷤ޤ
	 * 
	 * @param     shortcut 硼ȥå
	 * 
	 * @return    URL硼ȥåȤ¸ߤʤ <code>null</code>
	 */
	public String resolveShortcut(String shortcut)
	{
		if (shortcut == null || shortcut.length() == 0)
			return null;
		if (shortcut.indexOf(':') != -1)
			return null;
		int p = shortcut.indexOf(' ');
		if (p == -1)
			p = shortcut.indexOf('');
		
		String key    = shortcut;
		String option = null;
		if (p != -1) {
			key    = shortcut.substring(0,  p).trim();
			option = shortcut.substring(p + 1).trim();
		}
		
		Bookmark bookmark = (Bookmark)shortcuts.get(key);
		if (bookmark == null)
			return null;
		
		// ֥åޡƤʤå
		if (!checkInTree(bookmark))
			return null;
		
		String url = bookmark.getURLText();
		
		// 
		if (option != null && option.length() > 0) {
			//### TODO ֥åޡ lastCharset ˱ƥ󥳡ɤѤ
			option = URLEncoder.encode(option);
			if ((p = url.lastIndexOf("%s")) != 0)
				url = url.substring(0, p) + option + url.substring(p + 2);
			else
				url += option;
		}
		
		return url;
	}
	
	/**
	 * ѡʥġСե֤ޤ
	 * 
	 * @return    ѡʥġСե
	 *            ꤵƤʤ <code>null</code>
	 */
	public Bookmark getPersonalToolbarFolder()
	{
		synchronized (ptFolderLock) {
			if (ptFolder == null)
				return null;
			
			if (!checkInTree(ptFolder))
				ptFolder = null;
			
			return ptFolder;
		}
	}
	
	/** ѡʥġСե */
	void setPersonalToolbarFolder(Bookmark bookmark)
	{
		if (bookmark.getType() != Bookmark.FOLDER)
			throw new IllegalArgumentException("not folder");
		
		if (!checkInTree(bookmark))
			throw new IllegalArgumentException("invalid bookmark");
		
		synchronized (ptFolderLock) {
			if (ptFolder != null)
				ptFolder.setPersonalToolbarFolder(false);
			
			ptFolder = bookmark;
			ptFolder.setPersonalToolbarFolder(true);
		}
	}
	
	/** 硼ȥåȤѹޤ */
	boolean changeShortcut(String oldShortcut, String newShortcut, Bookmark bookmark)
	{
		if ((oldShortcut == null || oldShortcut.length() == 0) &&
		    (newShortcut == null || newShortcut.length() == 0))
			return true;
		
		if (oldShortcut != null &&
		    newShortcut != null &&
		    oldShortcut.compareTo(newShortcut) == 0)
			return true;
		
		if (newShortcut == null || newShortcut.length() == 0) {
			shortcuts.remove(oldShortcut);
			return true;
		}
		
		synchronized (shortcuts) {
			if (shortcuts.containsKey(newShortcut))
				return false;
			
			if (oldShortcut != null)
				shortcuts.remove(oldShortcut);
			
			shortcuts.put(newShortcut, bookmark);
		}
		
		return true;
	}
	
	/** ꡼ */
	private TagReader createTagReader(File file)
		throws IOException
	{
		for (int i = 0; i < 2; i++) {
			try {
				FileInputStream   fis = new FileInputStream(file);
				InputStreamReader isr = (encoding != null
				                        ? new InputStreamReader(fis, encoding)
				                        : new InputStreamReader(fis));
				return new TagReader(new BufferedReader(isr), null);
			} catch (UnsupportedEncodingException e) {
				encoding = null;
			}
		}
		
		throw new IOException("Can not open `" + file + "'");
	}
	
	/** 饻åȤȽ */
	private boolean checkCharset(Tag tag)
	{
		Hashtable param = tag.getAttribute();
		String s;
		if ((s = (String)param.get("HTTP-EQUIV")) == null)
			return false;
		
		if (s.toLowerCase().compareTo("content-type") != 0)
			return false;
		
		if ((s = (String)param.get("CONTENT")) == null)
			return false;
		
		String charset = null;
		try {
			charset = (new ContentType(s)).getParameter("charset");
		} catch (java.text.ParseException e) {}
		if (charset == null)
			return false;
		
		// charset ꤵƤΤǡ󥳡ǥ󥰤ϳꤷȤˤ
		this.commitEncoding = true;
		
		if (this.charset != null && this.charset.compareTo(charset) == 0)
			return false;
		
		this.charset = charset;
		
		String encoding = Charset.toEncoding(charset);
		if (encoding == null)
			return false;
		
		if (this.encoding != null && this.encoding.compareTo(encoding) == 0)
			return false;
		
		this.encoding = encoding;
		
		return true;
	}
	
	/** ֥åޡˤ뤫ɤå */
	private boolean checkInTree(Bookmark bookmark)
	{
		Bookmark parent = bookmark.parent;
		while (parent != rootFolder) {
			if (parent == null)
				return false;
			
			parent = parent.parent;
		}
		return true;
	}
}
