/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.awt.event.TabEvent;
import net.hizlab.kagetaka.awt.event.TabListener;
import net.hizlab.kagetaka.awt.image.OffscreenObserver;

import java.awt.AWTEvent;
import java.awt.Color;
import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Panel;
import java.awt.Rectangle;
import java.awt.SystemColor;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.MemoryImageSource;
import java.util.Hashtable;
import java.util.Vector;

/**
 * ڤؤѥͥǤ
 * 
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
public class TabbedContainer
	extends Container
{
	private static final int TABBAR_LEFT_FIX   = 4;
	private static final int TABBAR_RIGHT_FIX  = 5;
	private static final int TABBAR_IMAGE_FIX  = 2;
	private static final int TABBAR_HEIGHT_FIX = 4;
	
	private static final int NOACTIVE = 0;
	private static final int ACTIVE   = 1;
	private static final int BEFORE   = 2;
	
	private static final int[] LEFT_IMAGE =
	{
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	};
	private static final int[] RIGHT_IMAGE =
	{
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	};
	
	/** @serial ֥Сι⤵ */
	private int      tabbarHeight = 0;
	
	/** @serial ĤΥ֤κǾ */
	private int      minWidth =  50;
	/** @serial ĤΥ֤κ */
	private int      maxWidth = 300;
	/** @serial եΤե졼࿧ */
	private Color    focusFrameColor;
	/** @serial ƥ֥ȥե */
	private Font     activeTitleFont;
	/** @serial ƥ֥ȥʸ */
	private Color    activeTitleForeground;
	/** @serial ƥ֥ȥʸ */
	private Color    activeTitleBackground;
	/** @serial 󥢥ƥ֥ȥե */
	private Font     unactiveTitleFont;
	/** @serial 󥢥ƥ֥ȥʸ */
	private Color    unactiveTitleForeground;
	/** @serial 󥢥ƥ֥ȥʸ */
	private Color    unactiveTitleBackground;
	
	/** @serial ֥С */
	private Tabbar        tabbar;
	/** @serial ֥С쥤ȥޥ͡ */
	private AlignLayout   tabbarLayout;
	/** @serial ֥С */
	private TabScrollbar  tabScrollbar;
	/** @serial ᥤѥͥ */
	private Panel         main;
	/** @serial ᥤѥͥ쥤ȥޥ͡ */
	private CardLayout    mainLayout;
	/** @serial ƥ֥ѥͥ */
	private TabPanel      activeTabPanel;
	/** @serial ƥ֥ѥͥ */
	private int           activeTabPanelIndex;
	/** @serial ƥ֥ѥͥ뤬ɽƤ뤫ɤ */
	private boolean       activeVisible;
	/** @serial ѥͥΥꥹ */
	private Vector        panels = new Vector   ();
	/** @serial ѥͥΥϥå */
	private Hashtable     hash   = new Hashtable();
	/** @serial ѤƤ */
	private Vector        images = new Vector   ();
	
	private transient TabListener         tabListener;
	private transient ComponentListener   tabComponentListener;
	private transient FocusListener       tabFocusListener;
	private transient KeyListener         tabKeyListener;
	private transient MouseListener       tabMouseListener;
	private transient MouseMotionListener tabMouseMotionListener;
	
	/**
	 * ֥ѥͥޤ
	 * Dialog եȤѤƥ५顼ǽޤ
	 */
	public TabbedContainer()
	{
		// ֥С
		tabbar = new Tabbar();
		tabbar.setLayout(tabbarLayout = new AlignLayout(AlignLayout.HORIZONTAL));
		
		// ֥С
		tabScrollbar = new TabScrollbar();
		tabScrollbar.setVisible(false);
		
		// ᥤѥͥ
		main = new Panel(mainLayout = new CardLayout());
		
		// ̤
		GridBagLayout gbl    = new GridBagLayout();
		Insets        insets = new Insets(0, 0, 0, 0);
		setLayout(gbl);
		insets.left = 2;
		LayoutUtils.addGridBag(this, tabbar      , gbl, 0, 0, 1, 1, 1, 0, GridBagConstraints.NONE, GridBagConstraints.WEST     , insets);
		insets.left = 0;
		insets.bottom = 2;
		LayoutUtils.addGridBag(this, tabScrollbar, gbl, 1, 0, 1, 1, 0, 0, GridBagConstraints.NONE, GridBagConstraints.SOUTHEAST, insets);
		insets.bottom = 0;
		LayoutUtils.addGridBag(this, main        , gbl, 0, 1, 2, 1, 1, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER   , insets);
		
		// ꥹʤϿ
		addComponentListener(
			new ComponentAdapter()
			{
				/** ѹ */
				public void componentResized(ComponentEvent e)
				{
					relayoutTabs();
				}
			}
		);
		
		// ƥȥΥǥեȤ
		setForeground             (SystemColor.controlText);
		setBackground             (SystemColor.control    );
		setFocusFrameColor        (SystemColor.windowBorder   );
		setActiveTitleFont        (Font.decode("Dialog"      ));
		setActiveTitleForeground  (SystemColor.controlText    );
		setActiveTitleBackground  (SystemColor.control        );
		setUnActiveTitleFont      (Font.decode("Dialog"      ));
		setUnActiveTitleForeground(SystemColor.controlText    );
		setUnActiveTitleBackground(SystemColor.control        );
/*
setForeground  (Color.green);
setBackground  (Color.orange);
setFocusFrameColor        (Color.red);
setActiveTitleFont        (Font.decode("Dialog"      ));
setActiveTitleForeground  (Color.black);
setActiveTitleBackground  (Color.blue);
setUnActiveTitleFont      (Font.decode("Dialog"      ));
setUnActiveTitleForeground(Color.gray);
setUnActiveTitleBackground(Color.lightGray);
*/
	}
	
	/**
	 * ֥ѥͥɲäޤ
	 * 
	 * @param     key   ֤̤륭
	 * @param     title ȥ
	 * @param     image ᡼ɽʤ <code>null</code>
	 * @param     comp  ݡͥ
	 * @param     focus ɲä֤˥եư <code>true</code>
	 *                  ʳξ <code>false</code>
	 * 
	 * @return    ɲä֤̤륭
	 */
	public void addPanel(String key, String title, Image image, Component comp, boolean focus)
	{
		addPanel(key, title, image, comp, focus, -1);
	}
	
	/**
	 * ֥ѥͥꤷǥå֤ɲäޤ
	 * <p>
	 * <code>focus</code>  <code>true</code> ξ硢ɲä֤
	 * ƥ֤ˤʤޤܤΥ֤ξϡͤ˴طʤ
	 * Υ֤ƥ֤ˤʤޤΥ᥽åɤˤꥢƥ֤ʥ֤
	 * ѹƤ⡢{@link TabListener#activeChanged(TabEvent)}
	 * ٥Ȥȯޤ
	 * 
	 * @param     key   ֤̤륭
	 * @param     title ȥ
	 * @param     image ᡼ɽʤ <code>null</code>
	 * @param     comp  ݡͥ
	 * @param     focus ɲä֤˥եư <code>true</code>
	 *                  ʳξ <code>false</code>
	 * @param     index ɲä륿֥ѥͥΥǥå
	 * 
	 * @exception IllegalArgumentException ʣ
	 */
	public synchronized void addPanel(String key, String title, Image image,
	                                  Component comp, boolean focus, int index)
	{
		synchronized (panels) {
			if (hash.containsKey(key))
				throw new IllegalArgumentException("Already exists key `"+ key + "'");
			
			TabPanel panel = new TabPanel(key, title, image, comp);
			if (index == -1) {
				panels.addElement(panel);
				index = panels.size() - 1;
			} else {
				panels.insertElementAt(panel, index);
			}
			hash.put(panel.key, panel);
			
			panel.attach(index);
			if (tabComponentListener   != null) panel.tab.addComponentListener  (tabComponentListener  );
			if (tabFocusListener       != null) panel.tab.addFocusListener      (tabFocusListener      );
			if (tabKeyListener         != null) panel.tab.addKeyListener        (tabKeyListener        );
			if (tabMouseListener       != null) panel.tab.addMouseListener      (tabMouseListener      );
			if (tabMouseMotionListener != null) panel.tab.addMouseMotionListener(tabMouseMotionListener);
			
			if (panels.size() == 1 || focus)
				panel.setActived(false);
		}
		
		relayoutTabs();
	}
	
	/**
	 * ֥ѥͥޤ
	 * <p>
	 * ֤ƥ֤ä硢Υ֤κ¦Υ֤
	 * ƥ֤ˤʤޤ֤Ƭξϡ
	 * ƬΥ֤ƥ֤ˤʤޤ
	 * Υ᥽åɤˤꥢƥ֤ʥ֤
	 * ѹƤ⡢{@link TabListener#activeChanged(TabEvent)}
	 * ٥Ȥȯޤ
	 * 
	 * @param     key ֤̤륭
	 * 
	 * @exception IllegalArgumentException ꤵ줿֤¸ߤʤ
	 */
	public void removePanel(String key)
	{
		synchronized (panels) {
			TabPanel panel = (TabPanel)hash.get(key);
			if (panel == null)
				throw new IllegalArgumentException("Invalid key `"+ key + "'");
			
			int index = panels.indexOf(panel);
			
			hash.remove(key);
			panels.removeElementAt(index);
			
			panel.detach(index);
			
			if (panels.size() > 0) {
				if (panel == activeTabPanel) {
					if (index > 0)
						index--;
					
					TabPanel newFocusPanel = (TabPanel)panels.elementAt(index);
					newFocusPanel.setActived(false);
					if (panel.tab.focus)
						newFocusPanel.tab.requestFocus();
				} else if (index < activeTabPanelIndex) {
					activeTabPanelIndex--;
				}
			} else {
				activeTabPanel      = null;
				activeTabPanelIndex = 0;
			}
		}
		
		relayoutTabs();
	}
	
	/**
	 * ֤Υȥꤷޤ
	 * 
	 * @param     key   ֤̤륭
	 * @param     title ȥ
	 * 
	 * @exception IllegalArgumentException ꤵ줿֤¸ߤʤ
	 */
	public void setTitle(String key, String title)
	{
		TabPanel panel;
		boolean relayout = false;
		
		synchronized (panels) {
			panel = (TabPanel)hash.get(key);
			if (panel == null)
				throw new IllegalArgumentException("Invalid key `"+ key + "'");
			
			relayout = panel.tab.setTitle(title, panel.tab.image);
		}
		
		if (relayout)
			relayoutTabs();
		panel.tab.refresh();
	}
	
	/**
	 * ֤κɽ륤᡼ꤷޤ
	 * 
	 * @param     key   ֤̤륭
	 * @param     image ᡼
	 * 
	 * @exception IllegalArgumentException ꤵ줿֤¸ߤʤ
	 */
	public void setImage(String key, Image image)
	{
		TabPanel panel;
		boolean relayout = false;
		
		synchronized (panels) {
			panel = (TabPanel)hash.get(key);
			if (panel == null)
				throw new IllegalArgumentException("Invalid key `"+ key + "'");
			
			relayout = panel.tab.setTitle(panel.tab.title, image);
		}
		
		if (relayout)
			relayoutTabs();
		panel.tab.refresh();
	}
	
	/**
	 * ꤷ֤򥢥ƥ֤ˤޤ
	 * <p>
	 * Υ᥽åɤˤꥢƥ֤ʥ֤
	 * ѹƤ⡢{@link TabListener#activeChanged(TabEvent)}
	 * ٥Ȥȯޤ
	 * 
	 * @param     key ֤̤륭
	 * 
	 * @return    ƥ֤ʥ֤ѹ줿 <code>true</code>
	 *            Ǥ˻ꤵ줿֤ƥ֤ä <code>false</code>
	 * 
	 * @exception IllegalArgumentException ꤵ줿֤¸ߤʤ
	 */
	public boolean setActivePanel(String key)
	{
		synchronized (panels) {
			TabPanel panel = (TabPanel)hash.get(key);
			if (panel == null)
				throw new IllegalArgumentException("Invalid key `"+ key + "'");
			if (panel == activeTabPanel)
				return false;
			
			return panel.setActived(false);
		}
	}
	
	/**
	 * ƥ֤ʥ֤μ̥֤ޤ
	 * 
	 * @return    ֤̤륭
	 *            ֤Ĥʤ <code>null</code>
	 */
	public String getActivePanelKey()
	{
		synchronized (panels) {
			if (activeTabPanel == null)
				return null;
			
			return activeTabPanel.key;
		}
	}
	
	/**
	 * ƥ֤ʥ֤Υǥå֤ޤ
	 * 
	 * @return    ǥå
	 *            ֤Ĥʤ <code>-1</code>
	 */
	public int getActivePanelIndex()
	{
		synchronized (panels) {
			if (activeTabPanel == null)
				return -1;
			
			return activeTabPanelIndex;
		}
	}
	
	/**
	 * ĤΥ֤κǾ֤ޤ
	 * 
	 * @return    Ǿ
	 */
	public synchronized int getTabMinWidth()
	{
		return minWidth;
	}
	
	/**
	 * ĤΥ֤κǾꤷޤ
	 * 
	 * @param     width Ǿ
	 */
	public synchronized void setTabMinWidth(int width)
	{
		minWidth = width;
		relayoutTabs();
	}
	
	/**
	 * ĤΥ֤κ֤ޤ
	 * 
	 * @return    
	 */
	public synchronized int getTabMaxWidth()
	{
		return maxWidth;
	}
	
	/**
	 * ĤΥ֤κꤷޤ
	 * 
	 * @param     width 
	 */
	public synchronized void setTabMaxWidth(int width)
	{
		maxWidth = width;
		relayoutTabs();
	}
	
	/**
	 * ʿꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setForeground(Color color)
	{
		super.setForeground(color);
		tabScrollbar.setForeground(color);
	}
	
	/**
	 * طʿꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setBackground(Color color)
	{
		super.setBackground(color);
		tabScrollbar.setBackground(color);
	}
	
	/**
	 * եΤե졼࿧֤ޤ
	 * 
	 * @return    
	 */
	public synchronized Color getFocusFrameColor()
	{
		return focusFrameColor;
	}
	
	/**
	 * եΤե졼࿧ꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setFocusFrameColor(Color color)
	{
		focusFrameColor = color;
	}
	
	/**
	 * ƥ֥ȥѤեȤ֤ޤ
	 * 
	 * @return    ե
	 */
	public synchronized Font getActiveTitleFont()
	{
		return activeTitleFont;
	}
	
	/**
	 * ƥ֥ȥѤեȤꤷޤ
	 * 
	 * @param     font ե
	 */
	public synchronized void setActiveTitleFont(Font font)
	{
		activeTitleFont = font;
		calculateTabbarSize();
	}
	
	/**
	 * ƥ֥ȥʸ֤ޤ
	 * 
	 * @return    
	 */
	public synchronized Color getActiveTitleForeground()
	{
		return activeTitleForeground;
	}
	
	/**
	 * ƥ֥ȥʸꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setActiveTitleForeground(Color color)
	{
		activeTitleForeground = color;
	}
	
	/**
	 * ƥ֥ȥطʿ֤ޤ
	 * 
	 * @return    
	 */
	public synchronized Color getActiveTitleBackground()
	{
		return activeTitleBackground;
	}
	
	/**
	 * ƥ֥ȥطʿꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setActiveTitleBackground(Color color)
	{
		activeTitleBackground = color;
	}
	
	/**
	 * 󥢥ƥ֥ȥѤեȤ֤ޤ
	 * 
	 * @return    ե
	 */
	public synchronized Font getUnActiveTitleFont()
	{
		return unactiveTitleFont;
	}
	
	/**
	 * 󥢥ƥ֥ȥѤեȤꤷޤ
	 * 
	 * @param     font ե
	 */
	public synchronized void setUnActiveTitleFont(Font font)
	{
		unactiveTitleFont = font;
		calculateTabbarSize();
	}
	
	/**
	 * 󥢥ƥ֥ȥʸ֤ޤ
	 * 
	 * @return    
	 */
	public synchronized Color getUnActiveTitleForeground()
	{
		return unactiveTitleForeground;
	}
	
	/**
	 * 󥢥ƥ֥ȥʸꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setUnActiveTitleForeground(Color color)
	{
		unactiveTitleForeground = color;
	}
	
	/**
	 * 󥢥ƥ֥ȥطʿ֤ޤ
	 * 
	 * @return    
	 */
	public synchronized Color getUnActiveTitleBackground()
	{
		return unactiveTitleBackground;
	}
	
	/**
	 * 󥢥ƥ֥ȥطʿꤷޤ
	 * 
	 * @param     color 
	 */
	public synchronized void setUnActiveTitleBackground(Color color)
	{
		unactiveTitleBackground = color;
	}
	
	/**
	 * ֤򥹥뤵ܥꤷޤ
	 * 
	 * @param     left  Υܥ
	 * @param     right Υܥ
	 */
	public synchronized void setTabScrollButton(ImageButton left, ImageButton right)
	{
		tabScrollbar.setButton(left, right);
	}
	
	/**
	 * ֥ꥹʤϿޤ
	 * 
	 * @param     l Ͽ륿֥ꥹ
	 */
	public synchronized void addTabListener(TabListener l)
	{
		enableEvents(0);
		tabListener = AWTEventMulticaster.add(tabListener, l);
	}
	
	/**
	 * ֥ꥹʤޤ
	 * 
	 * @param     l 륿֥ꥹ
	 */
	public synchronized void removeTabListener(TabListener l)
	{
		tabListener = AWTEventMulticaster.remove(tabListener, l);
	}
	
	/**
	 * ݡͥȤΥ󥹥󥹤饿֤̤륭֤ޤ
	 * Υ᥽åɤϡ٥ȤΥ
	 * ݡͥȤΥ󥹥󥹤顢뤿Ѥޤ
	 * 
	 * @param     c ݡͥȡ
	 *              ʥݡͥȤξ <code>null</code>
	 */
	public String getKey(Component c)
	{
		if (c == null || !(c instanceof Tab))
			return null;
		
		Tab tab = (Tab)c;
		
		synchronized (panels) {
			TabPanel panel;
			for (int i = 0; i < panels.size(); i++) {
				panel = (TabPanel)panels.elementAt(i);
				if (tab == panel.tab)
					return panel.key;
			}
		}
		
		return null;
	}
	
	/**
	 * ֤Ф륳ݡͥȥꥹʤϿޤ
	 * 
	 * @param     l Ͽ륳ݡͥȥꥹ
	 */
	public synchronized void addTabComponentListener(ComponentListener l)
	{
		tabComponentListener = AWTEventMulticaster.add(tabComponentListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.addComponentListener(l);
	}
	
	/**
	 * ֤Ф륳ݡͥȥꥹʤޤ
	 * 
	 * @param     l 륳ݡͥȥꥹ
	 */
	public synchronized void removeTabComponentListener(ComponentListener l)
	{
		tabComponentListener = AWTEventMulticaster.remove(tabComponentListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.removeComponentListener(l);
	}
	
	/**
	 * ֤ФեꥹʤϿޤ
	 * 
	 * @param     l Ͽեꥹ
	 */
	public synchronized void addTabFocusListener(FocusListener l)
	{
		tabFocusListener = AWTEventMulticaster.add(tabFocusListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.addFocusListener(l);
	}
	
	/**
	 * ֤Фեꥹʤޤ
	 * 
	 * @param     l եꥹ
	 */
	public synchronized void removeTabFocusListener(FocusListener l)
	{
		tabFocusListener = AWTEventMulticaster.remove(tabFocusListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.removeFocusListener(l);
	}
	
	/**
	 * ֤Ф륭ꥹʤϿޤ
	 * 
	 * @param     l Ͽ륭ꥹ
	 */
	public synchronized void addTabKeyListener(KeyListener l)
	{
		tabKeyListener = AWTEventMulticaster.add(tabKeyListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.addKeyListener(l);
	}
	
	/**
	 * ֤Ф륭ꥹʤޤ
	 * 
	 * @param     l 륭ꥹ
	 */
	public synchronized void removeTabKeyListener(KeyListener l)
	{
		tabKeyListener = AWTEventMulticaster.remove(tabKeyListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.removeKeyListener(l);
	}
	
	/**
	 * ֤ФޥꥹʤϿޤ
	 * 
	 * @param     l Ͽޥꥹ
	 */
	public synchronized void addTabMouseListener(MouseListener l)
	{
		tabMouseListener = AWTEventMulticaster.add(tabMouseListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.addMouseListener(l);
	}
	
	/**
	 * ֤Фޥꥹʤޤ
	 * 
	 * @param     l ޥꥹ
	 */
	public synchronized void removeTabMouseListener(MouseListener l)
	{
		tabMouseListener = AWTEventMulticaster.remove(tabMouseListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.removeMouseListener(l);
	}
	
	/**
	 * ֤Фޥ⡼ꥹʤϿޤ
	 * 
	 * @param     l Ͽޥ⡼ꥹ
	 */
	public synchronized void addTabMouseMotionListener(MouseMotionListener l)
	{
		tabMouseMotionListener = AWTEventMulticaster.add(tabMouseMotionListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.addMouseMotionListener(l);
	}
	
	/**
	 * ֤Фޥ⡼ꥹʤޤ
	 * 
	 * @param     l ޥ⡼ꥹ
	 */
	public synchronized void removeTabMouseMotionListener(MouseMotionListener l)
	{
		tabMouseMotionListener = AWTEventMulticaster.remove(tabMouseMotionListener, l);
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.removeMouseMotionListener(l);
	}
	
//### Override
	/**
	 * ֤κǾΥ֤ޤ
	 * ⤵Τ¤Ƥޤ
	 * 
	 * @return    Ǿ
	 */
	public Dimension getMinimumSize()
	{
		return new Dimension(0, tabbarHeight);
	}
	
	/**
	 * ֥ѥͥκԤޤ
	 * 
	 * @param     g Graphics ɥ
	 */
	public void update(Graphics g)
	{
		paint(g);
	}
	
	/**
	 * ֥ѥͥԤޤ
	 * 
	 * @param     g Graphics ɥ
	 */
	public void paint(Graphics g)
	{
		Dimension size = getSize();
		
		size.height = tabbar.getSize().height;
		
		// طʿɤ
		g.setColor(getBackground());
		g.fillRect(0, 0, size.width, size.height);
		
		// 
		paintUnderLine(g, size);
		
		// Ƭƥ֤ξˡƬΥ֤κ¦
		if (tabbarLayout.getStatus().top == activeTabPanelIndex) {
			g.setColor(activeTitleBackground.brighter());
			g.drawLine(1, 1, 1, 1);
			g.drawLine(0, 2, 0, size.height - 2);
			
			g.setColor(activeTitleBackground);
			g.drawLine(1, 2, 1, size.height - 2);
		}
	}
	
	/**
	 * Υ֤Υѥ᡼ʸ֤ޤ
	 * 
	 * @return    ѥ᡼ʸ
	 */
	protected String paramString()
	{
		String str = super.paramString();
		return str;
	}
	
	/**
	 * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
	 * 
	 * @param     e ٥
	 */
	protected void processEvent(AWTEvent e) {
		if (e instanceof TabEvent) {
			processTabEvent((TabEvent)e);
			return;
		}
		super.processEvent(e);
	}
	
	/**
	 * ΥݡͥȤȯ륿֥٥Ȥ
	 * ϿƤ뤹٤Ƥ {@link TabListener} 뤳Ȥˤꡢ
	 * ֥٥Ȥޤ
	 * 
	 * @param     e ٥
	 */
	protected void processTabEvent(TabEvent e) {
		if (tabListener == null)
			return;
		
		switch (e.getID()) {
		case TabEvent.ACTIVE_CHANGED: tabListener.activeChanged(e); break;
		case TabEvent.TAB_ADDED     : tabListener.tabAdded     (e); break;
		case TabEvent.TAB_REMOVED   : tabListener.tabRemoved   (e); break;
		case TabEvent.TAB_CHANGED   : tabListener.tabChanged   (e); break;
		}
	}
	
//### private
	/** ֥С礭׻ */
	private void calculateTabbarSize()
	{
		// Ʒ׻
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.calculateWidth();
		
		calculateTabbarHeight();
		
		relayoutTabs();
	}
	
	/** ֥Сι⤵Ʒ׻ */
	private void calculateTabbarHeight()
	{
		synchronized (panels) {
			// ⤵Ʒ׻
			tabbarHeight = Math.max((  activeTitleFont != null ? getFontMetrics(  activeTitleFont).getHeight() : 0),
			                        (unactiveTitleFont != null ? getFontMetrics(unactiveTitleFont).getHeight() : 0));
			
			// βι⤵׻
			for (int i = 0; i < images.size(); i++)
				tabbarHeight = Math.max(tabbarHeight, ((Dimension)images.elementAt(i)).height);
			
			tabbarHeight += TABBAR_HEIGHT_FIX * 2;
		}
	}
	
	/** ٤ƤΥ̵֤ */
	private void invalidateAllTabs()
	{
		for (int i = 0; i < panels.size(); i++)
			((TabPanel)panels.elementAt(i)).tab.invalidate();
	}
	
	/** ٥Ȥȯ */
	private void postEvent(String key, int id)
	{
		getToolkit().getSystemEventQueue().postEvent(new TabEvent(this, key, id));
	}
	
	/** Τ֤ */
	private void relayoutTabs()
	{
		// ֥Сɽ뤫ɤꤹ
		if (tabbar != null && isVisible()) {
			boolean visible = (panels.size() > 1 &&
			                   getSize().width < tabbar.getPreferredSize().width);
			
			if (tabScrollbar.isVisible() != visible)
				tabScrollbar.setVisible(visible);
		}
		
		// 
		validate();
//		tabbar.repaint();
		
		// ̡֤ƥ֥֤ɽƤʤɽ
		synchronized (panels) {
			if (panels.size() <= 1)
				return;
			
			if (activeTabPanel != null)
				tabbarLayout.show(tabbar, activeTabPanelIndex);
		}
		
		// ֥Сɽ
		repaintTabScroll();
	}
	
	/** ֥Сɽ */
	private void repaintTabScroll()
	{
		AlignLayout.Status status = tabbarLayout.getStatus();
		
		boolean b = (status.top <= activeTabPanelIndex &&
		             activeTabPanelIndex <= status.last);
		if (b != activeVisible) {
			activeVisible = b;
			// ƥ֥ѥͥ뤬üФäꤷˡ
			// ƥ֥ѥͥκΥ֤褹
			if (status.last == activeTabPanelIndex && status.last > 0)
				((TabPanel)panels.elementAt(status.last - 1)).tab.refresh();
			else if (status.last >= 0 && status.last + 1 == activeTabPanelIndex)
				((TabPanel)panels.elementAt(status.last)).tab.refresh();
		}
		
		// üΥƥ֥ѥͥκ¦
		repaint(0, 0, 2, tabbar.getSize().height);
		
		tabScrollbar.setValues(status.top, status.last, status.total);
	}
	
	/**  */
	private void paintUnderLine(Graphics g, Dimension size)
	{
		g.setColor(activeTitleBackground);
		g.drawLine(0, size.height - 1, size.width - 1, size.height - 1);
		g.setColor(unactiveTitleBackground.brighter());
		g.drawLine(0, size.height - 2, size.width - 1, size.height - 2);
	}
	
//### TabPanel
	/** ֥ѥͥ */
	private class TabPanel
	{
		private String    key;
		private Tab       tab;
		private Component comp;
		
		private int       index;
		private int       active;
		
		/** 󥹥󥹤 */
		private TabPanel(String key, String title, Image image, Component comp)
		{
			this.key   = key;
			this.tab   = new Tab(title, image);
			this.comp  = comp;
			
			tab.addMouseListener(
				new MouseAdapter() {
					/** å줿 */
					public void mouseClicked(MouseEvent e)
					{
						if (((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK) &&
						    e.getClickCount() == 1) {
							// ʬ򥢥ƥ֤ˤ
							if (active != ACTIVE)
								setActived(true);
							tab.requestFocus();
						}
					}
				}
			);
			
			tab.addKeyListener(
				new KeyAdapter() {
					/** 줿 */
					public void keyPressed(KeyEvent e)
					{
						switch (e.getKeyCode()) {
						case KeyEvent.VK_LEFT:
						case KeyEvent.VK_PAGE_UP:
							synchronized (panels) {
								int index = panels.indexOf(TabPanel.this);
								if (index <= 0)
									return;
								TabPanel panel = (TabPanel)panels.elementAt(index - 1);
								panel.setActived(true);
								panel.tab.requestFocus();
							}
							break;
						case KeyEvent.VK_RIGHT:
						case KeyEvent.VK_PAGE_DOWN:
							synchronized (panels) {
								int index = panels.indexOf(TabPanel.this);
								if (index + 1 >= panels.size())
									return;
								TabPanel panel = (TabPanel)panels.elementAt(index + 1);
								panel.setActived(true);
								panel.tab.requestFocus();
							}
							break;
						case KeyEvent.VK_HOME:
							synchronized (panels) {
								TabPanel panel = (TabPanel)panels.elementAt(0);
								panel.setActived(true);
								panel.tab.requestFocus();
							}
							break;
						case KeyEvent.VK_END:
							synchronized (panels) {
								TabPanel panel = (TabPanel)panels.elementAt(panels.size() - 1);
								panel.setActived(true);
								panel.tab.requestFocus();
							}
							break;
						}
					}
				}
			);
		}
		
		/** ȥɲ */
		private void attach(int index)
		{
			this.index = index;
			
			// ǥեȤΥꤷƤ
			comp.setSize(main.getSize());
			
			tabbar.add(tab , new Integer(index));
			main  .add(comp, key);
			
			if (index > 0) {
				TabPanel panel = (TabPanel)panels.elementAt(index - 1);
				if (panel.active == BEFORE) {
					panel.changeActived(NOACTIVE);
					changeActived(BEFORE);
				}
			}
		}
		
		/** ȥ뤫 */
		private void detach(int index)
		{
			tabbar.remove(tab );
			main  .remove(comp);
			
			if (index > 0 && active == BEFORE)
				((TabPanel)panels.elementAt(index - 1)).changeActived(BEFORE);
		}
		
		/** ƥ֤ */
		private boolean setActived(boolean postEvent)
		{
			if (!changeActived(ACTIVE))
				return false;
			
			if (postEvent)
				postEvent(key, TabEvent.ACTIVE_CHANGED);
			
			return true;
		}
		
		/** ƥ֤ѹ */
		private boolean changeActived(int active)
		{
			if (active == this.active)
				return false;
			
			boolean focus = false;
			
			// ֤򸵤᤹
			if (this.active == ACTIVE)
				changeBeforeActive(NOACTIVE);
			if (active == ACTIVE &&
			    activeTabPanel != null &&
			    activeTabPanel != this) {
				focus = activeTabPanel.tab.focus;
				activeTabPanel.changeActived(NOACTIVE);
			}
			
			// ֤ѹ
			this.active = active;
			tab .active = active;
			tab .focusTraversable = (active == ACTIVE);
			if (focus)
				tab.requestFocus();
			
			if (active == ACTIVE) {
				int index = panels.indexOf(this);
				
				tabbarLayout.show(tabbar, index);
				mainLayout  .show(main  , key  );
				
				activeTabPanel      = this;
				activeTabPanelIndex = index;
				
				changeBeforeActive(BEFORE);
				
				repaintTabScroll();
			}
			
			tab.refresh();
			
			return true;
		}
		
		/** ꤵ줿֤Υ֤ξ֤ѹ */
		private void changeBeforeActive(int active)
		{
			int index = panels.indexOf(this);
			
			if (index > 0)
				((TabPanel)panels.elementAt(index - 1)).changeActived(active);
		}
	}
	
//### Tabbar
	/**  */
	private class Tabbar
		extends Panel
	{
		/** 󥹥󥹤 */
		private Tabbar()
		{
		}
		
		/** ѥͥ򹹿 */
		public void update(Graphics g)
		{
			paint(g);
		}
		
		/** ѥͥ */
		public void paint(Graphics g)
		{
			AlignLayout.Status status = tabbarLayout.getStatus();
			Rectangle rect = ((TabPanel)panels.elementAt(status.last)).tab.getBounds();
			Dimension size = getSize();
			int x = rect.x + rect.width;
			int w = size.width  - x;
			
			if (w > 0) {
				g.setColor(TabbedContainer.this.getBackground());
				g.fillRect(x, 0, w, size.height);
			}
			
			// 
			paintUnderLine(g, getSize());
			
			super.paint(g);
		}
	}
	
//### Tab
	/**  */
	private class Tab
		extends Component
	{
		private ImageCreatorComponent icc = new ImageCreatorComponent(this);
		private Object    lock = new Object();
		private String    title;
		private Image     image;
		private Dimension imageSize;
		private int       width;
		private int       active;
		private boolean   focus;
		private boolean   focusTraversable;
		private Image     offScreen;
		private Graphics  og;
		private OffscreenObserver oo;
		
		private Tab(String title, Image image)
		{
			setTitle(title, image);
			
			addFocusListener(
				new FocusListener()
				{
					/** ե */
					public void focusGained(FocusEvent e)
					{
						if (!focus) {
							focus = true;
							refresh();
						}
					}
					
					/** ե򼺤ä */
					public void focusLost(FocusEvent e)
					{
						if (focus) {
							focus = false;
							refresh();
						}
					}
				}
			);
			
			addComponentListener(
				new ComponentAdapter()
				{
					/** ѹ */
					public void componentResized(ComponentEvent e)
					{
						synchronized (lock) {
							if (offScreen != null) {
								if (oo != null) {
									oo.original.flush();
									oo.dispose();
									oo = null;
								}
								og.dispose();
								offScreen.flush();
							}
							
							Dimension size = getSize();
							if (size.width == 0 || size.height == 0) {
								offScreen = null;
								og        = null;
							} else {
								offScreen = createImage(size.width, size.height);
								og        = offScreen.getGraphics();
								refresh();
							}
						}
					}
				}
			);
		}
		
		/** ȥȥ᡼ */
		private boolean setTitle(String title, Image image)
		{
			boolean change = false;
			boolean recalc = false;
			
			if (this.title != title) {
				change = true;
				recalc = true;
				
				this.title = title;
			}
			
			if (this.image != image) {
				change = true;
				recalc = recalc || ((this.image == null) != (image == null));
				
				if (imageSize != null)
					images.removeElement(imageSize);
				
				if (image != null) {
					// ºݤ˥ɤƤߤ
					if (ImageUtils.load(image, TabbedContainer.this)) {
						Dimension newSize = new Dimension(image.getWidth (TabbedContainer.this),
						                                  image.getHeight(TabbedContainer.this));
						
						boolean heightChanged = (imageSize != null && imageSize.height != newSize.height);
						
						if (!recalc)
							recalc = (imageSize != null && imageSize.width != newSize.width);
						imageSize = newSize;
						images.addElement(imageSize);
						
						if (heightChanged)
							calculateTabbarHeight();
					} else {
						image = null;
					}
				}
				
				this.image = image;
				if (image == null)
					this.imageSize = null;
			}
			
			if (!change)
				return false;
			
			if (!recalc || !calculateWidth()) {
				refresh();
				return false;
			}
			
			return true;
		}
		
		/** ʬƷ׻ */
		private boolean calculateWidth()
		{
			int w = 0;
			
			if (image != null) {
				w += imageSize.width;
				w += TABBAR_IMAGE_FIX;
			}
			
			if (title != null)
				w += Math.max(getFontMetrics(  activeTitleFont).stringWidth(title),
				              getFontMetrics(unactiveTitleFont).stringWidth(title));
			w += TABBAR_LEFT_FIX + TABBAR_RIGHT_FIX;
			
			if (w == width)
				return false;
			
			width = w;
			invalidate();
			return true;
		}
		
		/** ե뤳Ȥ */
		public boolean isFocusTraversable()
		{
			return focusTraversable;
		}
		
		/** ǾΥ */
		public Dimension getMinimumSize()
		{
			return new Dimension(minWidth, tabbarHeight);
		}
		
		/** 祵 */
		public Dimension getMaximumSize()
		{
			return getPreferredSize();
		}
		
		/** 侩 */
		public Dimension getPreferredSize()
		{
			return new Dimension(Math.max(Math.min(width, maxWidth), minWidth), tabbarHeight);
		}
		
		/** ֤κԤ */
		public void update(Graphics g)
		{
			paint(g);
		}
		
		/** ֤Ԥ */
		public void paint(Graphics g)
		{
			if (offScreen != null)
				g.drawImage(offScreen, 0, 0, this);
		}
		
		/** ׵ */
		private void refresh()
		{
			synchronized (lock) {
				if (offScreen == null)
					return;
				
				// ξ˴
				if (oo != null) {
					oo.original.flush();
					oo.dispose();
					oo = null;
				}
				
				Dimension size = getSize();
				Color bg = null;
				
				// طʿɤ
				og.setColor(TabbedContainer.this.getBackground());
				og.fillRect(0, 0, size.width, size.height);
				if (active == ACTIVE) {
					og.setColor(bg = activeTitleBackground);
					og.fillRect(0, 1, size.width - 2, size.height - 2);
				} else {
					og.setColor(bg = unactiveTitleBackground);
					og.fillRect(1, 3, size.width - 3, size.height - 5);
				}
				
				int fix = (active == ACTIVE ? 0 : 1);
				int x   = TABBAR_LEFT_FIX + fix;
				
				// ɽ
				if (image != null) {
					int y = (int)((tabbarHeight - imageSize.height) / 2.0 + 0.5);
					
					Image original = createImage(imageSize.width, imageSize.height);
					if (original != null) {
						Graphics g = original.getGraphics();
						g.setColor(bg);
						g.fillRect(0, 0, imageSize.width, imageSize.height);
						g.dispose();
						
						oo = new OffscreenObserver(this, og, x, y + fix, original);
						
						og.drawImage(image, x, y + fix, oo);
						x += imageSize.width;
						x += TABBAR_IMAGE_FIX;
					}
				}
				
				// ʸɽ
				if (title != null) {
					Font font = null;
					if (active == ACTIVE) {
						og.setFont (font = activeTitleFont);
						og.setColor(activeTitleForeground );
					} else {
						og.setFont (font = unactiveTitleFont);
						og.setColor(unactiveTitleForeground );
					}
					FontData    fd = FontData.getInstance(icc, font);
					FontMetrics fm = fd.getFontMetrics();
					int offset = (int)((tabbarHeight - fd.getFullSize().height) / 2.0 + 0.5);
					int y1 = fd.getHalfBase() + offset + fix;
					int y2 = fd.getFullBase() + offset + fix;
					
					// ­ʤɤå
					int     end = size.width - TABBAR_RIGHT_FIX + fix;
					boolean cut = false;
					if (fm.stringWidth(title) + x > end) {
						end -= fm.stringWidth("...");
						cut = true;
					}
					
					char[] cs = new char[1];
					char   c;
					int    w;
					for (int i = 0; i < title.length(); i++) {
						cs[0] = c = title.charAt(i);
						w = fm.charWidth(c);
						if (x + w > end)
							break;
						og.drawChars(cs, 0, 1, x, ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? y1 : y2));
						x += w;
					}
					if (cut)
						og.drawString("...", x, y1);
				}
				
				// Ȥɽ
				if (active == ACTIVE) {
					og.setColor(bg.brighter());
					og.drawLine(0, 0, size.width - 3,  0);
					
					og.setColor(bg.darker());
					og.drawLine(size.width - 2, 2, size.width - 2, size.height - 2);
					
					og.setColor(bg.darker().darker());
					og.drawLine(size.width - 2, 1, size.width - 2, 1);
					og.drawLine(size.width - 1, 2, size.width - 1, size.height - 2);
				} else {
					og.setColor(bg.brighter());
					og.drawLine(1, 3, 1, 3);
					og.drawLine(2, 2, size.width - 3,  2);
					og.drawLine(0, 4, 0, size.height - 1);
					og.drawLine(0, size.height - 2, size.width - 1, size.height - 2);
					
					if (active == BEFORE && activeVisible) {
						og.setColor(activeTitleBackground.brighter());
						og.drawLine(size.width - 1, 1, size.width - 1, 1);
						og.drawLine(size.width - 2, 2, size.width - 2, size.height - 2);
						
						og.setColor(activeTitleBackground);
						og.drawLine(size.width - 1, 2, size.width - 1, size.height - 2);
					} else {
						og.setColor(bg.darker());
						og.drawLine(size.width - 2, 4, size.width - 2, size.height - 3);
						
						og.setColor(bg.darker().darker());
						og.drawLine(size.width - 2, 3, size.width - 2, 3);
						og.drawLine(size.width - 1, 4, size.width - 1, size.height - 3);
					}
				}
				
				// ǲɽ
				og.setColor(activeTitleBackground);
				og.drawLine(0, size.height - 1, size.width - 1, size.height - 1);
				
				// եȤɽ
				if (focus) {
					og.setColor(focusFrameColor);
					GraphicsUtils.drawDashed(og, 1, 3, size.width  - 6, 1, 1, 1, GraphicsUtils.HORIZONTAL);
					GraphicsUtils.drawDashed(og, 1, 3, 1, size.height - 5, 1, 1, GraphicsUtils.VERTICAL  );
					GraphicsUtils.drawDashed(og, 1, size.height - 3, size.width - 6, 1, 1, 1, GraphicsUtils.HORIZONTAL);
					GraphicsUtils.drawDashed(og, size.width - 5, 3, 1, size.height - 5, 1, 1, GraphicsUtils.VERTICAL  );
				}
				
				repaint();
			}
		}
		
		/** ʸĹڤ */
		private String cutWidth(FontMetrics fm, String value, int width)
		{
			if (fm.stringWidth(value) < width)
				return value;
			
			width -= fm.stringWidth("...");
			int length = value.length();
			int w;
			int i = 0;
			for (; i < length; i++) {
				w = fm.charWidth(value.charAt(i));
				if (width - w < 0)
					break;
				width -= w;
			}
			
			return value.substring(0, i) + "...";
		}
	}
	
//### TabScrollbar
	/** ֥С */
	private class TabScrollbar
		extends Panel
	{
		private ImageButton left;
		private ImageButton right;
		
		/** 󥹥󥹤 */
		private TabScrollbar()
		{
			setLayout(new FlowLayout(FlowLayout.CENTER, 0, 0));
		}
		
		/** ܥ */
		private synchronized void setButton(ImageButton left, ImageButton right)
		{
			this.left  = left;
			this.right = right;
			
			Color bg = getBackground();
			left .setBackground(bg);
			right.setBackground(bg);
			
			left .addMouseListener(
				new MouseAdapter()
				{
					private boolean pressed;
					
					/** ޥ줿 */
					public void mousePressed(MouseEvent e)
					{
						if (check(e))
							pressed = true;
					}
					
					/** ޥ줿 */
					public void mouseReleased(MouseEvent e)
					{
						if (check(e)) {
							tabbarLayout.previous(tabbar);
							repaintTabScroll();
						}
						pressed = false;
					}
				}
			);
			right.addMouseListener(
				new MouseAdapter()
				{
					private boolean pressed;
					
					/** ޥ줿 */
					public void mousePressed(MouseEvent e)
					{
						if (check(e))
							pressed = true;
					}
					
					/** ޥ줿 */
					public void mouseReleased(MouseEvent e)
					{
						if (check(e)) {
							tabbarLayout.next(tabbar);
							repaintTabScroll();
						}
						pressed = false;
					}
				}
			);
			
			removeAll();
			add(left );
			add(right);
		}
		
		/** ֤ */
		private synchronized void setValues(int top, int last, int total)
		{
			if (left  != null)
				left .setEnabled((top > 0));
			if (right != null)
				right.setEnabled((last + 1 < total));
		}
		
		/** Ǿ֤ */
		public Dimension getMinimumSize()
		{
			return getPreferredSize();
		}
		
		/** 祵֤ */
		public Dimension getMaximumSize()
		{
			return getPreferredSize();
		}
		
		/** 侩֤ */
		public Dimension getPreferredSize()
		{
			Dimension leftSize  = null, rightSize = null;
			
			if (left  != null)
				leftSize  = left .getPreferredSize();
			if (right != null)
				rightSize = right.getPreferredSize();
			
			return new Dimension((leftSize  != null ? leftSize .width : 0) +
			                     (rightSize != null ? rightSize.width : 0),
			                     Math.max((leftSize  != null ? leftSize .height : 0),
			                              (rightSize != null ? rightSize.height : 0)));
		}
		
		/** ե뤳Ȥ뤫֤ */
		public boolean isFocusTraversable()
		{
			return false;
		}
		
		/** طʿ */
		public void setBackground(Color c)
		{
			super.setBackground(c);
			
			if (left  != null)
				left .setBackground(c);
			if (right != null)
				right.setBackground(c);
		}
		
		/** ɽɽڤؤ */
		public void setVisible(boolean v)
		{
			super.setVisible(v);
			
			if (v) {
				left .repaint();
				right.repaint();
			} else {
				activeVisible = true;
			}
		}
		
		/** ʬɲä줿 */
		public void addNotify()
		{
			super.addNotify();
			
			if (left == null || right == null)
				createDefaultButton();
		}
		
		/** ǥեȥܥ */
		private void createDefaultButton()
		{
			ImageButton left  = new ImageButton(null);
			ImageButton right = new ImageButton(null);
			setButton(left, right);
			
			Image image;
			MemoryImageSource mis;
			int[] bitmap;
			int fg = getForeground().getRGB() | 0xff000000;
			//int bg = getBackground().getRGB() | 0xff000000;
			int bg = 0;
			
			int w = 13;
			int h = 13;
			
			// left
			bitmap = new int[w * h];
			for (int i = 0; i < bitmap.length; i++)
				if (LEFT_IMAGE[i] == 1)
					bitmap[i] = fg;
				else
					bitmap[i] = bg;
			
			mis = new MemoryImageSource(w, h, bitmap, 0, w);
			mis.setAnimated(false);
			left.setImage(left.createImage(mis));
			
			// right
			bitmap = new int[w * h];
			for (int i = 0; i < bitmap.length; i++)
				if (RIGHT_IMAGE[i] == 1)
					bitmap[i] = fg;
				else
					bitmap[i] = bg;
			
			mis = new MemoryImageSource(w, h, bitmap, 0, w);
			mis.setAnimated(false);
			right.setImage(right.createImage(mis));
		}
		
		/** ܥ夫å */
		private boolean check(MouseEvent e)
		{
			Component c = e.getComponent();
			
			if (!c.isEnabled())
				return false;
			if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0)
				return false;
			int x = e.getX();
			int y = e.getY();
			Dimension size = c.getSize();
			return (0 <= x && x < size.width && 0 <= y && y <= size.height);
		}
	}
}
