/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.awt.event.StateEvent;
import net.hizlab.kagetaka.awt.event.StateListener;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.awt.image.OffscreenObserver;

import java.awt.AWTEvent;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.SystemColor;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.image.FilteredImageSource;
import java.awt.image.RGBImageFilter;

/**
 * ᡼դΥɽ饹Ǥ
 * ΥܥϡˤꣴĤΥ᡼̾С̵ˤѤޤ
 * ΤᡢĤξ֤٤ƤΥ᡼ꤹˡȡ
 * Ĥξ֡̾ˤΥ᡼ꤷ줫¾ΣĤ
 * ֤ <code>ImageButton</code> ǺФˡޤ
 * <p>
 * ޤ<code>setHotspot</code> ᥽åɤǡ
 * ֥˥塼ѤΥ᡼ɲä뤳Ȥޤ
 * <p>
 * Ĥξ֤٤ƤΥ᡼ꤷƤ硢
 * <code>setHotspot</code> ᥽åɤǤϡĤΥѥϿ
 * ɬפޤΥ᥽åɤϿĤΰΰ
 * ޥưȡ֥˥塼ѤưԤޤ
 * Ĥξ֤Υ᡼Ƥ硢<code>setHotspot</code> ᥽åɤϡ
 * ĤΰΥѥϿɬפޤξ硢ۥåȥݥåѤ
 * ᡼ϡ̾Υ᡼α¦ɲäޤ
 * <p>
 * Ĥξ֤Υ᡼Ƥ硢᡼Ȱɽʸ
 * Ǥޤξ硢᡼α¦ǥۥåȥݥåȤ꺸¦
 * ʸɽޤ
 * 
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 * 
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.8 $
 */
public class ImageButton
	extends SizedComponent
{
	private static final Font DEFAULT_FONT = Font.decode("Dialog-plain-12");
	private static final int  LABEL_RIGHT  = 3;
	
	/** ̾ξ */
	public static final int NORMAL       = 0;
	/** ޥ褿 */
	public static final int OVER         = 1;
	/** ޥ줿 */
	public static final int DOWN         = 2;
	/** ܥ̵ʾ */
	public static final int DISABLE      = 3;
	/** ޥۥåȥݥåȤξ褿 */
	public static final int HOTSPOT_OVER = 4;
	/** ۥåȥݥåȤ줿 */
	public static final int HOTSPOT_DOWN = 5;
	
	/** @serial طʥ᡼ */
	private Image     backImage;
	/** @serial طʥ᡼ */
	private Dimension backImageSize;
	
	/** @serial ᡼ꥨ */
	private ImageCreatorComponent icc = new ImageCreatorComponent(this);
	
	/** @serial  */
	private int       state;
	/** @serial Ρޥξ */
	private int       stateEnable;
	/** @serial ξ */
	private int       stateDown;
	/** @serial ɽƤ */
	private int       statePaint = -1;
	/** @serial ֥å */
	private Object    stateLock  = new Object();
	
	/** @serial 줿 X  */
	private int       pressX = -1;
	/** @serial 줿 Y  */
	private int       pressY = -1;
	
	/** @serial ۥåȥݥåȥꥢ */
	private Polygon   area;
	/** @serial ᡼ˤۥåȥݥåȥꥢ */
	private Polygon   imageArea;
	
	/** @serial ᡼꤫ */
	private boolean   all;
	/** @serial ̾᡼ */
	private Image     base;
	/** @serial С᡼ */
	private Image     over;
	/** @serial ᡼ */
	private Image     down;
	/** @serial ̵᡼ */
	private Image     disable;
	/** @serial ۥåȥݥåȥС᡼ */
	private Image     sover;
	/** @serial ۥåȥݥåȲ᡼ */
	private Image     sdown;
	/** @serial ۥåȥݥå̵᡼ */
	private Image     sdisable;
	/** @serial ٥ʸ */
	private String    label;
	
	/** @serial ٥륵 */
	private Dimension labelSize;
	/** @serial ̾磻᡼ */
	private Dimension baseSize;
	/** @serial ۥåȥݥåȥ᡼ */
	private Dimension spotSize;
	
	/** @serial ե꡼ */
	private Image    offScreen;
	/** @serial ե꡼󥰥եå */
	private Graphics og;
	/** @serial ե꡼å */
	private Object   ogLock = new Object();
	/** @serial ե꡼󥤥᡼ */
	private OffscreenObserver oob;
	/** @serial ե꡼󥹥ݥåȹ */
	private OffscreenObserver oos;
	
	private transient StateListener stateListener;
	
	/**
	 * ꤵ줿᡼򸵤ˡ󥹥󥹤ޤ
	 * <code>base</code> ǻꤵ줿᡼򸵤ˡ4 Ĥξ֤ޤ
	 * 
	 * @param     base ̾ξ֤Υ᡼
	 */
	public ImageButton(Image base)
	{
		this(base, null, null, null, null, null, null);
	}
	
	/**
	 * ꤵ줿᡼򸵤ˡ󥹥󥹤ޤ
	 * <code>base</code> ǻꤵ줿᡼򸵤ˡ4 Ĥξ֤ޤ
	 * 
	 * @param     label ܥΥ٥롢ɽʤ <code>null</code>
	 * @param     base  ̾ξ֤Υ᡼
	 * @param     spot  ۥåȥݥåȤΥ᡼
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 */
	public ImageButton(String label, Image base, Image spot)
	{
		this(base, null, null, null, null, spot, null);
		setLabel(label);
	}
	
	/**
	 * ꤵ줿᡼򸵤ˡ󥹥󥹤ޤ
	 * 
	 * @param     base ̾ξ
	 * @param     down ܥ󤬲줿
	 * @param     over ޥ˾ä
	 * @param     disable ܥ̵ξ
	 * @param     sarea ۥåȥݥåȤΰ֡
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 * @param     sover ۥåȥݥåȤ褿ȤΥ᡼
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 * @param     sdown ۥåȥݥåȤƤȤΥ᡼
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 */
	public ImageButton(Image base, Image down, Image over, Image disable,
	                   Polygon sarea, Image sover, Image sdown)
	{
		setImageImpl(base, down, over, disable, sarea, sover, sdown);
		
		addMouseListener(
			new MouseAdapter()
			{
				/** ޥ줿 */
				public void mousePressed(MouseEvent e)
				{
					if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0)
						return;
					int x = e.getX();
					int y = e.getY();
					
					if (x < 0 || y < 0 || x >= size.width || y > size.height)
						return;
					
					if (area != null && area.contains(x, y))
						changeState(HOTSPOT_DOWN);
					else
						changeState(DOWN);
					
					pressX = e.getX();
					pressY = e.getY();
				}
				
				/** ޥ줿 */
				public void mouseReleased(MouseEvent e)
				{
					if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0)
						return;
					int x = e.getX();
					int y = e.getY();
					
					if (x < 0 || y < 0 || x >= size.width || y > size.height)
						changeState(NORMAL);
					else if (area != null && area.contains(e.getX(), e.getY()))
						changeState(HOTSPOT_OVER);
					else
						changeState(OVER);
					
					if (e.getClickCount() > 0 &&
					    x >= 0 && y >= 0 && x < size.width && y < size.height &&
					    (x != pressX || y != pressY)) {
						getToolkit().getSystemEventQueue().postEvent(
							new MouseEvent(e.getComponent(),
							               MouseEvent.MOUSE_CLICKED,
							               e.getWhen(),
							               e.getModifiers(),
							               e.getX(),
							               e.getY(),
							               e.getClickCount(),
							               e.isPopupTrigger()));
					}
				}
				
				/** ޥä */
				public void mouseEntered(MouseEvent e)
				{
					if (area != null && area.contains(e.getX(), e.getY()))
						changeState(HOTSPOT_OVER);
					else
						changeState(OVER);
				}
				
				/** ޥФ */
				public void mouseExited(MouseEvent e)
				{
					changeState(NORMAL);
				}
			}
		);
		
		addMouseMotionListener(
			new MouseMotionAdapter()
			{
				/** ޥɥå줿 */
				public void mouseDragged(MouseEvent e)
				{
					if (state == NORMAL)
						return;
					
					int x = e.getX();
					int y = e.getY();
					
					if (x < 0 || y < 0 || x >= size.width || y > size.height)
						return;
					
					changeState(stateDown);
				}
				
				/** ޥư줿 */
				public void mouseMoved(MouseEvent e)
				{
					if (area == null)
						return;
					
					if (area.contains(e.getX(), e.getY()))
						changeState(HOTSPOT_OVER);
					else
						changeState(OVER);
				}
			}
		);
		
		addComponentListener(
			new ComponentAdapter()
			{
				/** ѹ */
				public void componentResized(ComponentEvent e)
				{
					synchronized (ogLock) {
						if (offScreen != null) {
							if (oob != null) { oob.dispose(); oob = null; }
							if (oos != null) { oos.dispose(); oos = null; }
							og.dispose();
							offScreen.flush();
						}
						
						Dimension nowSize = getSize();
						if (nowSize.width == 0 || nowSize.height == 0) {
							offScreen = null;
							og        = null;
						} else {
							offScreen = createImage(nowSize.width, nowSize.height);
							og        = offScreen.getGraphics();
							repaint();
						}
						
						if (area != null &&
						    (nowSize.width > size.width || nowSize.height > size.height))
							calculateArea();
					}
				}
			}
		);
		
		setFont      (DEFAULT_FONT           );
		setForeground(SystemColor.controlText);
		setBackground(SystemColor.control    );
	}
	
	/**
	 * ٤Ƥξ֡ʣġˤΥ᡼ꤵƤ뤫֤ޤ
	 * 
	 * @return    ٤Ƥξ֤Υ᡼ꤵƤ <code>true</code>
	 *            ʳξ <code>false</code>
	 */
	public boolean hasAllImage()
	{
		return all;
	}
	
	/**
	 * ۥåȥݥåȤꤵƤ뤫֤ޤ
	 * 
	 * @return    ۥåȥݥåȤꤵƤ <code>true</code>
	 *            ʳξ <code>false</code>
	 */
	public boolean hasHotspot()
	{
		return (area != null);
	}
	
	/**
	 * ܥɽ٥֤ޤ
	 * 
	 * @return    ܥΥ٥
	 */
	public String getLabel()
	{
		return label;
	}
	
	/**
	 * ܥɽ٥ꤷޤ
	 * 
	 * @param     label ܥΥ٥
	 */
	public void setLabel(String label)
	{
		synchronized (stateLock) {
			if (all)
				return;
			
			if ((this.label == label) ||
			    (this.label != null && label != null && this.label.compareTo(label) == 0))
				return;
			
			this.label     = label;
			this.labelSize = null;
			
			if (label != null) {
				FontMetrics fm = getFontMetrics(getFont());
				labelSize = new Dimension(fm.stringWidth(label) + LABEL_RIGHT, fm.getHeight());
			}
			
			calculateSize();
		}
	}
	
	/**
	 * ᡼ꤷޤ
	 * 
	 * @param     base ̾ξ
	 */
	public void setImage(Image base)
	{
		setImageImpl(base, null, null, null, null, null, null);
	}
	
	/**
	 * ᡼ꤷޤ
	 * 
	 * @param     base ̾ξ
	 * @param     spot ۥåȥݥåȤΥ᡼
	 *                 ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 */
	public void setImage(Image base, Image spot)
	{
		setImageImpl(base, null, null, null, null, spot, null);
	}
	
	/**
	 * ᡼ꤷޤ
	 * 
	 * @param     base ̾ξ
	 * @param     down ܥ󤬲줿
	 * @param     over ޥ˾ä
	 * @param     disable ܥ̵ξ
	 * @param     sarea ۥåȥݥåȤΰ֡
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 * @param     sover ۥåȥݥåȤ褿ȤΥ᡼
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 * @param     sdown ۥåȥݥåȤƤȤΥ᡼
	 *                  ۥåȥݥåȤ¸ߤʤ <code>null</code>
	 */
	public void setImage(Image base, Image down, Image over, Image disable,
	                     Polygon sarea, Image sover, Image sdown)
	{
		setImageImpl(base, down, over, disable, sarea, sover, sdown);
	}
	
	/** ºݤ˥᡼ */
	private void setImageImpl(Image base, Image down, Image over, Image disable,
	                          Polygon sarea, Image sover, Image sdown)
	{
		synchronized (stateLock) {
			if (base    != null && !ImageUtils.load(base   , this)) base    = null;
			if (down    != null && !ImageUtils.load(down   , this)) down    = null;
			if (over    != null && !ImageUtils.load(over   , this)) over    = null;
			if (disable != null && !ImageUtils.load(disable, this)) disable = null;
			if (sover   != null && !ImageUtils.load(sover  , this)) sover   = null;
			if (sdown   != null && !ImageUtils.load(sdown  , this)) sdown   = null;
			
			this.all       = (down != null);
			this.base      = base;
			this.down      = down;
			this.over      = over;
			this.disable   = disable;
			this.area      = (sover != null ? sarea : null);
			this.imageArea = area;
			this.sover     = sover;
			this.sdown     = sdown;
			this.baseSize  = new Dimension(0, 0);
			this.spotSize  = null;
			
			if (base != null) {
				baseSize.width  = base.getWidth (this);
				baseSize.height = base.getHeight(this);
				if (!all) {
					baseSize.width  += 3;
					baseSize.height += 3;
				}
			}
			
			// Ĥξ֥᡼ȡۥåȥݥåȤͭ
			if (!all && sover != null)
				this.spotSize = new Dimension(sover.getWidth (this) + 3,
				                              sover.getHeight(this) + 3);
			
			// ֤Υ᡼ꤵƤϡ٥뤬ѤǤʤ
			if (all) {
				this.label     = null;
				this.labelSize = null;
			}
			
			calculateSize();
			repaint();
		}
	}
	
	/** ΥܥΥ򽸷 */
	private void calculateSize()
	{
		Dimension old = new Dimension(size);
		
		size.width  = baseSize.width;
		size.height = baseSize.height;
		if (spotSize  != null) {
			size.width  += spotSize .width;
			size.height =  Math.max(size.height, spotSize .height);
		}
		if (labelSize != null) {
			size.width  += labelSize.width;
			size.height =  Math.max(size.height, labelSize.height);
		}
		
		calculateArea();
		
		// ȥ㤦Τߺƥ쥤оݤˤ
		if (!old.equals(size))
			invalidate();
	}
	
	/** ΥۥåȥݥåȥꥢΥ򽸷 */
	private void calculateArea()
	{
		if (sover == null)
			return;
		
		Dimension nowSize = getSize();
		
		if (all) {
			// ºݤΥ礭ϥ󥿥󥰤
			if (nowSize.width > size.width || nowSize.height > size.height) {
				int w = (nowSize.width  - size.width ) / 2;
				int h = (nowSize.height - size.height) / 2;
				int   npoints = imageArea.npoints;
				int[] xpoints = new int[npoints];
				int[] ypoints = new int[npoints];
				System.arraycopy(imageArea.xpoints, 0, xpoints, 0, npoints);
				System.arraycopy(imageArea.ypoints, 0, ypoints, 0, npoints);
				for (int i = 0; i < npoints; i++) {
					xpoints[i] += w;
					ypoints[i] += h;
				}
				area = new Polygon(xpoints, ypoints, npoints);
			}
		} else {
			// ºݤΥ礭ϡϥ󥿥󥰤岼Ϲ
			int w = (labelSize != null ? labelSize.width : 0);
			w += (nowSize.width - size.width) / 2;
			area = new Polygon(new int[]{baseSize.width + w,
			                             baseSize.width + w + spotSize.width,
			                             baseSize.width + w + spotSize.width,
			                             baseSize.width + w},
			                   new int[]{0,
			                             0,
			                             nowSize.height,
			                             nowSize.height},
			                   4);
		}
	}
	
	/**
	 * ۥåȥݥåȤȤƤꤵƤ֤֤ޤ
	 * 
	 * @return      ۥåȥݥåȤΰ
	 */
	public Polygon getHotspotArea()
	{
		return area;
	}
	
	/**
	 * ̤򥢥åץǡȤޤ
	 * 
	 * @param     g եå
	 */
	public void update(Graphics g)
	{
		paint(g);
	}
	
	/**
	 * ᡼ºݤ褷ޤ
	 * 
	 * @param     g եå
	 */
	public void paint(Graphics g)
	{
		synchronized (ogLock) {
			synchronized (stateLock) {
				if (state != statePaint)
					refresh();
			}
			
			if (offScreen != null)
				g.drawImage(offScreen, 0, 0, this);
		}
	}
	
	/**
	 * ݡͥȤѲǽˤޤ
	 * 
	 * @param     b Ѳǽˤ <code>true</code>
	 *              ԲĤˤ <code>false</code>
	 */
	public void setEnabled(boolean b)
	{
		super.setEnabled(b);
		
		changeState((b ? stateEnable : DISABLE));
	}
	
	/**
	 * Υ֤Υѥ᡼ʸ֤ޤ
	 * 
	 * @return    ѥ᡼ʸ
	 */
	protected String paramString()
	{
		String str = super.paramString();
		return str + ",state=" + state;
	}
	
	/**
	 * ֥ꥹʤϿޤ
	 * 
	 * @param     l Ͽ֥ꥹ
	 */
	public synchronized void addStateListener(StateListener l)
	{
		enableEvents(0);
		stateListener = AWTEventMulticaster.add(stateListener, l);
	}
	
	/**
	 * ֥ꥹʤޤ
	 * 
	 * @param     l ֥ꥹ
	 */
	public synchronized void removeStateListener(StateListener l)
	{
		stateListener = AWTEventMulticaster.remove(stateListener, l);
	}
	
	/**
	 * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
	 * 
	 * @param     e ٥
	 */
	protected void processEvent(AWTEvent e) {
		if (e instanceof StateEvent) {
			processStateEvent((StateEvent)e);
			return;
		}
		super.processEvent(e);
	}
	
	/**
	 * ΥݡͥȤȯѹ٥Ȥ
	 * ϿƤ뤹٤Ƥ {@link StateListener} 뤳Ȥˤꡢ
	 * ѹ٥Ȥޤ
	 * 
	 * @param     e ٥
	 */
	protected void processStateEvent(StateEvent e) {
		if (stateListener == null)
			return;
		
		switch (e.getID()) {
		case StateEvent.STATE_CHANGED: stateListener.stateChanged(e); break;
		}
	}
	
	/** ѹ٥Ȥȯޤ */
	private void postStateEvent(int state)
	{
		StateEvent e = new StateEvent(this, StateEvent.STATE_CHANGED, state);
		try {
			getToolkit().getSystemEventQueue().postEvent(e);
		} catch (SecurityException ex) {
			processStateEvent(e);
		}
	}
	
	/**
	 * طʲꤷޤ
	 * 
	 * @param     image 
	 */
	public void setBackImage(Image image)
	{
		backImage = image;
		if (image != null && !ImageUtils.load(image, this)) image = null;
		
		if (image != null)
			backImageSize = new Dimension(image.getWidth(this), image.getHeight(this));
	}
	
	/** ߤΥơˤä֤˺ */
	private void refresh()
	{
		synchronized (ogLock) {
			if (offScreen == null)
				return;
			
			if (oob != null) { oob.dispose(); oob = null; }
			if (oos != null) { oos.dispose(); oos = null; }
			
			Dimension nowSize = getSize();
			Point     p       = new Point((nowSize.width - size.width ) / 2, 0);
			
			Color bg = getBackground();
			og.setColor(bg);
			og.fillRect(0, 0, nowSize.width, nowSize.height);
			if (backImage != null) {
				int x = 0, y = 0;
				for (x = 0; x < nowSize.width; x += backImageSize.width)
					for (y = 0; y < nowSize.height; y += backImageSize.height)
						og.drawImage(backImage, x, y, this);
			}
			
			Image image = null, simage = null;
			if (all) {
				switch (state) {
				case NORMAL      : image = base   ; break;
				case OVER        : image = over   ; break;
				case DOWN        : image = down   ; break;
				case DISABLE     : image = disable; break;
				case HOTSPOT_OVER: image = sover  ; break;
				case HOTSPOT_DOWN: image = sdown  ; break;
				}
			} else {
				int sw = baseSize.width + (labelSize != null ? labelSize.width : 0);
				image  = base;
				simage = sover;
				switch (state) {
				case NORMAL:
					p.x += 1;
					p.y += 1;
					break;
				case OVER:
				case HOTSPOT_OVER:
					og.setColor(bg.brighter());
					og.drawLine(p.x, p.y, p.x + nowSize.width  - 2, p.y);
					og.drawLine(p.x, p.y, p.x, p.y + nowSize.height - 2);
					if (area != null)
						og.drawLine(p.x + sw, p.y, p.x + sw, p.y + nowSize.height - 2);
					og.setColor(bg.darker());
					og.drawLine(p.x, p.y + nowSize.height - 1, p.x + nowSize.width - 1, p.y + nowSize.height - 1);
					og.drawLine(p.x + nowSize.width  - 1, p.y, p.x + nowSize.width - 1, p.y + nowSize.height - 1);
					if (area != null)
						og.drawLine(p.x + sw - 1, p.y, p.x + sw - 1, p.y + nowSize.height - 1);
					p.x += 1;
					p.y += 1;
					break;
				case DOWN:
					og.setColor(bg.darker());
					og.drawLine(p.x, p.y, p.x + nowSize.width  - 1, p.y);
					og.drawLine(p.x, p.y, p.x, p.y + nowSize.height - 1);
					if (area != null)
						og.drawLine(p.x + sw, p.y, p.x + sw, p.y + nowSize.height - 2);
					og.setColor(bg.brighter());
					og.drawLine(p.x + 1, p.y + nowSize.height - 1, p.x + nowSize.width - 1, p.y + nowSize.height - 1);
					og.drawLine(p.x + nowSize.width  - 1, p.y + 1, p.x + nowSize.width - 1, p.y + nowSize.height - 1);
					if (area != null)
						og.drawLine(p.x + sw - 1, p.y, p.x + sw - 1, p.y + nowSize.height - 1);
					p.x += 2;
					p.y += 2;
					break;
				case DISABLE:
					if (disable == null)
						disable = createImage(new FilteredImageSource(base.getSource(),
						                                              new GrayFilter(getBackground())));
					if (area != null && sdisable == null)
						sdisable = createImage(new FilteredImageSource(sover.getSource(),
						                                               new GrayFilter(getBackground())));
					image  = disable;
					simage = sdisable;
					p.x += 1;
					p.y += 1;
					break;
				case HOTSPOT_DOWN:
					og.setColor(bg.brighter());
					og.drawLine(p.x, p.y, p.x + sw - 2, p.y);
					og.drawLine(p.x, p.y, p.x, p.y + nowSize.height - 2);
					og.drawLine(p.x + nowSize.width  - 1, p.y, p.x + nowSize.width - 1, p.y + nowSize.height - 1);
					og.drawLine(p.x + sw, p.y + nowSize.height - 1, p.x + nowSize.width - 2, p.y + nowSize.height - 1);
					og.setColor(bg.darker());
					og.drawLine(p.x, p.y + nowSize.height - 1, p.x + sw - 1, p.y + nowSize.height - 1);
					og.drawLine(p.x + sw - 1, p.y, p.x + sw - 1, p.y + nowSize.height - 2);
					og.drawLine(p.x + sw, p.y, p.x + nowSize.width - 2, p.y);
					og.drawLine(p.x + sw, p.y, p.x + sw, p.y + nowSize.height - 2);
					p.x += 1;
					p.y += 1;
					break;
				}
			}
			
			int offset;
			
			if (image != null) {
				offset = (nowSize.height - baseSize.height ) / 2;
				oob = new OffscreenObserver(this, og, p.x, p.y + offset, null);
				og.drawImage(image, p.x, p.y + offset, oob);
				p.x += baseSize.width;
			}
			
			// ٥
			if (label != null) {
				Font font = getFont();
				og.setFont (font);
				Color color = getForeground();
				if (state == DISABLE)
					color = new Color((new GrayFilter(getBackground())).filterRGB(0, 0, color.getRGB()));
				og.setColor(color);
				FontData    fd = FontData.getInstance(icc, font);
				FontMetrics fm = fd.getFontMetrics();
				offset = (int)((nowSize.height - fd.getFullSize().height - 3) / 2.0 + 0.5);
				int y1 = fd.getHalfBase() + offset + p.y;
				int y2 = fd.getFullBase() + offset + p.y;
				
				char[] cs = new char[1];
				char   c;
				int    w;
				for (int i = 0; i < label.length(); i++) {
					cs[0] = c = label.charAt(i);
					w = fm.charWidth(c);
					og.drawChars(cs, 0, 1, p.x, ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? y1 : y2));
					p.x += w;
				}
				p.x += LABEL_RIGHT;
			}
			
			// ݥåȥ᡼
			if (simage != null) {
				if (state == HOTSPOT_DOWN) {
					p.x += 1;
					p.y += 1;
				}
				
				offset = (nowSize.height - spotSize.height ) / 2;
				oos = new OffscreenObserver(this, og, p.x, p.y + offset, null);
				og.drawImage(simage, p.x, p.y + offset, oos);
			}
			
			statePaint = state;
		}
	}
	
	/**
	 * ֤֤ޤ
	 * 
	 * @return    
	 */
	public int getState()
	{
		return state;
	}
	
	/** ֤ѹ */
	private void changeState(int s)
	{
		synchronized (stateLock) {
			if (s != DISABLE)
				stateEnable = s;
			
			if (!isEnabled())
				s = DISABLE;
			
			if (state != s) {
				state = s;
				
				switch (state) {
				case DOWN        :
				case HOTSPOT_DOWN:
					stateDown = state; break;
				}
				
				if (isVisible()) {
					Graphics g = getGraphics();
					if (g != null)
						try {
							update(g);
						} finally {
							g.dispose();
						}
					else
						repaint();
				}
				
				postStateEvent(s);
			}
		}
	}
}
