/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer.ssl;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.InputBox;
import net.hizlab.kagetaka.io.SafetyFileOutputStream;
import net.hizlab.kagetaka.net.DistinguishedName;
import net.hizlab.kagetaka.net.SSL;
import net.hizlab.kagetaka.net.HostnameWrongException;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.awt.Frame;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URLConnection;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.cert.CertificateException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * SSL б뤿Υޥ͡Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.2 $
 */
public class SSLManager
       extends net.hizlab.kagetaka.viewer.SSLManager {

    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.ssl.Resources";

    private static final String STORE_JAVA_PREFIX   = "J.";
    private static final String STORE_SYSTEM_PREFIX = "S.";
    private static final String STORE_USER_PREFIX   = "U.";

    private ViewerOption   option;
    private Map            factories;
    private Map            trusted;
    private SecureRandom   secureRandom;
    private KeyStore       keyStore;
    private KeyStore       trustStore;
    private KeyManager  [] keyManagers;
    private TrustManager[] trustManagers;
    private WindowListener windowListener;

    private File           userTrustStorePath;
    private char[]         userTrustStorePass;

    /**
     * SSL ޥ͡ޤ
     *
     * @param  option ץ
     */
    SSLManager(ViewerOption option) {
        this.option    = option;
        this.factories = new HashMap();
    }

    /** ץ󤫤顢SSL  */
    private synchronized void initialize() {
        final String sep = File.separator;

        File   systemTrustStorePath = null;
        String javaHome = System.getProperty("java.home");
        if (javaHome != null) {
            systemTrustStorePath = new File(javaHome + sep + "lib" + sep + "security" + sep + "cacerts");
        }

        trusted      = Collections.synchronizedMap(new HashMap());
        secureRandom = new SecureRandom();

        load(option.getPropertyFile(ViewerOption.KEY_SSL_KEYSTORE_SYSTEM_PATH  ), STORE_SYSTEM_PREFIX,
             option.getPropertyFile(ViewerOption.KEY_SSL_KEYSTORE_USER_PATH    ), STORE_USER_PREFIX  ,
             systemTrustStorePath                                               , STORE_JAVA_PREFIX  ,
             option.getPropertyFile(ViewerOption.KEY_SSL_TRUSTSTORE_SYSTEM_PATH), STORE_SYSTEM_PREFIX,
             option.getPropertyFile(ViewerOption.KEY_SSL_TRUSTSTORE_USER_PATH  ), STORE_USER_PREFIX  );
    }

    /** ꤵ줿ե뤫 SSL ɤ߹ */
    private void load(File systemKeyStorePath  , String systemKeyStorePrefix  ,
                      File userKeyStorePath    , String userKeyStorePrefix    ,
                      File javaTrustStorePath  , String javaTrustStorePrefix  ,
                      File systemTrustStorePath, String systemTrustStorePrefix,
                      File userTrustStorePath  , String userTrustStorePrefix  ) {
        // KeyStore
        keyStore = getKeyStore(null, null, null, null);
        getKeyStore(systemKeyStorePath, null, keyStore, systemKeyStorePrefix);
        getKeyStore(userKeyStorePath  , null, keyStore, userKeyStorePrefix  );

        // TrustStore
        trustStore = getKeyStore(null, null, null, null);
        getKeyStore(javaTrustStorePath  , null, trustStore, javaTrustStorePrefix  );
        getKeyStore(systemTrustStorePath, null, trustStore, systemTrustStorePrefix);
        getKeyStore(userTrustStorePath  , null, trustStore, userTrustStorePrefix  );
        this.userTrustStorePath = userTrustStorePath;

        // 르ꥺ
        String keyManagerFactoryAlgorithm   = KeyManagerFactory  .getDefaultAlgorithm();
        String trustManagerFactoryAlgorithm = TrustManagerFactory.getDefaultAlgorithm();

        // KeyManagerFactory κ
        KeyManagerFactory keyManagerFactory;
        try {
            keyManagerFactory = KeyManagerFactory.getInstance(keyManagerFactoryAlgorithm);
        } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
            return;
        }
        try {
            keyManagerFactory.init(keyStore, null);
        } catch (UnrecoverableKeyException e) {
Debug.out.println(e);
            return;
        } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
            return;
        } catch (KeyStoreException e) {
Debug.out.println(e);
            return;
        }

        // TrustManagerFactory κ
        TrustManagerFactory trustManagerFactory;
        try {
            trustManagerFactory = TrustManagerFactory.getInstance(trustManagerFactoryAlgorithm);
        } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
            return;
        }
        try {
            trustManagerFactory.init(trustStore);
        } catch (KeyStoreException e) {
Debug.out.println(e);
            return;
        }

        keyManagers   = keyManagerFactory  .getKeyManagers  ();
        trustManagers = trustManagerFactory.getTrustManagers();

        // ꥹʡκ
        windowListener = new WindowAdapter() {
            /** ɥĤ */
            public void windowClosed(WindowEvent e) {
                factories.remove(e.getWindow());
            }
        };
    }

    /** ȥɤ߹ baseStore ˥ԡ */
    private KeyStore getKeyStore(File path, char[] pass,
                                 KeyStore baseStore, String prefix) {
        if (baseStore != null
                && (path == null || !path.exists() || !path.canRead())) {
            return null;
        }

        try {
            KeyStore keyStore = KeyStore.getInstance("JKS");

            try {
                keyStore.load((path != null
                               ? new FileInputStream(path)
                               : null),
                              pass);
            } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
                return null;
            } catch (CertificateException e) {
Debug.out.println(e);
                return null;
            } catch (FileNotFoundException e) {
Debug.out.println(e);
                return null;
            } catch (IOException e) {
Debug.out.println(e);
                return null;
            }

            if (baseStore != null) {
                String alias;
                for (Enumeration e = keyStore.aliases(); e.hasMoreElements();) {
                    alias = (String) e.nextElement();
                    baseStore.setCertificateEntry(prefix + alias,
                                                  keyStore.getCertificate(alias));
                }
            }

            return keyStore;
        } catch (KeyStoreException e) {
Debug.out.println(e);
        }

        return null;
    }

    /* ꤵ줿ͥ SSL 򥻥åȥå */
    /** {@inheritDoc} */
    public void setup(Frame owner, URLConnection connection) {
        if (!(connection instanceof HttpsURLConnection)) {
            return;
        }

        SSLGroup sslGroup;

        synchronized (this) {
            if ((sslGroup = (SSLGroup) factories.get(owner)) == null) {
                if (trusted == null) {
                    initialize();
                }

                // TrustManager 
                TrustManager[] trustManagers = this.trustManagers;
                if (trustManagers != null
                        && trustManagers.length > 0
                        && trustManagers[0] instanceof X509TrustManager) {
                    trustManagers = new TrustManager[1];
                    trustManagers[0] = new LooseX509TrustManager(this,
                                                                 (X509TrustManager) this.trustManagers[0],
                                                                 owner, trusted);
                }

                SSLContext sslContext;
                try {
                    sslContext = SSLContext.getInstance("SSL");
                } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
                    return;
                }
                try {
                    sslContext.init(keyManagers,
                                    trustManagers,
                                    secureRandom);
                } catch (KeyManagementException e) {
Debug.out.println(e);
                    return;
                }

                sslGroup = new SSLGroup(sslContext.getSocketFactory(),
                                        new LooseHostnameVerifier(owner));

                owner.addWindowListener(windowListener);
                factories.put(owner, sslGroup);
            }
        }

        HttpsURLConnection httpsConnection = (HttpsURLConnection) connection;
        httpsConnection.setSSLSocketFactory(sslGroup.sslSocketFactory);
        httpsConnection.setHostnameVerifier(sslGroup.hostnameVerifier);
    }

    /* ̵뤹㳰ɤ */
    /** {@inheritDoc} */
    public boolean ignored(IOException exception) {
        return (SSL.hasCertificateException(exception)
             || exception instanceof HostnameWrongException);
    }

    /* ͥ󤫤 SSL  */
    /** {@inheritDoc} */
    public net.hizlab.kagetaka.viewer.SSLCertification getSSLCertification(URLConnection connection) {
        if (!(connection instanceof HttpsURLConnection)) {
            return null;
        }

        HttpsURLConnection httpsConnection = (HttpsURLConnection) connection;
        Certificate[] localCertificates  = httpsConnection.getLocalCertificates();
        Certificate[] serverCertificates = null;
        try {
            serverCertificates = httpsConnection.getServerCertificates();
        } catch (SSLPeerUnverifiedException e) { }

        return new SSLCertification(localCertificates, serverCertificates);
    }

    /**
     * TrustStore ɲäޤ
     *
     * @param  certificate ɲä X509 
     * @param  save        ¸ <code>true</code>
     *                     ¸ʤ <code>false</code>
     * @param  owner       ʡ
     */
    synchronized void addCertificateEntry(X509Certificate certificate,
                                          boolean save, Frame owner) {
        DistinguishedName dn = new DistinguishedName(certificate.getSubjectX500Principal().getName());
        String alias  = dn.getCommonName();

        try {
            // alias ʣϡˡ.nפղ
            String suffix = "";
            int    index  = 0;
            while (trustStore.containsAlias(STORE_USER_PREFIX + alias + suffix)) {
                suffix = "." + (++index);
            }
            trustStore.setCertificateEntry(STORE_USER_PREFIX + alias + suffix,
                                           certificate);
        } catch (KeyStoreException e) {
Debug.out.println(e);
        }

        // ե¸
        if (save) {
            KeyStore ts;
            if (userTrustStorePath.exists()) {
                // ե뤬¸ߤ硢ɤ߹ǥѥɤå
                if (userTrustStorePass == null) {
                    if ((userTrustStorePass = getPassword(owner, "exists")) == null) {
                        return;
                    }
                }
                // ɤ߹
                for (;;) {
                    ts = getKeyStore(userTrustStorePath, userTrustStorePass, null, null);
                    if (ts != null) {
                        break;
                    }
                    if ((userTrustStorePass = getPassword(owner, "retry")) == null) {
                        return;
                    }
                }
            } else {
                // ե뤬¸ߤʤ硢ѥɤ
                if (userTrustStorePass == null) {
                    if ((userTrustStorePass = getPassword(owner, "new")) == null) {
                        return;
                    }
                }
                ts = getKeyStore(null, null, null, null);
            }
            if (ts == null) {
                return;
            }

            try {
                String suffix = "";

                if (ts.size() > 0) {
                    // 񤬴¸ߤʤå
                    Certificate c;
                    for (Enumeration e = ts.aliases(); e.hasMoreElements();) {
                        c = ts.getCertificate((String) e.nextElement());
                        if (c != null && c.equals(certificate)) {
                            return;
                        }
                    }

                    // alias ʣϡˡ.nפղ
                    int    index  = 0;
                    while (ts.containsAlias(alias + suffix)) {
                        suffix = "." + (++index);
                    }
                }

                ts.setCertificateEntry(alias + suffix, certificate);

                // ¸
                SafetyFileOutputStream os = null;
                try {
                    // ե˽񤭽Ф
                    os = new SafetyFileOutputStream(userTrustStorePath);
                    ts.store(os, userTrustStorePass);
                    os.close();
                    os = null;
                } finally {
                    if (os != null) {
                        try {
                            os.abort();
                        } catch (IOException e) { }
                    }
                }
            } catch (KeyStoreException e) {
Debug.out.println(e);
            } catch (IOException e) {
Debug.out.println(e);
            } catch (NoSuchAlgorithmException e) {
Debug.out.println(e);
            } catch (CertificateException e) {
Debug.out.println(e);
            }
        }
    }

    /** ѥɤ */
    private char[] getPassword(Frame owner, String key) {
        String pass =  InputBox.show(owner,
                                     getMessage("password.message." + key),
                                     getMessage("password.title"  ),
                                     "", true);
        return (pass != null ? pass.toCharArray() : null);
    }

    /** ꥽ʸ */
    private String getMessage(String key) {
        if (key == null) {
            return "";
        }

        return Resource.getMessage(RESOURCE, "sslmanager." + key, null);
    }

//### SSLGroup
    /** ɥȤ SSL Υǡ */
    private final class SSLGroup {
        private SSLSocketFactory sslSocketFactory;
        private HostnameVerifier hostnameVerifier;

        /** 󥹥󥹤 */
        private SSLGroup(SSLSocketFactory sslSocketFactory,
                         HostnameVerifier hostnameVerifier) {
            this.sslSocketFactory = sslSocketFactory;
            this.hostnameVerifier = hostnameVerifier;
        }
    }
}
