/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.io.TextOutputStream;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.UnknownHostException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.StringTokenizer;

import javax.net.ssl.HandshakeCompletedEvent;
import javax.net.ssl.HandshakeCompletedListener;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * HTTPS 饤ȤδŪʵǽ󶡤ޤ
 * Υ饹ϥåɥդǤϤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.1 $
 */
public final class HttpsConnection extends HttpConnection {
    /** ȥͥ */
    final HttpClient tunnel;
    /** SSL åȥեȥ */
    final SSLSocketFactory sslSocketFactory;

    private HostnameVerifier hostnameVerifier;
    private SSLSocket        sslSocket;
    private SSLSession       sslSession;

    /**
     * ꤷۥȤȥݡȤ³ HTTPS 饤Ȥޤ
     *
     * @param  protocol         ץȥ
     * @param  host             ФΥۥ
     * @param  port             ФΥݡ
     * @param  proxy            Фץξ <code>true</code>
     *                          ʳξ <code>false</code>
     * @param  sslSocketFactory SSL åȥեȥ
     * @param  hostnameVerifier ۥ̾٥ե
     *
     * @throws UnknownHostException aa ʥۥȤꤷ
     * @throws IOException IO 顼ȯ
     */
    HttpsConnection(String protocol, String host, int port,
                    boolean proxy,
                    SSLSocketFactory sslSocketFactory,
                    HostnameVerifier hostnameVerifier)
            throws UnknownHostException, IOException {
        super(protocol, host, port, proxy);
        this.tunnel           = null;
        this.sslSocketFactory = sslSocketFactory;
        this.hostnameVerifier = hostnameVerifier;
    }

    /**
     * ȥͥͳơꤷۥȤȥݡȤ³
     * HTTPS 饤Ȥޤ
     *
     * @param  protocol         ץȥ
     * @param  host             ФΥۥd
     * @param  port             ФΥݡ
     * @param  tunnel           ȥͥ
     * @param  sslSocketFactory SSL åȥեȥ
     * @param  hostnameVerifier ۥ̾٥ե
     *
     * @throws UnknownHostException ʥۥȤꤷ
     * @throws IOException IO 顼ȯ
     */
    HttpsConnection(String protocol, String host, int port,
                    HttpClient tunnel,
                    SSLSocketFactory sslSocketFactory,
                    HostnameVerifier hostnameVerifier)
            throws UnknownHostException, IOException {
        super(protocol, host, port, false);
        this.tunnel           = tunnel;
        this.sslSocketFactory = sslSocketFactory;
        this.hostnameVerifier = hostnameVerifier;
    }

    /* ۥȤ³ */
    /** {@inheritDoc} */
    void open() throws IOException {
        if (serverSocket != null) {
            throw new IOException("Already opened");
        }

        if (usingProxy) {
            // HTTPS ȥͥξ
            super.open();
            return;
        } else if (tunnel == null) {
            sslSocket = (SSLSocket) sslSocketFactory.createSocket(serverHost, serverPort);
        } else {
            sslSocket = (SSLSocket) sslSocketFactory.createSocket(tunnel.connection.serverSocket, serverHost, serverPort, false);
        }

        // SSLSocket 
        String[] protocols = getProtocols   ();
        String[] ciphers   = getCipherSuites();
        if (protocols != null) {
            sslSocket.setEnabledProtocols   (protocols);
        }
        if (ciphers   != null) {
            sslSocket.setEnabledCipherSuites(ciphers  );
        }
        sslSocket.addHandshakeCompletedListener(
            new HandshakeCompletedListener() {
                /** SSL ϥɥλ */
                public void handshakeCompleted(HandshakeCompletedEvent event) {
                    // Ф³ʤȤ餷Τ
                    sslSession = event.getSession();
                }
            }
        );

        // ³
        sslSocket.startHandshake();
        sslSession = sslSocket.getSession();

        serverSocket = sslSocket;
        serverOutput = new TextOutputStream(
                         new BufferedOutputStream(serverSocket.getOutputStream()),
                         NetworkClient.defaultOutputEncoding);
        serverInput  = serverSocket.getInputStream();

        serverSocket.setTcpNoDelay(true);

        // å
        checkURLSpoofing();

Debug.out.println("% " + Integer.toHexString(hashCode()) + (!usingProxy ? ", opened: " : ", openet: ") + protocol + "://" + serverHost + ":" + serverPort + (tunnel != null ? ", " + Integer.toHexString(tunnel.connection.hashCode()) : ""));
    }

    /* ³Ĥ */
    /** {@inheritDoc} */
    void closeImpl() throws IOException {
        try {
            super.closeImpl();
        } finally {
            if (tunnel != null) {
                tunnel.dispose();
            }
        }
    }

    /**
     * SSL Υå֤ޤ
     *
     * @return SSL å
     */
    SSLSession getSSLSession() {
        return sslSession;
    }

    /**
     * ۥ̾٥ե֤ޤ
     *
     * @return ۥ̾٥ե
     */
    HostnameVerifier getHostnameVerifier() {
        return hostnameVerifier;
    }

    /**
     * ۥ̾٥եꤷޤ
     *
     * @param  hostnameVerifier ۥ̾٥ե
     */
    void setHostnameVerifier(HostnameVerifier hostnameVerifier) {
        this.hostnameVerifier = hostnameVerifier;
    }

    /* Keep-Alive ͭ */
    /** {@inheritDoc} */
    boolean isKeepingAlive() {
        return (!usingProxy
                ? super.isKeepingAlive()
                : false);  // HTTPS ץ Keep-Alive ̵
    }

    /** Ѳǽ SSL ץȥ */
    private String[] getProtocols() {
        return getPropertyValues("https.protocols");
    }

    /** ѲǽʰŹ */
    private String[] getCipherSuites() {
        return getPropertyValues("https.cipherSuites");
    }

    /** ץѥƥͤ */
    private String[] getPropertyValues(String key) {
        String value = System.getProperty(key);
        if (value == null || value.length() == 0) {
            return null;
        }

        StringTokenizer st = new StringTokenizer(value, ",");
        ArrayList list = new ArrayList();

        while (st.hasMoreElements()) {
            list.add(st.nextElement());
        }

        return (String[]) list.toArray(new String[list.size()]);
    }

    /** URL Τʤå */
    private void checkURLSpoofing()
            throws IOException {

        String host = serverHost;

        // IPv6 б
        if (host != null && host.startsWith("[") && host.endsWith("]")) {
            host = host.substring(1, host.length() - 1);
        }

        try {
            Certificate[] certificate = sslSession.getPeerCertificates();
            if (certificate != null
                    && certificate.length > 0
                    && certificate[0] instanceof X509Certificate) {

                // å return
                if (HostnameChecker.match(host, (X509Certificate) certificate[0])) {
                    return;
                }
            }
        } catch (SSLPeerUnverifiedException e) {
            // ̵
        }

        String cipher = sslSession.getCipherSuite();
        if ((cipher != null) && (cipher.indexOf("_anon_") != -1)) {
            return;
        }

        // ۥ̾å
        if (hostnameVerifier != null
                && hostnameVerifier.verify(host, sslSession)) {
            return;
        }

        serverSocket.close();
        sslSession.invalidate();

        throw new HostnameWrongException("HTTPS hostname wrong:  should be <" + host + ">");
    }
}
