/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer.history;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.FileNotFoundException;
import java.io.InvalidObjectException;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;

/**
 * 륯饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class HistoryManager {
    private static final String ENCODING = "Unicode";
    private static final int LF = 0x0A;

    private ViewerOption  option;
    private HistoryEditor editor;

    private File      historyFile;
    private History   top;
    private History   last;
    private Hashtable hash = new Hashtable();

    /**
     * ҥȥޥ͡ޤ
     *
     * @param  option ץ
     */
    public HistoryManager(ViewerOption option) {
        this.option = option;
        load(option.getPropertyFile(ViewerOption.KEY_HISTORY_PATH));
    }

    /**
     * ե뤫ɤޤ
     *
     * @param  file ɤե
     */
    private synchronized void load(File file) {
        try {
            BufferedReader br = new BufferedReader(
                                  new InputStreamReader(
                                    new FileInputStream(file),
                                    ENCODING
                                  )
                                );

            historyFile = file;

            String  line;
            History h;

            READ:
            while ((line = br.readLine()) != null) {
                try {
                    h = new History(line, true);
                    hash.put(h.getURLText(), h);
                    if (last == null) {
                        top = last = h;
                    } else {
                        top.prev = h;
                        h.next = top;
                        top = h;
                    }
                } catch (InvalidObjectException e) { }
            }

            br.close();
        } catch (FileNotFoundException e) {
            return;
        } catch (IOException e) {
            //### ERROR
Debug.out.println(e);
e.printStackTrace(Debug.out);
        }
    }

    /**
     * ե¸ޤ
     */
    public synchronized void save() {
        File file = option.getPropertyFile   (ViewerOption.KEY_HISTORY_PATH);
        int  num  = option.getPropertyInteger(ViewerOption.KEY_HISTORY_SAVE_NUM, 500);

        if (!ViewerOption.ensureDirectory(file.getParent())) {
            //### ERROR
Debug.out.println("Can not create parent dir (" + file + ")");
            return;
        }

        try {
            BufferedWriter bw = new BufferedWriter(
                                  new OutputStreamWriter(
                                    new FileOutputStream(file),
                                    ENCODING
                                  )
                                );

            History h = last;
            while (h != null && num-- != 0) {
                h.write(bw);
                bw.write(LF);
                h = h.prev;
            }
            bw.flush();
            bw.close();
        } catch (IOException e) {
            //### ERROR
Debug.out.println(e);
e.printStackTrace(Debug.out);
        }
    }

    /**
     * ֤ޤ
     *
     * @param  reverse ֤ <code>true</code>
     *                 Ť֤ <code>false</code>
     *
     * @return 
     */
    public synchronized History[] getHistories(boolean reverse) {
        Vector list = new Vector(hash.size());

        if (reverse) {
            History h = last;
            while (h != null) {
                if (h.isVisible()) {
                    list.addElement(h);
                }
                h = h.prev;
            }
        } else {
            History h = top;
            while (h != null) {
                if (h.isVisible()) {
                    list.addElement(h);
                }
                h = h.next;
            }
        }

        History[] histories = new History[list.size()];
        list.copyInto(histories);
        return histories;
    }

    /**
     * ꤷ URL ˬ줿ȤΤ餻ޤ
     *
     * @param  url  URL
     * @param  list ꥹȤ˲ä <code>true</code>
     *              äʤ <code>false</code>
     *
     * @return 
     */
    public synchronized History visited(String url, boolean list) {
        History h = (History) hash.get(url);

        if (h == null) {
            h = new History(url);
            h.setVisible(list);
            hash.put(url, h);
        } else {
            h.visited();
            h.setVisible(list);
            if (h.prev != null && h.next != null) {
                h.prev.next = h.next;
                h.next.prev = h.prev;
            } else if (h.next != null) {
                h.next.prev = null;
                top = h.next;
            } else {
                return h;
            }
            h.prev = h.next = null;
        }

        if (top == null) {
            top = last = h;
        } else {
            h.prev = last;
            last.next = h;
            last = h;
        }

        return h;
    }

    /**
     * ꤷ URL ˬ줿ȤΤ餻ޤ
     *
     * @param  url URL
     *
     * @return ˬ䤷Ȥ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public synchronized boolean contains(String url) {
        return hash.containsKey(url);
    }

    /**
     * ꤷޤ
     *
     * @param  h 
     */
    public synchronized void remove(History h) {
        String url = h.getURLText();
        if (!hash.containsKey(url)) {
            return;
        }

        if (h == null) {
            return;
        }

        if (h.prev != null && h.next != null) {
            h.prev.next = h.next;
            h.next.prev = h.prev;
        } else if (h.next != null) {
            h.next.prev = null;
            top = h.next;
        } else if (h.prev != null) {
            h.prev.next = null;
            last = h.prev;
        } else {
            top = last = null;
        }
        h.prev = h.next = null;

        hash.remove(url);
    }

    /**
     * 򥨥ǥɽޤ
     */
    public synchronized void showEditor() {
        if (editor == null || !editor.isVisible()) {
            editor = new HistoryEditor(option);
        }

        editor.show();
    }
}
