/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer.bookmark;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.InputBox;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.Tree;
import net.hizlab.kagetaka.awt.TreeItem;
import net.hizlab.kagetaka.awt.image.OffscreenBuffer;
import net.hizlab.kagetaka.viewer.HawkViewer;
import net.hizlab.kagetaka.viewer.HawkWindow;
import net.hizlab.kagetaka.viewer.WindowManager;
import net.hizlab.kagetaka.viewer.option.Setter;
import net.hizlab.kagetaka.viewer.option.ViewerOption;
import net.hizlab.kagetaka.viewer.option.InvalidValueException;
import net.hizlab.kagetaka.viewer.theme.Theme;

import java.awt.Button;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Label;
import java.awt.MenuItem;
import java.awt.Point;
import java.awt.PopupMenu;
import java.awt.SystemColor;
import java.awt.TextField;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Stack;
import java.util.Vector;

/**
 * ֥åޡǥǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class BookmarkEditor extends Frame implements HawkWindow {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.bookmark.Resources";
    private static final int    MARGIN   = 3;

    private static final int MENU_OPEN         = 0;
    private static final int MENU_NEWBOOKMARK  = 1;
    private static final int MENU_NEWFOLDER    = 2;
    private static final int MENU_NEWSEPARATOR = 3;
    private static final int MENU_CUT          = 4;
    private static final int MENU_COPY         = 5;
    private static final int MENU_PASTE        = 6;
    private static final int MENU_DELETE       = 7;
    private static final int MENU_PTF          = 8;
    private static final int MENU_PROPERTY     = 9;

    private static final int MENU_SIZE         = 10;

    /** @serial å֥ */
    private Object     lock = new Object();
    /** @serial ץ */
    private ViewerOption    option;
    /** @serial ֥åޡޥ͡ */
    private BookmarkManager bm;
    /** @serial ɥޥ͡ */
    private WindowManager   wm;
    /** @serial  */
    private TextField  search;
    /** @serial ֥åޡĥ꡼ */
    private Tree       tree;
    /** @serial إܥ */
    private Button     up;
    /** @serial إܥ */
    private Button     down;
    /** @serial ƥȥ٥ */
    private Label      text;
    /** @serial URL ٥ */
    private Label      url;
    /** @serial 롼ȥƥ */
    private Item       root;
    /** @serial åȥХåե */
    private Bookmark   buffer;
    /** @serial ƥȥ˥塼 */
    private PopupMenu  popupMenu;
    /** @serial ˥塼 */
    private MenuItem[] menuItems;
    /** @serial ѹ줿ɤ */
    private boolean    changed;
    /** @serial եΥ᡼ */
    private Image      fileImage;
    /** @serial եγ᡼ */
    private Image      openfolderImage;
    /** @serial եĤ᡼ */
    private Image      closefolderImage;

    /** @serial ե꡼󥤥᡼ */
    private OffscreenBuffer offscreenBuffer;

    /**
     * ֥åޡǥޤ
     *
     * @param  option ץ
     */
    BookmarkEditor(ViewerOption option) {
        this.option = option;
        this.bm     = option.getBookmarkManager();
        this.wm     = WindowManager.getInstance();
        setTitle     (getMessage("title" ));
        setResizable (true                );
        setForeground(SystemColor.textText);
        setBackground(SystemColor.text    );

        this.offscreenBuffer = new OffscreenBuffer(this);

        Image image = Resource.getImageResource(RESOURCE, "bookmarkeditor.icon", getToolkit());
        if (image != null) {
            setIconImage(image);
        }

        // ᥤѥͥ
        GridBagLayout gbl    = new GridBagLayout();
        Insets        insets = new Insets(MARGIN, MARGIN, MARGIN, MARGIN);
        setLayout(gbl);

        LayoutUtils.addGridBag(this,           new Label    (getMessage("label.search")), gbl, 0, 0, 1, 1, 0, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST     , insets);
        LayoutUtils.addGridBag(this, search  = new TextField(                          ), gbl, 1, 0, 2, 1, 1, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST     , insets);
        LayoutUtils.addGridBag(this, tree    = new Tree     (10, 20                    ), gbl, 0, 1, 3, 1, 1, 1, GridBagConstraints.BOTH      , GridBagConstraints.CENTER   , insets);
        LayoutUtils.addGridBag(this, up      = new Button   (getMessage("label.up"    )), gbl, 0, 2, 1, 1, 0, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.NORTHWEST, insets);
        LayoutUtils.addGridBag(this,           new Label    (getMessage("label.detail")), gbl, 1, 2, 1, 2, 0, 0, GridBagConstraints.NONE      , GridBagConstraints.NORTH    , insets);
        LayoutUtils.addGridBag(this, text    = new Label    (                          ), gbl, 2, 2, 1, 1, 1, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST     , insets);
        LayoutUtils.addGridBag(this, down    = new Button   (getMessage("label.down"  )), gbl, 0, 3, 1, 1, 0, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.NORTHWEST, insets);
        LayoutUtils.addGridBag(this, url     = new Label    (                          ), gbl, 2, 3, 1, 1, 1, 0, GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST     , insets);

        // ˥塼
        popupMenu = new PopupMenu(getMessage("menu"));
        menuItems = new MenuItem[MENU_SIZE];
        createMenuItem(MENU_OPEN        , false);
        popupMenu.addSeparator();
        createMenuItem(MENU_NEWBOOKMARK , true );
        createMenuItem(MENU_NEWFOLDER   , true );
        createMenuItem(MENU_NEWSEPARATOR, true );
        popupMenu.addSeparator();
        createMenuItem(MENU_CUT         , false);
        createMenuItem(MENU_COPY        , false);
        createMenuItem(MENU_PASTE       , false);
        createMenuItem(MENU_DELETE      , false);
        popupMenu.addSeparator();
        createMenuItem(MENU_PTF         , false);
        popupMenu.addSeparator();
        createMenuItem(MENU_PROPERTY    , false);
        add(popupMenu);                         // tree ˤɲäʤʽͤ

        // ꥹʤϿ
        KeyListener findKeyListener;
        addKeyListener(findKeyListener =
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    boolean isShift = ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0);
                    boolean isCtrl  = ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0);

                    // F3 Ǽ򸡺
                    if (e.getKeyCode() == KeyEvent.VK_F3 && !isShift) {
                        searchBookmark(false);
                        e.consume();
                        return;
                    }

                    // Shift + F3 򸡺
                    if (e.getKeyCode() == KeyEvent.VK_F3 && isShift) {
                        searchBookmark(true);
                        e.consume();
                        return;
                    }

                    // Ctrl + F Ǹܤä˥ե
                    if (e.getKeyCode() == KeyEvent.VK_F && isCtrl) {
                        search.requestFocus();
                        return;
                    }

                    // Ctrl + W Ĥ
                    if (e.getKeyCode() == KeyEvent.VK_W && isCtrl) {
                        closeWindow();
                        return;
                    }

                    // Ctrl + Q ǽλ
                    if (e.getKeyCode() == KeyEvent.VK_Q && isCtrl) {
                        wm.exit();
                        return;
                    }
                }
            }
        );
        addWindowListener(
            new WindowAdapter() {
                /** ɥ줿Ȥ */
                public void windowOpened(WindowEvent e) {
                    wm.addWindow(BookmarkEditor.this);
                }

                /** ɥĤ褦ȤȤ */
                public void windowClosing(WindowEvent e) {
                    closeWindow();
                }

                /** ɥĤ줿Ȥ */
                public void windowClosed(WindowEvent e) {
                    wm.removeWindow(BookmarkEditor.this);

                    // ץ¸
                    Point position = getLocation();
                    if (position.x >= 0 || position.y >= 0) {
                        Setter setter = BookmarkEditor.this.option.getSetter();
                        try {
                            setter.putPropertyPoint    (ViewerOption.KEY_BOOKMARKS_WINDOW_POSITION, position);
                            setter.putPropertyDimension(ViewerOption.KEY_BOOKMARKS_WINDOW_SIZE    , BookmarkEditor.this.getSize());
                            setter.commit();
                        } catch (InvalidValueException ex) { }
                    }

                    if (!changed) {
                        return;
                    }

                    // ɥ
                    BookmarkEditor.this.wm.bookmarkChanged();
                    // ֥åޡ¸
                    BookmarkEditor.this.bm.save();
                }
            }
        );

        search.addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    // ENTER ǳ
                    if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                        searchBookmark(false);
                        e.consume();
                        return;
                    }
                }
            }
        );

        tree.addItemListener(
            new ItemListener() {
                /** 򤵤줿 */
                public void itemStateChanged(ItemEvent e) {
                    int index = tree.getSelectedIndex();
                    if (index < 0) {
                        return;
                    }

                    // ܥ̵ͭڤؤ
                    up  .setEnabled((index != 0));
                    down.setEnabled((index + 1 < tree.getViewItemCount()));

                    Item item = (Item) tree.getSelectedTreeItem();
                    if (item == null) {
                        return;
                    }

                    if (item.type == Bookmark.BOOKMARK) {
                        text.setText(item.bookmark.getText   ());
                        url .setText(item.bookmark.getURLText());
                    } else {
                        text.setText("");
                        url .setText("");
                    }
                }
            }
        );
        tree.addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    boolean isAlt = ((e.getModifiers() & InputEvent.ALT_MASK) != 0);

                    // ALT + ENTER ǥץѥƥ
                    if (e.getKeyCode() == KeyEvent.VK_ENTER && isAlt) {
                        showProperty();
                        e.consume();
                        return;
                    }

                    // Ctrl Ȥ߹碌
                    if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
                        switch (e.getKeyCode()) {
                        case KeyEvent.VK_D:
                            deleteBookmark();
                            e.consume();
                            return;
                        case KeyEvent.VK_X:
                            cutBookmark();
                            e.consume();
                            return;
                        case KeyEvent.VK_C:
                            copyBookmark();
                            e.consume();
                            return;
                        case KeyEvent.VK_V:
                            pasteBookmark();
                            e.consume();
                            return;
                        default: // AVOID
                        }
                    }

                    switch (e.getKeyCode()) {
                    case KeyEvent.VK_ENTER:           // ENTER ǳ
                        openBookmark();
                        e.consume();
                        return;
                    case KeyEvent.VK_DELETE:          // DELETE Ǻ
                        deleteBookmark();
                        e.consume();
                        return;
                    default: // AVOID
                    }
                }
            }
        );
        tree.addMouseListener(
            new MouseAdapter() {
                /** å */
                public void mouseClicked(MouseEvent e) {
                    // å
                    if (/*---*/e.isPopupTrigger()
                            || ((e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK)) {
                        Item item = (Item) tree.getSelectedTreeItem();
                        if (item != null) {
                            menuItems[MENU_OPEN        ].setEnabled((item.type == Bookmark.BOOKMARK ));
                            menuItems[MENU_CUT         ].setEnabled(true);
                            menuItems[MENU_COPY        ].setEnabled(true);
                            menuItems[MENU_DELETE      ].setEnabled(true);
                            menuItems[MENU_PTF         ].setEnabled((!item.bookmark.getPersonalToolbarFolder() && item.type == Bookmark.FOLDER));
                            menuItems[MENU_PROPERTY    ].setEnabled((item.type != Bookmark.SEPARATOR));
                        }
                        menuItems[MENU_PASTE].setEnabled((buffer != null));

                        Font font = menuItems[MENU_OPEN].getFont();
                        menuItems[MENU_OPEN].setFont(new Font(font.getName(), Font.BOLD, font.getSize()));

                        popupMenu.show(e.getComponent(), e.getX(), e.getY());
                        e.consume();
                        return;
                    }

                    // ֥륯å
                    if (/*---*/((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK)
                            && e.getClickCount() == 2) {
                        openBookmark();
                        e.consume();
                        return;
                    }
                }
            }
        );

        up.addMouseListener(
            new MouseAdapter() {
                private boolean pressed;

                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    if (checkInnerComponent(e)) {
                        pressed = true;
                    }
                }

                /** ޥ줿 */
                public void mouseReleased(MouseEvent e) {
                    if (pressed && checkInnerComponent(e)) {
                        moveUp();
                    }
                    pressed = false;
                }
            }
        );
        up.addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyTyped(KeyEvent e) {
                    // ENTER ǳ
                    if (e.getKeyChar() == KeyEvent.VK_SPACE) {
                        moveUp();
                    }
                }
            }
        );

        down.addMouseListener(
            new MouseAdapter() {
                private boolean pressed;

                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    if (checkInnerComponent(e)) {
                        pressed = true;
                    }
                }

                /** ޥ줿 */
                public void mouseReleased(MouseEvent e) {
                    if (pressed && checkInnerComponent(e)) {
                        moveDown();
                    }
                    pressed = false;
                }
            }
        );
        down.addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyTyped(KeyEvent e) {
                    // ENTER ǳ
                    if (e.getKeyChar() == KeyEvent.VK_SPACE) {
                        moveDown();
                    }
                }
            }
        );

        // ٤ƤΥȥ˥ꥹʡɲ
        search.addKeyListener(findKeyListener);
        tree  .addKeyListener(findKeyListener);
        up    .addKeyListener(findKeyListener);
        down  .addKeyListener(findKeyListener);

        validate();

        // ֤
        setSize(option.getPropertyDimension(ViewerOption.KEY_BOOKMARKS_WINDOW_SIZE));

        Point position = option.getPropertyPoint(ViewerOption.KEY_BOOKMARKS_WINDOW_POSITION);
        if (position != null) {
            setLocation(position);
        } else {
            Dimension size        = getSize();
            Dimension desktopSize = getToolkit().getScreenSize();
            setLocation((desktopSize.width  - size.width ) / 2,
                        (desktopSize.height - size.height) / 2);
        }

        Theme theme = option.getThemeManager().getTheme();
        fileImage        = theme.getImage(Theme.BOOKMARK_FILE       , getToolkit());
        openfolderImage  = theme.getImage(Theme.BOOKMARK_OPENFOLDER , getToolkit());
        closefolderImage = theme.getImage(Theme.BOOKMARK_CLOSEFOLDER, getToolkit());

        // ƥȥΥǥեȤ
        setForeground(SystemColor.controlText);
        setBackground(SystemColor.control    );
        tree.setRootTreeItem((root = new Item(null, bm.getRootBookmark())));
        tree.setBackground(SystemColor.window);
        text.setFont(new Font("Dialog", Font.BOLD, 12));
        up  .setEnabled(false);
        down.setEnabled(false);
    }

    /** ˥塼ܤ */
    private void createMenuItem(int index, boolean enable) {
        String menuKey = null;

        /// MENU
        switch (index) {
        case MENU_OPEN        : menuKey = "menu.open"        ; break;
        case MENU_NEWBOOKMARK : menuKey = "menu.newbookmark" ; break;
        case MENU_NEWFOLDER   : menuKey = "menu.newfolder"   ; break;
        case MENU_NEWSEPARATOR: menuKey = "menu.newseparator"; break;
        case MENU_CUT         : menuKey = "menu.cut"         ; break;
        case MENU_COPY        : menuKey = "menu.copy"        ; break;
        case MENU_PASTE       : menuKey = "menu.paste"       ; break;
        case MENU_DELETE      : menuKey = "menu.delete"      ; break;
        case MENU_PTF         : menuKey = "menu.ptf"         ; break;
        case MENU_PROPERTY    : menuKey = "menu.property"    ; break;
        default:
            //### ERROR
            return;
        }

        MenuItem mi = new MenuItem(getMessage(menuKey));
        mi.setEnabled(enable);
        mi.addActionListener(new MenuActionListener(index));
        menuItems[index] = mi;
        popupMenu.add(mi);
    }

    /** ꥽ʸ */
    private String getMessage(String key) {
        if (key == null) {
            return "";
        }

        return Resource.getMessage(RESOURCE, "bookmarkeditor." + key, null);
    }

//### HawkWindow
    /**
     * ɥĤޤ
     */
    public void closeWindow() {
        dispose();
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#addWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void addWindowMenu(HawkWindow window) {
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#removeWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void removeWindowMenu(int index) {
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#changeWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void changeWindowMenu(int index, HawkWindow window) {
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#addToBookmark(Bookmark, Bookmark)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void addToBookmark(Bookmark parent, Bookmark bookmark) {
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#bookmarkChanged()}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void bookmarkChanged() {
    }

//### Overwrite
    /**
     * Ԥޤ
     *
     * @param  g Graphics ɥ
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * Ԥޤ
     *
     * @param  g Graphics ɥ
     */
    public void paint(Graphics g) {
        offscreenBuffer.paint(g);
    }

//### private
    /**  */
    private void searchBookmark(boolean reverse) {
        String search = this.search.getText();
        if (search == null || search.length() == 0) {
            return;
        }
        search = search.toLowerCase();

        Stack stack = new Stack();
        int   pos   = -2;
        Item item = (Item) tree.getSelectedTreeItem();
        if (item == null) {
            item = root;
        } else if (item.type != Bookmark.FOLDER || reverse) {
            pos  = item.getNumber();
            item = item.parent;
        }

        for (;;) {
            if (item.search(search, stack, reverse, pos)) {
                // Ĥä

                while (item.parent != null) {
                    stack.push(new Integer(item.getNumber()));
                    item = item.parent;
                }

                int[] indices = new int[stack.size()];
                int   i = 0;
                while (!stack.empty()) {
                    indices[i++] = ((Integer) stack.pop()).intValue();
                }

                tree.selectTreeItem(indices);
                return;
            }

            if (item.parent == null) {
                break;
            }

            pos  = item.getNumber();
            item = item.parent;
        }
    }

    /**  */
    private void openBookmark() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null || item.type != Bookmark.BOOKMARK) {
                return;
            }

            HawkViewer viewer = wm.getActiveViewer();
            if (viewer == null) {
                return;
            }

            viewer.open(item.bookmark.getURL());
        }
    }

    /** ֥åޡ */
    private void createBookmark() {
        synchronized (lock) {
            Bookmark bookmark = Detail.show(this, bm, null);
            if (bookmark != null) {
                addBookmark(bookmark);
            }
        }

        tree.refresh();
    }

    /** ե */
    private void createFolder() {
        synchronized (lock) {
            String name = InputBox.show(this,
                                        getMessage("newfolder.message"),
                                        getMessage("newfolder.title"  ),
                                        "", false);
            if (name == null) {
                return;
            }

            addBookmark(new Bookmark(name));
        }

        tree.refresh();
    }

    /** ѥ졼 */
    private void createSeparator() {
        synchronized (lock) {
            addBookmark(new Bookmark());
        }

        tree.refresh();
    }

    /** ֥åޡڤ */
    private void cutBookmark() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null) {
                return;
            }

            item.removeFromParent();
            changed = true;

            buffer = item.bookmark;

            if (buffer.getType() == Bookmark.BOOKMARK) {
                copyClipboard(buffer.getURLText());
            }
        }

        tree.refresh();
    }

    /** ֥åޡ򥳥ԡ */
    private void copyBookmark() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null) {
                return;
            }

            buffer = (Bookmark) item.bookmark.clone();

            if (buffer.getType() == Bookmark.BOOKMARK) {
                copyClipboard(buffer.getURLText());
            }
        }
    }

    /** ֥åޡŽդ */
    private void pasteBookmark() {
        synchronized (lock) {
            if (buffer == null) {
                return;
            }

            addBookmark((Bookmark) buffer.clone());
        }

        tree.refresh();
    }

    /** ֥åޡ */
    private void deleteBookmark() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null) {
                return;
            }

            item.removeFromParent();
            changed = true;
        }

        tree.refresh();
    }

    /** ѡʥġСե */
    private void changePtf() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null) {
                return;
            }

            bm.setPersonalToolbarFolder(item.bookmark);
            changed = true;
        }
    }

    /** ץѥƥɽ */
    private void showProperty() {
        synchronized (lock) {
            Item item = (Item) tree.getSelectedTreeItem();
            if (item == null) {
                return;
            }

            switch (item.type) {
            case Bookmark.BOOKMARK:
                Bookmark bookmark = Detail.show(this, bm, item.bookmark);
                if (bookmark == null) {
                    return;
                }
                text.setText(item.bookmark.getText   ());
                url .setText(item.bookmark.getURLText());
                break;
            case Bookmark.FOLDER:
                String name = InputBox.show(this,
                                            getMessage("editfolder.message"),
                                            getMessage("editfolder.title"  ),
                                            item.bookmark.getText(), false);
                if (name == null) {
                    return;
                }
                item.bookmark.setText(name);
                break;
            default:
                return;
            }

            changed = true;
        }
        tree.refresh();
    }

    /** ذư */
    private void moveUp() {
        synchronized (lock) {
            int index = tree.getSelectedIndex();
            if (index < 1) {
                return;
            }

            Item item1 = (Item) tree.getTreeItem(index    );
            Item item2 = (Item) tree.getTreeItem(index - 1);

            int level1 = tree.getTreeItemLevel (index    );
            int level2 = tree.getTreeItemLevel (index - 1);
            int state2 = tree.getTreeItemState (index - 1);

            up  .setEnabled((index > 1));
            down.setEnabled(true       );

            item1.removeFromParent();

            if (level1 < level2) {
                item2.addAfter    (item1);
            } else if (state2 == Tree.ITEM_FOLDER_NONE) {
                item2.addLastChild(item1);
                tree.openTreeItem(index - 1);
            } else {
                item2.addBefore   (item1);
            }

            changed = true;
        }
        tree.refresh();
    }

    /** ذư */
    private void moveDown() {
        synchronized (lock) {
            int index = tree.getSelectedIndex();
            if (index < 0 || index >= tree.getViewItemCount()) {
                return;
            }

            Item item1 = (Item) tree.getTreeItem(index    );
            Item item2 = (Item) tree.getTreeItem(index + 1);

            int level1 = tree.getTreeItemLevel (index    );
            int level2 = tree.getTreeItemLevel (index + 1);
            int state2 = tree.getTreeItemState (index + 1);

            up  .setEnabled(true           );
            down.setEnabled((item2 != null));

            // Ǿ̤ξ硢ʾ岼ؤϰưǤʤ
            if (item2 == null && level1 == 1) {
                return;
            }

            item1.removeFromParent();

            if (item2 == null) {
                root .addLastChild(item1);
            } else if (level1 > level2) {
                item2.addBefore   (item1);
            } else if (state2 == Tree.ITEM_FOLDER_NONE) {
                item2.addTopChild (item1);
                tree.openTreeItem(index + 1);
            } else if (state2 == Tree.ITEM_FOLDER_OPEN) {
                item2.addTopChild (item1);
            } else {
                item2.addAfter    (item1);
            }

            changed = true;
        }
        tree.refresh();
    }

    /** ֥åޡɲáץå */
    private void addBookmark(Bookmark bookmark) {
        Item item1 = new Item(null, bookmark);

        int index = tree.getSelectedIndex();
        if (index < 0) {
            root.addLastChild(item1);
        } else {
            Item item2 = (Item) tree.getTreeItem(index);

            // ɲоݤƥ䡢Ĥեξϡμ
            switch (tree.getTreeItemState(index)) {
            case Tree.ITEM_FOLDER_NONE:
            case Tree.ITEM_FOLDER_CLOSE:
            case Tree.ITEM_NORMAL:
                item2.addBefore   (item1);
                break;
            case Tree.ITEM_FOLDER_OPEN:
                item2.addLastChild(item1);
                break;
            default: // AVOID
            }
        }

        changed = true;
    }

    /** ե */
    private Vector createItems(Item parent, Vector bookmarks) {
        Vector   v = new Vector();
        Bookmark b;
        for (int i = 0; i < bookmarks.size(); i++) {
            b = (Bookmark) bookmarks.elementAt(i);
            v.addElement(new Item(parent, b));
        }
        return v;
    }

    /** åץܡɤ˥ԡ */
    private void copyClipboard(String value) {
        Clipboard       clipboard = getToolkit().getSystemClipboard();
        StringSelection data      = new StringSelection(value);
        clipboard.setContents(data, data);
    }

    /** ܥ夫å */
    private boolean checkInnerComponent(MouseEvent e) {
        Component c = (Component) e.getSource();
        if (!c.isEnabled()) {
            return false;
        }
        if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
            return false;
        }
        int x = e.getX();
        int y = e.getY();
        Dimension size = c.getSize();
        return (0 < x && x <= size.width && 0 < y && y <= size.height);
    }

//### Item
    /** ֥åޡƥ */
    private final class Item implements TreeItem {
        private int      type;
        private Item     parent;
        private Bookmark bookmark;
        private Vector   items;

        /** 󥹥󥹤 */
        private Item(Item parent, Bookmark bookmark) {
            this.parent   = parent;
            this.type     = bookmark.getType();
            this.bookmark = bookmark;
        }

        /** ĥ꡼ɽ륢֤ */
        public Image getTreeIcon(boolean opened) {
            if (type != Bookmark.FOLDER) {
                return fileImage;
            }

            return (opened ? openfolderImage : closefolderImage);
        }

        /** ĥ꡼ɽ٥֤ */
        public String getTreeLabel() {
            if (type == Bookmark.SEPARATOR) {
                return null;
            }

            return bookmark.getText();
        }

        /** ҤΥƥ֤ */
        public Vector getTreeItems() {
            if (type != Bookmark.FOLDER) {
                return null;
            }

            if (items != null) {
                return items;
            }

            return (items = createItems(this, bookmark.getBookmarks()));
        }

        /** ʬμɲ */
        private void addAfter(Item item) {
            parent.insertTreeItem(item, getNumber() + 1);
        }

        /** ʬμɲ */
        private void addBefore(Item item) {
            parent.insertTreeItem(item, getNumber());
        }

        /** ʬλäƤҤκǽɲ */
        private void addTopChild(Item item) {
            insertTreeItem(item, 0);
        }

        /** ʬλäƤҤκǸɲ */
        private void addLastChild(Item item) {
            insertTreeItem(item, -1);
        }

        /** ʬλäƤҤλ֤ */
        private void insertTreeItem(Item item, int index) {
            Vector v = getTreeItems();
            if (v == null) {
                return;
            }

            if (index < 0) {
                v.addElement(item);
            } else {
                v.insertElementAt(item, index);
            }

            item.parent = this;
            bookmark.addBookmark(item.bookmark, index);
        }

        /** Ƥ鼫ʬ */
        private void removeFromParent() {
            parent.bookmark.removeBookmark(bookmark);
            if (parent.items != null) {
                parent.items.removeElement(this);
            }
            parent = null;
        }

        /** ʬƥեβܤΥ֥åޡ */
        private int getNumber() {
            return parent.getTreeItems().indexOf(this);
        }

        /** ʸ򸡺 */
        private boolean search(String search, Stack stack, boolean reverse, int pos) {
            if (/*---*/reverse
                    && type == Bookmark.FOLDER
                    && searchSub(search, stack, reverse, pos)) {
                return true;
            }

            // 椫θξϡʬʥեξΤߡˤ򸡺ʤ
            // üΥƥξϡɬ pos == -1 
            if (reverse || pos == -1) {
                switch (type) {
                case Bookmark.SEPARATOR:
                    return false;
                case Bookmark.BOOKMARK:
                    if (/*---*/bookmark.getURLText() != null
                            && bookmark.getURLText().toLowerCase().indexOf(search) != -1) {
                        return true;
                    }
                    /* ե륹롼 */
                case Bookmark.FOLDER:
                    if (/*---*/bookmark.getText() != null
                            && bookmark.getText().toLowerCase().indexOf(search) != -1) {
                        return true;
                    }
                default: // AVOID
                }
            }

            if (/*---*/!reverse
                    && type == Bookmark.FOLDER
                    && searchSub(search, stack, reverse, pos)) {
                return true;
            }

            return false;
        }

        /** ʸǤ鸡 */
        private boolean searchSub(String search, Stack stack, boolean reverse, int pos) {
            Vector v = getTreeItems();
            if (v == null) {
                return false;
            }

            int start = (pos >= 0 ? pos + 1 : 0), end = v.size(), step = 1;
            if (reverse) {
                start = (pos >= 0 ? pos - 1 : v.size() - 1); end = -1; step = -1;
            }

            for (int i = start; (reverse ? (i > end) : (i < end)); i += step) {
                if (((Item) v.elementAt(i)).search(search, stack, reverse, -1)) {
                    stack.push(new Integer(i));
                    return true;
                }
            }

            return false;
        }
    }

//### MenuActionListener
    /** ˥塼ѤΥꥹ */
    private final class MenuActionListener implements ActionListener {
        private int index = 0;

        /** ˥塼ꥹ  */
        private MenuActionListener(int index) {
            this.index = index;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            switch (index) {
            case MENU_OPEN        : openBookmark   (); break;
            case MENU_NEWBOOKMARK : createBookmark (); break;
            case MENU_NEWFOLDER   : createFolder   (); break;
            case MENU_NEWSEPARATOR: createSeparator(); break;
            case MENU_CUT         : cutBookmark    (); break;
            case MENU_COPY        : copyBookmark   (); break;
            case MENU_PASTE       : pasteBookmark  (); break;
            case MENU_DELETE      : deleteBookmark (); break;
            case MENU_PTF         : changePtf      (); break;
            case MENU_PROPERTY    : showProperty   (); break;
            default: // AVOID
            }
        }
    }
}
