/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer.bookmark;

import net.hizlab.kagetaka.parser.Tag;
import net.hizlab.kagetaka.util.StringUtils;
import net.hizlab.kagetaka.util.TextFormat;

import java.io.StringWriter;
import java.io.Writer;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;
import java.net.URL;
import java.net.MalformedURLException;

/**
 * ֥åޡɽ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.1.1.1 $
 */
public class Bookmark implements Cloneable {
    private static final int LF = 0x0A;

    /** ե */
    public static final int FOLDER    = 1;
    /** ֥åޡ */
    public static final int BOOKMARK  = 2;
    /** ѥ졼 */
    public static final int SEPARATOR = 3;

    private int     type;
    private Vector  bookmarks;
    private String  text;
    private String  urlText;
    private URL     url;
    private long    addDate;
    private long    lastVisit;
    private long    lastModified;
    private String  lastCharset;
    private String  shortcuturl;
    private boolean folderGroup;
    private boolean ptFolder;

    Bookmark parent = null;                   // ƤΥ֥åޡ

    /**
     * ֥åޡޤ
     *
     * @param  text ɽʸ
     * @param  url URL
     */
    public Bookmark(String text, URL url) {
        this.type    = BOOKMARK;
        this.text    = text;
        this.urlText = url.toString();
        this.url     = url;
        this.addDate = System.currentTimeMillis();
    }

    /**
     * ֥åޡޤ
     *
     * @param  text ɽʸ
     * @param  url URL
     */
    public Bookmark(String text, String url) {
        this.type    = BOOKMARK;
        this.text    = text;
        this.urlText = url;
        this.addDate = System.currentTimeMillis();
    }

    /**
     * եޤ
     *
     * @param  text ɽʸ
     */
    public Bookmark(String text) {
        this.type      = FOLDER;
        this.bookmarks = new Vector();
        this.text      = text;
        this.addDate   = System.currentTimeMillis();
    }

    /**
     * ѥ졼ޤ
     */
    public Bookmark() {
        this.type = SEPARATOR;
    }

    /**
     * ֥åޡޤ
     *
     * @param  text ʸ
     * @param  tag  
     */
    Bookmark(String text, Tag tag) {
        String    element = tag.getElement();
        Hashtable param   = tag.getAttribute();

        if (element.compareTo("A") == 0) {
            this.type = BOOKMARK;
            if (param != null) {
                this.lastVisit   = toLong   (param.get("LAST_VISIT"   ));
                this.lastCharset = (String)  param.get("LAST_CHARSET" ) ;
                this.shortcuturl = (String)  param.get("SHORTCUTURL"  ) ;
            }
        } else {
            this.type      = FOLDER;
            this.bookmarks = new Vector();
            if (param != null) {
                this.folderGroup = toBoolean(param.get("FOLDER_GROUP"           ));
                this.ptFolder    = toBoolean(param.get("PERSONAL_TOOLBAR_FOLDER"));
            }
        }
        this.text = text;
        if (param != null) {
            this.urlText      = (String)  param.get("HREF"         ) ;
            this.addDate      = toLong   (param.get("ADD_DATE"     ));
            this.lastModified = toLong   (param.get("LAST_MODIFIED"));
        }
    }

    /** ʸͤѴ */
    private long toLong(Object value) {
        if (value == null) {
            return 0;
        }

        return Long.parseLong((String) value);
    }

    /** ʸ򿿵ͤѴ */
    private boolean toBoolean(Object value) {
        if (value == null) {
            return false;
        }

        return Boolean.valueOf((String) value).booleanValue();
    }

    /**
     * ֥åޡμफɤ֤ޤ
     *
     * @return ֥åޡμ
     */
    public int getType() {
        return type;
    }

    /**
     * ʬοƥե֤ޤ
     *
     * @return ƥեΩƤ <code>null</code>
     */
    public Bookmark getParentFolder() {
        return parent;
    }

    /**
     * ʬեξˡΥեΥ֥åޡ֤ޤ
     *
     * @return եΥ֥åޡ
     */
    public Vector getBookmarks() {
        return bookmarks;
    }

    /**
     * ʬեξˡΥեΥ֥åޡκǸɲäޤ
     *
     * @param  bookmark ɲä֥åޡ
     */
    public void addBookmark(Bookmark bookmark) {
        addBookmark(bookmark, -1);
    }

    /**
     * ʬեξˡΥեΥ֥åޡ
     * Υǥå֤ɲäޤ
     *
     * @param  bookmark ɲä֥åޡ
     * @param  index    ǥå
     *                  Ǹɲä <code>-1</code>
     */
    public void addBookmark(Bookmark bookmark, int index) {
        if (type != FOLDER) {
            throw new IllegalArgumentException("not folder");
        }

        if (index < 0) {
            bookmarks.addElement(bookmark);
        } else {
            bookmarks.insertElementAt(bookmark, index);
        }

        bookmark.parent = this;
    }

    /**
     * ʬեξˡ
     * Υեˤꤵ줿֥åޡޤ
     *
     * @param  bookmark ֥åޡ
     */
    public void removeBookmark(Bookmark bookmark) {
        if (type != FOLDER) {
            throw new IllegalArgumentException("not folder");
        }

        if (bookmark.parent == null) {
            return;
        }

        bookmark.parent = null;
        bookmarks.removeElement(bookmark);
    }

    /**
     * ʬƥեβܤΥ֥åޡ֤ޤ
     *
     * @return <code>0</code> Ϥޤֹ
     */
    public int getNumber() {
        return parent.bookmarks.indexOf(this);
    }

    /**
     * ɽʸޤ
     *
     * @return ɽʸ
     */
    public String getText() {
        return text;
    }

    /**
     * ɽʸꤷޤ
     *
     * @param  text ɽʸ
     */
    public void setText(String text) {
        if (type == SEPARATOR) {
            throw new IllegalArgumentException("separator");
        }

        this.text = text;
    }

    /**
     * URL ޤ
     *
     * @return URL
     */
    public URL getURL() {
        if (type != BOOKMARK) {
            return null;
        }

        if (url != null) {
            return url;
        }

        try {
            url = StringUtils.toURL(TextFormat.convertXhtml(urlText, true, true, true, true));
        } catch (MalformedURLException e) {
            return null;
        }

        return url;
    }

    /**
     * URL ꤷޤ
     *
     * @param  url URL
     */
    public void setURL(URL url) {
        if (type != BOOKMARK) {
            throw new IllegalArgumentException("not bookmark");
        }

        this.urlText = url.toString();
        this.url     = url;
    }

    /**
     * URL ʸɽޤ
     *
     * @return URL
     */
    public String getURLText() {
        if (type != BOOKMARK) {
            return null;
        }

        return urlText;
    }

    /**
     * URL ʸɽꤷޤ
     *
     * @param  url URL
     *
     * @throws MalformedURLException  URL Ǥ̵
     */
    public void setURLText(String url)
            throws MalformedURLException {
        if (type != BOOKMARK) {
            throw new IllegalArgumentException("not bookmark");
        }

        this.url     = StringUtils.toURL(url);
        this.urlText = url;
    }

    /**
     * 硼ȥå URL ޤ
     *
     * @return 硼ȥå URL
     */
    public String getShortcutURL() {
        return shortcuturl;
    }

    /**
     * 硼ȥå URL ꤷޤ
     *
     * @param  text 硼ȥå URL
     */
    public void setShortcutURL(String text) {
        if (type != BOOKMARK) {
            throw new IllegalArgumentException("not bookmark");
        }

        this.shortcuturl = text;
    }

    /**
     * Ǹˬ䤷ݤʸåȤꤷޤ
     *
     * @return Ǹˬ䤷ݤʸåȡ
     *         ꤵƤʤ <code>null</code>
     */
    public String getLastCharset() {
        return lastCharset;
    }

    /**
     * ˬѤߤޡޤ
     *
     * @param  lastModified Ǹˬ줿
     * @param  charset      Ǹˬ줿Ȥʸå
     */
    public void visited(long lastModified, String charset) {
        if (type != BOOKMARK) {
            throw new IllegalArgumentException("not bookmark");
        }

        this.lastVisit    = System.currentTimeMillis();
        this.lastModified = lastModified;
        this.lastCharset  = charset;
    }

    /**
     * ѡʥġСեɤ֤ޤ
     *
     * @return ѡʥġСեξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean getPersonalToolbarFolder() {
        return ptFolder;
    }

    /**
     * ѡʥġСեꤷޤ
     *
     * @param  b ѡʥġСեˤ <code>true</code>
     *           ʳξ <code>false</code>
     */
    void setPersonalToolbarFolder(boolean b) {
        this.ptFolder = b;
    }

    /**
     * ʸѴޤ
     *
     * @param  w 饤
     * @param  level ٥
     *
     * @throws IOException I/O 顼ȯ
     */
    void write(Writer w, int level)
            throws IOException {
        String s;

        int space = level * 4;
        for (int i = 0; i < space; i++) {
            w.write(' ');
        }

        switch (type) {
        case FOLDER:
            w.write("<DT><H3");
            break;
        case BOOKMARK:
            w.write("<DT><A HREF=\"");
            s = urlText;
            s = StringUtils.replace(s, "#" , "#x23;");
            s = StringUtils.replace(s, "\"", "#x22;");
            w.write(s);
            w.write('"');
            break;
        case SEPARATOR:
            w.write("<HR");
            break;
        default: // AVOID
        }

        if (addDate > 0) {
            w.write(" ADD_DATE=\"");
            w.write(String.valueOf(addDate));
            w.write('"');
        }

        if (lastVisit > 0) {
            w.write(" LAST_VISIT=\"");
            w.write(String.valueOf(lastVisit));
            w.write('"');
        }

        if (lastModified > 0) {
            w.write(" LAST_MODIFIED=\"");
            w.write(String.valueOf(lastModified));
            w.write('"');
        }

        if (lastCharset != null) {
            w.write(" LAST_CHARSET=\"");
            w.write(lastCharset);
            w.write('"');
        }

        if (shortcuturl != null) {
            w.write(" SHORTCUTURL=\"");
            w.write(shortcuturl);
            w.write('"');
        }

        if (folderGroup) {
            w.write(" FOLDER_GROUP=\"true\"");
        }

        if (ptFolder) {
            w.write(" PERSONAL_TOOLBAR_FOLDER=\"true\"");
        }

        w.write('>');

        if (type != SEPARATOR) {
            s = text;
            s = StringUtils.replace(s, "&" , "&amp;" );
            s = StringUtils.replace(s, "<" , "&lt;"  );
            s = StringUtils.replace(s, ">" , "&gt;"  );
            s = StringUtils.replace(s, "\"", "&quot;");
            w.write(s);
        }

        switch (type) {
        case FOLDER:
            w.write("</H3>");
            break;
        case BOOKMARK:
            w.write("</A>");
            break;
        default: // AVOID
        }

        w.write(LF);

        if (type == FOLDER) {
            for (int i = 0; i < space; i++) {
                w.write(' ');
            }
            w.write("<DL><p>");
            w.write(LF);

            level++;
            for (int i = 0; i < bookmarks.size(); i++) {
                ((Bookmark) bookmarks.elementAt(i)).write(w, level);
            }

            for (int i = 0; i < space; i++) {
                w.write(' ');
            }
            w.write("</DL><p>");
            w.write(LF);
        }

        w.flush();
    }

    /**
     * ޤ
     * Υ֥åޡ¸ߤϡ餹٤Ƥʣޤ
     *
     * @return 
     */
    public Object clone() {
        try {
            Bookmark base = (Bookmark) super.clone();
            base.parent = null;

            // Ҥʣ
            if (base.bookmarks != null) {
                int size = bookmarks.size();
                Vector v = new Vector(size);
                for (int i = 0; i < size; i++) {
                    v.addElement(((Bookmark) base.bookmarks.elementAt(i)).clone());
                }
                base.bookmarks = v;
            }

            return base;
        } catch (CloneNotSupportedException e) {
            return null;
        }
    }

    /**
     * ʸѴޤ
     *
     * @return ʸ
     */
    public String toString() {
        StringWriter w = new StringWriter();
        try {
            write(w, 0);
        } catch (IOException e) { }
        return w.toString();
    }
}
