/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.ImageButton;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.Toolbar;
import net.hizlab.kagetaka.awt.QuietContainer;
import net.hizlab.kagetaka.awt.event.StateEvent;
import net.hizlab.kagetaka.awt.event.StateListener;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.viewer.bookmark.Bookmark;
import net.hizlab.kagetaka.viewer.bookmark.BookmarkManager;
import net.hizlab.kagetaka.viewer.event.ContextFocusEvent;
import net.hizlab.kagetaka.viewer.event.ContextFocusListener;
import net.hizlab.kagetaka.viewer.event.ContextHistoryEvent;
import net.hizlab.kagetaka.viewer.event.ContextHistoryListener;
import net.hizlab.kagetaka.viewer.event.ContextInformationEvent;
import net.hizlab.kagetaka.viewer.event.ContextInformationListener;
import net.hizlab.kagetaka.viewer.event.ContextRequestEvent;
import net.hizlab.kagetaka.viewer.event.ContextRequestListener;
import net.hizlab.kagetaka.viewer.option.ViewerOption;
import net.hizlab.kagetaka.viewer.theme.Theme;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Font;
import java.awt.Image;
import java.awt.Insets;
import java.awt.MenuItem;
import java.awt.Toolkit;
import java.awt.Polygon;
import java.awt.PopupMenu;
import java.awt.SystemColor;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.Vector;

/**
 * ӥ塼ѥġС󶡤ޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
public class ViewerToolbar extends Toolbar
        implements ContextFocusListener,
                   ContextHistoryListener,
                   ContextInformationListener,
                   ContextRequestListener {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    /**                  */ public static final String BACK     = "back"    ;
    /**                  */ public static final String FORWARD  = "forward" ;
    /** ɹ               */ public static final String RELOAD   = "reload"  ;
    /**                  */ public static final String STOP     = "stop"    ;
    /** GO                   */ public static final String GO       = "go"      ;
    /** ۡ               */ public static final String HOME     = "home"    ;
    /** ֥åޡ         */ public static final String BOOKMARK = "bookmark";
    /** ѡʥġС */ public static final String PTF      = "ptf"     ;
    /** ɥ쥹С         */ public static final String ADDRESS  = "address" ;

    static final int TYPE_NAVI = 1;
    static final int TYPE_LINK = 2;

    private static final int BUTTON_BACK       =  0;
    private static final int BUTTON_FORWARD    =  1;
    private static final int BUTTON_RELOAD     =  2;
    private static final int BUTTON_STOP       =  3;
    private static final int BUTTON_GO         =  4;
    private static final int BUTTON_HOME       =  5;
    private static final int BUTTON_BOOKMARK   =  6;
    private static final int BUTTONS_NUM       =  7;

    private static final int TEXTFIELD_ADDRESS =  0;
//    private static final int TEXTFIELD_SEARCH  =  1;
    private static final int TEXTFIELDS_NUM    =  2;

    private static final int PTF_BOOKMARK      = -1;

    private static final int MENU_SIZE         = 20;

    private static final int LOAD_BEFORE  = 0;
    private static final int LOAD_LOADING = 1;
    private static final int LOAD_LOADED  = 2;

    /** @serial ӥ塼 */
    private HawkViewer    viewer;
    /** @serial  */
    private int           type;
    /** @serial ץ */
    private ViewerOption  option;
    /** @serial ơ */
    private Theme         theme;
    /** @serial ơå */
    private Object        stateLock = new Object();

    /** @serial ġСե */
    private Font  toolbarFont;
    /** @serial ġСե顼 */
    private Color toolbarFore;
    /** @serial ġСХå顼 */
    private Color toolbarBack;
    /** @serial ġС᡼ */
    private Image toolbarImage;

    /** @serial ܥ */
    private ImageButton[]  buttons;
    /** @serial ƥȥܥå */
    private TextField  []  textFields;
    /** @serial ᥤѥͥ */
    private QuietContainer ptPanel;
    /** @serial ᥤѥͥ쥤 */
    private GridBagLayout  ptPanelLayout;
    /** @serial ݥåץåץ˥塼 */
    private PopupMenu      popupMenu;
    /** @serial ݥåץåץ˥塼ɽ椫 */
    private boolean        showPopupMenu;
    /** @serial ֥åޡ */
    private Bookmark[]     bookmarks;
    /** @serial ֥åޡ */
    private String[][]     values;
    /** @serial եβ */
    private Image          fileImage;
    /** @serial եβ */
    private Image          openfolderImage;
    /** @serial ĤƤեβ */
    private Image          closefolderImage;

    /** @serial ƥ֤ʥƥ */
    private ViewerContext context;
    /** @serial ߤͭʥꥯ */
    private Request       request;

    /**
     * ӥ塼ѥġССޤ
     *
     * @param  viewer ӥ塼
     * @param  type   
     */
    ViewerToolbar(HawkViewer viewer, int type) {
        this.viewer = viewer;
        this.type   = type;
        this.option = viewer.getOption();
        this.theme  = option.getThemeManager().getTheme();

        Toolkit tk = getToolkit();

        toolbarFont  = theme.getFont (Theme.BROWSER_TOOLBAR_FONT      );
        toolbarFore  = theme.getColor(Theme.BROWSER_TOOLBAR_FOREGROUND);
        toolbarBack  = theme.getColor(Theme.BROWSER_TOOLBAR_BACKGROUND);
        toolbarImage = theme.getImage(Theme.BROWSER_TOOLBAR_BACKGROUND_IMAGE, tk);

        buttons    = new ImageButton[BUTTONS_NUM   ];
        textFields = new TextField  [TEXTFIELDS_NUM];

        int index = 0;
        String key = null;
        String[] list = null;

        switch (type) {
        case TYPE_NAVI: list = option.getPropertyStrings(ViewerOption.KEY_SHOW_NAVIBAR_BUTTONS); break;
        case TYPE_LINK: list = option.getPropertyStrings(ViewerOption.KEY_SHOW_LINKBAR_BUTTONS); break;
        default: // AVOID
        }

        for (int i = 0; i < list.length; i++) {
            if (list[i].compareTo(BACK    ) == 0) { index = BUTTON_BACK    ; key = Theme.BROWSER_TOOLBAR_BACK    ; } else
            if (list[i].compareTo(FORWARD ) == 0) { index = BUTTON_FORWARD ; key = Theme.BROWSER_TOOLBAR_FORWARD ; } else
            if (list[i].compareTo(RELOAD  ) == 0) { index = BUTTON_RELOAD  ; key = Theme.BROWSER_TOOLBAR_RELOAD  ; } else
            if (list[i].compareTo(STOP    ) == 0) { index = BUTTON_STOP    ; key = Theme.BROWSER_TOOLBAR_STOP    ; } else
            if (list[i].compareTo(GO      ) == 0) { index = BUTTON_GO      ; key = Theme.BROWSER_TOOLBAR_GO      ; } else
            if (list[i].compareTo(HOME    ) == 0) { index = BUTTON_HOME    ; key = Theme.BROWSER_TOOLBAR_HOME    ; } else
            if (list[i].compareTo(BOOKMARK) == 0) { index = BUTTON_BOOKMARK; key = Theme.BROWSER_TOOLBAR_BOOKMARK; } else
            if (list[i].compareTo(PTF     ) == 0) {
                if (ptPanel != null) {
                    continue;
                }

                ptPanel = new QuietContainer(ptPanelLayout = new GridBagLayout());
                add(ptPanel);
                continue;
            } else
            if (list[i].compareTo(ADDRESS ) == 0) {
                if (textFields[TEXTFIELD_ADDRESS] != null) {
                    continue;
                }

                textFields[TEXTFIELD_ADDRESS] = new TextField();
                textFields[TEXTFIELD_ADDRESS].setFont      (theme.getFont (Theme.BROWSER_TOOLBAR_ADDRESSBAR_FONT      ));
                textFields[TEXTFIELD_ADDRESS].setForeground(theme.getColor(Theme.BROWSER_TOOLBAR_ADDRESSBAR_FOREGROUND));
                textFields[TEXTFIELD_ADDRESS].setBackground(theme.getColor(Theme.BROWSER_TOOLBAR_ADDRESSBAR_BACKGROUND));
                textFields[TEXTFIELD_ADDRESS].addKeyListener  (new AddressBarKeyListener  ());
                textFields[TEXTFIELD_ADDRESS].addFocusListener(new AddressBarFocusListener());
                add(textFields[TEXTFIELD_ADDRESS], MAXIMUM);
                continue;
            } else {
                continue;
            }

            if (buttons[index] != null) {
                continue;
            }

            if ((buttons[index] = theme.getImageButton(key, tk)) == null) {
                continue;
            }

            if (toolbarFont  != null) { buttons[index].setFont      (toolbarFont ); }
            if (toolbarFore  != null) { buttons[index].setForeground(toolbarFore ); }
            if (toolbarBack  != null) { buttons[index].setBackground(toolbarBack ); }
            if (toolbarImage != null) { buttons[index].setBackImage (toolbarImage); }
            buttons[index].addMouseListener(new ButtonMouseListener(index));

            if (theme.getBoolean(key + Theme.EXT_LABEL)) {
                buttons[index].setLabel(Resource.getMessage(RESOURCE, "toolbar." + list[i] + ".label", null));
            }

            add(buttons[index]);
        }

        if (buttons[BUTTON_BACK    ] != null) {
            buttons[BUTTON_BACK    ].addStateListener(new ButtonStateListener(BUTTON_BACK    ));
        }
        if (buttons[BUTTON_FORWARD ] != null) {
            buttons[BUTTON_FORWARD ].addStateListener(new ButtonStateListener(BUTTON_FORWARD ));
        }
        if (buttons[BUTTON_BOOKMARK] != null) {
            buttons[BUTTON_BOOKMARK].addStateListener(new ButtonStateListener(BUTTON_BOOKMARK));
        }

        setForeground(SystemColor.controlText);
        setBackground(SystemColor.control    );

        if (toolbarFont  != null) { setFont      (toolbarFont ); }
        if (toolbarFore  != null) { setForeground(toolbarFore ); }
        if (toolbarBack  != null) { setBackground(toolbarBack ); }
        if (toolbarImage != null) { setBackImage (toolbarImage); }

        bookmarkChanged();
    }

    /**
     * ɥ쥹ޤ
     *
     * @return ɥ쥹
     */
    public String getAddress() {
        if (textFields[TEXTFIELD_ADDRESS] == null) {
            return null;
        }

        return textFields[TEXTFIELD_ADDRESS].getText();
    }

    /**
     * ɥ쥹С˥եưޤ
     *
     * @return ɥ쥹С¸ߤ <code>true</code>
     *         ¸ߤʤ <code>false</code>
     */
    public boolean focusAddress() {
        if (textFields[TEXTFIELD_ADDRESS] == null) {
            return false;
        }

        textFields[TEXTFIELD_ADDRESS].requestFocus();
        return true;
    }

//### Bookmark
    /**
     * ֥åޡɲäΤޤ
     *
     * @param  parent   ƤΥ֥åޡ
     * @param  bookmark ɲä֥åޡ
     */
    void addToBookmark(Bookmark parent, Bookmark bookmark) {
        bookmarkChanged();
    }

    /**
     * ֥åޡѹΤޤ
     */
    void bookmarkChanged() {
        if (ptPanel == null) {
            return;
        }

        BookmarkManager bm = option.getBookmarkManager();
        synchronized (bm) {
            Bookmark root = bm.getPersonalToolbarFolder();
            Vector   v    = null;
            if (root != null) {
                v = root.getBookmarks();
            }

            if (v == null || v.size() == 0) {
                if (this.bookmarks != null) {
                    ptPanel.removeAll();
                    this.bookmarks = null;
                }
                return;
            }

            Bookmark[] bookmarks = new Bookmark[v.size()];
            String[][] values    = new String[v.size()][2];
            v.copyInto(bookmarks);
            Bookmark b;
            for (int i = 0; i < values.length; i++) {
                b = (Bookmark) v.elementAt(i);
                values[i][0] = b.getText   ();
                values[i][1] = b.getURLText();
            }

            // ֥åޡѹʤɤå
            if (this.values != null && this.values.length == values.length) {
                boolean same = true;
                String[] b1, b2;
                for (int i = 0; i < values.length; i++) {
                    b1 = this.values[i];
                    b2 = values[i];
                    if (/*---*/!compareString(b1[0], b2[0])
                            || !compareString(b1[1], b2[1])) {
                        same = false;
                        break;
                    }
                }
                if (same) {
                    return;
                }
            }

            ImageButton      ib       = null;
            Insets           insets   = new Insets(0, 0, 0, 0);
            BookmarkListener listener = null;
            ptPanel.removeAll();
            for (int i = 0; i < values.length; i++) {
                switch (bookmarks[i].getType()) {
                case Bookmark.FOLDER:
                    if (openfolderImage  == null) {
                        openfolderImage  = theme.getImage(Theme.BOOKMARK_OPENFOLDER , getToolkit());
                    }
                    if (closefolderImage == null) {
                        closefolderImage = theme.getImage(Theme.BOOKMARK_CLOSEFOLDER, getToolkit());
                    }
                    ib = new ImageButton(bookmarks[i].getText(), true, closefolderImage, openfolderImage, null, null, null, null, null, null);
                    break;
                case Bookmark.BOOKMARK:
                    if (fileImage   == null) {
                        fileImage   = theme.getImage(Theme.BOOKMARK_FILE  , getToolkit());
                    }
                    ib = new ImageButton(bookmarks[i].getText(), fileImage  , null);
                    break;
                default:
                    continue;
                }

                ib.setFont      (toolbarFont );
                ib.setForeground(toolbarFore );
                ib.setBackground(toolbarBack );
                ib.setBackImage (toolbarImage);
                listener = new BookmarkListener(bookmarks[i]);
                ib.addMouseListener(listener);
                ib.addStateListener(listener);
                LayoutUtils.addGridBag(ptPanel, ib, ptPanelLayout, i, 0, 1, 1, 0, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
            }

            validate();
            this.bookmarks = bookmarks;
            this.values    = values;
        }
    }

    /** Ʊå */
    private static boolean compareString(String s1, String s2) {
        if (s1 == null) {
            return (s2 == null);
        } else if (s2 == null) {
            return false;
        }

        return (s1.compareTo(s2) == 0);
    }

//### ContextFocusListener
    /* ƥȤեȤ */
    /** {@inheritDoc} */
    public void focusGained(ContextFocusEvent e) {
        synchronized (stateLock) {
            Request request = e.getRequest();

            this.context = e.getContext();
            this.request = request;

            // ɾ
            if (request == null) {
                setLoaded(LOAD_BEFORE );
            } else {
                if (e.isLoading()) {
                    setLoaded(LOAD_LOADING);
                } else {
                    setLoaded(LOAD_LOADED );
                }
            }
        }
    }

    /* ƥȤե򼺤Ȥ */
    /** {@inheritDoc} */
    public void focusLost(ContextFocusEvent e) {
        ViewerContext context = this.context;
        if (context != null && textFields[TEXTFIELD_ADDRESS] != null) {
           context.setAddress(textFields[TEXTFIELD_ADDRESS].getText());
        }
    }

//### ContextHistoryListener
    /* ҥȥ꤬ѹ줿 */
    /** {@inheritDoc} */
    public void historyChanged(ContextHistoryEvent e) {
        synchronized (stateLock) {
            ViewerContext context = e.getContext();
            Request       request = e.getRequest();

            if (this.context != context) {
                return;
            }
            this.request = request;

            setNavi(request);
        }
    }

//### ContextInformationListener
    /* ɥ쥹ѹ줿 */
    /** {@inheritDoc} */
    public void addressChanged(ContextInformationEvent e) {
        synchronized (stateLock) {
            if (this.context != e.getContext()) {
                return;
            }

            setAddress((String) e.getData());
        }
    }

    /* ȥ뤬ѹ줿 */
    /** {@inheritDoc} */
    public void titleChanged(ContextInformationEvent e) {
    }

    /* ơѹ줿 */
    /** {@inheritDoc} */
    public void statusChanged(ContextInformationEvent e) {
    }

//### ContextRequestListener
    /* ꥯȤդ */
    /** {@inheritDoc} */
    public void requestAccepted(ContextRequestEvent e) {
        synchronized (stateLock) {
            if (this.context != e.getContext()) {
                return;
            }

            setLoaded (LOAD_LOADING);
            setAddress(e.getRequest().url.toString());
        }
    }

    /* ꥯȤλ */
    /** {@inheritDoc} */
    public void requestDone(ContextRequestEvent e) {
        synchronized (stateLock) {
            if (this.context != e.getContext()) {
                return;
            }

            setLoaded(LOAD_LOADED);
        }
    }

//### private
    /** ɤ߹ޤƤ뤫ɤ stateLock */
    private void setLoaded(int load) {
        if (buttons[BUTTON_RELOAD] != null) { buttons[BUTTON_RELOAD].setEnabled(load != LOAD_BEFORE ); }
        if (buttons[BUTTON_STOP  ] != null) { buttons[BUTTON_STOP  ].setEnabled(load == LOAD_LOADING); }
    }

    /** 롦إܥ stateLock */
    private void setNavi(Request request) {
        if (buttons[BUTTON_BACK   ] != null) { buttons[BUTTON_BACK   ].setEnabled((request != null && request.getPrevious() != null)); }
        if (buttons[BUTTON_FORWARD] != null) { buttons[BUTTON_FORWARD].setEnabled((request != null && request.getNext    () != null)); }
    }

    /** ɥ쥹 stateLock */
    private void setAddress(String address) {
        if (textFields[TEXTFIELD_ADDRESS] != null) {
            textFields[TEXTFIELD_ADDRESS].setText(address);
        }
    }

    /** ݥåץåץ˥塼ɽ */
    private void showPopupMenu(Component comp, int index, Bookmark bookmark) {
        ViewerContext context;
        Request       request;
        synchronized (stateLock) {
            context = this.context;
            request = this.request;
        }

        synchronized (popupMenu) {
            if (showPopupMenu) {
                return;
            }

            popupMenu.removeAll();

            switch (index) {
            case BUTTON_BACK    :
            case BUTTON_FORWARD :
            {
                int      num  = 0;
                MenuItem mi;
                String   title;
                boolean  back = (index == BUTTON_BACK);

                while (num++ < MENU_SIZE
                        && (request = (back
                                       ? request.getPrevious()
                                       : request.getNext    ())) != null) {

                    title = request.getDocument().getTitle();
                    if (title == null) {
                        title = "";
                    }

                    mi = new MenuItem(title);
                    mi.addActionListener(new MoveActionListener(context, (back ? -num : num)));

                    popupMenu.add(mi);
                }

                break;
            }
            case BUTTON_BOOKMARK:
                viewer.createBookmarkMenu(popupMenu, true);
                break;
            case PTF_BOOKMARK:
                viewer.createBookmarkMenu(popupMenu, bookmark.getBookmarks());
                break;
            default: // AVOID
            }

            showPopupMenu = true;
            popupMenu.show(comp, 0, comp.getSize().height);
        }
    }

    /** ϤƤ륢ɥ쥹˰ư */
    private void goAddress(boolean isCtrl) {
        if (textFields[TEXTFIELD_ADDRESS] == null) {
            return;
        }

        String path = textFields[TEXTFIELD_ADDRESS].getText();
        if (path != null && path.length() > 0) {
            String surl = option.getBookmarkManager().resolveShortcut(path);
            if (surl != null) {
                path = surl;
            }
            viewer.open((isCtrl ? HawkViewer.NEW_TAB : HawkViewer.CURRENT_TAB), path);
        }
    }

//### ButtonMouseListener
    /** ޥꥹ */
    private final class ButtonMouseListener extends MouseAdapter {
        private int index;

        /** 󥹥󥹤 */
        private ButtonMouseListener(int index) {
            this.index = index;
        }

        /** å줿 */
        public void mouseClicked(MouseEvent e) {
            if (!e.getComponent().isEnabled()) {
                return;
            }

            // å
            if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK) {
                // ۥåȥݥå
                switch (index) {
                case BUTTON_BACK    :
                case BUTTON_FORWARD :
                    Polygon spot = ((ImageButton) e.getComponent()).getHotspotArea();
                    if (spot != null && spot.contains(e.getX(), e.getY())) {
                        return;
                    }
                default: // AVOID
                }

                boolean isShift = ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0);
                boolean isCtrl  = ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0);
                ViewerContext context = ViewerToolbar.this.context;

                switch (index) {
                case BUTTON_BACK    : context.back   (1      ); break;
                case BUTTON_FORWARD : context.forward(1      ); break;
                case BUTTON_RELOAD  : context.reload (isShift); break;
                case BUTTON_STOP    : context.stop   (       ); break;
                case BUTTON_GO      : goAddress(isCtrl)       ; break;
                case BUTTON_HOME    : viewer.moveHome(isCtrl ? HawkViewer.NEW_TAB : HawkViewer.CURRENT_TAB); break;
                case BUTTON_BOOKMARK: break;
                default: // AVOID
                }
            }
        }
    }

//### ButtonStateListener
    /** ֥ꥹ */
    private final class ButtonStateListener implements StateListener {
        private int index;
        private int downState;

        /** 󥹥󥹤 */
        private ButtonStateListener(int index) {
            this.index = index;

            if (ViewerToolbar.this.popupMenu == null) {
                ViewerToolbar.this.popupMenu = new PopupMenu();
                ViewerToolbar.this.add(popupMenu);
            }

            switch (index) {
            case BUTTON_BACK    :
            case BUTTON_FORWARD :
                downState = ImageButton.HOTSPOT_DOWN;
                break;
            case BUTTON_BOOKMARK:
                downState = ImageButton.DOWN;
                break;
            default: // AVOID
            }
        }

        /** ֤ѹ줿 */
        public void stateChanged(StateEvent e) {
            if (((ImageButton) e.getComponent()).getState() == downState) {
                showPopupMenu(e.getComponent(), index, null);
            } else {
                showPopupMenu = false;
            }
        }
    }

//### MoveActionListener
    /** ưѤΥƥꥹ */
    private final class MoveActionListener implements ActionListener {
        private ViewerContext context;
        private int           num;

        /** ˥塼ꥹ  */
        private MoveActionListener(ViewerContext context, int num) {
            this.context = context;
            this.num     = num;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            if (num > 0) {
                context.forward( num);
            } else {
                context.back   (-num);
            }
        }
    }

//### AddressBarKeyListener
    /** ɥ쥹СΥꥹ */
    private final class AddressBarKeyListener extends KeyAdapter {
        /** 󥹥󥹤 */
        private AddressBarKeyListener() {
        }

        /** Υ줿Ȥ */
        public void keyPressed(KeyEvent e) {
            // keyTyped Ǥ ENTER 򥭥åǤʤĶ
            if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                e.consume();
                goAddress(((e.getModifiers() & InputEvent.CTRL_MASK) != 0));
            }
        }
    }

//### AddressBarFocusListener
    /** եϥɥ */
    private final class AddressBarFocusListener implements FocusListener {
        /** 󥹥󥹤 */
        private AddressBarFocusListener() {
        }

        /** եμ */
        public void focusGained(FocusEvent e) {
            TextField c = (TextField) e.getSource();
            c.setCaretPosition(c.getText().length());
            c.selectAll();
        }

        /** եΥ */
        public void focusLost(FocusEvent e) {
            TextField c = (TextField) e.getSource();
            c.select(0, 0);
        }
    }

//### BookmarkListener
    /** ֥åޡꥹ */
    private final class BookmarkListener extends MouseAdapter implements StateListener {
        private Bookmark bookmark;
        private boolean  folder;

        /** 󥹥󥹤 */
        private BookmarkListener(Bookmark bookmark) {
            this.bookmark = bookmark;
            this.folder   = (bookmark.getType() == Bookmark.FOLDER);

            if (ViewerToolbar.this.popupMenu == null) {
                ViewerToolbar.this.popupMenu = new PopupMenu();
                ViewerToolbar.this.add(popupMenu);
            }
        }

        /** å줿 */
        public void mouseClicked(MouseEvent e) {
            if (folder || !e.getComponent().isEnabled()) {
                return;
            }

            // å
            if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK) {
                URL url = bookmark.getURL();
                if (url != null) {
                    viewer.open(((e.getModifiers() & InputEvent.CTRL_MASK) != 0
                                 ? HawkViewer.NEW_TAB
                                 : HawkViewer.CURRENT_TAB),
                                url);
                }
            }
        }

        /** ֤ѹ줿 */
        public void stateChanged(StateEvent e) {
            if (!folder) {
                return;
            }

            if (((ImageButton) e.getComponent()).getState() == ImageButton.DOWN) {
                showPopupMenu(e.getComponent(), PTF_BOOKMARK, bookmark);
            } else {
                showPopupMenu = false;
            }
        }
    }
}
