/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.protocol.about.About;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.net.URL;
import java.net.MalformedURLException;

/**
 * ӥ塼ѤΥݡǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
class ViewerReporter implements Reporter {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    private HawkViewer    viewer;
    private ViewerContext context;
    private ViewerOption  option;
    private int           logLevel;

    /**
     * ݡޤ
     *
     * @param  viewer  ӥ塼
     * @param  context ƥ
     */
    ViewerReporter(HawkViewer viewer, ViewerContext context) {
        this.viewer   = viewer;
        this.context  = context;
        this.option   = viewer.getOption();
        this.logLevel = option.getPropertyInteger(ViewerOption.KEY_LOG_LEVEL, Reporter.ERROR);
    }

    /** {@inheritDoc} */
    public void report(int facility, int level, int emphasis,
                       Content source, int line, int column,
                       String tag, String message) {
        if (level >= logLevel) {
            StringBuffer sb = new StringBuffer();
            sb.append("#");

            switch (level) {
            case Reporter.ERROR  : sb.append("ERROR"  ); break;
            case Reporter.PROF   : sb.append("PROF"   ); break;
            case Reporter.WARNING: sb.append("WARNING"); break;
            default              : sb.append("MESSAGE");
            }

            sb.append(":");
            sb.append(line);
            sb.append(":");
            sb.append(column);
            sb.append(":");
            sb.append(tag);
            sb.append(":");
            sb.append(message);
            Debug.out.println(sb.toString());
        }

        switch (emphasis) {
        case ALERT:
            if (source != null && context.containsRequest(source)) {
                String title;
                int    icon;
                switch (level) {
                case Reporter.ERROR  : title = "error"  ; icon = MessageBox.ICON_STOP       ; break;
                case Reporter.WARNING: title = "warning"; icon = MessageBox.ICON_EXCLAMATION; break;
                default              : title = "message"; icon = MessageBox.ICON_INFORMATION;
                }
                MessageBox.show(viewer, message,
                                Resource.getMessage(RESOURCE, "message.title." + title, null),
                                MessageBox.BUTTON_OK | icon);
            }
            break;
        case FYI:
            if (source == null || context.containsRequest(source)) {
                context.setStatus(message);
            }
            break;
        default: // AVOID
        }
    }

    /** {@inheritDoc} */
    public void report(Throwable t, Request request) {
        t.printStackTrace(Debug.err);

        if (!option.getPropertyBoolean(ViewerOption.KEY_SYSTEM_REPORT_SEND, true)) {
            return;
        }

        URL url = request.url;
        if (About.ABOUT.equalsIgnoreCase(url.getProtocol() + ":")) {
            return;
        }

        if (request.getDocument() != null) {
            url = request.getContentURL();
        }
        Bug bug = new Bug(url.toString(), t.toString(), t);

        try {
            viewer.open(new Request(new URL(About.CRASH_REPORT), bug, null, null, Request.OPEN_NEWTAB, Request.CACHE_NONE));
        } catch (MalformedURLException e) { }
    }

    /** {@inheritDoc} */
    public boolean isReported(int facility, int level) {
        return (level >= logLevel);
    }
}
