/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.java2.DnDListener;
import net.hizlab.kagetaka.addin.java2.DnDWrapper;
import net.hizlab.kagetaka.addin.java2.WheelListener;
import net.hizlab.kagetaka.addin.java2.WheelWrapper;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.awt.panel.HawkPanel;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Document;
import net.hizlab.kagetaka.rendering.FrameItem;
import net.hizlab.kagetaka.rendering.Link;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.viewer.option.InvalidValueException;
import net.hizlab.kagetaka.viewer.option.OptionListener;
import net.hizlab.kagetaka.viewer.option.ViewerController;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Frame;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.Menu;
import java.awt.MenuItem;
import java.awt.Rectangle;
import java.awt.SystemColor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.Vector;

/**
 * ӥ塼Ѥ襭ХѥѥͥǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.9 $
 */
class ViewerPanel extends HawkPanel implements OptionListener {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    private Object            frameLock = new Object();
    private ViewerBaseContext context;
    private ViewerOption      option;

    private MenuItem[][] appMenu;

    private boolean        frameMode;
    private LayoutManager  normalLayoutManager;
    private FramesetLayout framesetLayoutManager;
    private FrameItem      rootFrameItem;

    /**
     * 襭Хѥڥޤ
     *
     * @param  context       륳ƥ
     * @param  option        ӥ塼ץ
     * @param  scrollbarMode Сɽ⡼
     * @param  dndListener   ɥåɥɥåץꥹʡ
     *                       ݡȤƤʤ <code>null</code>
     */
    public ViewerPanel(ViewerBaseContext context, ViewerOption option,
                       int scrollbarMode, DnDListener  dndListener) {
        super(context,
              new ExScrollbar(ExScrollbar.HORIZONTAL),
              new ExScrollbar(ExScrollbar.VERTICAL  ),
              scrollbarMode);

        this.context = context;
        this.option  = option;

        // ˥塼ѹ
        initContextMenu();

        // ޥۥ륵ݡ
        WheelWrapper wheelWrapper = WheelWrapper.getInstance();
        if (wheelWrapper != null) {
            wheelWrapper.addWheelListener(getScreen(), new WheelListener() {
                    /** ޥۥ뤬ž줿 */
                    public void spin(int type, int value, int modifiers) {
                        if (!ViewerPanel.this.option.getPropertyBoolean(ViewerOption.KEY_MOUSE_WHEEL, true)) {
                            return;
                        }
                        if (ViewerPanel.this.option.getPropertyBoolean(ViewerOption.KEY_MOUSE_WHEEL_SWAP, true)) {
                            scroll(SCROLL_HORIZONTAL, type, value);
                        } else {
                            scroll(SCROLL_VERTICAL  , type, value);
                        }
                    }
                }
            );
        }
        // ɥåɥɥåץݡ
        if (dndListener != null) {
            DnDWrapper.getInstance().addDnDListener(getScreen(),
                                                    dndListener,
                                                    new Class[]{File.class,
                                                                String.class});
        }

        // ꥹʤϿ
        option.addListener(this);
    }

    /** ѥͥΥƥȥ˥塼 */
    private void initContextMenu() {
        Vector         items    = null;
        MenuItem       mi       = null;
        ActionListener listener = null;

        // ǳ
        items    = getLinkMenuItems();
        listener = new NewActionListener();
        mi = new MenuItem(getMessage("contextmenu.opennewwindow", null));
        mi.setActionCommand (NewActionListener.WINDOW);
        mi.addActionListener(listener);
        items.insertElementAt(mi, 0);
        mi = new MenuItem(getMessage("contextmenu.opennewtab"   , null));
        mi.setActionCommand (NewActionListener.TAB   );
        mi.addActionListener(listener);
        items.insertElementAt(mi, 1);
        items.insertElementAt(new MenuItem("-"), 2);

        // ץꥱǳ˥塼
        initApplicationMenu();
        // ġå
        initToolTip();
    }

    /** ѥͥΥƥȥ˥塼ʥץꥱˤ */
    private void initApplicationMenu() {
        Menu                      menu     = null;
        Vector                    items    = null;
        ApplicationActionListener listener = null;

        // ץꥱǳ˥塼
        ViewerOption.Application[] apps = option.getApplications();

        if (apps == null || apps.length == 0) {
            if (appMenu == null) {
                return;
            }
            // Ͽ
            items = getMainMenuItems();
            items.removeElement(appMenu[0][0]);
            items.removeElement(appMenu[0][1]);
            items = getLinkMenuItems();
            items.removeElement(appMenu[1][0]);
            items.removeElement(appMenu[1][1]);
            appMenu = null;
            return;
        }

        String label = getMessage("contextmenu.openapp", null);
        if (appMenu == null) {
            appMenu = new MenuItem[2][2];
        }

        for (int i = 0; i < 2; i++) {
            switch (i) {
            case 0:
                listener = new ApplicationActionListener(ApplicationActionListener.MAIN, option);
                items    = getMainMenuItems();
                break;
            case 1:
                listener = new ApplicationActionListener(ApplicationActionListener.LINK, option);
                items    = getLinkMenuItems();
                break;
            default: // AVOID
            }

            if (appMenu[i][0] == null) {
                items.addElement(appMenu[i][0] = new MenuItem("-"  ));
                items.addElement(appMenu[i][1] = new Menu    (label));
            }
            menu = (Menu) appMenu[i][1];
            menu.removeAll();

            for (int j = 0; j < apps.length; j++) {
                createApplicationMenu(apps[j], menu, listener);
            }
        }
    }

    /** ġåפ */
    private void initToolTip() {
        setToolTipOption(option.getPropertyBoolean(ViewerOption.KEY_TOOLTIP_BROWSER_USE  , true),
                         option.getPropertyInteger(ViewerOption.KEY_TOOLTIP_BROWSER_DELAY, 0   ),
                         option.getPropertyInteger(ViewerOption.KEY_TOOLTIP_BROWSER_TERM , 0   ));
    }

    /** ץꥱ˥塼 */
    private void createApplicationMenu(ViewerOption.Application app,
                                       Menu parent,
                                       ApplicationActionListener listener) {
        if (listener.type == ApplicationActionListener.LINK && !app.getOnline()) {
            return;
        }

        MenuItem mi = new MenuItem(app.getName());

        mi.setActionCommand (app.getId());
        mi.addActionListener(listener   );

        parent.add(mi);
    }

//### override
    /** {@inheritDoc} */
    public void setup(Document document) {
        synchronized (frameLock) {
            // ե졼⡼ɤä
            if (frameMode) {
                frameMode = false;

                // 쥤ȥޥ͡򸵤᤹
                setLayout(normalLayoutManager);
                framesetLayoutManager.dispose();
                framesetLayoutManager = null;
                rootFrameItem         = null;

                // Хɽ
                getScreen().setVisible(true);
                validate();
            }
        }

        super.setup(document, true);
    }

    /**
     * 褷ޤ
     *
     * @param  g Graphics
     */
    public void paint(Graphics g) {
        FrameItem rootItem = this.rootFrameItem;
        if (rootItem == null) {
            super.paint(g);
            return;
        }

        Dimension size = getSize();
        g.setColor(SystemColor.control);
        g.fillRect(0, 0, size.width, size.height);

        super.paint(g);

        Component[] components = getComponents();
        int index = 0;
        Rectangle bounds;
        int x, y;
        for (int i = 0; i < components.length; i++) {
            if (components[index].isShowing()
                    && components[index] instanceof ViewerPanel
                    && ((ViewerPanel) components[index]).context.getFrameItem().getBorder()) {
                bounds = components[index].getBounds();
                x = bounds.x - 2;
                y = bounds.y - 2;
                g.setColor(SystemColor.controlShadow);
                g.drawLine(x, y, x + bounds.width  + 3, y);
                g.drawLine(x, y, x, y + bounds.height + 3);
                g.setColor(SystemColor.controlDkShadow);
                g.drawLine(x + 1, y + 1, x + bounds.width  + 2, y + 1);
                g.drawLine(x + 1, y + 1, x + 1, y + bounds.height + 2);
                g.setColor(SystemColor.controlHighlight);
                g.drawLine(x + bounds.width  + 2, y + 2, x + bounds.width + 2, y + bounds.height + 2);
                g.drawLine(x + 2, y + bounds.height + 2, x + bounds.width + 2, y + bounds.height + 2);
                g.setColor(SystemColor.controlLtHighlight);
                g.drawLine(x + bounds.width  + 3, y + 1, x + bounds.width + 3, y + bounds.height + 3);
                g.drawLine(x + 1, y + bounds.height + 3, x + bounds.width + 3, y + bounds.height + 3);
            }
            index++;
        }
    }

//### package
    /**
     * ѥͥե졼Ѥ˥åȥåפޤ
     *
     * @param  request  ꥯ
     * @param  document ɥ
     * @param  rootItem ե졼ॢƥ
     */
    void setup(Request request, Document document, FrameItem rootItem) {
        super.setup(document, false);

        FrameItem[][] items  = rootItem.getItems();
        Dimension     size   = getSize  ();
        Insets        insets = getInsets();
        size.width  -= insets.top + insets.bottom;
        size.height -= insets.left + insets.right;

        synchronized (frameLock) {
            if (!frameMode) {
                frameMode = true;

                // 쥤ȥޥ͡¸
                normalLayoutManager = getLayout();

                // Хɽˤ
                getScreen().setVisible(false);
            }

            // 쥤ȥޥ͡ѹ
            if (framesetLayoutManager != null) {
                framesetLayoutManager.dispose();
            }
            framesetLayoutManager = new FramesetLayout(this, rootItem);
            rootFrameItem         = rootItem;
            setLayout(framesetLayoutManager);

            createFrameset(request, document, items);
            validate();
        }
    }

//### private
    /** ե졼ॻåȤ */
    private void createFrameset(Request request, Document document,
                                FrameItem[][] items) {
        FrameItem[][] children;

        for (int r = 0; r < items.length; r++) {
            for (int c = 0; c < items[r].length; c++) {
                if ((children = items[r][c].getItems()) != null) {
                    createFrameset(request, document, children);
                } else {
                    createFrame(request, document, items[r][c]);
                }
            }
        }
    }

    /** ե졼 */
    private void createFrame(Request request, Document document,
                             FrameItem item) {
        ViewerBaseContext subContext = context.createSubContext(item);

        // ե졼ΥɥȤɤ߹
        String src = item.getSrc();
        if (src != null) {
            try {
                if (item instanceof RequestSet.Frame) {
                    subContext.load(((RequestSet.Frame) item).request, false);
                } else {
                    subContext.load(new Request(document.createURL(src), null,
                                                document.content.url, null,
                                                Request.OPEN_DEFAULT,
                                                request.getUseCache()),
                                    false);
                }
            } catch (MalformedURLException e) {
                Reporter reporter = context.getReporter();
                if (reporter != null) {
                    reporter.report(Reporter.RENDERING,
                                    Reporter.WARNING,
                                    Reporter.NONE,
                                    document.content, 0, 0, "Frame",
                                    Resource.getMessage("render.warning.href",
                                                        new String[]{src, e.toString()}));
                }
            }
        }
    }

    /** ꥽ʸ */
    private String getMessage(String key, String[] args) {
        return Resource.getMessage(RESOURCE, key, args);
    }

//### OptionListener
    /* ͤѹȤ */
    /** {@inheritDoc} */
    public void propertyChange(ViewerOption option, String key, Object oldValue, Object newValue)
            throws InvalidValueException {
    }

    /* ͤѹꡢѹꤵ줿 */
    /** {@inheritDoc} */
    public void propertiesChanged(ViewerOption option, ViewerController c, Enumeration list) {
        String key;
        while (list.hasMoreElements()) {
            key = (String) list.nextElement();

            // ץꥱ
            if (key.startsWith(ViewerOption.KEY_APP)) {
                initApplicationMenu();
                return;
            }

            // ʡġåפ
            if (/*---*/key.compareTo(ViewerOption.KEY_TOOLTIP_BROWSER_USE  ) == 0
                    || key.compareTo(ViewerOption.KEY_TOOLTIP_BROWSER_DELAY) == 0
                    || key.compareTo(ViewerOption.KEY_TOOLTIP_BROWSER_TERM ) == 0) {
                initToolTip();
            }
        }
    }

//### NewActionListener
    /** ǳѤΥꥹ */
    private final class NewActionListener implements ActionListener {
        private static final String WINDOW = "open.newwindow";
        private static final String TAB    = "open.newtab";

        /** ꥹʤ */
        private NewActionListener() {
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            String command = e.getActionCommand();

            Link link = getSelectedLink();
            if (link == null) {
                return;
            }

            Request request = context.getRequest();
            URL     referer = (request != null ? request.getDocument().content.url : null);
            String  target  = (link.target != null ? link.target : request.getDocument().getBaseTarget());

            if (WINDOW.compareTo(command) == 0) {
                context.openHawk(new Request(link.url, null, referer, target, Request.OPEN_NEWWINDOW, Request.CACHE_NORMAL));
            } else
            if (TAB   .compareTo(command) == 0) {
                context.openHawk(new Request(link.url, null, referer, target, Request.OPEN_NEWTAB   , Request.CACHE_NORMAL));
            }
        }
    }

//### ApplicationActionListener
    /** ץꥱ󤫤鳫ѤΥꥹ */
    private final class ApplicationActionListener implements ActionListener {
        private static final int MAIN = 0;
        private static final int LINK = 1;

        private int          type;
        private ViewerOption option;

        /** ꥹʤ */
        private ApplicationActionListener(int type, ViewerOption option) {
            this.type   = type;
            this.option = option;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            String command = e.getActionCommand();
            ViewerOption.Application[] apps = option.getApplications();
            if (apps == null) {
                return;
            }

            for (int i = 0; i < apps.length; i++) {
                ViewerOption.Application app = apps[i];
                if (command.compareTo(app.getId()) != 0) {
                    continue;
                }

                Request request = context.getRequest();
                Content content = null;
                String  cache   = null;

                URL url = null;
                if (type == MAIN) {
                    url = ((request != null) ? request.url : null);
                } else {
                    Link link = getSelectedLink();
                    if (link != null) {
                        url = link.url;
                    }
                }
                if (url == null) {
                    return;
                }

                if (/*---*/type == MAIN
                        && request != null
                        && (content = request.getDocument().content) instanceof ViewerContent) {
                    cache = ((ViewerContent) content).getCachePath();
                }

                StringBuffer exec = new StringBuffer();

                try {
                    app.exec(url, cache, exec);
                } catch (IOException ex) {
                    Frame owner = LayoutUtils.getFrame(ViewerPanel.this);
                    if (owner == null) {
ex.printStackTrace(Debug.out);
                        //### ERROR
                        return;
                    }
                    MessageBox.show(owner,
                                    getMessage("message.appnotopen.text" , new String[]{app.getName(), url.toString(), ex.getMessage()}),
                                    getMessage("message.appnotopen.title", null),
                                    MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
                }
            }
        }
    }
}
