/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.viewer.bookmark.Bookmark;
import net.hizlab.kagetaka.viewer.bookmark.BookmarkManager;
import net.hizlab.kagetaka.viewer.event.ContextFocusEvent;
import net.hizlab.kagetaka.viewer.event.ContextFocusListener;
import net.hizlab.kagetaka.viewer.event.ContextHistoryEvent;
import net.hizlab.kagetaka.viewer.event.ContextHistoryListener;
import net.hizlab.kagetaka.viewer.event.ContextInformationEvent;
import net.hizlab.kagetaka.viewer.event.ContextInformationListener;
import net.hizlab.kagetaka.viewer.event.ContextRequestEvent;
import net.hizlab.kagetaka.viewer.event.ContextRequestListener;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.awt.CheckboxMenuItem;
import java.awt.Component;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.MenuShortcut;
import java.awt.PopupMenu;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.net.URL;
import java.util.Stack;
import java.util.Vector;

/**
 * 륦ɥΥ˥塼򥳥ȥ뤹륯饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
class ViewerMenu
        implements ContextFocusListener,
                   ContextHistoryListener,
                   ContextInformationListener,
                   ContextRequestListener {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    private static final int MENU_MENU  = 0;
    private static final int MENU_POPUP = 1;
    private static final int MENU_ITEM  = 2;
    private static final int MENU_CHECK = 3;

    private static final int MENU_SEPARATOR             =  -1;

    private static final int MENU_FILE                  =  0;
    private static final int MENU_FILE_NEWWINDOW        =  1;
    private static final int MENU_FILE_NEWTAB           =  2;
    private static final int MENU_FILE_OPENURL          =  3;
    private static final int MENU_FILE_OPENFILE         =  4;
    private static final int MENU_FILE_CLOSEWINDOW      =  5;
    private static final int MENU_FILE_CLOSETAB         =  6;
    private static final int MENU_FILE_SAVEAS           =  7;
    private static final int MENU_FILE_PRINTSET         =  8;
    private static final int MENU_FILE_PRINTPREVIEW     =  9;
    private static final int MENU_FILE_PRINT            = 10;
    private static final int MENU_FILE_EXIT             = 11;
    private static final int MENU_EDIT                  = 12;
    private static final int MENU_EDIT_CUT              = 13;
    private static final int MENU_EDIT_COPY             = 14;
    private static final int MENU_EDIT_PASTE            = 15;
    private static final int MENU_EDIT_DELETE           = 16;
    private static final int MENU_EDIT_SELECTALL        = 17;
    private static final int MENU_EDIT_FIND             = 18;
    private static final int MENU_EDIT_FINDNEXT         = 19;
    private static final int MENU_EDIT_FINDBEFORE       = 20;
    private static final int MENU_VIEW                  = 21;
    private static final int MENU_VIEW_SHOW             = 22;
    private static final int MENU_VIEW_SHOW_NAVIBAR     = 23;
    private static final int MENU_VIEW_SHOW_SEARCHBAR   = 24;
    private static final int MENU_VIEW_SHOW_LINKBAR     = 25;
    private static final int MENU_VIEW_SHOW_STATUSBAR   = 26;
    private static final int MENU_VIEW_STOP             = 27;
    private static final int MENU_VIEW_RELOAD           = 28;
    private static final int MENU_VIEW_SIZE             = 29;
    private static final int MENU_VIEW_SIZE_SMALL       = 30;
    private static final int MENU_VIEW_SIZE_LARGE       = 31;
    private static final int MENU_VIEW_SIZE_M3          = 32;
    private static final int MENU_VIEW_SIZE_M2          = 33;
    private static final int MENU_VIEW_SIZE_M1          = 34;
    private static final int MENU_VIEW_SIZE_0           = 35;
    private static final int MENU_VIEW_SIZE_P1          = 36;
    private static final int MENU_VIEW_SIZE_P2          = 37;
    private static final int MENU_VIEW_SIZE_P3          = 38;
    private static final int MENU_VIEW_SIZE_INPUT       = 39;
    private static final int MENU_VIEW_CODE             = 40;
    private static final int MENU_VIEW_CODE_JA_AUTO     = 41;
    private static final int MENU_VIEW_CODE_SHIFT_JIS   = 42;
    private static final int MENU_VIEW_CODE_EUC_JP      = 43;
    private static final int MENU_VIEW_CODE_ISO_2022_JP = 44;
    private static final int MENU_VIEW_CODE_ISO_8859_1  = 45;
    private static final int MENU_VIEW_CODE_UTF_8       = 46;
    private static final int MENU_VIEW_SOURCE           = 47;
    private static final int MENU_VIEW_INFO             = 48;
    private static final int MENU_MOVE                  = 49;
    private static final int MENU_MOVE_BACK             = 50;
    private static final int MENU_MOVE_FORWARD          = 51;
    private static final int MENU_MOVE_HOME             = 52;
    private static final int MENU_MOVE_HISTORY          = 53;
    private static final int MENU_BOOKMARK              = 54;
    private static final int MENU_BOOKMARK_ADD          = 55;
    private static final int MENU_BOOKMARK_ADDEX        = 56;
    private static final int MENU_BOOKMARK_EDITOR       = 57;
    private static final int MENU_TOOL                  = 58;
    private static final int MENU_TOOL_AUTH             = 59;
    private static final int MENU_TOOL_COOKIE           = 60;
    private static final int MENU_TOOL_WEB              = 61;
    private static final int MENU_TOOL_WEB_CONSOLE      = 62;
    private static final int MENU_TOOL_OPTION           = 63;
    private static final int MENU_WINDOW                = 64;
    private static final int MENU_HELP                  = 65;
    private static final int MENU_HELP_ABOUT            = 66;
    private static final int MENU_TAB                   = 67;
    private static final int MENU_TAB_CLOSE             = 68;
    private static final int MENU_TAB_RELOAD            = 69;
    private static final int MENU_TAB_CREATE            = 70;
    private static final int MENU_TAB_CLOSEOTHER        = 71;
    private static final int MENU_TAB_CLOSELEFT         = 72;
    private static final int MENU_TAB_CLOSERIGHT        = 73;
    private static final int MENU_TAB_LOCK              = 74;

    private static final int MENU_SIZE                  = 76;
    private static final int MENU_SIZE_MOVE_MIN         =  6;
    private static final int MENU_SIZE_MOVE_MAX         = 20;
    private static final int MENU_SIZE_BOOKMARK         =  5;

    private static final int LOAD_BEFORE  = 0;
    private static final int LOAD_LOADING = 1;
    private static final int LOAD_LOADED  = 2;

    private HawkViewer   viewer;
    private ViewerOption option;

    private MenuBar    menuBar;
    private MenuItem[] menus;
    private Menu       moveMenu;
    private Menu       bookmarkMenu;
    private Menu       windowMenu;
    private PopupMenu  tabPopupMenu;
    private MenuItem   activeMenuItem;

    private Object stateLock = new Object();               // ֤Υå

    private ViewerContext context;          // ƥ֤ʥƥ
    private Request       request;          // ߤΥꥯ

    /**
     * Υɥ˥塼ޤ
     *
     * @param  viewer ӥ塼
     */
    ViewerMenu(HawkViewer viewer) {
        this.viewer = viewer;
        this.option = viewer.getOption();

        initializeMenu();
    }

    /** ˥塼Сۤ */
    private void initializeMenu() {
        WindowManager windowManager = WindowManager.getInstance();

        menuBar = new MenuBar();
        menus   = new MenuItem[MENU_SIZE];

        Menu menu = null, sub = null;

        // ե ˥塼
        menu = (Menu)
        createMenuItem(MENU_FILE                 , MENU_MENU , null,  0                   );
        createMenuItem(MENU_FILE_NEWWINDOW       , MENU_ITEM , menu,  KeyEvent.VK_N       );
        createMenuItem(MENU_FILE_NEWTAB          , MENU_ITEM , menu,  KeyEvent.VK_T       );
        createMenuItem(MENU_FILE_OPENURL         , MENU_ITEM , menu,  KeyEvent.VK_L       );
        createMenuItem(MENU_FILE_OPENFILE        , MENU_ITEM , menu,  KeyEvent.VK_O       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_CLOSEWINDOW     , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_CLOSETAB        , MENU_ITEM , menu,  KeyEvent.VK_W       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_SAVEAS          , MENU_ITEM , menu,  KeyEvent.VK_S       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_PRINTSET        , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_PRINTPREVIEW    , MENU_ITEM , menu, -KeyEvent.VK_P       );
        createMenuItem(MENU_FILE_PRINT           , MENU_ITEM , menu,  KeyEvent.VK_P       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_FILE_EXIT            , MENU_ITEM , menu,  KeyEvent.VK_Q       );
        menuBar.add(menu);

        // Խ ˥塼
        menu = (Menu)
        createMenuItem(MENU_EDIT                 , MENU_MENU , null,  0                   );
        createMenuItem(MENU_EDIT_CUT             , MENU_ITEM , menu,  KeyEvent.VK_X       );
        createMenuItem(MENU_EDIT_COPY            , MENU_ITEM , menu,  KeyEvent.VK_C       );
        createMenuItem(MENU_EDIT_PASTE           , MENU_ITEM , menu,  KeyEvent.VK_V       );
        createMenuItem(MENU_EDIT_DELETE          , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_EDIT_SELECTALL       , MENU_ITEM , menu,  KeyEvent.VK_A       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_EDIT_FIND            , MENU_ITEM , menu,  KeyEvent.VK_F       );
        createMenuItem(MENU_EDIT_FINDNEXT        , MENU_ITEM , menu,  KeyEvent.VK_G       );
        createMenuItem(MENU_EDIT_FINDBEFORE      , MENU_ITEM , menu, -KeyEvent.VK_G       );
        menuBar.add(menu);

        // ɽ ˥塼
        menu = (Menu)
        createMenuItem(MENU_VIEW                 , MENU_MENU , null,  0                   );
        sub  = (Menu)
        createMenuItem(MENU_VIEW_SHOW            , MENU_MENU , menu,  0                   );
        createMenuItem(MENU_VIEW_SHOW_NAVIBAR    , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SHOW_SEARCHBAR  , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SHOW_LINKBAR    , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SHOW_STATUSBAR  , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_VIEW_STOP            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_VIEW_RELOAD          , MENU_ITEM , menu,  KeyEvent.VK_R       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        sub  = (Menu)
        createMenuItem(MENU_VIEW_SIZE            , MENU_MENU , menu,  0                   );
        createMenuItem(MENU_VIEW_SIZE_SMALL      , MENU_CHECK, sub ,  KeyEvent.VK_ADD     );
        createMenuItem(MENU_VIEW_SIZE_LARGE      , MENU_CHECK, sub ,  KeyEvent.VK_SUBTRACT);
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_M3         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_M2         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_M1         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_0          , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_P1         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_P2         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_P3         , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , sub ,  0                   );
        createMenuItem(MENU_VIEW_SIZE_INPUT      , MENU_CHECK, sub ,  0                   );
        sub  = (Menu)
        createMenuItem(MENU_VIEW_CODE            , MENU_MENU , menu,  0                   );
        createMenuItem(MENU_VIEW_CODE_JA_AUTO    , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_CODE_SHIFT_JIS  , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_CODE_EUC_JP     , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_CODE_ISO_2022_JP, MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_CODE_ISO_8859_1 , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_VIEW_CODE_UTF_8      , MENU_CHECK, sub ,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_VIEW_SOURCE          , MENU_ITEM , menu,  KeyEvent.VK_U       );
        createMenuItem(MENU_VIEW_INFO            , MENU_ITEM , menu,  KeyEvent.VK_I       );
        menuBar.add(menu);

        // ư ˥塼
        menu = moveMenu = (Menu)
        createMenuItem(MENU_MOVE                 , MENU_MENU , null,  0                   );
        createMenuItem(MENU_MOVE_BACK            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_MOVE_FORWARD         , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_MOVE_HOME            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_MOVE_HISTORY         , MENU_ITEM , menu,  KeyEvent.VK_H       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        menuBar.add(menu);

        // ֥åޡ ˥塼
        menu = bookmarkMenu = (Menu)
        createMenuItem(MENU_BOOKMARK             , MENU_MENU , null,  0                   );
        createMenuItem(MENU_BOOKMARK_ADD         , MENU_ITEM , menu,  KeyEvent.VK_D       );
        createMenuItem(MENU_BOOKMARK_ADDEX       , MENU_ITEM , menu, -KeyEvent.VK_D       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_BOOKMARK_EDITOR      , MENU_ITEM , menu,  KeyEvent.VK_B       );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        menuBar.add(menu);
        bookmarkChanged();

        // ġ ˥塼
        menu = (Menu)
        createMenuItem(MENU_TOOL                 , MENU_MENU , null,  0                   );
        createMenuItem(MENU_TOOL_AUTH            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TOOL_COOKIE          , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        sub  = (Menu)
        createMenuItem(MENU_TOOL_WEB             , MENU_MENU , menu,  0                   );
        createMenuItem(MENU_TOOL_WEB_CONSOLE     , MENU_ITEM , sub ,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TOOL_OPTION          , MENU_ITEM , menu,  0                   );
        menuBar.add(menu);

        // ɥ ˥塼
        menu = windowMenu = (Menu)
        createMenuItem(MENU_WINDOW               , MENU_MENU , null,  0                   );
        menuBar.add(menu);
        HawkWindow[] windows = windowManager.getWindows();
        for (int i = 0; i < windows.length; i++) {
            addWindowMenu(windows[i]);
        }

        // إ ˥塼
        menu = (Menu)
        createMenuItem(MENU_HELP                 , MENU_MENU , null,  0                   );
        createMenuItem(MENU_HELP_ABOUT           , MENU_ITEM , menu,  0                   );
        menuBar.setHelpMenu(menu);

        //  ˥塼
        menu = tabPopupMenu = (PopupMenu)
        createMenuItem(MENU_TAB                  , MENU_POPUP, null,  0                   );
        createMenuItem(MENU_TAB_CLOSE            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_RELOAD           , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_CREATE           , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_CLOSEOTHER       , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_CLOSELEFT        , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_CLOSERIGHT       , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_SEPARATOR            , MENU_ITEM , menu,  0                   );
        createMenuItem(MENU_TAB_LOCK             , MENU_CHECK, menu,  0                   );
        viewer.add(tabPopupMenu);

        //### TODO ̤ʤΤ̵ˤƤ
        menus[MENU_FILE_SAVEAS          ].setEnabled(false);
        menus[MENU_FILE_PRINTSET        ].setEnabled(false);
        menus[MENU_FILE_PRINTPREVIEW    ].setEnabled(false);
        menus[MENU_FILE_PRINT           ].setEnabled(false);
        menus[MENU_EDIT_CUT             ].setEnabled(false);
        menus[MENU_EDIT_COPY            ].setEnabled(false);
        menus[MENU_EDIT_PASTE           ].setEnabled(false);
        menus[MENU_EDIT_DELETE          ].setEnabled(false);
        menus[MENU_EDIT_SELECTALL       ].setEnabled(false);
        menus[MENU_EDIT_FIND            ].setEnabled(false);
        menus[MENU_EDIT_FINDNEXT        ].setEnabled(false);
        menus[MENU_EDIT_FINDBEFORE      ].setEnabled(false);
        menus[MENU_VIEW_SHOW_SEARCHBAR  ].setEnabled(false);
        menus[MENU_VIEW_SIZE            ].setEnabled(false);
        menus[MENU_VIEW_CODE            ].setEnabled(false);
        menus[MENU_VIEW_INFO            ].setEnabled(false);
        menus[MENU_TOOL_AUTH            ].setEnabled(false);
        menus[MENU_TOOL_COOKIE          ].setEnabled(false);
        menus[MENU_TOOL_WEB_CONSOLE     ].setEnabled(false);

        viewer.setMenuBar(menuBar);
    }

    /** ˥塼ܤ */
    private MenuItem createMenuItem(int index, int type, Menu parent, int key) {
        String menuKey = null;

        /// MENU
        switch (index) {
        case MENU_SEPARATOR                   : break;
        case MENU_FILE                        : menuKey = "menu.file"                 ; break;
        case MENU_FILE_NEWWINDOW              : menuKey = "menu.file.newwindow"       ; break;
        case MENU_FILE_NEWTAB                 : menuKey = "menu.file.newtab"          ; break;
        case MENU_FILE_OPENURL                : menuKey = "menu.file.openurl"         ; break;
        case MENU_FILE_OPENFILE               : menuKey = "menu.file.openfile"        ; break;
        case MENU_FILE_CLOSEWINDOW            : menuKey = "menu.file.closewindow"     ; break;
        case MENU_FILE_CLOSETAB               : menuKey = "menu.file.closetab"        ; break;
        case MENU_FILE_SAVEAS                 : menuKey = "menu.file.saveas"          ; break;
        case MENU_FILE_PRINTSET               : menuKey = "menu.file.printset"        ; break;
        case MENU_FILE_PRINTPREVIEW           : menuKey = "menu.file.printpreview"    ; break;
        case MENU_FILE_PRINT                  : menuKey = "menu.file.print"           ; break;
        case MENU_FILE_EXIT                   : menuKey = "menu.file.exit"            ; break;
        case MENU_EDIT                        : menuKey = "menu.edit"                 ; break;
        case MENU_EDIT_CUT                    : menuKey = "menu.edit.cut"             ; break;
        case MENU_EDIT_COPY                   : menuKey = "menu.edit.copy"            ; break;
        case MENU_EDIT_PASTE                  : menuKey = "menu.edit.paste"           ; break;
        case MENU_EDIT_DELETE                 : menuKey = "menu.edit.delete"          ; break;
        case MENU_EDIT_SELECTALL              : menuKey = "menu.edit.selectall"       ; break;
        case MENU_EDIT_FIND                   : menuKey = "menu.edit.find"            ; break;
        case MENU_EDIT_FINDNEXT               : menuKey = "menu.edit.findnext"        ; break;
        case MENU_EDIT_FINDBEFORE             : menuKey = "menu.edit.findbefore"      ; break;
        case MENU_VIEW                        : menuKey = "menu.view"                 ; break;
        case MENU_VIEW_SHOW                   : menuKey = "menu.view.show"            ; break;
        case MENU_VIEW_SHOW_NAVIBAR           : menuKey = "menu.view.show.navibar"    ; break;
        case MENU_VIEW_SHOW_SEARCHBAR         : menuKey = "menu.view.show.searchbar"  ; break;
        case MENU_VIEW_SHOW_LINKBAR           : menuKey = "menu.view.show.linkbar"    ; break;
        case MENU_VIEW_SHOW_STATUSBAR         : menuKey = "menu.view.show.statusbar"  ; break;
        case MENU_VIEW_STOP                   : menuKey = "menu.view.stop"            ; break;
        case MENU_VIEW_RELOAD                 : menuKey = "menu.view.reload"          ; break;
        case MENU_VIEW_SIZE                   : menuKey = "menu.view.size"            ; break;
        case MENU_VIEW_SIZE_SMALL             : menuKey = "menu.view.size.small"      ; break;
        case MENU_VIEW_SIZE_LARGE             : menuKey = "menu.view.size.large"      ; break;
        case MENU_VIEW_SIZE_M3                : menuKey = "menu.view.size.-3"         ; break;
        case MENU_VIEW_SIZE_M2                : menuKey = "menu.view.size.-2"         ; break;
        case MENU_VIEW_SIZE_M1                : menuKey = "menu.view.size.-1"         ; break;
        case MENU_VIEW_SIZE_0                 : menuKey = "menu.view.size.0"          ; break;
        case MENU_VIEW_SIZE_P1                : menuKey = "menu.view.size.+1"         ; break;
        case MENU_VIEW_SIZE_P2                : menuKey = "menu.view.size.+2"         ; break;
        case MENU_VIEW_SIZE_P3                : menuKey = "menu.view.size.+3"         ; break;
        case MENU_VIEW_SIZE_INPUT             : menuKey = "menu.view.size.input"      ; break;
        case MENU_VIEW_CODE                   : menuKey = "menu.view.code"            ; break;
        case MENU_VIEW_CODE_JA_AUTO           : menuKey = "menu.view.code.ja_auto"    ; break;
        case MENU_VIEW_CODE_SHIFT_JIS         : menuKey = "menu.view.code.shift_jis"  ; break;
        case MENU_VIEW_CODE_EUC_JP            : menuKey = "menu.view.code.euc-jp"     ; break;
        case MENU_VIEW_CODE_ISO_2022_JP       : menuKey = "menu.view.code.iso-2022-jp"; break;
        case MENU_VIEW_CODE_ISO_8859_1        : menuKey = "menu.view.code.iso-8859-1" ; break;
        case MENU_VIEW_CODE_UTF_8             : menuKey = "menu.view.code.utf-8"      ; break;
        case MENU_VIEW_SOURCE                 : menuKey = "menu.view.source"          ; break;
        case MENU_VIEW_INFO                   : menuKey = "menu.view.info"            ; break;
        case MENU_MOVE                        : menuKey = "menu.move"                 ; break;
        case MENU_MOVE_BACK                   : menuKey = "menu.move.back"            ; break;
        case MENU_MOVE_FORWARD                : menuKey = "menu.move.forward"         ; break;
        case MENU_MOVE_HOME                   : menuKey = "menu.move.home"            ; break;
        case MENU_MOVE_HISTORY                : menuKey = "menu.move.history"         ; break;
        case MENU_BOOKMARK                    : menuKey = "menu.bookmark"             ; break;
        case MENU_BOOKMARK_ADD                : menuKey = "menu.bookmark.add"         ; break;
        case MENU_BOOKMARK_ADDEX              : menuKey = "menu.bookmark.addex"       ; break;
        case MENU_BOOKMARK_EDITOR             : menuKey = "menu.bookmark.editor"      ; break;
        case MENU_TOOL                        : menuKey = "menu.tool"                 ; break;
        case MENU_TOOL_AUTH                   : menuKey = "menu.tool.auth"            ; break;
        case MENU_TOOL_COOKIE                 : menuKey = "menu.tool.cookie"          ; break;
        case MENU_TOOL_WEB                    : menuKey = "menu.tool.web"             ; break;
        case MENU_TOOL_WEB_CONSOLE            : menuKey = "menu.tool.web.console"     ; break;
        case MENU_TOOL_OPTION                 : menuKey = "menu.tool.option"          ; break;
        case MENU_WINDOW                      : menuKey = "menu.window"               ; break;
        case MENU_HELP                        : menuKey = "menu.help"                 ; break;
        case MENU_HELP_ABOUT                  : menuKey = "menu.help.about"           ; break;
        case MENU_TAB                         : menuKey = "menu.tab"                  ; break;
        case MENU_TAB_CLOSE                   : menuKey = "menu.tab.close"            ; break;
        case MENU_TAB_RELOAD                  : menuKey = "menu.tab.reload"           ; break;
        case MENU_TAB_CREATE                  : menuKey = "menu.tab.create"           ; break;
        case MENU_TAB_CLOSEOTHER              : menuKey = "menu.tab.closeother"       ; break;
        case MENU_TAB_CLOSELEFT               : menuKey = "menu.tab.closeleft"        ; break;
        case MENU_TAB_CLOSERIGHT              : menuKey = "menu.tab.closeright"       ; break;
        case MENU_TAB_LOCK                    : menuKey = "menu.tab.lock"             ; break;
        default:
            //### ERROR
            return null;
        }

        String   label = (menuKey == null ? "-" : Resource.getMessage(RESOURCE, menuKey, null));
        MenuItem mi    = null;
        switch (type) {
        case MENU_MENU:
            mi = new Menu(label);
            break;
        case MENU_POPUP:
            mi = new PopupMenu(label);
            break;
        case MENU_ITEM:
            mi = new MenuItem(label);
            if (index != MENU_SEPARATOR) {
                mi.addActionListener(new MenuActionListner(index));
            }
            break;
        case MENU_CHECK:
        {
            CheckboxMenuItem cmi = new CheckboxMenuItem(label);
            cmi.addItemListener(new MenuActionListner(index));
            mi = cmi;

            break;
        }
        default: // AVOID
        }

        if (key != 0) {
            mi.setShortcut(new MenuShortcut(Math.abs(key), key < 0));
        }
        mi.setActionCommand(menuKey != null ? menuKey : "-");

        if (index != MENU_SEPARATOR) {
            menus[index] = mi;
        }
        if (parent != null) {
            parent.add(mi);
        }

        return mi;
    }

//### package
    /**
     * ѤΥݥåץåץ˥塼ɽޤ
     *
     * @param  c ݡͥ
     * @param  x X
     * @param  y Y
     */
    void showTabPopupMenu(Component c, int x, int y) {
        // ֤
        ((CheckboxMenuItem) menus[MENU_TAB_LOCK]).setState(
            (context.getTabMode() & HawkViewer.TABMODE_LOCK) != 0);
        tabPopupMenu.show(c, x, y);
    }

//### HawkWindow
    /**
     * ɥ˥塼ɲäޤ
     *
     * @param  window ɥ
     */
    void addWindowMenu(HawkWindow window) {
        MenuItem mi;

        if (this.viewer == window) {
            CheckboxMenuItem cmi = new CheckboxMenuItem(window.getTitle(), true);
            cmi.addItemListener(new CheckboxMenuItemListener());
            mi = cmi;
        } else {
            mi = new MenuItem(window.getTitle());
            mi.addActionListener(new WindowActionListener(window));
        }

        windowMenu.add(mi);
    }

    /**
     * ɥ˥塼ޤ
     *
     * @param  index ɥǥå
     */
    void removeWindowMenu(int index) {
        windowMenu.remove(index);
    }

    /**
     * ɥ˥塼ѹޤ
     *
     * @param  index  ɥǥå
     * @param  window ɥ
     */
    void changeWindowMenu(int index, HawkWindow window) {
        // ɲäʤȡ˥塼Ѥʤ VM 
        MenuItem mi = windowMenu.getItem(index);
        windowMenu.remove(index);
        mi.setLabel(window.getTitle());
        windowMenu.insert(mi, index);
    }

    /**
     * ֥åޡѹΤޤ
     */
    void bookmarkChanged() {
        BookmarkManager bm = option.getBookmarkManager();
        Vector   bookmarks = bm.getBookmarks();

        // ֥åޡ˥塼õ
        for (int i = bookmarkMenu.getItemCount() - 1; i >= MENU_SIZE_BOOKMARK; i--) {
            bookmarkMenu.remove(i);
        }

        createBookmarkMenu(bookmarkMenu, bookmarks);
    }

    /**
     * ֥åޡɲäΤޤ
     *
     * @param  parent   ƥ֥åޡ
     * @param  bookmark ɲä֥åޡ
     */
    void addToBookmark(Bookmark parent, Bookmark bookmark) {
        Menu menu = bookmarkMenu;

        // parent  root ξ null 
        if (parent != null && parent.getParentFolder() == null) {
            parent = null;
        }

        if (parent != null) {
            Stack stack = new Stack();
            while (parent.getParentFolder() != null) {
                stack.push(new Integer(parent.getNumber()));
                parent = parent.getParentFolder();
            }
            menu = (Menu) menu.getItem(((Integer) stack.pop()).intValue() + MENU_SIZE_BOOKMARK);
            while (!stack.empty()) {
                menu = (Menu) menu.getItem(((Integer) stack.pop()).intValue());
            }
        }

        createBookmarkMenu(menu, bookmark);
    }

    /**
     * ֥åޡ˥塼ޤ
     *
     * @param  menu ɲΥ˥塼
     * @param  tool ֥åޡΥ˥塼ɲä <code>true</code>
     *              ɲäʤ <code>false</code>
     */
    void createBookmarkMenu(Menu menu, boolean tool) {
        BookmarkManager bm = option.getBookmarkManager();
        Vector   bookmarks = bm.getBookmarks();

        if (tool) {
            MenuItem mi;
            boolean canReload = context.canReload();
                 createMenuItem(MENU_BOOKMARK       , MENU_MENU , null, 0);
            mi = createMenuItem(MENU_BOOKMARK_ADD   , MENU_ITEM , menu, 0); if (!canReload) { mi.setEnabled(false); }
            mi = createMenuItem(MENU_BOOKMARK_ADDEX , MENU_ITEM , menu, 0); if (!canReload) { mi.setEnabled(false); }
                 createMenuItem(MENU_SEPARATOR      , MENU_ITEM , menu, 0);
                 createMenuItem(MENU_BOOKMARK_EDITOR, MENU_ITEM , menu, 0);
                 createMenuItem(MENU_SEPARATOR      , MENU_ITEM , menu, 0);
        }

        createBookmarkMenu(menu, bookmarks);
    }
    /**
     * Ʊ٥ƤΥ֥åޡ˥塼ޤ
        *
     * @param  menu      ɲΥ˥塼
     * @param  bookmarks ɲä֥åޡ
     */
    void createBookmarkMenu(Menu menu, Vector bookmarks) {
        int size = bookmarks.size();

        for (int i = 0; i < size; i++) {
            createBookmarkMenu(menu, (Bookmark) bookmarks.elementAt(i));
        }
    }

    /** ֥åޡ˥塼κñʲ */
    private void createBookmarkMenu(Menu menu, Bookmark bookmark) {
        Menu     sub;
        MenuItem mi;

        switch (bookmark.getType()) {
        case Bookmark.FOLDER:
            sub = new Menu(bookmark.getText());
            createBookmarkMenu(sub, bookmark.getBookmarks());
            menu.add(sub);
            break;
        case Bookmark.BOOKMARK:
            mi = new MenuItem(bookmark.getText());
            mi.addActionListener(new BookmarkActionListener(bookmark));
            menu.add(mi);
            break;
        case Bookmark.SEPARATOR:
            menu.addSeparator();
            break;
        default: // AVOID
        }
    }

//### ContextFocusListener
    /* ƥȤեȤ */
    /** {@inheritDoc} */
    public void focusGained(ContextFocusEvent e) {
        ViewerContext context = e.getContext();
        Request       request = e.getRequest();
        synchronized (stateLock) {
            this.context = context;
            this.request = request;
        }

        if (request == null) {
            setLoaded(LOAD_BEFORE , request);
        } else if (context.canStop()) {
            setLoaded(LOAD_LOADING, request);
        } else {
            setLoaded(LOAD_LOADED , request);
        }
    }

    /* ƥȤե򼺤Ȥ */
    /** {@inheritDoc} */
    public void focusLost(ContextFocusEvent e) {
    }

//### ContextHistoryListener
    /* ҥȥ꤬ѹ줿 */
    /** {@inheritDoc} */
    public void historyChanged(ContextHistoryEvent e) {
        ViewerContext context = e.getContext();
        Request       request = e.getRequest();
        synchronized (stateLock) {
            if (this.context != context) {
                return;
            }
            this.request = request;
        }

        historyChanged(context, request);
    }

//### ContextInformationListener
    /* ɥ쥹ѹ줿 */
    /** {@inheritDoc} */
    public void addressChanged(ContextInformationEvent e) {
    }

    /* ȥ뤬ѹ줿 */
    /** {@inheritDoc} */
    public void titleChanged(ContextInformationEvent e) {
        MenuItem mi = activeMenuItem;
        if (mi != null) {
            String title = (String) e.getData();
            mi.setLabel(title != null ? title : "");
        }
    }

    /* ơѹ줿 */
    /** {@inheritDoc} */
    public void statusChanged(ContextInformationEvent e) {
    }

//### ContextRequestListener
    /* ꥯȤդ */
    /** {@inheritDoc} */
    public void requestAccepted(ContextRequestEvent e) {
        ViewerContext context = e.getContext();
        synchronized (stateLock) {
            if (this.context != context) {
                return;
            }
        }

        setSelected(false);
        setLoaded  (LOAD_LOADING, e.getRequest());
    }

    /* ꥯȤλ */
    /** {@inheritDoc} */
    public void requestDone(ContextRequestEvent e) {
        ViewerContext context = e.getContext();
        synchronized (stateLock) {
            if (this.context != context) {
                return;
            }
        }

        setLoaded(LOAD_LOADED, e.getRequest());
    }

//### package
    /**
     * ɽݡͥȤѹȼʤ˥塼ޤ
     *
     * @param  target å
     * @param  b      ɽ줿 <code>true</code>
     *                ɽˤʤä <code>false</code>
     */
    void changeVisibleComponent(int target, boolean b) {
        switch (target) {
        case HawkViewer.COMPONENT_NAVIBAR  : ((CheckboxMenuItem) menus[MENU_VIEW_SHOW_NAVIBAR   ]).setState(b); break;
        case HawkViewer.COMPONENT_SEARCHBAR: ((CheckboxMenuItem) menus[MENU_VIEW_SHOW_SEARCHBAR ]).setState(b); break;
        case HawkViewer.COMPONENT_LINKBAR  : ((CheckboxMenuItem) menus[MENU_VIEW_SHOW_LINKBAR   ]).setState(b); break;
        case HawkViewer.COMPONENT_STATUSBAR: ((CheckboxMenuItem) menus[MENU_VIEW_SHOW_STATUSBAR ]).setState(b); break;
        default: // AVOID
        }
    }

//### private
    /** ɤ߹ޤƤ뤫ɤ */
    private void setLoaded(int load, Request request) {
        Content content;

        //### TODO ̤ʤΤ̵ˤƤ
//        menus[MENU_FILE_SAVEAS      ].setEnabled(load == LOAD_LOADED );
//        menus[MENU_FILE_PRINTPREVIEW].setEnabled(load == LOAD_LOADED );
//        menus[MENU_FILE_PRINT       ].setEnabled(load == LOAD_LOADED );
        menus[MENU_VIEW_STOP        ].setEnabled(load == LOAD_LOADING);
        menus[MENU_VIEW_RELOAD      ].setEnabled(load != LOAD_BEFORE );
        menus[MENU_VIEW_SOURCE      ].setEnabled(load == LOAD_LOADED
                                              && (request != null
                                               && request.getDocument() != null
                                               && (content = request.getDocument().content) instanceof ViewerContent
                                               && ((ViewerContent) content).getCachePath() != null));
//        menus[MENU_VIEW_INFO        ].setEnabled(load == LOAD_LOADED );
        menus[MENU_BOOKMARK_ADD     ].setEnabled(load != LOAD_BEFORE );
        menus[MENU_BOOKMARK_ADDEX   ].setEnabled(load != LOAD_BEFORE );
    }

    /**
     * ϰ򤵤Ƥ뤫ɤȼʤ˥塼ޤ
     *
     * @param  sel ϰ򤵤줿 <code>true</code>
     *             줿 <code>false</code>
     */
    void setSelected(boolean sel) {
        //### TODO ̤ʤΤ̵ˤƤ
//        menus[MENU_EDIT_CUT   ].setEnabled(sel);
//        menus[MENU_EDIT_COPY  ].setEnabled(sel);
//        menus[MENU_EDIT_PASTE ].setEnabled(sel);
//        menus[MENU_EDIT_DELETE].setEnabled(sel);
    }

    /** ֥⡼ɤѹ */
    private void setTabMode(boolean add, long mode) {
        if (add) {
            viewer.addTabMode   (viewer.getTabContextIndex(), mode);
        } else {
            viewer.removeTabMode(viewer.getTabContextIndex(), mode);
        }
    }

    /** ˥塼ѹ */
    private void historyChanged(ViewerContext context, Request request) {
        if (request == null) {
            menus[MENU_MOVE_BACK   ].setEnabled(false);
            menus[MENU_MOVE_FORWARD].setEnabled(false);
            return;
        }

        Request r, currentRequest = request;

        Vector next = new Vector(MENU_SIZE_MOVE_MAX);
        Vector prev = new Vector(MENU_SIZE_MOVE_MAX);
        Vector all  = new Vector(MENU_SIZE_MOVE_MAX);

        // ߤ¦
        int num = MENU_SIZE_MOVE_MAX / 2 - 1;
        while (num-- > 0 && (r = request.getNext()) != null) {
            next.addElement(request = r);
        }

        // ߤ겼¦
        num     = MENU_SIZE_MOVE_MAX - next.size();
        request = currentRequest;
        while (num-- > 0 && (r = request.getPrevious()) != null) {
            prev.addElement(request = r);
        }

        // ٤Ƥ¤٤
        for (int i = next.size() - 1; i >= 0; i--) {
            all.addElement(next.elementAt(i));
        }
        all.addElement(currentRequest);
        for (int i = 0; i < prev.size(); i++) {
            all.addElement(prev.elementAt(i));
        }


        // ư˥塼õ
        for (int i = moveMenu.getItemCount() - 1; i >= MENU_SIZE_MOVE_MIN; i--) {
            moveMenu.remove(i);
        }

        menus[MENU_MOVE_BACK   ].setEnabled(prev.size() > 0);
        menus[MENU_MOVE_FORWARD].setEnabled(next.size() > 0);


        MenuItem mi;
        String   title;
        int      move = next.size();
        num = all.size();
        for (int i = 0; i < num; i++) {
            request = (Request) all.elementAt(i);
            title   = request.getDocument().getTitle();
            if (title == null) {
                title = "";
            }

            if (move == 0) {
                CheckboxMenuItem cmi = new CheckboxMenuItem(title, true);
                cmi.addItemListener(new CheckboxMenuItemListener());
                mi = activeMenuItem = cmi;
            } else {
                mi = new MenuItem(title);
                mi.addActionListener(new MoveActionListener(context, move));
            }
            move--;

            moveMenu.add(mi);
        }
    }

//### MenuActionListner
    /** ˥塼ѤΥꥹ */
    private final class MenuActionListner implements ActionListener, ItemListener {
        private int index = 0;

        /** ˥塼ꥹ  */
        private MenuActionListner(int index) {
            this.index = index;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            boolean isShift = ((e.getModifiers() & ActionEvent.SHIFT_MASK) != 0);
            // e.getModifiers() = 0x80 = 128 Sun JDK 1.4.2_04 ǤϾʤȤ⤳ͤˤʤ롣ʤ
            boolean isCtrl  = ((e.getModifiers() & (ActionEvent.CTRL_MASK | 0x80)) != 0);

            switch (index) {
            case MENU_FILE_NEWWINDOW       : viewer.createWindow       (); break;
            case MENU_FILE_NEWTAB          : viewer.createTab          (HawkViewer.CURRENT_TAB, true); break;
            case MENU_FILE_OPENURL         : viewer.openURL            (HawkViewer.CURRENT_TAB); break;
            case MENU_FILE_OPENFILE        : viewer.openFile           (HawkViewer.CURRENT_TAB); break;
            case MENU_FILE_CLOSEWINDOW     : viewer.closeWindow        (); break;
            case MENU_FILE_CLOSETAB        : viewer.closeTab           (HawkViewer.CURRENT_TAB); break;
//            case MENU_FILE_SAVEAS          : viewer.saveAs             (HawkViewer.CURRENT_TAB); break;
//            case MENU_FILE_PRINTSET        : viewer.setPrint           (); break;
//            case MENU_FILE_PRINTPREVIEW    : viewer.showPrintPreview   (HawkViewer.CURRENT_TAB); break;
//            case MENU_FILE_PRINT           : viewer.printDocument      (HawkViewer.CURRENT_TAB); break;
            case MENU_FILE_EXIT            : viewer.exitViewer         (); break;
//            case MENU_EDIT_CUT             : viewer.cutText            (); break;
//            case MENU_EDIT_COPY            : viewer.copyText           (); break;
//            case MENU_EDIT_PASTE           : viewer.pasteText          (); break;
//            case MENU_EDIT_DELETE          : viewer.deleteText         (); break;
//            case MENU_EDIT_SELECTALL       : viewer.selectAll          (); break;
//            case MENU_EDIT_FIND            : viewer.findText           (); break;
//            case MENU_EDIT_FINDNEXT        : viewer.findNextText       (); break;
//            case MENU_EDIT_FINDBEFORE      : viewer.findBeforeText     (); break;
            case MENU_VIEW_STOP            : viewer.stop               (HawkViewer.CURRENT_TAB); break;
            case MENU_VIEW_RELOAD          : viewer.reload             (HawkViewer.CURRENT_TAB, isShift); break;
            case MENU_VIEW_SOURCE          : viewer.showSource         (HawkViewer.CURRENT_TAB); break;
//            case MENU_VIEW_INFO            : viewer.showInfo           (HawkViewer.CURRENT_TAB); break;
            case MENU_MOVE_BACK            : viewer.moveHistory        (HawkViewer.CURRENT_TAB, -1); break;
            case MENU_MOVE_FORWARD         : viewer.moveHistory        (HawkViewer.CURRENT_TAB, +1); break;
            case MENU_MOVE_HOME            : viewer.moveHome           (isCtrl ? HawkViewer.NEW_TAB : HawkViewer.CURRENT_TAB); break;
            case MENU_MOVE_HISTORY         : viewer.showHistoryEditor  (); break;
            case MENU_BOOKMARK_ADD         : viewer.addBookmark        (HawkViewer.CURRENT_TAB); break;
            case MENU_BOOKMARK_ADDEX       : viewer.addBookmarkAs      (HawkViewer.CURRENT_TAB); break;
            case MENU_BOOKMARK_EDITOR      : viewer.showBookmarkEditor (); break;
//            case MENU_TOOL_AUTH            : viewer.showAuthManager    (); break;
//            case MENU_TOOL_COOKIE          : viewer.showCookieManager  (); break;
            case MENU_TOOL_OPTION          : viewer.showOptionManager  (); break;
//            case MENU_TOOL_WEB_CONSOLE     : viewer.showWebConsole     (); break;
            case MENU_HELP_ABOUT           : viewer.showAbout          (); break;
            case MENU_TAB_CLOSE            : viewer.closeTab           (viewer.getTabContextIndex()); break;
            case MENU_TAB_RELOAD           : viewer.reload             (viewer.getTabContextIndex(), isShift); break;
            case MENU_TAB_CREATE           : viewer.createTab          (viewer.getTabContextIndex() + 1, true); break;
            case MENU_TAB_CLOSEOTHER       : viewer.closeTab           (HawkViewer.OTHERALL_TAB); break;
            case MENU_TAB_CLOSELEFT        : viewer.closeTab           (HawkViewer.LEFTALL_TAB ); break;
            case MENU_TAB_CLOSERIGHT       : viewer.closeTab           (HawkViewer.RIGHTALL_TAB); break;
            default: // AVOID
            }
        }
        /** ϥɥ */
        public void itemStateChanged(ItemEvent e) {
            boolean selected = (e.getStateChange() == ItemEvent.SELECTED);

            // ȥ
            switch (index) {
            case MENU_VIEW_SHOW_NAVIBAR  : viewer.setVisibleComponent(HawkViewer.COMPONENT_NAVIBAR  , selected); break;
            case MENU_VIEW_SHOW_SEARCHBAR: viewer.setVisibleComponent(HawkViewer.COMPONENT_SEARCHBAR, selected); break;
            case MENU_VIEW_SHOW_LINKBAR  : viewer.setVisibleComponent(HawkViewer.COMPONENT_LINKBAR  , selected); break;
            case MENU_VIEW_SHOW_STATUSBAR: viewer.setVisibleComponent(HawkViewer.COMPONENT_STATUSBAR, selected); break;
            case MENU_TAB_LOCK           : setTabMode(selected, HawkViewer.TABMODE_LOCK); break;
            default: // AVOID
            }

            if (!selected) {
                return;
            }

            // 
//            switch (index) {
//            case MENU_VIEW_SIZE_SMALL      : viewer.VIEW_SIZE_SMALL        (); break;
//            case MENU_VIEW_SIZE_LARGE      : viewer.VIEW_SIZE_LARGE        (); break;
//            case MENU_VIEW_SIZE_M3         : viewer.VIEW_SIZE_P3           (); break;
//            case MENU_VIEW_SIZE_M2         : viewer.VIEW_SIZE_P2           (); break;
//            case MENU_VIEW_SIZE_M1         : viewer.VIEW_SIZE_P1           (); break;
//            case MENU_VIEW_SIZE_0          : viewer.VIEW_SIZE_0            (); break;
//            case MENU_VIEW_SIZE_P1         : viewer.VIEW_SIZE_M1           (); break;
//            case MENU_VIEW_SIZE_P2         : viewer.VIEW_SIZE_M2           (); break;
//            case MENU_VIEW_SIZE_P3         : viewer.VIEW_SIZE_M3           (); break;
//            case MENU_VIEW_SIZE_INPUT      : viewer.VIEW_SIZE_INPUT        (); break;
//            case MENU_VIEW_CODE_JA_AUTO    : viewer.VIEW_CODE_JA_AUTO      (); break;
//            case MENU_VIEW_CODE_SHIFT_JIS  : viewer.VIEW_CODE_SHIFT_JIS    (); break;
//            case MENU_VIEW_CODE_EUC_JP     : viewer.VIEW_CODE_EUC_JP       (); break;
//            case MENU_VIEW_CODE_ISO_2022_JP: viewer.VIEW_CODE_ISO_2022_JP  (); break;
//            case MENU_VIEW_CODE_ISO_8859_1 : viewer.VIEW_CODE_ISO_8859_1   (); break;
//            case MENU_VIEW_CODE_UTF_8      : viewer.VIEW_CODE_UTF_8        (); break;
//            }
        }
    }

//### MoveActionListener
    /** ưѤΥƥꥹ */
    private final class MoveActionListener implements ActionListener {
        private ViewerContext context;
        private int           num;

        /** ˥塼ꥹ  */
        private MoveActionListener(ViewerContext context, int num) {
            this.context = context;
            this.num     = num;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            if (num > 0) {
                context.forward( num);
            } else {
                context.back   (-num);
            }
        }
    }

//### BookmarkActionListener
    /** ưѤΥƥꥹ */
    private final class BookmarkActionListener implements ActionListener {
        private Bookmark bookmark;

        /** ˥塼ꥹ  */
        private BookmarkActionListener(Bookmark bookmark) {
            this.bookmark = bookmark;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            URL url = bookmark.getURL();
            if (url != null) {
                // e.getModifiers() = 0x80 = 128 Sun JDK 1.4.2_04 ǤϾʤȤ⤳ͤˤʤ롣ʤ
                viewer.open(((e.getModifiers() & (ActionEvent.CTRL_MASK | 0x80)) != 0
                             ? HawkViewer.NEW_TAB
                             : HawkViewer.CURRENT_TAB),
                            url);
            }
        }
    }

//### CheckboxMenuItemListener
    /** ɥѤΥƥꥹ */
    private final class CheckboxMenuItemListener implements ItemListener {
        /** ˥塼ƥꥹ  */
        private CheckboxMenuItemListener() {
        }

        /** ϥɥ */
        public void itemStateChanged(ItemEvent e) {
            CheckboxMenuItem menu = (CheckboxMenuItem) e.getSource();
            menu.setState(true);
        }
    }

//### WindowActionListener
    /** ɥѤΥƥꥹ */
    private final class WindowActionListener implements ActionListener {
        private HawkWindow window;

        /** ˥塼ꥹ  */
        private WindowActionListener(HawkWindow window) {
            this.window = window;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            window.show();
        }
    }
}
