/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.java2.LangWrapper;
import net.hizlab.kagetaka.addin.java2.NETBWrapper;
import net.hizlab.kagetaka.io.MeteredInputStream;
import net.hizlab.kagetaka.net.HttpURLConnectionWrapper;
import net.hizlab.kagetaka.net.URLConnectionWrapper;
import net.hizlab.kagetaka.protocol.CacheSupported;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.util.ContentHeaderParser;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.util.Environment;
import net.hizlab.kagetaka.util.StringUtils;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.awt.Toolkit;
import java.awt.image.ImageProducer;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.net.ContentHandler;
import java.net.ContentHandlerFactory;
import java.net.HttpURLConnection;
import java.net.URLConnection;
import java.net.UnknownServiceException;
import java.text.ParseException;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * ӥ塼ѤγĥƥĤɽ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.10 $
 */
public class ViewerContent extends Content {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    private static final String CONTENT_PATH_PROPERTY = "java.content.handler.pkgs";
    private static final String CONTENT_CLASS_PREFIX  = "sun.net.www.content";

    // ƥĥϥɥեȥ
    private static ContentHandlerFactory factory;
    // ƥĥϥɥ饭å
    private static Hashtable handlers = new Hashtable();

    private SSLCertification sslCertification;
    private CacheEntry       cacheEntry;

    private int validTimes = 1;

    /**
     * ƥĤޤ
     *
     * @param  connection       ͥ
     * @param  sslCertification SSL 
     */
    public ViewerContent(URLConnection connection,
                         SSLCertification sslCertification) {
        super(connection);

        this.sslCertification = sslCertification;
    }

    /**
     * å奨ȥ򸵤ˤƥĤޤ
     *
     * @param  connection ͥ
     * @param  cacheEntry å奨ȥ
     */
    public ViewerContent(URLConnection connection,
                         CacheEntry cacheEntry) {
        this(connection, cacheEntry.sslCertification);

        this.cacheEntry = cacheEntry;
    }

    /** {@inheritDoc} */
    public synchronized InputStream getInputStream(Reporter reporter)
            throws IOException {
        InputStream is = super.getInputStream(reporter);

        // ȽƤϡ𤹤褦ˤ
        if (is != null && reporter != null && contentLength > 0) {
            is = new Chatty(is, reporter);
        }

        return is;
    }

    /** {@inheritDoc} */
    public synchronized Object getObject(int type)
            throws IOException {
        if (object != null) {
            return object;
        }

        // ƥĥפ׵ᤵƤΤȰ㤦
        if (!containsType(type)) {
Debug.out.println("### WARING ### : ViewerContent.getObject : invalid content-type : " + type + ", " + contentType);
            // HTTP 쥹ݥ󥹤 300 ʾξ null ֤
            if (connection instanceof HttpURLConnection) {
                if (((HttpURLConnection) connection).getResponseCode() >= 300) {
                    return null;
                }
            }

            // ǡƬɤ߹
            InputStream is = connection.getInputStream();
            if (!is.markSupported()) {
                is = new BufferedInputStream(is);
            }
            is.mark(8);
            int c1 = is.read();
            int c2 = is.read();
            int c3 = is.read();
            int c4 = is.read();
            int c5 = is.read();
            int c6 = is.read();
            int c7 = is.read();
            int c8 = is.read();
            is.reset();
            is.mark(-1);

            ContentType newContentType = null;
            if (c1 == 'G' && c2 == 'I' && c3 == 'F' && c4 == '8') {
                // GIF
                newContentType = new ContentType("image", "gif");
            } else if (c1 == '\377' && c2 == '\330' && c3 == '\377') {
                // JPEG
                newContentType = new ContentType("image", "jpeg");
            } else if (c1 == '#' && c2 == 'd' && c3 == 'e' && c4 == 'f') {
                // Xbm
                newContentType = new ContentType("image", "x-xbitmap");
            } else if (c1 == 137 && c2 == 80 && c3 == 78
                    && c4 ==  71 && c5 == 13 && c6 == 10
                    && c7 ==  26 && c8 == 10) {
                // PNG
                newContentType = new ContentType("image", "png");
            }
Debug.out.println("### WARING ### : ViewerContent.getObject : reset content-type : " + newContentType);

            // פǻƤߤ
            if (newContentType != null
                    && (object = getContentHandler(newContentType.toString()).getContent(createURLConnection(is))) != null) {
                // פѴ
                if (object instanceof ImageProducer) {
                    object = Toolkit.getDefaultToolkit().createImage((ImageProducer) object);
                }
                contentType = newContentType;
                return object;
            }
        }

        return super.getObject(type);
    }

    /** {@inheritDoc} */
    public boolean isCached() {
        return (cacheEntry != null);
    }

    /**
     * ݡɬפȤʤץåȥȥ꡼֤ޤ
     *
     * @return ץåȥȥ꡼
     */
    public String getFileName() {
        String fileName = null;

        // Content-Disposition إåե̾
        String cd = connection.getHeaderField("Content-Disposition");
        if (cd != null) {
            try {
                ContentHeaderParser.Data data = ContentHeaderParser.parse(cd);
                fileName = (String) data.parameters.get("filename");
            } catch (ParseException e) {
                //### ERROR
            }
        }

        // ƥĥפ
        if (fileName == null) {
            fileName = contentType.getParameter("filename");
        }

        // URL ե̾
        if (fileName == null) {
            fileName = StringUtils.getFile(url);
        }

        return fileName;
    }

    /**
     * SSL ֤ޤ
     *
     * @return SSL 
     */
    public SSLCertification getSSLCertification() {
        return sslCertification;
    }

    /**
     * ƥĤͭǺѲǽɤ֤ޤ
     * ͭ¤꤬Ƥʤ硢ץˤäơ
     * Ѳǽɤ֤ޤ
     *
     * @param  option  ץ
     * @param  request ꥯ
     *
     * @return ƥĤѲǽʾ <code>true</code>
     *         ʳξ <code>false</code>
     */
    boolean validateExpiration(ViewerOption option, Request request) {
        int times = (cacheEntry != null ? cacheEntry.incrementTimes() : 0);

        // Expire ꤵƤϡ򻲾Ȥ
        if (expiration > 0) {
           return (expiration > System.currentTimeMillis());
        }

        // ꥯȤȥåפǤϤʤ䡢
        // Last-Modifed  ETag ꡢåλѲ
        // ʲξϺѲǽMozilla  - ֤
        return (request.parent != null
             || ((lastModified > 0 || eTag != null)
              && times <= option.getPropertyInteger(ViewerOption.KEY_CACHE_VALID_TIMES, 1)));
    }

    /**
     * åեΥѥ֤ޤ
     *
     * @return åեΥѥ
     *         ¸ߤʤ <code>null</code>
     */
    String getCachePath() {
        if (connection instanceof CacheSupported) {
            return ((CacheSupported) connection).getCachePath();
        }

        if (url.getProtocol().toLowerCase().compareTo("file") == 0) {
            String path = url.getFile();
            if (Environment.isWindows) {
                path = path.substring(1);
                path = path.replace('/', File.separatorChar);
            }
            return path;
        }

        return null;
    }

    /**
     * åϿѤߤΥ꥽˴ޤ
     */
    void removeCache() {
        if (connection instanceof CacheSupported) {
            ((CacheSupported) connection).removeCache();
        }
    }

    /**
     * åλѲꥻåȤޤ
     */
    void resetTimes() {
        if (cacheEntry != null) {
            cacheEntry.resetTimes();
        }
    }

    /** ƥĥפ type ˴ޤޤƤ뤫֤ */
    private boolean containsType(int type) {
        String mainType = contentType.getType();
        int    resType  = 0;
        if (mainType.compareTo("image") == 0) {
            resType = TYPE_IMAGE;
        } else if (mainType.compareTo("text") == 0) {
            resType = TYPE_TEXT;
        } else if (mainType.compareTo("audio") == 0) {
            resType = TYPE_AUDIO;
        } else {
            return true;
        }

        return ((type & resType) != 0);
    }

    /** ƥĥϥɥ */
    private ContentHandler getContentHandler(String type)
            throws UnknownServiceException {
        ContentHandler handler = (ContentHandler) handlers.get(type);
        if (handler != null) {
            return handler;
        }

/*
        // factory Ͼ nullʤɤä URLConnection Τ
        if (factory != null
            && (handler = factory.createContentHandler(type)) == null) {
            return null;
        }
*/
        if ((handler = lookupContentHandlerClassFor(type)) == null) {
            return null;
        }

        handlers.put(type, handler);
        return handler;
    }

    /** ƥĥפ򥯥饹̾Ȥץեåղäƥåå */
    private ContentHandler lookupContentHandlerClassFor(String type) {
        String typeName = convertTypeToPackageName(type);
        String prefixes = getContentHandlerPkgPrefixes();

        StringTokenizer st = new StringTokenizer(prefixes, "|");

        while (st.hasMoreTokens()) {
            String prefix = st.nextToken().trim();

            try {
                String className = prefix + "." + typeName;
                Class clazz = null;
                try {
                    clazz = Class.forName(className);
                } catch (ClassNotFoundException e) {
                    LangWrapper langWrapper = LangWrapper.getInstance();
                    if (langWrapper != null) {
                        ClassLoader cl = langWrapper.getSystemClassLoader();
                        if (cl != null) {
                            clazz = cl.loadClass(className);
                        }
                    }
                }
                if (clazz != null) {
                    ContentHandler handler = (ContentHandler) clazz.newInstance();
                    return handler;
                }
            } catch (Exception e) {
                //### ERROR
e.printStackTrace(Debug.out);
            }
        }

        return null;
    }

    /** ƥĥפѥå̾Ѵ */
    private String convertTypeToPackageName(String type) {
        type = type.toLowerCase();

        int    length  = type.length();
        char[] newName = new char[length];
        type.getChars(0, length, newName, 0);

        for (int i = 0; i < length; i++) {
            char c = newName[i];
            if (c == '/') {
                newName[i] = '.';
            } else if (!(('A' <= c && c <= 'Z')
                      || ('a' <= c && c <= 'z')
                      || ('0' <= c && c <= '9'))) {
                newName[i] = '_';
            }
        }

        return new String(newName);
    }


    /** ƥĥϥɥΥץեåꥹȤ */
    private String getContentHandlerPkgPrefixes() {
        String packagePrefixList = System.getProperty(CONTENT_PATH_PROPERTY, "");

        if (packagePrefixList != "") {
            packagePrefixList += "|";
        }

        return packagePrefixList + CONTENT_CLASS_PREFIX;
    }

    /** ꤷϥȥ꡼֤ URLConnection  */
    private URLConnection createURLConnection(InputStream is) {
        NETBWrapper netbWrapper = NETBWrapper.getInstance();
        return (netbWrapper != null
                ? netbWrapper.getURLConnectionWrapper(url, connection, is)
                : (connection instanceof HttpURLConnection
                   ? (URLConnection) new HttpURLConnectionWrapper(url, (HttpURLConnection) connection, is)
                   : (URLConnection) new URLConnectionWrapper    (url, connection, is)));
    }

//### Chatty
    /** ɤ߹Ψ𤹤ꥹ */
    private final class Chatty extends MeteredInputStream {
        private Reporter reporter;
        private String   name;

        private int percent = 0;

        /** 󥹥󥹤 */
        private Chatty(InputStream in, Reporter reporter) {
            super(in);
            this.reporter = reporter;
            this.name     = url.toString();
        }

        /** ɤ߹ХȿѤä */
        protected void changed(long length) {
            int p = (int) (length * 100 / contentLength);

            // 100 % ˤϤʤʲɤ߹ߤ뤫⤷ʤ
            if (p >= 100) {
                p = 99;
            }

            if (p != percent) {
                // ΥѡȤɽ
                if (percent > 0) {
                    //### ERROR ˤ?
                    reporter.report(Reporter.NETWORK, Reporter.INFO, Reporter.FYI,
                                    ViewerContent.this, 0, 0,
                                    "LOAD", Resource.getMessage(RESOURCE, "content.load.percent",
                                                                new Object[]{name, new Integer(percent)}));
                }
                percent = p;
            }
        }
    }
}
