/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.BindException;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.text.ParsePosition;
import java.util.Random;
import java.util.StringTokenizer;

/**
 * ¾ΥץꥱʤɤꥯȤդơ
 * ɽ뤿Υ饹Ǥ
 * Ϥ˥ݡȤ򥪡ץ󤷥Х⡼ɤȤƵư褦Ȥޤ
 * ξ硢¾ꥯȤդ֤ˤʤޤ
 * ˥ݡȤѤƤϥ饤ȥ⡼ɤˤʤꡢ
 * ¾ΥФ˥ꥯȤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.5 $
 */
class RequestBroker {
    // ߤΥС
    private static final long   CURRENT_VERSION = 1;
    // եΥץեå
    private static final String PREFIX          = "lock.";
    // ǡΥ󥳡ǥ
    private static final String ENCODING        = "SJIS";

    // ޥ
    private static final String COMMAND_HELO    = "HELO";
    private static final String COMMAND_GET     = "GET";
    private static final String COMMAND_QUIT    = "QUIT";
    private static final String COMMAND_EXIT    = "EXIT";
    // 쥹ݥ
    private static final String RESPONSE_OK             = "200";
    private static final String RESPONSE_BAD_REQUEST    = "400";
    private static final String RESPONSE_FORBIDDEN      = "403";
    private static final String RESPONSE_BAD_METHOD     = "405";
    private static final String RESPONSE_INTERNAL_ERROR = "500";
    private static final String RESPONSE_VERSION        = "505";

    private ViewerOption option;
    private String       userName;
    private Listener     listener;
    private Sender       sender;

    /**
     * ꥯȥ֥ޤ
     *
     * @param  option ץ
     */
    RequestBroker(ViewerOption option) {
        this.option   = option;
        this.userName = System.getProperty("user.name", "?");

        connect();
    }

    /** ³򳫻 */
    private boolean connect() {
        String home = option.getUserHome();

        if (!ViewerOption.ensureDirectory(home)) {
            return false;
        }

        String version = Resource.getMessage("kagetaka.version"  , null);

        int              port   = -1;
        Random           random = null;
        RandomAccessFile raf    = null;

        try {
            // åե򳫤
            for (int loopCounter = 0; loopCounter < 5; loopCounter++) {
                try {
                    raf = new RandomAccessFile(home + File.separator + PREFIX + version, "rw");
                    break;
                } catch (FileNotFoundException e) {
                    //### ERROR ƥǥ쥯ȥ꤬ʤʥե뤬̵ʤ餳㳰ϵʤ
Debug.out.println(e);
                    return false;
                } catch (IOException e) {
Debug.out.println(e);
                }
                try {
                    Thread.sleep(200);
                } catch (InterruptedException e) {
                    return false;
                }
            }
            if (raf == null) {
                return false;
            }

            // ݡȤɤ߹
            try {
                port = raf.readInt();
            } catch (EOFException e) {
            } catch (IOException e) {
                //### ERROR
e.printStackTrace(Debug.out);
            }

            try {
                // ݡȤõ
                for (int tryCounter = 0; tryCounter < 100; tryCounter++) {
                    if (port <= 0) {
                        if (random == null) {
                            random = new Random(userName.hashCode());
                        }
                        port = getPort(random);
                    }
//Debug.out.println("try = " + port);
                    // ³Ƥߤ
                    try {
                        listener = new Listener(port);
                        listener.start();
                        return true;
                    } catch (BindException e) {
                        // ˻ѺѤߤξϡ¾ԤƤʤϤ
                        try {
                            sender = new Sender(port);
                            return true;
                        } catch (IllegalArgumentException ex) {
                            // 桼äꤷ
                        //### ERROR
Debug.out.println(ex);
                        }
                    }
                    port = 0;
                }
            } catch (IOException e) {
                //### ERROR
e.printStackTrace(Debug.out);
            }
        } finally {
            if (raf != null) {
                // ѲǽʥݡȤ񤭹
                if (listener != null || sender != null) {
                    try {
                        raf.seek(0);
                        raf.writeInt(port);
                    } catch (IOException e) {
                        //### ERROR
e.printStackTrace(Debug.out);
                    }
                }
                try {
                    raf.close();
                } catch (IOException e) { }
            }
        }

        return false;
    }

    /** ݡȤ */
    private int getPort(Random random) {
        int min  = option.getPropertyInteger(ViewerOption.KEY_LISTENER_MIN, 26201);
        int max  = option.getPropertyInteger(ViewerOption.KEY_LISTENER_MAX, 26500);
        return (int) (random.nextDouble() * (max - min) + min);
    }

    /**
     * Х⡼ɤȤƼ¹椫ɤ֤ޤ
     *
     * @return Х⡼ɤξ <code>true</code>
     *         饤ȥ⡼ɤȤʤäƤ <code>false</code>
     */
    synchronized boolean isServerMode() {
        return (sender == null);
    }

    /**
     * ꥯȤ¸α륤󥹥󥹤ޤ
     *
     * @param  url ꥯȤ URL
     */
    synchronized void send(String url) {
        if (sender == null) {
            throw new IllegalStateException("server mode");
        }

        sender.send(url);
    }

    /**
     * ߤޤ
     */
    synchronized void stop() {
        if (listener != null) {
            listener.interrupt();
            listener = null;
        }
        if (sender != null) {
            sender.close();
            sender = null;
        }
    }

//### Sender
    /**  */
    private final class Sender {
        private Socket         client;
        private FlushWriter    fw;
        private BufferedReader br;

        /** 󥹥󥹤κ */
        private Sender(int port) throws IOException, IllegalArgumentException {
            client = new Socket("127.0.0.1", port);
            fw     = new FlushWriter(
                       new OutputStreamWriter(client.getOutputStream(),
                                              ENCODING));
            br     = new BufferedReader(
                       new InputStreamReader (client.getInputStream(),
                                              ENCODING));

            // ͥ
            fw.println(COMMAND_HELO + " " + CURRENT_VERSION + " " + userName);
            String response = null;
            try {
                if ((response = br.readLine()) != null) {
                    String code = getCode(response);
                    if (code.compareTo(RESPONSE_OK) == 0) {
                        return;
                    }
                }
            } catch (IOException e) {
                //### ERROR
e.printStackTrace(Debug.out);
            }

            close();
            throw new IllegalArgumentException("response is " + response);
        }

        /**  */
        private void send(String url) {
            fw.println(COMMAND_GET + " " + url);
            try {
                String response = br.readLine();
//Debug.out.println(response);
                if (getCode(response).compareTo(RESPONSE_OK) != 0) {
                    //### ERROR
Debug.out.println("Response is " + response + " for GET");
                }
            } catch (IOException e) { }
        }

        /** 쥹ݥ󥹥ɤ */
        private String getCode(String line) {
            int p = line.indexOf(' ');
            return (p != -1
                    ? line.substring(0, p)
                    : line);
        }

        /** Ĥ */
        private void close() {
            try {
                fw.close();
            } catch (IOException e) { }
            try {
                client.close();
            } catch (IOException e) { }
        }
    }

//### Listener
    /** ꥹ */
    private final class Listener extends Thread {
        private ServerSocket server;

        /** 󥹥󥹤κ */
        private Listener(int port) throws IOException {
            setName("RequestBroker-Listener-" + port);

            InetAddress local;
            try {
                local = InetAddress.getByName("127.0.0.1");
            } catch (UnknownHostException e) {
                //### ERROR
Debug.out.println(e);
                local = null;
            }
            server = new ServerSocket(port, 10, local);
        }

        /** ¹ */
        public void run() {
            try {
                Socket accept;
                for (;;) {
                    accept = server.accept();
                    (new Agent(accept)).start();
                }
            } catch (IOException e) {
                //### ERROR
e.printStackTrace(Debug.out);
            } finally {
                try {
                    server.close();
                } catch (IOException e) { }
            }
        }
    }

//### Agent
    /** ¹ */
    private final class Agent extends Thread {
        private Socket accept;

        /** 󥹥󥹤 */
        private Agent(Socket accept) {
            this.accept = accept;

            setName("RequestBroker-Agent-" + accept.getLocalPort());
        }

        /** ¹ */
        public void run() {
            FlushWriter    fw = null;
            BufferedReader br = null;
            try {
                fw = new FlushWriter(
                       new OutputStreamWriter(accept.getOutputStream(),
                                              ENCODING));
                br = new BufferedReader(
                       new InputStreamReader(accept.getInputStream(),
                                             ENCODING));
                String     line;
                Header     header;
                Request    request;
                HawkViewer viewer;

                // إå
                for (;;) {
                    if ((line = br.readLine()) == null) {
                        return;
                    }
                    try {
                        header = new Header(line);
                        fw.println(RESPONSE_OK + " OK");

                        // EXIT or QUIT
                        if (header.version < 0) {
                            return;
                        }
                        break;
                    } catch (IllegalArgumentException e) {
                        fw.println(e.getMessage());
                    }
                }

                // ꥯ
                while ((line = br.readLine()) != null) {
                    try {
                        request = new Request(line);

                        // EXIT or QUIT
                        if (request.url == null) {
                            fw.println(RESPONSE_OK + " OK");
                            return;
                        }

                        // ɽ
                        viewer  = WindowManager.getInstance().getActiveViewer();
                        if (viewer != null) {
                            viewer.open(HawkViewer.NEW_TAB, request.url);
                            fw.println(RESPONSE_OK + " OK");
                        } else {
                            //### ERROR
                            fw.println(RESPONSE_INTERNAL_ERROR + " Internal Error");
                        }
                    } catch (IllegalArgumentException e) {
                        fw.println(e.getMessage());
                    }
                }
            } catch (IOException e) {
                        //### ERROR
e.printStackTrace(Debug.out);
            } finally {
                if (br != null) {
                    try {
                        br.close();
                    } catch (IOException e) { }
                }
                if (fw != null) {
                    try {
                        fw.flush();
                        fw.close();
                    } catch (IOException e) { }
                }
                try {
                    accept.close();
                } catch (IOException e) { }
            }
        }
    }

//### Header
    /** إå */
    private final class Header {
        private long   version;
        private String userName;

        /** إåʬ */
        private Header(String line) throws IllegalArgumentException {
            StringTokenizer st = new StringTokenizer(line, " ");
            String hello   = (st.hasMoreTokens() ? st.nextToken().toUpperCase() : null);
            String version = (st.hasMoreTokens() ? st.nextToken().toUpperCase() : null);
            String user    = (st.hasMoreTokens() ? st.nextToken()               : null);

            //// QUIT, EXIT
            if (hello != null
                    && (hello.compareTo(COMMAND_QUIT) == 0
                     || hello.compareTo(COMMAND_EXIT) == 0)) {
                this.version = -1;
                return;
            }

            // ɬܥå
            if (hello == null || version == null || user == null) {
                throw new IllegalArgumentException(RESPONSE_BAD_REQUEST + " Bad Request (no hello, version and user)");
            }
            if (hello.compareTo(COMMAND_HELO) != 0) {
                throw new IllegalArgumentException(RESPONSE_BAD_REQUEST + " Bad Request (invalid hello " + hello + ")");
            }
            // СΥå
            DecimalFormat df = new DecimalFormat("0");
            ParsePosition pp = new ParsePosition(0);
            Number number = df.parse(version, pp);
            if (number == null
                    || pp.getIndex() != version.length()
                    || !(number instanceof Long)) {
                throw new IllegalArgumentException(RESPONSE_BAD_REQUEST + " Bad Request (invalid version " + version + ")");
            }
            long versionNumber = number.longValue();

            if (versionNumber > CURRENT_VERSION) {
                throw new IllegalArgumentException(RESPONSE_VERSION + " Version Not Supported (version " + versionNumber + " > " + CURRENT_VERSION + ")");
            }

            // 桼̾Υå
            if (user.compareTo(RequestBroker.this.userName) != 0) {
                throw new IllegalArgumentException(RESPONSE_FORBIDDEN + " Forbidden (invalid user " + user + ")");
            }

            this.version  = versionNumber;
            this.userName = user;
        }
    }

//### Request
    /** ꥯ */
    private final class Request {
        private final String url;

        /** ꥯȤιԤʬ */
        private Request(String line) throws IllegalArgumentException {
            int p = line.indexOf(' ');
            String command = (p == -1
                              ? line.trim().toUpperCase()
                              : line.substring(0, p).trim().toUpperCase());

            //// GET
            if (command.compareTo(COMMAND_GET) == 0) {
                //  URL 
                if (p != -1) {
                    int length = line.length();
                    while (++p < length) {
                        if (line.charAt(p) != ' ') {
                            break;
                        }
                    }
                    if (p < length) {
                        url = line.substring(p);
                        return;
                    }
                }
                throw new IllegalArgumentException(RESPONSE_BAD_REQUEST + " Bad Request");
            }

            //// QUIT, EXIT
            if (command.compareTo(COMMAND_QUIT) == 0
                    || command.compareTo(COMMAND_EXIT) == 0) {
                url = null;
                return;
            }

            throw new IllegalArgumentException(RESPONSE_BAD_METHOD + " Method Not Allowed");
        }
    }

//### FlushWriter
    /** ɬեåԤ饤 */
    private final class FlushWriter extends BufferedWriter {
        private IOException exception;

        /** 󥹥󥹤 */
        private FlushWriter(Writer w) {
            super(w);
        }

        /** Խ񤭹ߤԤեå */
        private void println(String value) {
            try {
                write(value, 0, value.length());
                write("\r\n");
                flush();
            } catch (IOException e) {
                if (exception == null) {
                    exception = e;
                }
            }
        }
    }
}
