/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.java2.AWTWrapper;
import net.hizlab.kagetaka.addin.java2.DnDListener;
import net.hizlab.kagetaka.addin.java2.DnDWrapper;
import net.hizlab.kagetaka.awt.ColorConverter;
import net.hizlab.kagetaka.awt.ImageButton;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.awt.StatusBarIcon;
import net.hizlab.kagetaka.awt.TabPanel;
import net.hizlab.kagetaka.awt.event.TabEvent;
import net.hizlab.kagetaka.awt.event.TabListener;
import net.hizlab.kagetaka.awt.image.OffscreenBuffer;
import net.hizlab.kagetaka.protocol.about.About;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.Environment;
import net.hizlab.kagetaka.util.StringUtils;
import net.hizlab.kagetaka.util.windows.Shortcut;
import net.hizlab.kagetaka.viewer.bookmark.Bookmark;
import net.hizlab.kagetaka.viewer.bookmark.BookmarkAs;
import net.hizlab.kagetaka.viewer.bookmark.BookmarkManager;
import net.hizlab.kagetaka.viewer.event.ContextInformationEvent;
import net.hizlab.kagetaka.viewer.event.ContextInformationListener;
import net.hizlab.kagetaka.viewer.event.ContextRequestEvent;
import net.hizlab.kagetaka.viewer.event.ContextRequestListener;
import net.hizlab.kagetaka.viewer.history.HistoryManager;
import net.hizlab.kagetaka.viewer.option.Setter;
import net.hizlab.kagetaka.viewer.option.ViewerOption;
import net.hizlab.kagetaka.viewer.option.InvalidValueException;
import net.hizlab.kagetaka.viewer.theme.Theme;
import net.hizlab.kagetaka.viewer.theme.ThemeListener;
import net.hizlab.kagetaka.viewer.theme.ThemeManager;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Menu;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Hashtable;
import java.util.Vector;

/**
 * 褹뤿ΡΩѤΥɥǤ
 * ΥɥˤϡѤΥ˥塼ʤɤ֤졢
 * Τ٤ƤεǽѤ뤳Ȥޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.14 $
 */
public class HawkViewer extends Frame implements HawkWindow {
    private static final String    RESOURCE          = "net.hizlab.kagetaka.viewer.Resources";
    private static final Dimension MINIMUM_SIZE      = new Dimension(100, 70);
    private static final String    URL_FILE_PROTOCOL = "file";
    private static final String    URL_FILE_SCHEME   = URL_FILE_PROTOCOL + ":/";

    /** ȥ֤ɽǥå */
    public static final int CURRENT_TAB  = -1;
    /** ٤ƤΥ֤ɽǥå */
    public static final int ALL_TAB      = -2;
    /** Ȥ꺸¦ɽǥå */
    public static final int LEFTALL_TAB  = -3;
    /** Ȥ걦¦ɽǥå */
    public static final int RIGHTALL_TAB = -4;
    /** ȰʳΤ٤ƤΥ֤ɽǥå */
    public static final int OTHERALL_TAB = -5;
    /** ֺΥ֤ɽǥå */
    public static final int TOP_TAB      = -6;
    /** ֱΥ֤ɽǥå */
    public static final int LAST_TAB     = -7;
    /** ֤ɽǥå */
    public static final int NEW_TAB      = -8;

    /** ʥӥС */
    public static final int COMPONENT_NAVIBAR    =  1;
    /** С */
    public static final int COMPONENT_SEARCHBAR  =  2;
    /** 󥯥С */
    public static final int COMPONENT_LINKBAR    =  3;
    /** ơС */
    public static final int COMPONENT_STATUSBAR  =  4;

    /** Referer: ƱۥȤΤ */
    public static final int REFERER_SAMEHOST     =   0;
    /** Referer: ʤ */
    public static final int REFERER_NONE         =  -1;
    /** Referer: Ƥ */
    public static final int REFERER_ALL          =  -2;

    /** С: ɽ */
    public static final String SCROLLBAR_YES     = "yes";
    /** С: ư */
    public static final String SCROLLBAR_AUTO    = "auto";
    /** С: ɽ */
    public static final String SCROLLBAR_NO      = "no";

    /** ֥å */
    public static final long TABMODE_LOCK = 1;

    /** @serial ǥեȥɥꥹ */
    private static Vector windowListeners = new Vector();
    /** @serial ɥ */
    private static int    windowCounter;

    /** @serial ɥֹ */
    private int                 windowNumber;
    /** @serial ӥ塼ץ */
    private ViewerOption        option;
    /** @serial ɥޥ͡ */
    private WindowManager       windowManager;
    /** @serial ֥åޡޥ͡ */
    private BookmarkManager     bookmarkManager;
    /** @serial ҥȥޥ͡ */
    private HistoryManager      historyManager;
    /** @serial ơޥޥ͡ */
    private ThemeManager        themeManager;
    /** @serial ƥȥ٥ȥޥ͡ */
    private ContextEventManager contextEventManager;
    /** @serial ơ */
    private Theme               theme;

    /** @serial ե꡼󥤥᡼ */
    private OffscreenBuffer offscreenBuffer;
    /** @serial ˥塼 */
    private ViewerMenu      viewerMenu;
    /** @serial ᥤ쥤ȥޥ͡ */
    private GridBagLayout   gbl;
    /** @serial ᥤޡ */
    private Insets          insets;
    /** @serial ʥӥС */
    private ViewerToolbar   naviBar;
    /** @serial 󥯥С */
    private ViewerToolbar   linkBar;
    /** @serial ᥤѥͥ */
    private TabPanel        mainPanel;
    /** @serial ơС */
    private ViewerStatusBar statusBar;
    /** @serial ƥȥꥹ */
    private ViewerContextListener contextListener;

    /** @serial ɥåɥɥåץꥹ */
    private DnDListener     dndListener;

    /** @serial ᥤ󥿥ȥ */
    private String          titleMain;
    /** @serial ȥղäʸ */
    private String          titleSuffix;
    /** @serial ơʸ */
    private String          statusText;
    /** @serial ơ */
    private StatusBarIcon[] statusIcons;
    /** @serial ɥ */
    private Point           viewerPosition;
    /** @serial ɥ */
    private Dimension       viewerSize;

    /** @serial ȱ륳ƥ */
    private Vector        contexts    = new Vector();
    /** @serial ȱ륳ƥ */
    private Hashtable     contextHash = new Hashtable();
    /** @serial ȱ륳ƥ */
    private ViewerContext context;

    /** @serial ֤Υƥȥ˥塼 */
    private int    tabContextIndex = -1;

    // ɬ¹Ԥ
    static {
        addDefaultWindowListener(new DefaultWindowListener());
    }

    /**
     * Υɥޤ
     *
     * @param  option ץ
     */
    public HawkViewer(ViewerOption option) {
        this.windowNumber        = ++windowCounter;
        this.option              = option;
        this.windowManager       = WindowManager.getInstance();
        this.bookmarkManager     = option.getBookmarkManager();
        this.historyManager      = option.getHistoryManager ();
        this.themeManager        = option.getThemeManager   ();
        this.contextEventManager = new ContextEventManager(windowNumber);
        this.theme               = themeManager.getTheme();

        this.offscreenBuffer     = new OffscreenBuffer(this);
        this.viewerMenu          = new ViewerMenu(this);
        this.mainPanel           = new TabPanel();
        this.contextListener     = new ViewerContextListener();

        setTitleSuffix(getMessage("title", null));
        Image image = Resource.getImageResource(RESOURCE, "icon", getToolkit());
        if (image != null) {
            setIconImage(image);
        }

        gbl    = new GridBagLayout();
        insets = new Insets(0, 0, 0, 0);
        setLayout(gbl);

        LayoutUtils.addGridBag(this, new MenuPanel(), gbl, 0, 0, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
        LayoutUtils.addGridBag(this, mainPanel      , gbl, 0, 3, 1, 1, 1, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);

        // ɲåݡͥȤ
        if (option.getPropertyBoolean(ViewerOption.KEY_SHOW_NAVIBAR  , true)) {
            setVisibleComponent(COMPONENT_NAVIBAR  , true);
        }
        if (option.getPropertyBoolean(ViewerOption.KEY_SHOW_LINKBAR  , true)) {
            setVisibleComponent(COMPONENT_LINKBAR  , true);
        }
        if (option.getPropertyBoolean(ViewerOption.KEY_SHOW_STATUSBAR, true)) {
            setVisibleComponent(COMPONENT_STATUSBAR, true);
        }

        validate();

        // ꥹʡϿ
        addWindowListener(new MainWindowListener());
        for (int i = 0; i < windowListeners.size(); i++) {
            addWindowListener((WindowListener) windowListeners.elementAt(i));
        }
        mainPanel.addTabListener(new MainPanelTabListener());
        MainPanelTabMouseListener tabMouseListener = new MainPanelTabMouseListener();
        mainPanel.addMouseListener   (tabMouseListener);
        mainPanel.addTabMouseListener(tabMouseListener);
        addComponentListener(
            new ComponentAdapter() {
                /** ѹ줿 */
                public void componentResized(ComponentEvent e) {
                    if (!isMaximized()) {
                        viewerSize = getSize();
                    }
                }

                /** ư줿 */
                public void componentMoved(ComponentEvent e) {
                    if (!isMaximized()) {
                        viewerPosition = getLocation();
                    }
                }
            }
        );

        // ֤
        viewerSize = option.getPropertyDimension(ViewerOption.KEY_WINDOW_SIZE);
        setSize(viewerSize);

        Point position = option.getPropertyPoint(ViewerOption.KEY_WINDOW_POSITION);
        if (position != null) {
            setLocation(position);
        } else {
            Dimension size        = getSize();
            Dimension desktopSize = getToolkit().getScreenSize();
            position = new Point((desktopSize.width  - size.width ) / 2,
                                 (desktopSize.height - size.height) / 2);
            // ǥȥåפ
            setLocation(position.x, position.y);
        }
        viewerPosition = position;
        // 粽
        if (Environment.javaVersion >= 104) {
            AWTWrapper awtWrapper = AWTWrapper.getInstance();
            if (awtWrapper != null) {
                int maximized = option.getPropertyInteger(ViewerOption.KEY_WINDOW_MAXIMIZED,
                                                          AWTWrapper.MAXIMIZED_NONE);
                if (maximized != AWTWrapper.MAXIMIZED_NONE) {
                    awtWrapper.setFrameMaximized(this, maximized);
                }
            }
        }

        // ɥåɥɥå
        DnDWrapper dndWrapper = DnDWrapper.getInstance();
        if (dndWrapper != null) {
            dndListener = new DnDListener() {
                /** ɥåפ */
                public boolean isActive() {
                    return true;
                }

                /** ɥåפ줿 */
                public void drop(Object[] objects) {
                    int tabIndex = CURRENT_TAB;
                    for (int i = 0; i < objects.length; i++) {
                        Object object = objects[i];
                        if (object instanceof File) {
                            open(tabIndex, ((File) object).getAbsolutePath());
                        } else if (object instanceof String) {
                            open(tabIndex, (String) object);
                        }
                        // ĤʹߤϿ֤ǳ
                        tabIndex = NEW_TAB;
                    }
                }
            };
            dndWrapper.addDnDListener(this,
                                      dndListener,
                                      new Class[]{File.class,
                                                  String.class});
        }

        // ơ
        themeManager.addListener(
            new ThemeListener() {
                /** ơޤѹ줿 */
                public void themeChanged(Theme theme) {
                    HawkViewer.this.theme = theme;
                    applyTheme();

                    // ġСɤ߹ľ
                    if (naviBar != null) { changeVisibleComponent(COMPONENT_NAVIBAR, false); changeVisibleComponent(COMPONENT_NAVIBAR, true); }
                    if (linkBar != null) { changeVisibleComponent(COMPONENT_LINKBAR, false); changeVisibleComponent(COMPONENT_LINKBAR, true); }

                    // ֤Υɤ߹ľ
                    Image normal  = theme.getImage(Theme.BROWSER_TAB_ICON_NORMAL , getToolkit());
                    Image loading = theme.getImage(Theme.BROWSER_TAB_ICON_LOADING, getToolkit());
                    ViewerContext c;
                    synchronized (contexts) {
                        for (int i = 0; i < contexts.size(); i++) {
                            c = (ViewerContext) contexts.elementAt(i);
                            c.getTab().setLabel(c.canStop() ? loading : normal);
                        }
                    }

                    validate();
                }
            }
        );

        // ơޤŬ
        applyTheme();

        // ֤ĺ
        createTab(TOP_TAB, true);
        if (contexts.size() > 0) {
            context = (ViewerContext) contexts.elementAt(0);
        }
    }

//### Original
    /**
     * Υɥ򳫤ȤɬϿ륦ɥꥹʤɲäޤ
     *
     * @param  listener ɥꥹ
     */
    public static void addDefaultWindowListener(WindowListener listener) {
        windowListeners.addElement(listener);
    }

    /**
     * Υɥ򳫤ȤɬϿ륦ɥꥹʤޤ
     *
     * @param  listener ɥꥹ
     */
    public static void removeDefaultWindowListener(WindowListener listener) {
        windowListeners.removeElement(listener);
    }

    /**
     * ΥɥΥȥղäʸꤷޤ
     * ⤷ɬפʤ <code>null</code> ꤷޤ
     *
     * @param  suffix ȥղäʸ
     */
    public synchronized void setTitleSuffix(String suffix) {
        this.titleSuffix = suffix;
        setTitle(titleMain);
    }

    /**
     * ץ֤ޤ
     *
     * @return ץ
     */
    public ViewerOption getOption() {
        return option;
    }

    /**
     * ꥯȤɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  request ꥯ
     */
    public void open(Request request) {
        switch (request.openMode) {
        case Request.OPEN_NEWWINDOW:
            createWindow().open(CURRENT_TAB, request);
            break;
        case Request.OPEN_NEWTAB:
            open(createTab(CURRENT_TAB, false), request);
            break;
        default:
            open(CURRENT_TAB, request);
            break;
        }
    }

    /**
     * ꥯȤɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  index   ֥ǥå
     * @param  request ꥯ
     */
    public void open(int index, Request request) {
        ViewerContext[] contextList = getContexts(index);
        // ǽΥƥȤǤΤɤ߹
        if (contextList.length > 0) {
            contextList[0].load(request, true);
        }
    }

    /**
     * ˻ꤷ URL ɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  url ɤ߹ URL
     */
    public void open(URL url) {
        open(CURRENT_TAB, url);
    }

    /**
     * ˻ꤷ URL ɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  index ֥ǥå
     * @param  url   ɤ߹ URL
     */
    public void open(int index, URL url) {
        int openMode = Request.OPEN_DEFAULT;
        if (index == NEW_TAB) {
            openMode = Request.OPEN_NEWTAB;
            index    = createTab(CURRENT_TAB, true);
        }
        open(index, new Request(url, null, null, "_top", openMode, Request.CACHE_NORMAL));
    }

    /**
     * ˻ꤷѥɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  path ɤ߹ѥ
     */
    public void open(String path) {
        open(CURRENT_TAB, path);
    }

    /**
     * ˻ꤷѥɤ߹ߡ̤ɽޤ
     * Υ᥽åɤϡ̥åɤǽԤᡢ褬
     * ᥽åɤλޤ
     *
     * @param  index ֥ǥå
     * @param  path  ɤ߹ѥ
     */
    public void open(int index, String path) {
        if (path == null) {
            return;
        }

        URL url = null;
        try {
            url = StringUtils.toURL(path);

            // file:/  Windows ξϥ硼ȥåȥեβ򤹤
            if (url.getProtocol().compareTo(URL_FILE_PROTOCOL) == 0) {
                if (Environment.isWindows) {
                    String filePath = url.toString().substring(URL_FILE_SCHEME.length());
                    // ?  # ͭϡ硼ȥåȤβ򤷤ʤ
                    if (filePath.indexOf('?') == -1 && filePath.indexOf('#') == -1) {
                        String realPath = Shortcut.resolve(filePath);
                        if (realPath != null) {
                            url = StringUtils.toURL(realPath);
                        }
                    }
                }
            }
        } catch (MalformedURLException e) {
            MessageBox.show(HawkViewer.this,
                            getMessage("message.invalidurl.text" , new String[]{path, e.toString()}),
                            getMessage("message.invalidurl.title", null),
                            MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
            return;
        }

        open(index, url);
    }

    /**
     * ץ¸ޤ
     */
    public void saveOption() {
        option.save();
    }

    /**
     * ɽƤ륢ɥ쥹֤ޤ
     *
     * @return ɽƤ륢ɥ쥹
     *         ɽƤʤ <code>null</code>
     */
    public String getAddress() {
        if (naviBar == null) {
            return null;
        }

        return naviBar.getAddress();
    }

    /**
     * ˻ꤷʸ򡢥ơСɽ褦׵ᤷޤ
     * ơСʸõϡ<code>null</code> ꤷޤ
     *
     * @param  status ơСɽʸ
     */
    public void showStatus(String status) {
        if (statusBar != null) {
            statusBar.setText(status);
            statusText = status;
        }
    }

    /**
     * ꤷݡͥȤɽƤ뤫֤ޤ
     *
     * @param  target оݤΥݡͥ
     *
     * @return ɽƤ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean isVisibleComponent(int target) {
        switch (target) {
        case COMPONENT_NAVIBAR  : return (naviBar   != null);
        case COMPONENT_SEARCHBAR: return false;
        case COMPONENT_LINKBAR  : return (linkBar   != null);
        case COMPONENT_STATUSBAR: return (statusBar != null);
        default: // AVOID
        }
        return false;
    }

    /**
     * ꤷݡͥȤɽ걣ꤷޤ
     *
     * @param  target оݤΥݡͥ
     * @param  b ɽ <code>true</code>
     *           ʳξ <code>false</code>
     */
    public void setVisibleComponent(int target, boolean b) {
        if (isVisibleComponent(target) == b) {
            return;
        }

        changeVisibleComponent(target, b);

        validate();
        viewerMenu.changeVisibleComponent(target, b);

        // ¸
        Setter setter = option.getSetter();
        try {
            switch (target) {
            case COMPONENT_NAVIBAR   : setter.putPropertyBoolean(ViewerOption.KEY_SHOW_NAVIBAR  , b); break;
            case COMPONENT_SEARCHBAR : setter.putPropertyBoolean(ViewerOption.KEY_SHOW_SEARCHBAR, b); break;
            case COMPONENT_LINKBAR   : setter.putPropertyBoolean(ViewerOption.KEY_SHOW_LINKBAR  , b); break;
            case COMPONENT_STATUSBAR : setter.putPropertyBoolean(ViewerOption.KEY_SHOW_STATUSBAR, b); break;
            default: // AVOID
            }
            setter.commit();
        } catch (InvalidValueException e) { }
    }

    /**
     * ƥ֤ʥ֤Υǥå֤ޤ
     *
     * @return ֥ǥå
     *         ƥ֤ʥ֤¸ߤʤ <code>-1</code>
     */
    public int getActiveTab() {
        synchronized (contexts) {
            if (context == null) {
                return -1;
            }

            return contexts.indexOf(context);
        }
    }

    /**
     * ꤷ֤򥢥ƥ֤ˤޤ
     *
     * @param  index ֥ǥå
     *
     * @throws IndexOutOfBoundsException ϰϳΥǥåꤷ
     */
    public void setActiveTab(int index) {
        synchronized (contexts) {
            ViewerContext c = (ViewerContext) contexts.elementAt(index);

            if (c == context) {
                return;
            }

            c.getTab().requestActive();
        }
    }

    /**
     * ֤Υ⡼ɤɲäޤ
     *
     * @param  index ֥ǥå
     * @param  mode  ɲä⡼
     */
    public void addTabMode(int index, long mode) {
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            contextList[index].addTabMode(mode);
        }
    }

    /**
     * ֤Υ⡼ɤޤ
     *
     * @param  index ֥ǥå
     * @param  mode  ɲä⡼
     */
    public void removeTabMode(int index, long mode) {
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            contextList[index].removeTabMode(mode);
        }
    }

    /**
     * ֤Υƥȥ˥塼򳫤ǥå֤ޤ
     *
     * @return ǥå
     */
    int getTabContextIndex() {
        return tabContextIndex;
    }

    /**
     * ֥åޡ˥塼ޤ
     *
     * @param  menu ֥åޡۤ˥塼
     * @param  tool ֥åޡΥ˥塼ɲä
     *              <code>true</code>ʤ <code>false</code>
     */
    public void createBookmarkMenu(Menu menu, boolean tool) {
        viewerMenu.createBookmarkMenu(menu, tool);
    }

    /**
     * ֥åޡ˥塼ޤ
     *
     * @param  menu ֥åޡۤ˥塼
     * @param  bookmarks ˥塼ɽ֥åޡ
     */
    public void createBookmarkMenu(Menu menu, Vector bookmarks) {
        viewerMenu.createBookmarkMenu(menu, bookmarks);
    }

//### Override
    /**
     *  ɥɽޤ
     *  ɥǤ˸ƤФ˽Фޤ
     */
    public void show() {
        Dimension size1 = getSize();
        super.show();
        Dimension size2 = getSize();

        if (!size1.equals(size2)) {
            Environment.windowSizeOffset = new Dimension(size2.width  - size1.width ,
                                                         size2.height - size1.height);
        }
    }

    /**
     * ΥɥκǾ֤ޤ
     *
     * @return Ǿ
     */
    public Dimension getMinimumSize() {
        return MINIMUM_SIZE;
    }

    /**
     * ΥɥΥȥ򡢻ꤵ줿ͤꤷޤ
     * ºݤˤϡǻꤷͤθˡ{@link #setTitleSuffix(String)}
     * ˤäꤵ줿ʸղäޤ
     *
     * @param  title ΥɥΥȥ
     */
    public synchronized void setTitle(String title) {
        this.titleMain = title;

        if (titleMain == null && titleSuffix == null) {
            super.setTitle(null);
        } else if (titleMain   == null) {
            super.setTitle(titleSuffix);
        } else if (titleSuffix == null) {
            super.setTitle(titleMain);
        } else {
            super.setTitle(titleMain + " - " + titleSuffix);
        }

        windowManager.changeWindow(this);
    }

    /**
     * ̤򥢥åץǡȤޤ
     *
     * @param  g եå
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ᡼ºݤ褷ޤ
     *
     * @param  g եå
     */
    public void paint(Graphics g) {
//Debug.out.println("h.paint="+g.getClipBounds());
        offscreenBuffer.paint(g);
    }

    /**
     * ɥõޤ
     * ɥѤƤ꥽뤿
     * Υ᥽åɤƤФʤФʤޤ
     */
    public void dispose() {
        //### TODO ٤ƤΥƥȤ˴

        super.dispose();
    }

    /**
     * ΥɥΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        if (titleSuffix != null) {
            str += ",suffix=" + titleSuffix;
        }
        return str;
    }

//### Action
    /**
     * ֿɥ򳫤פ¹Ԥޤ
     *
     * @return ӥ塼
     */
    public HawkViewer createWindow() {
        Point location = getLocationOnScreen();
        location.x += 32;
        location.y += 32;

        Dimension desktopSize = getToolkit().getScreenSize();
        if (/*---*/desktopSize.width  <= location.x
                || desktopSize.height <= location.y) {
            location.x = location.y = 0;
        }

        Dimension size = getSize();
        size.width  -= Environment.windowSizeOffset.width;
        size.height -= Environment.windowSizeOffset.height;

        HawkViewer hawkWindow = new HawkViewer(option);
//        hawkWindow.setSize    (size    );
        hawkWindow.setLocation(location);
        hawkWindow.show();
        return hawkWindow;
    }

    /**
     * ֿ֤򳫤פ¹Ԥޤ
     *
     * @param  index  ֥ǥå
     * @param  active ֤򥢥ƥ֤ˤ <code>true</code>
     *                ʳξ <code>false</code>
     *
     * @return ɲä֤Υǥå
     */
    public int createTab(int index, boolean active) {
        // Сɽˡ
        String showScrollbar = option.getPropertyString(ViewerOption.KEY_SHOW_SCROLLBAR);
        int    scrollbarMode;
        if (SCROLLBAR_AUTO.compareTo(showScrollbar) == 0) {
            scrollbarMode = Value.SCROLLING_AUTO;
        } else if (SCROLLBAR_NO.compareTo(showScrollbar) == 0) {
            scrollbarMode = Value.SCROLLING_NO;
        } else {
            scrollbarMode = Value.SCROLLING_YES;
        }

        ViewerContext c = new ViewerContext(this, scrollbarMode);
        c.addContextInformationListener(contextListener);
        c.addContextRequestListener    (contextListener);
        c.addContextFocusListener      (viewerMenu     );
        c.addContextHistoryListener    (viewerMenu     );
        c.addContextInformationListener(viewerMenu     );
        c.addContextRequestListener    (viewerMenu     );
        c.addContextFocusListener      (naviBar        );
        c.addContextHistoryListener    (naviBar        );
        c.addContextInformationListener(naviBar        );
        c.addContextRequestListener    (naviBar        );
        c.addContextFocusListener      (linkBar        );
        c.addContextHistoryListener    (linkBar        );
        c.addContextInformationListener(linkBar        );
        c.addContextRequestListener    (linkBar        );

        synchronized (contexts) {
            switch (index) {
            case CURRENT_TAB: index = contexts.indexOf(context) + 1; break;
            case TOP_TAB    : index =  0; break;
            case LAST_TAB   : index = -1; break;
            default: // AVOID
            }

            if (index < 0 || contexts.size() < index) {
                index = contexts.size();
            }

            if (index <= tabContextIndex) {
                tabContextIndex++;
            }

            TabPanel.Tab tab = mainPanel.addPanel(c.getTitle(),
                                                  theme.getImage(Theme.BROWSER_TAB_ICON_NORMAL,
                                                                 getToolkit()),
                                                  c.getComponent(),
                                                  active,
                                                  index);
            c.setTab(tab);

            contexts.insertElementAt(c, index);
            contextHash.put(tab, c);

        }

        return index;
    }

    /**
     * Web 򳫤פ¹Ԥޤ
     *
     * @param  index ֥ǥå
     */
    public void openURL(int index) {
        if (!option.getPropertyBoolean(ViewerOption.KEY_WINDOW_OWD_USE, false)) {
            if ((naviBar != null && naviBar.focusAddress())
                    || (linkBar != null && linkBar.focusAddress())) {
                return;
            }
        }

        String     path   = null;
        OpenDialog dialog = new OpenDialog(this);
        dialog.show();

        if (dialog.getResult() == OpenDialog.RESULT_OPEN) {
            path = dialog.getPath();
            if (path != null && path.length() == 0) {
                path = null;
            }
        }
        dialog.dispose();

        if (path != null) {
            open(index, path);
        }
    }

    /**
     * ֥ե򳫤פ¹Ԥޤ
     *
     * @param  index ֥ǥå
     */
    public void openFile(int index) {
        FileDialog fd = new FileDialog(this,
                                       getMessage("dialog.file.open", null),
                                       FileDialog.LOAD);
        fd.show();

        String path = fd.getFile();
        if (path == null) {
            fd.dispose();
            return;
        }

        path = (fd.getDirectory() != null ? fd.getDirectory() : "") + path;
        fd.dispose();

        open(index, URL_FILE_SCHEME + path);
    }

    /**
     * ֥ɥĤפ¹Ԥޤ
     * <p>
     * Υ᥽åɤˤꡢɥõ٤ƤΥ꥽ޤ
     * äơΥ᥽åɤƤǥɥõϡ
     * {@link #dispose()} ƤӽФɬפϤޤ
     */
    public void closeWindow() {
        try {
            Point     position = viewerPosition;
            Dimension size     = viewerSize;

            size.width  -= Environment.windowSizeOffset.width;
            size.height -= Environment.windowSizeOffset.height;
            Setter setter = option.getSetter();
            try {
                setter.putPropertyPoint    (ViewerOption.KEY_WINDOW_POSITION, position);
                setter.putPropertyDimension(ViewerOption.KEY_WINDOW_SIZE    , size    );
                if (Environment.javaVersion >= 104) {
                    AWTWrapper awtWrapper = AWTWrapper.getInstance();
                    if (awtWrapper != null) {
                        setter.putPropertyInteger(ViewerOption.KEY_WINDOW_MAXIMIZED,
                                                  awtWrapper.getFrameMaximized(this));
                    }
                }
                setter.commit();
            } catch (InvalidValueException e) { }
        } finally {
            dispose();
        }
    }

    /**
     * ֥֤Ĥפ¹Ԥޤ
     *
     * @param  index ֥ǥå
     */
    public void closeTab(int index) {
        boolean needCreate = false;

        synchronized (contexts) {
            ViewerContext[] contextList = getContexts(index);
            if (contextList.length == 0) {
                return;
            }

            ViewerContext c;
            TabPanel.Tab  tab;
            int           i;
            int           activeIndex = 0;
            if (context != null) {
                activeIndex = Math.max(contexts.indexOf(context), 0);
            }

            for (index = 0; index < contextList.length; index++) {
                c   = contextList[index];

                if ((c.getTabMode() & TABMODE_LOCK) != 0) {
                    continue;
                }

                tab = c.getTab();

                // Ǥ˺Ѥߤɤå
                if (!contextHash.containsKey(tab)) {
                    continue;
                }

                // ֤
                tab.dispose();

                i = contexts.indexOf(c);

                if (i < tabContextIndex) {
                    tabContextIndex--;
                } else if (i == tabContextIndex) {
                    tabContextIndex = -1;
                }

                if (i <= activeIndex) {
                    activeIndex--;
                }

                // ƥȤ
                c.dispose();
                contexts.removeElementAt(i);
                contextHash.remove(tab);
            }

            // ƥ֤ʥ֤
            if (contexts.size() > 0) {
                setActiveTab(Math.max(activeIndex, 0));
            } else {
                needCreate = true;
            }
        }

        // Ĥ⥿̵֤ʤäȤĺƤ
        if (needCreate) {
            createTab(TOP_TAB, true);
        }
    }

    /**
     * ֽλפ¹Ԥޤ
     */
    public void exitViewer() {
        windowManager.exit();
    }

    /**
     * ߡפ¹Ԥޤ
     *
     * @param  index ֥ǥå
     */
    public void stop(int index) {
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            contextList[index].stop();
        }
    }

    /**
     * ֺɤ߹ߡפ¹Ԥޤ
     *
     * @param  index ֥ǥå
     * @param  force Ū˺ɹ <code>true</code>
     *               ʳξ <code>false</code>
     */
    public void reload(int index, boolean force) {
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            contextList[index].reload(force);
        }
    }

    /**
     * ֥ڡΥפ¹Ԥޤ
     *
     * @param  index ֥ǥå
     */
    public void showSource(int index) {
        Request request;
        Content content;
        String  path;

        File        appPath   = option.getPropertyFile  (ViewerOption.KEY_TOOLS_SV_PATH  );
        String      appOption = option.getPropertyString(ViewerOption.KEY_TOOLS_SV_OPTION);
        IOException exception = null;
        String      command   = null;

        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            if ((request = contextList[index].getRequest()) != null
                    && (content = request.getDocument().content) instanceof ViewerContent
                    && (path = ((ViewerContent) content).getCachePath()) != null) {

                if (appOption == null || appOption.length() == 0) {
                    appOption = path;
                } else {
                    int p = appOption.indexOf("%s");
                    if (p != -1) {
                        appOption = appOption.substring(0, p) + path + appOption.substring(p + 2);
                    }
                }

                command = appPath.getPath() + " " + appOption;

                try {
                    Runtime.getRuntime().exec(command);
                } catch (IOException e) {
                    exception = e;
                    break;
                }
            }
        }

        if (exception != null) {
            MessageBox.show(HawkViewer.this,
                            getMessage("message.cannotviewsource.text" , new String[]{command, exception.getMessage()}),
                            getMessage("message.cannotviewsource.title", null),
                            MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
        }
    }

    /**
     * ưޤ
     *
     * @param  index ֥ǥå
     * @param  num ư̤ꤷޤ
     *             οʤ鼡ءοʤذưޤ
     */
    public void moveHistory(int index, int num) {
        boolean forward = (num > 0);
        num = Math.abs(num);

        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            if (forward) {
                contextList[index].forward(num);
            } else {
                contextList[index].back   (num);
            }
        }
    }

    /**
     * ۡɽޤ
     *
     * @param  index ֥ǥå
     */
    public void moveHome(int index) {
        String home = option.getPropertyString(ViewerOption.KEY_URL_HOME);
        if (home == null || home.length() == 0) {
            home = About.BLANK;
        }
        open(index, home);
    }

    /**
     * 򥨥ǥɽޤ
     */
    public void showHistoryEditor() {
        historyManager.showEditor();
    }

    /**
     * ɽΥڡ֥åޡɲäޤ
     *
     * @param  index ֥ǥå
     */
    public void addBookmark(int index) {
        Request request;
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            if ((request = contextList[index].getRequest()) == null) {
                continue;
            }

            Bookmark bookmark = new Bookmark(request.getDocument().getTitle(), request.url);
            Bookmark parent   = bookmarkManager.getRootBookmark();
            parent.addBookmark(bookmark);

            // ɥ
            windowManager.addToBookmark(parent, bookmark);
        }

        bookmarkManager.save();
    }

    /**
     * ɽΥڡ֥åޡ˾ܺ٤ꤷɲäޤ
     *
     * @param  index ֥ǥå
     */
    public void addBookmarkAs(int index) {
        Request request;
        ViewerContext[] contextList = getContexts(index);
        for (index = 0; index < contextList.length; index++) {
            if ((request = contextList[index].getRequest()) == null) {
                continue;
            }

            BookmarkAs.Entry[] entries = BookmarkAs.show(this, option,
                                                         request.getDocument().getTitle(),
                                                         request.url.toString());

            if (entries == null) {
                continue;
            }

            // ɥ
            synchronized (windowManager) {
                for (int i = 0; i < entries.length; i++) {
                    entries[i].parent.addBookmark(entries[i].bookmark);
                    windowManager.addToBookmark(entries[i].parent, entries[i].bookmark);
                }
            }
        }

        bookmarkManager.save();
    }

    /**
     * ֥åޡǥ򳫤ޤ
     */
    public void showBookmarkEditor() {
        bookmarkManager.showEditor();
    }

    /**
     * ץޥ͡ɽޤ
     */
    public void showOptionManager() {
        option.showManager(this);
    }

    /**
     * About ɽޤ
     */
    public void showAbout() {
        createWindow().open(CURRENT_TAB, About.ABOUT);
    }

//### HawkWindow
    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#addWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void addWindowMenu(HawkWindow window) {
        viewerMenu.addWindowMenu(window);
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#removeWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void removeWindowMenu(int index) {
        viewerMenu.removeWindowMenu(index);
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#changeWindow(HawkWindow)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void changeWindowMenu(int index, HawkWindow window) {
        viewerMenu.changeWindowMenu(index, window);
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#addToBookmark(Bookmark, Bookmark)}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void addToBookmark(Bookmark parent, Bookmark bookmark) {
        viewerMenu.addToBookmark(parent, bookmark);
        if (naviBar != null) { naviBar.addToBookmark(parent, bookmark); }
        if (linkBar != null) { linkBar.addToBookmark(parent, bookmark); }
    }

    /**
     * Υ᥽åɤľܸƤӽФƤϹԤޤ
     * ˡ{@link WindowManager#bookmarkChanged()}
     * ƤӽФɬפޤ
     *
     * {@inheritDoc}
     */
    public void bookmarkChanged() {
        viewerMenu.bookmarkChanged();
        if (naviBar != null) { naviBar.bookmarkChanged(); }
        if (linkBar != null) { linkBar.bookmarkChanged(); }
    }

//### package
    /**
     * ɥåɥɥåץꥹʤ֤ޤ
     *
     * @return ɥåɥɥåץꥹ
     */
    DnDListener getDnDListener() {
        return dndListener;
    }

    /**
     * ƥȥ٥ȥޥ֤͡ޤ
     *
     * @return ƥȥ٥ȥޥ͡
     */
    ContextEventManager getContextEventManager() {
        return contextEventManager;
    }

    /**
     * ơޤ֤ޤ
     *
     * @return ơ
     */
    Theme getTheme() {
        return theme;
    }

//### private
    /** ꥽ʸ */
    private String getMessage(String key, String[] args) {
        return Resource.getMessage(RESOURCE, key, args);
    }

    /** ꤵ줿֤ΰ */
    private ViewerContext[] getContexts(int index) {
        Vector v = new Vector();

        try {
            if (index >= 0) {
                v.addElement(contexts.elementAt(index));
            } else {
                switch (index) {
                case CURRENT_TAB:
                    v.addElement(context);
                    break;
                case ALL_TAB:
                    for (int i = 0; i < contexts.size(); i++) {
                        v.addElement(contexts.elementAt(i));
                    }
                    break;
                case LEFTALL_TAB:
                    for (int i = 0; i < contexts.size(); i++) {
                        if (contexts.elementAt(i) == context) {
                            break;
                        }
                        v.addElement(contexts.elementAt(i));
                    }
                    break;
                case RIGHTALL_TAB:
                {
                    int current = 0;
                    while (/**/current < contexts.size()
                            && contexts.elementAt(current++) != context) {
                        // AVOID
                    }
                    for (int i = current; i < contexts.size(); i++) {
                        v.addElement(contexts.elementAt(i));
                    }

                    break;
                }
                case OTHERALL_TAB:
                    for (int i = 0; i < contexts.size(); i++) {
                        if (contexts.elementAt(i) != context) {
                            v.addElement(contexts.elementAt(i));
                        }
                    }
                    break;
                case TOP_TAB:
                    v.addElement(contexts.elementAt(0));
                    break;
                case LAST_TAB:
                    v.addElement(contexts.elementAt(contexts.size() - 1));
                    break;
                default: // AVOID
                }
            }
        } catch (ArrayIndexOutOfBoundsException e) { }

        ViewerContext[] cs = new ViewerContext[v.size()];
        v.copyInto(cs);

        return cs;
    }

    /** ƥݡͥȤɽڤؤ */
    private void changeVisibleComponent(int target, boolean b) {
        if (b) {
            // ɽ
            switch (target) {
            case COMPONENT_NAVIBAR  :
                this.naviBar = new ViewerToolbar(this, ViewerToolbar.TYPE_NAVI);
                synchronized (contexts) {
                    ViewerContext c;
                    for (int i = 0; i < contexts.size(); i++) {
                        c = (ViewerContext) contexts.elementAt(i);
                        c.addContextFocusListener      (naviBar);
                        c.addContextHistoryListener    (naviBar);
                        c.addContextInformationListener(naviBar);
                        c.addContextRequestListener    (naviBar);
                    }
                    if (context != null) {
                        context.setActived(false);
                        context.setActived(true );
                    }
                }
                LayoutUtils.addGridBag(this, naviBar, gbl, 0, 1, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
                break;
            case COMPONENT_SEARCHBAR:
                return;
            case COMPONENT_LINKBAR  :
                this.linkBar = new ViewerToolbar(this, ViewerToolbar.TYPE_LINK);
                synchronized (contexts) {
                    ViewerContext c;
                    for (int i = 0; i < contexts.size(); i++) {
                        c = (ViewerContext) contexts.elementAt(i);
                        c.addContextFocusListener      (linkBar);
                        c.addContextHistoryListener    (linkBar);
                        c.addContextInformationListener(linkBar);
                        c.addContextRequestListener    (linkBar);
                    }
                    if (context != null) {
                        context.setActived(false);
                        context.setActived(true );
                    }
                }
                LayoutUtils.addGridBag(this, linkBar, gbl, 0, 2, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
                break;
            case COMPONENT_STATUSBAR:
                this.statusBar = new ViewerStatusBar(theme);
                LayoutUtils.addGridBag(this, statusBar , gbl, 0, 4, 1, 1, 1, 0, GridBagConstraints.BOTH, GridBagConstraints.CENTER, insets);
                if (statusText  != null) {
                    statusBar.setText (statusText );
                }
                if (statusIcons != null) {
                    statusBar.setIcons(statusIcons);
                }
                break;
            default: // AVOID
            }
        } else {
            // ɽ
            switch (target) {
            case COMPONENT_NAVIBAR   :
                synchronized (contexts) {
                    ViewerContext c;
                    for (int i = 0; i < contexts.size(); i++) {
                        c = (ViewerContext) contexts.elementAt(i);
                        c.removeContextFocusListener      (naviBar);
                        c.removeContextHistoryListener    (naviBar);
                        c.removeContextInformationListener(naviBar);
                        c.removeContextRequestListener    (naviBar);
                    }
                }
                remove(naviBar);
                naviBar = null;
                break;
            case COMPONENT_SEARCHBAR :
                return;
            case COMPONENT_LINKBAR   :
                synchronized (contexts) {
                    ViewerContext c;
                    for (int i = 0; i < contexts.size(); i++) {
                        c = (ViewerContext) contexts.elementAt(i);
                        c.removeContextFocusListener      (linkBar);
                        c.removeContextHistoryListener    (linkBar);
                        c.removeContextInformationListener(linkBar);
                        c.removeContextRequestListener    (linkBar);
                    }
                }
                remove(linkBar);
                linkBar = null;
                break;
            case COMPONENT_STATUSBAR :
                remove(statusBar);
                statusBar = null;
                break;
            default: // AVOID
            }
        }
    }

    /** ƥݡͥȤ˥ơޤŬ */
    private void applyTheme() {
        Color c;
        Font  f;
        ImageButton b1, b2;
        Toolkit tk = getToolkit();

        // ֥С
        mainPanel.setForeground             ((c = theme.getColor(Theme.BROWSER_TAB_FOREGROUND         )) != null ? c : SystemColor.controlText );
        mainPanel.setBackground             ((c = theme.getColor(Theme.BROWSER_TAB_BACKGROUND         )) != null ? c : SystemColor.control     );
        mainPanel.setFocusFrameColor        ((c = theme.getColor(Theme.BROWSER_TAB_FOCUS              )) != null ? c : SystemColor.windowBorder);
        mainPanel.setActiveTitleFont        ((f = theme.getFont (Theme.BROWSER_TAB_ACTIVE_FONT        )) != null ? f : getFont()               );
        mainPanel.setActiveTitleForeground  ((c = theme.getColor(Theme.BROWSER_TAB_ACTIVE_FOREGROUND  )) != null ? c : SystemColor.controlText );
        mainPanel.setActiveTitleBackground  ((c = theme.getColor(Theme.BROWSER_TAB_ACTIVE_BACKGROUND  )) != null ? c : SystemColor.control     );
        mainPanel.setUnActiveTitleFont      ((f = theme.getFont (Theme.BROWSER_TAB_UNACTIVE_FONT      )) != null ? f : getFont()               );
        mainPanel.setUnActiveTitleForeground((c = theme.getColor(Theme.BROWSER_TAB_UNACTIVE_FOREGROUND)) != null ? c : SystemColor.controlText );
        mainPanel.setUnActiveTitleBackground((c = theme.getColor(Theme.BROWSER_TAB_UNACTIVE_BACKGROUND)) != null ? c : SystemColor.control     );
        if (/*---*/(b1 = theme.getImageButton(Theme.BROWSER_TAB_SCROLLBAR_LEFT , tk)) != null
                && (b2 = theme.getImageButton(Theme.BROWSER_TAB_SCROLLBAR_RIGHT, tk)) != null) {
            mainPanel.setTabScrollButton(b1, b2);
        } else {
            mainPanel.resetTabScrollButton();
        }

        // ơС
        if (statusBar != null) {
            statusBar.setTheme(theme);
        }
    }

    /** ˻ꤷ򡢥ơСɽ */
    private void setStatusIcons(StatusBarIcon[] icons) {
        if (statusBar != null) {
            statusBar.setIcons(icons);
            statusIcons = icons;
        }
    }

    /** 粽Ƥ뤫ɤ֤ */
    private boolean isMaximized() {
        // 1.4 ʾξ API Ȥ
        if (Environment.javaVersion >= 104) {
            AWTWrapper awtWrapper = AWTWrapper.getInstance();
            if (awtWrapper != null) {
                return (awtWrapper.getFrameMaximized(HawkViewer.this) != AWTWrapper.MAXIMIZED_NONE);
            }
        }
        // ʳξϡ粽 location ͤˤʤΤԤWindows Τߡ
        Point position = getLocation();
        return (position.x < 0 && position.y < 0);
    }

//### MainWindowListener
    /** ɥϥɥ */
    private final class MainWindowListener extends WindowAdapter {
        /** 󥹥󥹤 */
        private MainWindowListener() {
        }

        /** ɥƥ֤ˤʤäȤ */
        public void windowActivated(WindowEvent e) {
            windowManager.changeActiveViewer(HawkViewer.this);
        }

        /** ɥ줿Ȥ */
        public void windowOpened(WindowEvent e) {
            windowManager.addWindow(HawkViewer.this);
        }

        /** ɥĤ褦ȤȤ */
        public void windowClosing(WindowEvent e) {
            closeWindow();
        }

        /** ɥĤ줿Ȥ */
        public void windowClosed(WindowEvent e) {
            windowManager.removeWindow(HawkViewer.this);
        }
    }

//### ContextListener
    /** ƥȥꥹ */
    private final class ViewerContextListener
            implements ContextInformationListener, ContextRequestListener {
        /** 󥹥󥹤 */
        private ViewerContextListener() {
        }

        /** ɥ쥹ѹ줿 */
        public void addressChanged(ContextInformationEvent e) {
        }

        /** ȥ뤬ѹ줿 */
        public void titleChanged(ContextInformationEvent e) {
            e.getContext().getTab().setLabel((String) e.getData());
            synchronized (contexts) {
                if (context == e.getContext()) {
                    setTitle((String) e.getData());
                }
            }
        }

        /** ơѹ줿 */
        public void statusChanged(ContextInformationEvent e) {
            synchronized (contexts) {
                if (context == e.getContext()) {
                    Object data = e.getData();
                    if (data instanceof String) {
                        showStatus((String) data);
                    } else if (data != null) {
                        setStatusIcons((StatusBarIcon[]) data);
                    }
                }
            }
        }

        /** ꥯȤդ */
        public void requestAccepted(ContextRequestEvent e) {
            e.getContext().getTab().setLabel(theme.getImage(Theme.BROWSER_TAB_ICON_LOADING, getToolkit()));
        }

        /** ꥯȤɤ߹ߤϤ줿 */
        public void requestLoading(ContextRequestEvent e) {
        }

        /** ꥯȤλ */
        public void requestDone(ContextRequestEvent e) {
            e.getContext().getTab().setLabel(theme.getImage(Theme.BROWSER_TAB_ICON_NORMAL, getToolkit()));
        }
    }

//### MainPanelTabListener
    /** ֥ꥹ */
    private final class MainPanelTabListener implements TabListener {
        /** 󥹥󥹤 */
        private MainPanelTabListener() {
        }

        /** ƥ֤ʥ֤ѹ줿 */
        public void activeChanged(TabEvent e) {
            synchronized (contexts) {
                if (context != null) {
                    context.setActived(false);
                }

                context = (ViewerContext) contextHash.get(e.getTab());
                context.setActived(true);
            }
        }

        /** ֤ɲä줿 */
        public void tabAdded(TabEvent e) {
        }

        /** ֤줿 */
        public void tabRemoved(TabEvent e) {
        }

        /** ֤ѹ줿 */
        public void tabChanged(TabEvent e) {
        }
    }

//### MainPanelTabMouseListener
    /** ֤Υޥꥹ */
    private final class MainPanelTabMouseListener extends MouseAdapter {
        /** 󥹥󥹤 */
        private MainPanelTabMouseListener() {
        }

        /** å줿 */
        public void mouseClicked(MouseEvent e) {
            // å
            if (/*---*/e.isPopupTrigger()
                    || ((e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK)) {

                synchronized (contexts) {
                    viewerMenu.showTabPopupMenu(e.getComponent(), e.getX(), e.getY());
                }
            }

            // 楯å
            if ((e.getModifiers() & InputEvent.BUTTON2_MASK) == InputEvent.BUTTON2_MASK) {
                synchronized (contexts) {
                    if (!(e.getComponent() instanceof TabPanel.Tab)) {
                        return;
                    }

                    ViewerContext c = (ViewerContext) contextHash.get((TabPanel.Tab) e.getComponent());
                    if (c == null) {
                        return;
                    }
                    int index = contexts.indexOf(c);
                    if (index == -1) {
                        return;
                    }
                    closeTab(index);
                }
                e.consume();
            }
        }
    }

//### MenuPanel
    /** ˥塼ѥͥ */
    private final class MenuPanel extends Component {
        /** 󥹥󥹤 */
        private MenuPanel() {
        }

        /**  */
        public void update(Graphics g) {
            paint(g);
        }

        /**  */
        public void paint(Graphics g) {
            Dimension size = getSize();

            g.setColor(ColorConverter.getDarker(SystemColor.menu));
            g.drawLine(0, size.height - 1, size.width - 1, size.height - 1);
        }

        /** 侩֤ */
        public Dimension getPreferredSize() {
            return new Dimension(0, 1);
        }

        /** Ǿ֤ */
        public Dimension getMinimumSize() {
            return new Dimension(1, 1);
        }
    }

//### DefaultWindowListener
    /** ɥϥɥ */
    private static final class DefaultWindowListener extends WindowAdapter {
        int count = 0;

        /** 󥹥󥹤 */
        private DefaultWindowListener() {
        }

        /** ɥ줿Ȥ */
        public synchronized void windowOpened(WindowEvent e) {
            count++;
        }

        /** ɥĤ줿Ȥ */
        public synchronized void windowClosed(WindowEvent e) {
            if (--count == 0) {
                CacheManager.dispose();
            }
        }
    }
}
