/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.rendering.FrameItem;
import net.hizlab.kagetaka.token.Value;

import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager2;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.Serializable;
import java.util.Vector;

/**
 * ե졼ॻåѤΥ쥤ȥޥ͡Ǥ
 * <p>
 * Υ쥤ȥޥ͡ϡƥʤ˥ݡͥȤɲäˡ
 * {@link Container#setLayout(LayoutManager)} Ѥ
 * ϿƤɬפޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
class FramesetLayout implements LayoutManager2, Serializable {
    private static final Cursor DEFAULT_CURSOR    = Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR );
    private static final Cursor VERTICAL_CURSOR   = Cursor.getPredefinedCursor(Cursor.S_RESIZE_CURSOR);
    private static final Cursor HORIZONTAL_CURSOR = Cursor.getPredefinedCursor(Cursor.E_RESIZE_CURSOR);

    private static final int RESIZE_NONE       = 0;
    private static final int RESIZE_VERTICAL   = 1;
    private static final int RESIZE_HORIZONTAL = 2;

    private static final int MIN_SIZE = 8;

    /** @serial ӥ塼ѥͥ */
    private ViewerPanel viewerPanel;
    /** @serial ե졼ॻåȤΥ롼ȥƥ */
    private FrameItem rootItem;
    /** @serial ݡͥ */
    private Vector components = new Vector();

    /** @serial ե졼Υ饤 */
    private Vector frameLines = new Vector();
    /** @serial ꥵΤᳫϰ */
    private Drag   resizeStart;

    /** @serial 롼ѥޥꥹ */
    private MouseListener       mouseListener;
    /** @serial 롼ѥޥ⡼ꥹ */
    private MouseMotionListener mouseMotionListener;

    /**
     * ե졼ॻåȤ֤
     * 쥤ȥޥ͡Υ󥹥󥹤ޤ
     *
     * @param  viewerPanel ӥ塼ѥͥ
     * @param  rootItem    ե졼ॢƥ
     */
    public FramesetLayout(ViewerPanel viewerPanel, FrameItem rootItem) {
        this.viewerPanel = viewerPanel;
        this.rootItem    = rootItem;

        // ե졼ȤΥɥåݡ
        viewerPanel.addMouseListener(
            this.mouseListener = new MouseAdapter() {
                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    // ̵ʥܥξ
                    if ((e.getModifiers()
                            & (InputEvent.BUTTON2_MASK
                             | InputEvent.BUTTON3_MASK)) != 0) {
                        resetMouseCursor();
                        resizeStart = null;
                        return;
                    }

                    // ͭʥܥξ
                    Line line = setupMouseCursor(e.getX(), e.getY());
                    if (line != null) {
                        resizeStart = new Drag(line, e.getX(), e.getY());
                    }
                }

                /** ޥ줿 */
                public void mouseReleased(MouseEvent e) {
                    Drag start = resizeStart;
                    if (start != null) {
                        resizeStart = null;

                        // ܥΥ줿ϰư
                        if ((e.getModifiers() & InputEvent.BUTTON1_MASK) != 0
                                && moveTo(start, e.getX(), e.getY())) {
                            return;
                        }

                        resetMouseCursor();
                    }
                }

                /** ޥä */
                public void mouseEntered(MouseEvent e) {
                    mouseMotionListener.mouseMoved(e);
                }

                /** ޥФ */
                public void mouseExited(MouseEvent e) {
                    if (resizeStart == null) {
                        resetMouseCursor();
                    }
                }
            }
        );

        viewerPanel.addMouseMotionListener(
            this.mouseMotionListener = new MouseMotionListener() {
                /** ޥɥå줿 */
                public void mouseDragged(MouseEvent e) {
                    //### TODO ɥå
                }

                /** ޥư줿 */
                public void mouseMoved(MouseEvent e) {
                    // ˥ޥܥ󤬲Ƥ̵
                    if ((e.getModifiers()
                             & (InputEvent.BUTTON1_MASK
                              | InputEvent.BUTTON2_MASK
                              | InputEvent.BUTTON3_MASK)) != 0) {
                        return;
                    }

                    setupMouseCursor(e.getX(), e.getY());
                }
            }
        );
    }

    /**
     * ꤵ줿󥪥֥ȤѤơ
     * ꤵ줿ݡͥȤ쥤Ȥɲäޤ
     *
     * @param  comp ɲä륳ݡͥ
     * @param  constraints 쥤ȤؤΥݡͥȤɲð֤ɲˡ
     */
    public void addLayoutComponent(Component comp, Object constraints) {
        synchronized (comp.getTreeLock()) {
            components.addElement(comp);
        }
    }

    /**
     * @deprecated  {@link #addLayoutComponent(Component, Object)}
     *             ѤƤ
     *
     * @param  name 
     * @param  comp ݡͥ
     */
    public void addLayoutComponent(String name, Component comp) {
        synchronized (comp.getTreeLock()) {
            addLayoutComponent(comp, null);
        }
    }

    /**
     * ꤵ줿ݡͥȤ쥤Ȥޤ
     *
     * @param  comp ݡͥ
     */
    public void removeLayoutComponent(Component comp) {
        synchronized (comp.getTreeLock()) {
            components.removeElement(comp);
        }
    }

    /**
     * ꤵ줿ѥͥο侩򻻽Фޤ
     *
     * @param  parent ƥƥ
     *
     * @return ΥѥͥΥ
     */
    public Dimension preferredLayoutSize(Container parent) {
        synchronized (parent.getTreeLock()) {
            Insets insets = parent.getInsets();
            Dimension size = new Dimension(0, 0);

            if (components.size() > 0) {
                preferredFrameset(0, rootItem.getItems(), size, true);
            }

            return new Dimension(insets.left + insets.right + size.width ,
                                 insets.top + insets.bottom + size.height);
        }
    }

    /**
     * ꤵ줿ѥͥκǾ򻻽Фޤ
     *
     * @param  parent ƥƥ
     *
     * @return ΥѥͥΥ
     */
    public Dimension minimumLayoutSize(Container parent) {
        synchronized (parent.getTreeLock()) {
            Insets insets = parent.getInsets();
            Dimension size = new Dimension(0, 0);

            if (components.size() > 0) {
                preferredFrameset(0, rootItem.getItems(), size, false);
            }

            return new Dimension(insets.left + insets.right + size.width ,
                                 insets.top + insets.bottom + size.height);
        }
    }

    /**
     * ꤵ줿åȥƥ˥ݡͥȤ쥤ȤȤ
     * 祵֤ޤ
     *
     * @param  target ֤ɬפʥݡͥ
     *
     * @return 祵
     */
    public Dimension maximumLayoutSize(Container target) {
        return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
    }

    /**
     * x ˡ֤ޤ
     *
     * @param  parent ƥƥ
     *
     * @return x ˡ
     */
    public float getLayoutAlignmentX(Container parent) {
        return 0.5f;
    }

    /**
     * y ˡ֤ޤ
     *
     * @param  parent ƥƥ
     *
     * @return y ˡ
     */
    public float getLayoutAlignmentY(Container parent) {
        return 0.5f;
    }

    /**
     * 쥤Ȥ̵ˤޤ
     *
     * @param  target ֤Ƥ륳ݡͥ
     */
    public void invalidateLayout(Container target) {
    }

    /**
     * ꤵ줿ѥͥΥ쥤ȤԤޤ
     *
     * @param  parent ƥƥ
     */
    public void layoutContainer(Container parent) {
        synchronized (parent.getTreeLock()) {
            if (components.size() == 0) {
                return;
            }

            Insets    insets = parent.getInsets();
            Dimension size   = parent.getSize();
            size.width  -= (insets.left + insets.right);
            size.height -= (insets.top + insets.bottom);

            Vector lines = new Vector();

            layoutFrameset(0, rootItem.getItems(),
                           insets.top, insets.left, size.width, size.height,
                           lines);

            frameLines = lines;
        }
    }

    /**
     * ΥݡͥȤʸɽ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        return super.toString();
    }

//### package
    /**
     * ˴ޤ
     */
    void dispose() {
        viewerPanel.removeMouseListener      (mouseListener      );
        viewerPanel.removeMouseMotionListener(mouseMotionListener);
    }

//### private
    /** ե졼ॻåȤ */
    private int preferredFrameset(int index, FrameItem[][] items,
                                  Dimension psize, boolean preferred) {
        FrameItem[][] children;
        Dimension size;

        for (int c = 0; c < items.length; c++) {
            Dimension rsize = new Dimension(0, 0);

            for (int r = 0; r < items[c].length; r++) {
                if ((children = items[c][r].getItems()) != null) {
                    size  = new Dimension(0, 0);
                    index = preferredFrameset(index, children, size, preferred);
                } else {
                    if (index < components.size()) {
                        if (preferred) {
                            size = ((Component) components.elementAt(index)).getPreferredSize();
                        } else {
                            size = ((Component) components.elementAt(index)).getMinimumSize();
                        }
                    } else {
                        size = new Dimension(0, 0);
                    }
                    index++;
                }
                rsize.width  = Math.max(size.width, rsize.width);
                rsize.height += size.height;
            }

            psize.width  += rsize.width;
            psize.height = Math.max(rsize.height, psize.height);
        }

        return index;
    }

    /** ե졼ॻåȤ */
    private int layoutFrameset(int index, FrameItem[][] items,
                               int x, int y, int width, int height,
                               Vector lines) {
        FrameItem[][] children;
//Debug.out.println("frameset[" + index + "]=" + x + "," + y + "," + width + "," + height);

        // ׻
        int[] widths  = calculateSize(items, true , width );
        int[] heights = calculateSize(items, false, height);
        int lx = x + width, ly;
        boolean b;

        // Ѥη
        int fixRight, fixLeft, fixTop, fixBottom;
        boolean[] vs = new boolean[widths .length + 1];
        boolean[] hs = new boolean[heights.length + 1];
        CHECK:
        for (int c = widths.length - 1; c >= 0; c--) {
            for (int r = heights.length - 1; r >= 0 ; r--) {
                if (items[c][r].getBorder()) {
                    vs[c] = vs[c + 1] = true;
                    continue CHECK;
                }
            }
        }
        CHECK:
        for (int r = heights.length - 1; r >= 0; r--) {
            for (int c = widths.length - 1; c >= 0 ; c--) {
                if (items[c][r].getBorder()) {
                    hs[r] = hs[r + 1] = true;
                    continue CHECK;
                }
            }
        }
        hs[0] = hs[hs.length - 1] = vs[0] = vs[vs.length - 1] = false;

        for (int c = 0; c < items.length; c++) {
            lx -= widths[c];
            ly =  y;
            for (int r = 0; r < items[c].length; r++) {
                if ((children = items[c][r].getItems()) != null) {
                    index = layoutFrameset(index, children, lx, ly, widths[c], heights[r], lines);
                } else {
                    if (index < components.size()) {
//Debug.out.println("frame[" + index + "]=" + lx + "," + ly + "," + widths[c] + "," + heights[r]);
                        b = items[c][r].getBorder();
                        fixRight  = (b ? 2 : 0) + (vs[c    ] ? 1 : 0);
                        fixLeft   = (b ? 2 : 0) + (vs[c + 1] ? 1 : 0);
                        fixTop    = (b ? 2 : 0) + (hs[r    ] ? 1 : 0);
                        fixBottom = (b ? 2 : 0) + (hs[r + 1] ? 1 : 0);

                        // 饤¸
                        if (fixRight > 0 && vs[c]) {
                            lines.addElement(new Line(RESIZE_VERTICAL,
                                                      lx + widths[c] - fixRight,
                                                      ly,
                                                      fixRight,
                                                      heights[r],
                                                      items [c - 1][0],
                                                      items [c    ][0],
                                                      widths[c - 1],
                                                      widths[c    ]));
                        }
                        if (fixLeft > 0 && vs[c + 1]) {
                            lines.addElement(new Line(RESIZE_VERTICAL,
                                                      lx,
                                                      ly,
                                                      fixLeft,
                                                      heights[r],
                                                      items [c    ][0],
                                                      items [c + 1][0],
                                                      widths[c    ],
                                                      widths[c + 1]));
                        }
                        if (fixTop > 0 && hs[r]) {
                            lines.addElement(new Line(RESIZE_HORIZONTAL,
                                                      lx,
                                                      ly,
                                                      widths[c],
                                                      fixTop,
                                                      items[0][r - 1],
                                                      items[0][r    ],
                                                      heights [r - 1],
                                                      heights [r    ]));
                        }
                        if (fixBottom > 0 && hs[r + 1]) {
                            lines.addElement(new Line(RESIZE_HORIZONTAL,
                                                      lx,
                                                      ly + heights[r] - fixBottom,
                                                      widths[c],
                                                      fixBottom,
                                                      items[0][r    ],
                                                      items[0][r + 1],
                                                      heights [r    ],
                                                      heights [r + 1]));
                        }

                        // ݡͥȤΰ֤
                        ((Component) components.elementAt(index)).setBounds(lx + fixLeft,
                                                                            ly + fixTop,
                                                                            widths [c] - fixLeft - fixRight,
                                                                            heights[r] - fixTop - fixBottom);
                    }
                    index++;
                }
                ly += heights[r];
            }
        }

        return index;
    }

    /** ׻ */
    private int[] calculateSize(FrameItem[][] items, boolean row, int size) {
        int[] fixed  = new int[(row ? items.length : items[0].length)];
        int fixedSum = 0;
        int asteSum  = 0;
        int odd, n;
        Value v;

        // ׻
        for (int i = 0; i < fixed.length; i++) {
            v = (row ? items[i][0].getHeight()
                     : items[0][i].getWidth ());
            switch (v.getUnit()) {
            case Value.UNIT_PERCENT:
                fixedSum += fixed[i] = Math.max(v.getValue(1, size, null, Value.DATA_NONE), MIN_SIZE);
                break;
            case Value.UNIT_ASTERISK:
                asteSum  += Math.max(v.intValue(), 1);
                break;
            default:
                fixedSum += (fixed[i] = Math.max(v.intValue(), MIN_SIZE));
            }
        }

        odd = size - fixedSum;

        // * ʬ
        if (asteSum > 0 && odd > 0) {
            for (int i = 0; i < fixed.length; i++) {
                v = (row ? items[i][0].getHeight()
                         : items[0][i].getWidth ());
                if (v.getUnit() == Value.UNIT_ASTERISK) {
                    n        =  v.intValue();
                    fixed[i] =  Math.max((odd * n) / asteSum, MIN_SIZE);
                    asteSum  -= n;
                    odd      -= fixed[i];
                }
            }
        } else
        // ;ʬ or ̤
        if (odd != 0) {
            for (int i = 0; i < fixed.length; i++) {
                n        =  (odd * fixed[i]) / fixedSum;
                odd      -= n;
                fixedSum -= fixed[i];
                fixed[i] += n;
                if (odd == 0) {
                    break;
                }
            }
        }

        return fixed;
    }

    /** ޥѹ */
    private Line setupMouseCursor(int x, int y) {
        Line line;

        for (int i = frameLines.size() - 1; i >= 0; i--) {
            if ((line = (Line) frameLines.elementAt(i)).contains(x, y)) {
                viewerPanel.setCursor(line.sense == RESIZE_VERTICAL
                                      ? HORIZONTAL_CURSOR
                                      : VERTICAL_CURSOR);
                return line;
            }
        }

        // ʳ
        resetMouseCursor();
        return null;
    }

    /** ޥξ֤򸵤᤹ */
    private void resetMouseCursor() {
        setMouseCursor(DEFAULT_CURSOR);
    }

    /** ޥ */
    private void setMouseCursor(Cursor cursor) {
        if (cursor != viewerPanel.getCursor()) {
            viewerPanel.setCursor(cursor);
        }
    }

    /** ư */
    private boolean moveTo(Drag start, int x, int y) {
        Line line = start.line;

        Insets    insets = viewerPanel.getInsets();
        Dimension size   = viewerPanel.getSize();
        boolean   valid  = true;

        if (line.sense == RESIZE_VERTICAL) {
            size.width  -= (insets.left + insets.right);
            x -= start.x;
            if (x == 0) {
                return true;
            }
            if (x > 0) {
                if (line.size1 - x <= MIN_SIZE) {
                    x = line.size1 - MIN_SIZE;
                    valid = false;
                }
            } else {
                if (line.size2 + x <= MIN_SIZE) {
                    x = -(line.size2 - MIN_SIZE);
                    valid = false;
                }
            }
            line.item1.setHeight(new Value((line.size1 - x) * 100 / (double) size.width , Value.UNIT_PERCENT));
            line.item2.setHeight(new Value((line.size2 + x) * 100 / (double) size.width , Value.UNIT_PERCENT));
        } else {
            size.height -= (insets.top + insets.bottom);
            y -= start.y;
            line.item1.setWidth (new Value((line.size1 + y) * 100 / (double) size.height, Value.UNIT_PERCENT));
            line.item2.setWidth (new Value((line.size2 - y) * 100 / (double) size.height, Value.UNIT_PERCENT));
        }

        viewerPanel.invalidate();
        viewerPanel.validate  ();
        viewerPanel.repaint   ();
        return valid;
    }

//### Line
    /** 饤 */
    private final class Line {
        private int       sense;
        private int       x;
        private int       y;
        private int       width;
        private int       height;
        private FrameItem item1;
        private FrameItem item2;
        private int       size1;
        private int       size2;

        /** 饤 */
        private Line(int sense, int x, int y, int width, int height,
                     FrameItem item1, FrameItem item2,
                     int size1, int size2) {
            this.sense  = sense;
            this.x      = x;
            this.y      = y;
            this.width  = width;
            this.height = height;
            this.item1  = item1;
            this.item2  = item2;
            this.size1  = size1;
            this.size2  = size2;
        }

        /** ޤޤƤ뤫ǧ */
        private boolean contains(int x, int y) {
            return (this.x <= x && x <= this.x + this.width
                 && this.y <= y && y <= this.y + this.height);
        }
    }

//### Drag
    /** ɥå */
    private final class Drag {
        private Line line;
        private int  x;
        private int  y;

        /** ɥå */
        private Drag(Line line, int x, int y) {
            this.line = line;
            this.x    = x;
            this.y    = y;
        }
    }
}
