/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.awt.InnerScrollbar;
import net.hizlab.kagetaka.awt.event.AdjustmentEvent;
import net.hizlab.kagetaka.awt.image.OffscreenImage;
import net.hizlab.kagetaka.awt.panel.PanelScrollbar;

import java.awt.Adjustable;
import java.awt.AWTEvent;
import java.awt.AWTEventMulticaster;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.AdjustmentListener;

/**
 * ѥͥѤΥСǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
class ExScrollbar extends Component implements Adjustable, PanelScrollbar {
    /** @serial å֥ */
    private Object lock = new Object();

    /** @serial ե꡼ */
    private OffscreenImage offscreenImage;

    /** @serial С */
    private Scrollbar scrollbar;
    /** @serial  */
    private int       orientation;
    /** @serial  */
    private boolean   isHorizontal;
    /** @serial ̵ͭ */
    private boolean   enabled = true;

    private transient AdjustmentListener adjustmentListener;

    /**
     * Сޤ
     *
     * @param  orientation 
     */
    ExScrollbar(int orientation) {
        scrollbar = new Scrollbar(orientation);
        scrollbar.setValues(0, 1, 0, 1);
        scrollbar.setEnabled(false);

        this.orientation  = orientation;
        this.isHorizontal = (orientation == HORIZONTAL);

        setUnitIncrement (1);
        setBlockIncrement(1);
    }

//### Override
    /**
     * 侩֤ޤ
     *
     * @return 侩
     */
    public Dimension getPreferredSize() {
        return (isHorizontal
                ? new Dimension(0, 16)
                : new Dimension(16, 0));
    }

    /**
     * 祵֤ޤ
     *
     * @return 祵
     */
    public Dimension getMaximumSize() {
        return getPreferredSize();
    }

    /**
     * ե뤳Ȥ뤫֤ޤ
     *
     * @return ե뤳Ȥ <code>true</code>
     *         ʤ <code>false</code>
     */
    public boolean isFocusTraversable() {
        return false;
    }

    /**
     * ̵ͭꤷޤ
     *
     * @param  b ͭˤ <code>true</code>
     *           ̵ˤ <code>false</code>
     */
    public void setEnabled(boolean b) {
        synchronized (scrollbar) {
            if (enabled == b) {
                return;
            }

            enabled = b;
            super.setEnabled(b);
            scrollbar.setEnabled(b);
        }

        repaintForce();
    }

    /**
     * ̤򥢥åץǡȤޤ
     *
     * @param  g եå
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ᡼ºݤ褷ޤ
     *
     * @param  g եå
     */
    public void paint(Graphics g) {
        offscreenImage.paint(g);
    }

    /**
     * ݡͥȤƥʤɲä줿ȤΤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();

            // ե꡼
            offscreenImage = new OffscreenImage(this) {
                public void update(Image offscreen, Graphics g, Dimension size) {
                    refresh(offscreen, g, size);
                }
            };
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();
            scrollbar.dispose();

            // ե꡼
            if (offscreenImage != null) {
                offscreenImage.dispose();
                offscreenImage = null;
            }
        }
    }

    /**
     * ΥơСΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString()
                   + (isHorizontal ? ",horizontal" : ",vertical")
                   + ",value="   + scrollbar.getValue()
                   + ",amount="  + scrollbar.getVisibleAmount()
                   + ",minimum=" + scrollbar.getMinimum()
                   + ",maximum=" + scrollbar.getMaximum();
        return str;
    }

    /** ե꡼ */
    private void refresh(Image offscreen, Graphics g, Dimension size) {
        scrollbar.setSize(size.width, size.height);
        scrollbar.paint(g, true);
    }

//### Adjustable, PanelScrollbar
    /* ݡͥȤ֤ */
    /** {@inheritDoc} */
    public Component getComponent() {
        return this;
    }

    /* ֤ */
    /** {@inheritDoc} */
    public int getOrientation() {
        return orientation;
    }

    /* ֤ͤ */
    /** {@inheritDoc} */
    public int getValue() {
        return scrollbar.getValue();
    }

    /* ͤꤹ */
    /** {@inheritDoc} */
    public void setValue(int newValue) {
        synchronized (scrollbar) {
            if (scrollbar.getValue() == newValue) {
                return;
            }

            scrollbar.setValue(newValue);
        }

        repaintForce();
    }

    /* Ǿ֤ͤ */
    /** {@inheritDoc} */
    public int getMinimum() {
        return scrollbar.getMinimum();
    }

    /* Ǿͤꤹ */
    /** {@inheritDoc} */
    public void setMinimum(int newMinimum) {
        if (scrollbar.getMinimum() == newMinimum) {
            return;
        }

        int value = scrollbar.getValue();
        if (isHorizontal) {
            value -= newMinimum - scrollbar.getMinimum();
        }

        fixValue(scrollbar.setValues(value, scrollbar.getVisibleAmount(), newMinimum, scrollbar.getMaximum()));
    }

    /* ֤ͤ */
    /** {@inheritDoc} */
    public int getMaximum() {
        return scrollbar.getMaximum();
    }

    /* ͤꤹ */
    /** {@inheritDoc} */
    public void setMaximum(int newMaximum) {
        synchronized (scrollbar) {
            if (scrollbar.getMaximum() == newMaximum) {
                return;
            }

            int value = scrollbar.getValue();
            if (isHorizontal) {
                value += newMaximum - scrollbar.getMaximum();
            }

            fixValue(scrollbar.setValues(value, scrollbar.getVisibleAmount(), scrollbar.getMinimum(), newMaximum));
        }
    }

    /* Ļʬ֤ */
    /** {@inheritDoc} */
    public int getVisibleAmount() {
        return scrollbar.getVisibleAmount();
    }

    /* Ļʬꤹ */
    /** {@inheritDoc} */
    public void setVisibleAmount(int newVisible) {
        synchronized (scrollbar) {
            if (scrollbar.getVisibleAmount() == newVisible) {
                return;
            }

            int value = scrollbar.getValue();
            if (isHorizontal) {
                value -= newVisible - scrollbar.getVisibleAmount();
            }

            fixValue(scrollbar.setValues(value, newVisible, scrollbar.getMinimum(), scrollbar.getMaximum()));
        }
    }

    /* ˥åȥ󥯥Ȥ֤ */
    /** {@inheritDoc} */
    public int getUnitIncrement() {
        return scrollbar.getUnitIncrement();
    }

    /* ˥åȥ󥯥Ȥꤹ */
    /** {@inheritDoc} */
    public void setUnitIncrement(int v) {
        scrollbar.setUnitIncrement(v);
    }

    /* ֥å󥯥Ȥ֤ */
    /** {@inheritDoc} */
    public int getBlockIncrement() {
        return scrollbar.getBlockIncrement();
    }

    /* ֥å󥯥Ȥꤹ */
    /** {@inheritDoc} */
    public void setBlockIncrement(int v) {
        scrollbar.setBlockIncrement(v);
    }

    /* ٥ȥꥹʤϿ */
    /** {@inheritDoc} */
    public void addAdjustmentListener(AdjustmentListener l) {
        enableEvents(AWTEvent.ADJUSTMENT_EVENT_MASK);
        adjustmentListener = AWTEventMulticaster.add(adjustmentListener, l);
    }

    /* ٥ȥꥹʤ */
    /** {@inheritDoc} */
    public void removeAdjustmentListener(AdjustmentListener l) {
        adjustmentListener = AWTEventMulticaster.remove(adjustmentListener, l);
    }

//### Action
    /**
     * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processEvent(AWTEvent e) {
        if (e instanceof AdjustmentEvent) {
            processAdjustmentEvent((AdjustmentEvent) e);
            return;
        }
        super.processEvent(e);
    }

    /**
     * ΥݡͥȤȯ륢㥹ȥȥ٥Ȥ
     * ϿƤ뤹٤Ƥ {@link AdjustmentListener} 뤳Ȥˤꡢ
     * 㥹ȥȥ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processAdjustmentEvent(AdjustmentEvent e) {
        if (adjustmentListener != null) {
            adjustmentListener.adjustmentValueChanged(e);
        }
    }

    /** 㥹ȥȥ٥Ȥȯޤ */
    private void postAdjustmentEvent(int value) {
        AdjustmentEvent e = new AdjustmentEvent(this,
                                                AdjustmentEvent.ADJUSTMENT_VALUE_USER_CHANGED,
                                                AdjustmentEvent.TRACK,
                                                value);
        try {
            getToolkit().getSystemEventQueue().postEvent(e);
        } catch (SecurityException ex) {
            processAdjustmentEvent(e);
        }
    }

//### private
    /** Ĵ᤹ */
    private void fixValue(boolean repaint) {
        boolean b = (scrollbar.getMaximum() > scrollbar.getVisibleAmount());
        if (enabled != b) {
            setEnabled(b);
            return;
        }

        if (repaint) {
            repaintForce();
        }
    }

    /** Ū˺ɽ */
    private void repaintForce() {
        OffscreenImage oi = offscreenImage;
        if (oi != null) {
            oi.repaint();
        }
    }

//### Scrollbar
    /** С */
    private final class Scrollbar extends InnerScrollbar {
        private int width;
        private int height;

        /** 󥹥󥹤 */
        private Scrollbar(int orientation) {
            super(orientation, ExScrollbar.this);
        }

        /**  */
        public void repaint() {
            repaintForce();
        }

        /** ͤѹ줿 */
        public void changedValue() {
            postAdjustmentEvent(scrollbar.getValue());
        }

        /** ѹ */
        public void setSize(int width, int height) {
            if (this.width == width && this.height == height) {
                return;
            }
            this.width  = width;
            this.height = height;
            super.setBounds(0, 0, width, height);
        }
    }
}
