/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.cookie.Cookie;
import net.hizlab.kagetaka.addin.java2.NETBWrapper;
import net.hizlab.kagetaka.awt.MessageBox;
import net.hizlab.kagetaka.net.HttpURLConnectionWrapper;
import net.hizlab.kagetaka.net.NetUtils;
import net.hizlab.kagetaka.net.URLConnectionWrapper;
import net.hizlab.kagetaka.protocol.CacheSupported;
import net.hizlab.kagetaka.protocol.ProxySupported;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.PostData;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.viewer.cookie.CookieManager;
import net.hizlab.kagetaka.viewer.option.OptionListener;
import net.hizlab.kagetaka.viewer.option.ViewerController;
import net.hizlab.kagetaka.viewer.option.ViewerOption;
import net.hizlab.kagetaka.viewer.option.InvalidValueException;

import java.awt.Frame;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.Enumeration;

/**
 * ץȥ³򥵥ݡȤ륯饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.9 $
 */
class Connector implements OptionListener {
    private static final String RESOURCE = "net.hizlab.kagetaka.viewer.Resources";

    private Frame         owner;
    private ViewerOption  option;
    private Reporter      reporter;
    private CookieManager cookieManager;
    private SSLManager    sslManager;
    private CacheManager  cacheManager;
    private boolean       useCache;
    private NETBWrapper   netbWrapper;

    /**
     * ͥΥ󥹥󥹤ޤ
     *
     * @param  owner    ʡ
     * @param  option   ץ
     * @param  reporter ݡ
     */
    Connector(Frame owner, ViewerOption option, ViewerReporter reporter) {
        this.owner         = owner;
        this.reporter      = reporter;
        this.option        = option;
        this.cookieManager = option.getCookieManager();
        this.sslManager    = option.getSSLManager();
        this.cacheManager  = CacheManager.getInstance(option);
        this.useCache      = option.getPropertyBoolean(ViewerOption.KEY_CACHE_STORE, true);
        this.netbWrapper   = NETBWrapper.getInstance();
    }

    /**
     * ꤵ줿 URL ΥƥĤ֤ޤ
     *
     * @param  request ꥯ
     *
     * @return ƥ
     *
     * @throws IOException IO 顼ȯ
     * @throws InterruptedException Ǥ줿
     */
    Content getContent(Request request)
            throws IOException, InterruptedException {
        URL               url            = request.url;
        URL               proxyUrl       = null;
        boolean           debug          = (reporter != null && reporter.isReported(Reporter.NETWORK, Reporter.DEBUG));

        ViewerContent cachedContent = null;

        try {
            if (reporter != null) {
                reporter.report(Reporter.NETWORK, Reporter.INFO, Reporter.NONE, null, 0, 0,
                                "Connect", getMessage("report.connect.open", new String[]{url.toString()}));
            }

            // BUGS ΥǥХå
            if (Debug.isDebug) {
                String crashUrl = option.getPropertyString(ViewerOption.KEY_DEBUG_CRASH_URL);
                if (crashUrl != null) {
                    if (url.toString().startsWith(crashUrl)) {
                        throw new RuntimeException("Crash Test (" + crashUrl + ")");
                    }
                }
            }

            URLConnection      connection           = null;
            HttpURLConnection  httpConnection       = null;
            URL                baseUrl              = url;
            URL                connectUrl           = url;
            PostData           pd                   = request.postData;
            boolean            isNewUrl             = true;
            boolean            isNewProxyAuth       = false;
            AuthenticationInfo proxyAuthentication  = null;
            boolean            isNewServerAuth      = false;
            AuthenticationInfo serverAuthentication = null;
            boolean            repostCheck          = true;

            CONNECT:
            for (;;) {
                // ǧھ
                if (isNewUrl) {
                    // ǧھ
                    serverAuthentication = AuthenticationInfo.get(AuthenticationInfo.SERVER, url);

                    // ץǧھ
                    URL newProxyUrl = option.getProxyURL(url);
                    if (newProxyUrl != proxyUrl) {
                        proxyUrl = newProxyUrl;
                        if (proxyUrl != null) {
                            proxyAuthentication = AuthenticationInfo.get(AuthenticationInfo.PROXY, proxyUrl);
                        } else {
                            proxyAuthentication = null;
                        }
                    }

                    // URL ץ
                    if (proxyUrl != null) {
                        connectUrl = new URL(url.getProtocol(), proxyUrl.getHost(), proxyUrl.getPort(), url.toString());
                    } else {
                        connectUrl = url;
                    }
                }

                // å夫
                if (request.getUseCache() != Request.CACHE_NONE) {
                    ViewerContent content = cacheManager.getContent(url, pd);
                    switch (request.getUseCache()) {
                    case Request.CACHE_MUST:
                        if (content != null) {
                            return content;
                        }
                        if (!"file".equalsIgnoreCase(url.getProtocol())) {
                            // ˲ɤ߹ޤʤξ
                            return null;
                        }
                        break;
                    case Request.CACHE_SOFT:
                        if (content != null) {
                            return content;
                        }
                        break;
                    case Request.CACHE_NORMAL:
                        if (content != null && content.validateExpiration(option, request)) {
                            return content;
                        }
                        break;
                    default: // AVOID
                    }
                    cachedContent = content;
                }

                // ɤ߹ߤξǡPOST ǡ¸ߤϳǧ
                if (repostCheck                             // ٤䤹
                        && !request.url.getProtocol().equalsIgnoreCase("about")
                        && request.getDocument() != null    // ɤ߹߻ˤ Document ͭ
                        && request.postData      != null) {
                    if (MessageBox.show(owner,
                                        getMessage("message.dorepost.text" , null),
                                        getMessage("message.dorepost.title", null),
                                        MessageBox.BUTTON_OKCANCEL | MessageBox.ICON_QUESTION) != MessageBox.BUTTON_OK) {
                        cachedContent = null;
                        return null;
                    }
                    repostCheck = false;
                }

                // ͥ
                connection = connectUrl.openConnection();

                // å奵ݡ
                if (useCache && connection instanceof CacheSupported) {
                    ((CacheSupported) connection).setupCache(cacheManager, pd);
                }

                // SSL ݡ
                if (sslManager != null
                        && "https".equalsIgnoreCase(url.getProtocol())) {
                    sslManager.setup(owner, connection);
                }

                if (connection instanceof HttpURLConnection) {
                    httpConnection = (HttpURLConnection) connection;
                } else {
                    httpConnection = null;
                }

                // ꥯȥإå
                setHTTPHeader(url, connection, request, cachedContent,
                              proxyAuthentication, serverAuthentication);

                // POST ǡ³
                if (pd != null) {
                    pd.send(connection);
                }

                // ³
                try {
                    connection.connect();

                    // ǳμ¤³
                    if (httpConnection != null) {
                        httpConnection.getResponseCode();
                    } else {
                        connection.getInputStream();
                    }
                } finally {
                    if (Thread.interrupted()) {
                        throw new InterruptedException("connect");
                    }
                }
                isNewUrl = false;

                // 쥹ݥ󥹥إå
                getHTTPHeader(url, connection);

                if (debug) {
                    dumpConnection(connection);
                }

                // ʹߤ HTTP ν
                if (httpConnection == null) {
                    break;
                }

                int code = httpConnection.getResponseCode();

                // ѹʤ
                if (cachedContent != null) {
                    if (code == HttpURLConnection.HTTP_NOT_MODIFIED) {
                        httpConnection.disconnect();
                        cachedContent.resetTimes();
                        ViewerContent content = cachedContent;
                        cachedContent = null;
                        return content;
                    }

                    cachedContent.removeCache();
                    cachedContent = null;
                }

                if (code == HttpURLConnection.HTTP_UNAUTHORIZED) {
                    // ǧڽ
                    if ((serverAuthentication = getWWWAuthenticate(url, connection, isNewServerAuth, false)) == null) {
                        break CONNECT;
                    }
                    isNewServerAuth = true;

                    // ɹԤ
                    if (reporter != null) {
                        reporter.report(Reporter.NETWORK, Reporter.INFO, Reporter.NONE, null, 0, 0,
                                        "Connect", getMessage("report.connect.reopen", new String[]{url.toString()}));
                    }
                } else if (code == HttpURLConnection.HTTP_PROXY_AUTH) {
                    // ǧڽ
                    if ((proxyAuthentication = getWWWAuthenticate(proxyUrl, connection, isNewProxyAuth, true)) == null) {
                        break CONNECT;
                    }
                    isNewProxyAuth = true;

                    // ɹԤ
                    if (reporter != null) {
                        reporter.report(Reporter.NETWORK, Reporter.INFO, Reporter.NONE, null, 0, 0,
                                        "Connect", getMessage("report.connect.reopen", new String[]{url.toString()}));
                    }
                } else {
                    // ǧھϤ줬ϥåɲ
                    if (isNewProxyAuth ) {
                        AuthenticationInfo.put(AuthenticationInfo.PROXY , proxyAuthentication , proxyUrl);
                    }
                    if (isNewServerAuth) {
                        AuthenticationInfo.put(AuthenticationInfo.SERVER, serverAuthentication, url     );
                    }

                    switch (code) {
                    case HttpURLConnection.HTTP_MULT_CHOICE:         // 300
                    case HttpURLConnection.HTTP_MOVED_PERM:          // 301
                    case 307:                                        // 307 Temporary Redirect
                        break;
                    case HttpURLConnection.HTTP_MOVED_TEMP:          // 302
                        if (pd != null && !option.getPropertyBoolean(ViewerOption.KEY_HTTP_STRICT, false)) {
                            pd = null;
                        }
                        break;
                    case HttpURLConnection.HTTP_SEE_OTHER :          // 303
                        if (pd != null) {
                            pd = null;
                        }
                        break;
                    default:
                        break CONNECT;
                    }

                    // 쥯Ƚ
                    if ((url = getRedirectURL(url, connection, pd, baseUrl)) == null) {
                        break CONNECT;
                    }

                    // ǧھ
                    isNewUrl             = true;
                    isNewProxyAuth       = false;
                    isNewServerAuth      = false;
                    serverAuthentication = null;

                    if (reporter != null) {
                        reporter.report(Reporter.NETWORK, Reporter.INFO, Reporter.NONE, null, 0, 0,
                                        "Connect", getMessage("report.connect.redirect", new String[]{url.toString()}));
                    }
                }

                httpConnection.disconnect();
            }

            return new ViewerContent(((proxyUrl == null) || (connection instanceof ProxySupported)
                                      ? connection
                                      : (netbWrapper != null
                                         ? netbWrapper.getURLConnectionWrapper(url, connection, null)
                                         : (httpConnection != null
                                            ? (URLConnection) new HttpURLConnectionWrapper(url, httpConnection, null)
                                            : (URLConnection) new URLConnectionWrapper    (url, connection    , null)))),
                                     (sslManager != null
                                      ? sslManager.getSSLCertification(connection)
                                      : null));
        } catch (IOException e) {
            // ǧ㳰̵뤷ƽλ
            if (sslManager != null
                    && sslManager.ignored(e)) {
                return null;
            }

            // ץ³顼ʤ⤷ʤ˾
            if (/*---*/proxyUrl != null
                    && (e instanceof UnknownHostException
                     || e instanceof ConnectException
                     || e instanceof SocketException)) {
                String option = (e instanceof UnknownHostException ? "notfound" : "notconnect");
                MessageBox.show(owner,
                                getMessage("message.proxy." + option + ".text" , new String[]{proxyUrl.getHost()}),
                                getMessage("message.proxy." + option + ".title", null),
                                MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
                return null;
            }

            throw e;
        } finally {
            // λ
            if (cachedContent != null) {
                cachedContent.removeCache();
            }
        }
    }

    /**
     * åꤷޤ
     *
     * @param  value 
     * @param  url   URL
     */
    void setCookie(String value, URL url) {
        Cookie cookie = cookieManager.createCookie(value, url);
        if (cookie == null) {
            return;
        }

        boolean strage = (cookie.getExpires() >= 0);
        String  accept;
        if (strage) {
            accept = option.getPropertyString(ViewerOption.KEY_COOKIE_ACCEPT_STRAGE );
        } else {
            accept = option.getPropertyString(ViewerOption.KEY_COOKIE_ACCEPT_SESSION);
        }

        if (accept == null || accept.compareTo(CookieManager.ASC) == 0) {
            if (MessageBox.show(owner,
                                getMessage("message.cookie.text" ,
                                           new Object[]{cookie.getDomain(),
                                                        cookie.getName  (),
                                                        cookie.getValue (),
                                                        cookie.getDomain(),
                                                        cookie.getPath  (),
                                                        new Integer((strage ? 1 : 0)),
                                                        (strage ? new Date(cookie.getExpires()) : null)}),
                                getMessage("message.cookie.title", null),
                                MessageBox.BUTTON_YESNO | MessageBox.ICON_QUESTION) == MessageBox.RESULT_YES) {
                accept = CookieManager.YES;
            } else {
                accept = CookieManager.NO;
            }
        }

        if (accept.compareTo(CookieManager.YES) == 0) {
            cookieManager.setCookie(cookie);
        }
    }

    /** HTTP ꥯȥإå */
    private void setHTTPHeader(URL url, URLConnection connection, Request request,
                               ViewerContent cachedContent,
                               AuthenticationInfo proxyAuthentication,
                               AuthenticationInfo serverAuthentication)
            throws IOException {
        connection.setRequestProperty("Host"           , url.getHost() + (url.getPort() > 0 ? ":" + url.getPort() : ""));
        connection.setRequestProperty("User-Agent"     , option.getPropertyString(ViewerOption.KEY_HTTP_USERAGENT     ));
        connection.setRequestProperty("Accept"         , option.getPropertyString(ViewerOption.KEY_HTTP_ACCEPT        ));
        connection.setRequestProperty("Accept-Language", option.getPropertyString(ViewerOption.KEY_HTTP_ACCEPTLANGUAGE));
        connection.setRequestProperty("Accept-Encoding", option.getPropertyString(ViewerOption.KEY_HTTP_ACCEPTENCODING));
        connection.setRequestProperty("Accept-Charset" , option.getPropertyString(ViewerOption.KEY_HTTP_ACCEPTCHARSET ));

        int referer = option.getPropertyInteger(ViewerOption.KEY_HTTP_REFERER, HawkViewer.REFERER_SAMEHOST);
        if (request.referer != null
                && (referer == HawkViewer.REFERER_ALL
                 || (referer != HawkViewer.REFERER_SAMEHOST
                  && NetUtils.isSamePath(request.referer, url, referer)))) {
            connection.setRequestProperty("Referer", request.referer.toExternalForm());
        }

        String cookie = cookieManager.getCookie(url);
        if (cookie != null) {
            connection.setRequestProperty("Cookie", cookie);
        }

        if (proxyAuthentication  != null) {
            connection.setRequestProperty("Proxy-Authorization", proxyAuthentication .getValue());
        }
        if (serverAuthentication != null) {
            connection.setRequestProperty("Authorization"      , serverAuthentication.getValue());
        }

        switch (request.getUseCache()) {
        case Request.CACHE_NONE:
            connection.setRequestProperty("Cache-Control", "no-cache");
            connection.setRequestProperty("Pragma"       , "no-cache");
            break;
        case Request.CACHE_CHECK:
            connection.setRequestProperty("Cache-Control", "max-age=0");
            break;
        default: // AVOID
        }

        // å
        if (cachedContent != null) {
            connection.setIfModifiedSince(cachedContent.lastModified);

            String eTag = cachedContent.eTag;
            if (eTag != null) {
                connection.setRequestProperty("If-None-Match", eTag);
            }
        }
    }

    /** HTTP ꥯȥإå */
    private void getHTTPHeader(URL url, URLConnection connection)
            throws IOException {
        String key;
        int    index = 1;

        for (;;) {
            if ((key = connection.getHeaderFieldKey(index)) == null) {
                break;
            }

            key = key.toLowerCase();
            if (key.compareTo("set-cookie") == 0) {
                setCookie(connection.getHeaderField(index), url);
            }

            index++;
        }
    }

    /** ǧھ桼 */
    private AuthenticationInfo getWWWAuthenticate(URL url, URLConnection connection,
                                                  boolean retry, boolean proxy) {
        String realm;
        if (proxy) {
            realm = connection.getHeaderField("Proxy-Authenticate");
        } else {
            realm = connection.getHeaderField("WWW-Authenticate");
        }
        if (realm == null) {
            return null;
        }

        // ǧھʬ (WWW-Authenticate: Basic realm=dokoka)
        String type = realm;
        int p = realm.indexOf(' ');
        if (p != -1) {
            type  = realm.substring(0, p);
            realm = realm.substring(p + 1);
            if (realm.toLowerCase().startsWith("realm")) {
                for (p = 5; p < realm.length(); p++) {
                    if (realm.charAt(p) != ' ') {
                        if (realm.charAt(p) == '=') {
                            realm = realm.substring(p + 1).trim();
                        }
                        break;
                    }
                }
            }
        } else {
            realm = null;
        }

        char atype;
        if (type.toLowerCase().compareTo("basic") == 0) {
            atype = AuthenticationInfo.BASIC;
        } else {
            // Basic ʳǧڤϥݡȳ
            MessageBox.show(owner,
                            getMessage("message.unknown_authtype.text" , new String[]{type, url.toString()}),
                            getMessage("message.unknown_authtype.title", null),
                            MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
            return null;
        }

        // realm ʸ Shift_JIS Ȥʣ礷Ƥߤ
        int length = realm.length();
        for (int i = 0; i < length; i++) {
            if (realm.charAt(i) >= 0x80) {
                try {
                    realm = new String(realm.getBytes("8859_1"), "SJIS");
                } catch (UnsupportedEncodingException e) { }
                break;
            }
        }

        return AuthenticationInfo.get(!retry, owner,
                                      (proxy ? AuthenticationInfo.PROXY : AuthenticationInfo.SERVER),
                                      url, realm, atype);
    }

    /** 쥯ȤԤ֤ */
    private URL getRedirectURL(URL url, URLConnection connection, PostData pd, URL baseUrl)
            throws IOException {
        String location = connection.getHeaderField("Location");
        if (location == null) {
            return null;
        }

        try {
            url = new URL(url, location);         //  Location ϥեѥΤϤ
            if (/*---*/pd != null
                    && MessageBox.show(owner,
                                       getMessage("message.redirect.text" , new String[]{baseUrl.toString(), location}),
                                       getMessage("message.redirect.title", null),
                                       MessageBox.BUTTON_OKCANCEL | MessageBox.ICON_QUESTION) != MessageBox.BUTTON_OK) {
                return null;
            }
        } catch (MalformedURLException e) {
            MessageBox.show(owner,
                            getMessage("message.invalidurl.text" , new String[]{location, e.toString()}),
                            getMessage("message.invalidurl.title", null),
                            MessageBox.BUTTON_OK | MessageBox.ICON_EXCLAMATION);
            return null;
        }

        return url;
    }

    /** ꥽ʸ */
    private String getMessage(String key, Object[] args) {
        return Resource.getMessage(RESOURCE, key, args);
    }

    /** ͥ */
    private void dumpConnection(URLConnection connection)
            throws IOException {
        if (connection instanceof HttpURLConnection) {
            HttpURLConnection httpConnection = (HttpURLConnection) connection;
            reporter.report(Reporter.NETWORK, Reporter.DEBUG, Reporter.NONE, null, 0, 0,
                            "Connect", "HTTP Response: " + httpConnection.getResponseCode()
                                                         + " " + httpConnection.getResponseMessage());
        }

        int index = 0;
        String key;

        while ((key = connection.getHeaderFieldKey(++index)) != null) {
            reporter.report(Reporter.NETWORK, Reporter.DEBUG, Reporter.NONE, null, 0, 0,
                            "Connect", "HTTP Response: [" + key + "]:[" + connection.getHeaderField(index) + "]");
        }
    }

    //### OptionListener
    /** {@inheritDoc} */
    public void propertyChange(ViewerOption option, String key, Object oldValue, Object newValue)
            throws InvalidValueException {
    }

    /** {@inheritDoc} */
    public void propertiesChanged(ViewerOption option, ViewerController c, Enumeration list) {
        String key;
        while (list.hasMoreElements()) {
            key = (String) list.nextElement();

            // å
            if (key.compareTo(ViewerOption.KEY_CACHE_STORE) == 0) {
                useCache = option.getPropertyBoolean(ViewerOption.KEY_CACHE_STORE, true);
            }
        }
    }
}
