/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.viewer;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.addin.java2.LangWrapper;
import net.hizlab.kagetaka.protocol.CacheFile;
import net.hizlab.kagetaka.rendering.PostData;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.util.StringUtils;
import net.hizlab.kagetaka.viewer.option.ViewerOption;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.InvalidObjectException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.text.ParseException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Random;

/**
 * ɤ߹ߥǡ򥭥å夹롢åޥ͡󶡤ޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
final class CacheManager implements net.hizlab.kagetaka.protocol.CacheManager {
    private static int serverNumber = 0;

    private static final String INDEX_FILE          = "00000000";
    private static final String INDEX_FILE_NOW      = ".dat";
    private static final String INDEX_FILE_NEW      = ".new";
    private static final String INDEX_FILE_ENCODING = "UTF-8";
    private static final char[] DIGITS =
    {
        '0', '1', '2', '3', '4', '5', '6', '7',
        '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'
    };

    private static Hashtable cacheManagers = new Hashtable();

    /** ƥå */
    static {
        LangWrapper wrapper = LangWrapper.getInstance();
        if (wrapper != null) {
            wrapper.addShutdownHook(
                new Thread() {
                    /** ¹ */
                    public void run() {
                        dispose();
                    }
                }
            );
        }
    }

    /**
     * åǥ쥯ȥꤴȤΥ󥹥󥹤ޤ
     *
     * @param  option ӥ塼ץ
     *
     * @return å奤󥹥
     */
    static CacheManager getInstance(ViewerOption option) {
        File cacheDir = option.getPropertyFile(ViewerOption.KEY_CACHE_STORE_PATH);
        if (cacheDir == null) {
            return null;
        }

        try {
            // åǥ쥯ȥ
            if (!cacheDir.exists()) {
                cacheDir.mkdirs();
                if (!cacheDir.exists()) {
                    return null;
                }
            }
        } catch (SecurityException e) {
            //### ERROR
e.printStackTrace(Debug.out);
            return null;
        }

        synchronized (cacheManagers) {
            CacheManager cm = (CacheManager) cacheManagers.get(cacheDir);
            if (cm == null) {
                cacheManagers.put(cacheDir, (cm = new CacheManager(option, cacheDir)));
            }

            return cm;
        }
    }

    /**
     * åޥ͡򤹤٤˴ޤ
     */
    static void dispose() {
        synchronized (cacheManagers) {
            for (Enumeration e = cacheManagers.elements(); e.hasMoreElements();) {
                ((CacheManager) e.nextElement()).exit();
            }

            cacheManagers.clear();
        }
    }

    private ViewerOption option;
    private SSLManager   sslManager;
    private Hashtable    cache = new Hashtable(500);
    private CacheEntry   top;
    private CacheEntry   last;
    private File         cacheDir;
    private String       cacheDirString;
    private Random       random;
    private Saver        saver;

    /** 󥹥󥹤 */
    private CacheManager(ViewerOption option, File cacheDir) {
        this.option         = option;
        this.sslManager     = option.getSSLManager();
        this.cacheDir       = cacheDir;
        this.cacheDirString = cacheDir.getPath();
        this.random         = new Random(System.currentTimeMillis());
        this.saver          = new Saver();

        if (option.getPropertyBoolean(ViewerOption.KEY_CACHE_STORE, true)) {
            load();
        }

        saver.start();
    }

    /** {@inheritDoc} */
    public CacheFile getFile(URLConnection connection) {
        String ext = StringUtils.getFile(connection.getURL().getFile());
        int p = ext.lastIndexOf('.');
        if (p != -1) {
            ext = ext.substring(p);
        } else {
            //### TODO MIME ơ֥򻲾
            ext = "";
            try {
                ContentType ct = ContentType.valueOf(connection.getContentType(), connection.getURL());
                if (ct != null) {
                    String subtype = ct.getSubType();
                    if (subtype != null) {
                        ext = "." + subtype;
                    }
                }
            } catch (ParseException e) { }
        }

        synchronized (this) {
            CacheFile file;

            for (;;) {
                do {
                    file = new CacheFile(cacheDirString, toHexString((long) (random.nextDouble() * 0xFFFFFFFEL) + 1) + ext);
                } while (file.exists());

                // ե
                try {
                    FileOutputStream fos = new FileOutputStream(file);
                    fos.close();
                    break;
                } catch (IOException e) { }
            }

            return file;
        }
    }

    /** {@inheritDoc} */
    public void addEntry(URLConnection connection, PostData pd, CacheFile file) {
        String key = createKey(connection.getURL(), pd);
Debug.out.println("* addc=" + file.getName() + "," + key);
if (file.length() == 0) {
Debug.err.println("### ERROR ### CacheManager.addEntry : file length is 0 : " + file.getName() + "," + key);
(new RuntimeException()).printStackTrace(Debug.err);
}
        CacheEntry entry = new CacheEntry(key,
                                          (pd != null),
                                          file,
                                          connection.getURL         (),
                                          connection.getContentType (),
                                          connection.getExpiration  (),
                                          connection.getDate        (),
                                          connection.getLastModified(),
                                          connection.getHeaderField ("ETag"),
                                          (sslManager != null
                                           ? sslManager.getSSLCertification(connection)
                                           : null));

        synchronized (cache) {
            addLink(entry);
            entry = (CacheEntry) cache.put(key, entry);
        }

        // CACHE_NONE ʤɤξ
        if (entry != null) {
            entry.dispose();
            saver.save();
        }

        saver.save();
    }

    /** {@inheritDoc} */
    public void removeEntry(URLConnection connection, PostData pd) {
        synchronized (cache) {
            CacheEntry entry = (CacheEntry) cache.remove(createKey(connection.getURL(), pd));
            if (entry != null) {
                removeEntryImpl(entry);
            }
        }
    }

    /**
     * å奨ȥޤ
     *
     * @param  entry 륨ȥ
     */
    void removeEntry(CacheEntry entry) {
        synchronized (cache) {
            cache.remove(entry.key);
            removeEntryImpl(entry);
        }
    }

    /** å奨ȥץå */
    private void removeEntryImpl(CacheEntry entry) {
        entry.dispose();
        removeLink(entry);
        saver.save();
Debug.out.println("* delc=" + entry.file.getName() + "," + entry.key);
    }

    /**
     * ꤵ줿 URL Υå夵줿ƥĤ֤ޤ
     * å¸ߤʤ <code>null</code> ֤ޤ
     *
     * @param  url URL
     * @param  pd  POST ǡ
     *
     * @return å夵줿ƥġ
     *         ¸ߤʤ <code>null</code>
     */
    ViewerContent getContent(URL url, PostData pd) {
        String     key = createKey(url, pd);
        CacheEntry entry;

        synchronized (cache) {
            entry = (CacheEntry) cache.get(key);
            if (entry == null) {
                return null;
            }

            if (top != entry) {
                removeLink(entry);
                addLink(entry);
            }
        }

        try {
Debug.out.println("* hitc=" + entry.file.getName() + "," + key);
            return new ViewerContent(new CachedURLConnection(this, url, entry),
                                     entry);
        } catch (IOException e) {
            //### ERROR
e.printStackTrace(Debug.out);
            removeEntry(entry);
        }

        return null;
    }

    /** åΥ֤ޤ */
    private String createKey(URL url, PostData pd) {
        StringBuffer sb = new StringBuffer();
        sb.append(url.getProtocol());
        sb.append("://");
        sb.append(url.getHost());
        if (url.getPort() > 0) {
            sb.append(":");
            sb.append(url.getPort());
        }
        sb.append(url.getFile());

        if (pd != null) {
            sb.append(((char) 0));
            sb.append(pd.hashCode());
        }

        return sb.toString();
    }

    /** 16ʿѴ */
    private String toHexString(long value) {
        char[] buffer = new char[8];
        int    mask   = 0xF;
        int    p      = 8;

        do {
            buffer[--p] = DIGITS[(int) (value & mask)];
            value >>>= 4;

        } while (value != 0);
        while (--p >= 0) {
            buffer[p] = '0';
        }

        return new String(buffer);
    }

    /**  */
    private void load() {
        BufferedReader br = null;

        try {
            String indexFileName = INDEX_FILE + INDEX_FILE_NOW;

            String[]  fileList = cacheDir.list();
            Hashtable fileHash = new Hashtable();
            for (int i = 0; i < fileList.length; i++) {
                fileHash.put(fileList[i], "");
            }
            fileHash.remove(indexFileName);

            File indexFile = new File(cacheDirString, indexFileName);
            if (indexFile.exists()) {
                br = new BufferedReader(
                       new InputStreamReader(
                         new FileInputStream(indexFile),
                         INDEX_FILE_ENCODING));

                String     line;
                int        number = 0;
                CacheEntry entry;

                while ((line = br.readLine()) != null) {
                    try {
                        number++;
                        entry = new CacheEntry(cacheDirString, line);

                        if (top == null) {
                            top = last = entry;
                        } else {
                            last.next = entry;
                            entry.prev = last;
                            last = entry;
                        }

                        cache.put(entry.key, entry);
                        fileHash.remove(entry.file.getName());
//Debug.out.println("load=" + entry.file.getName() + "," + entry.key);
                    } catch (InvalidObjectException e) {
                    //### ERROR
Debug.out.println(number + "," + e);
                    } catch (FileNotFoundException e) {
                    //### ERROR
Debug.out.println(number + "," + e);
                    }
                }
            }

            File deleteFile;
            for (Enumeration e = fileHash.keys(); e.hasMoreElements();) {
                deleteFile = new File(cacheDir, (String) e.nextElement());
                deleteFile.delete();
Debug.out.println("* delf=" + deleteFile.getName());
            }
        } catch (IOException e) {
                //### ERROR
e.printStackTrace(Debug.out);
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) { }
            }
        }
    }

    /** å󥯤ɲáץå */
    private void addLink(CacheEntry entry) {
        if (top != null) {
            top.prev = entry;
            entry.next = top;
        } else {
            last = entry;
        }
        top = entry;
    }

    /** å󥯤ץå */
    private void removeLink(CacheEntry entry) {
        if (entry.next == null) {        // Ǹξ
            last = entry.prev;
            if (last != null) {
                last.next = entry.prev = null;
            }
        } else if (entry.prev == null) { // ǽξ
            top = entry.next;
            top.prev = entry.next = null;
        } else {                         // ֤ξ
            entry.next.prev = entry.prev;
            entry.prev.next = entry.next;
            entry.prev = entry.next = null;
        }
    }

    /** å¸ƽλ */
    private void exit() {
        saver.exit();
    }

//### Saver
    /** ǥå¸Ԥ */
    private final class Saver extends Thread {
        private boolean reset;
        private boolean doSave;
        private Object  exitLock = new Object(); // join ǤϤޤʤΤ
        private boolean exit;

        /** 󥹥󥹤 */
        private Saver() {
            setName("CacheManager-" + (serverNumber++));
        }

        /** ¹ */
        public void run() {
            boolean doSave;
            try {
                while (!isInterrupted()) {
                    synchronized (this) {
                        if (!reset) {
                            wait();
                        }

                        do {
                            reset = false;
                            wait(5000);
                        } while (reset);

                        doSave = this.doSave;
                        this.doSave = false;
                    }

                    if (isInterrupted()) {
                        break;
                    }

                    if (doSave) {
                        saveImpl(false);
                    }
                }
            } catch (InterruptedException e) { }

            // λ¸
            synchronized (exitLock) {
                try {
                    saveImpl(true);
                } finally {
                    exit = true;
                    exitLock.notifyAll();
                }
            }
        }

        /** ¸ */
        private synchronized void save() {
            reset  = true;
            doSave = true;
            notify();
        }

        /** λ */
        private void exit() {
            if (exit) {
                return;
            }

            interrupt();
            try {
                synchronized (exitLock) {
                    if (!exit) {
                        exitLock.wait();
                    }
                }
            } catch (InterruptedException e) { }
        }

        /** ºݤ¸Ԥ */
        private void saveImpl(boolean exit) {
            if (!option.getPropertyBoolean(ViewerOption.KEY_CACHE_STORE, true)) {
                return;
            }

            try {
                File newFile = new File(cacheDirString, INDEX_FILE + INDEX_FILE_NEW);
                PrintWriter pw = new PrintWriter(
                                   new OutputStreamWriter(
                                     new BufferedOutputStream(
                                       new FileOutputStream(newFile),
                                       4096),
                                     INDEX_FILE_ENCODING));

                try {
                    CacheEntry   entry   = top;
                    int          num     = 0;
                    CacheEntry[] entries;

                    synchronized (this) {
                        int size = cache.size();
                        entries = new CacheEntry[size];
                        while (num < size) {
                            entries[num++] = entry;
                            entry = entry.next;
                        }
                    }

                    long maxDiskSize = option.getPropertyInteger(ViewerOption.KEY_CACHE_STORE_TOTAL  , 0) * 0x100000L;
                    long maxFileSize = option.getPropertyInteger(ViewerOption.KEY_CACHE_STORE_MAXSIZE, 0) * 0x100000L;
                    long useDiskSize = 0;

                    for (int i = 0; i < num; i++) {
                        entry = entries[i];

                        if (/*---*/(entry.length > maxFileSize
                                 || (useDiskSize += entry.length) > maxDiskSize
                                 || !entry.save(pw))
                                && exit) {
                            entry.dispose();
                        }
                    }
                } finally {
                    pw.flush();
                    pw.close();
                }

                File nowFile = new File(cacheDirString, INDEX_FILE + INDEX_FILE_NOW);
                nowFile.delete();
                newFile.renameTo(nowFile);
            } catch (IOException e) {
                //### ERROR
Debug.out.println("### WARNING ### CacheManager.Saver.saveImpl : " + e);
e.printStackTrace(Debug.out);
            } catch (RuntimeException e) {
                //### ERROR
Debug.err.println("### ERROR ### CacheManager.Saver.saveImpl : " + e);
e.printStackTrace(Debug.err);
            }

            doSave = false;
Debug.out.println("* cache saved (" + exit + ")");
        }
    }
}
