/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.util;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.java2.NETWrapper;

import java.net.FileNameMap;
import java.net.URL;
import java.text.ParseException;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * Content-Type Ϥ륯饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class ContentType {
    /** text/html */
    public static final ContentType CT_TEXT_HTML       = new ContentType("text"   , "html"   );
    /** text/css */
    public static final ContentType CT_TEXT_CSS        = new ContentType("text"   , "css"    );
    /** text/plain */
    public static final ContentType CT_TEXT_PLAIN      = new ContentType("text"   , "plain"  );
    /** content/unknown */
    public static final ContentType CT_CONTENT_UNKNOWN = new ContentType("content", "unknown");

    private String    type       = null;      // ǥ
    private String    subtype    = null;      // ǥ֥
    private Hashtable parameters = new Hashtable();          // ѥ᡼

    /**
     * ꤵ줿ʸ򸵤ˡʸ񥿥פޤ
     *
     * @param  value Content-Type
     *
     * @throws ParseException ǽ Content-Type ꤵ줿
     */
    public ContentType(String value)
            throws ParseException {
        ContentHeaderParser.Data data = ContentHeaderParser.parse(value);

        parseMedia(data.type);
        parameters = data.parameters;
    }

    /**
     * ꤵ줿פ򸵤ˡʸ񥿥פޤ
     *
     * @param  type 
     * @param  subtype ֥
     */
    public ContentType(String type, String subtype) {
        this.type    = type;
        this.subtype = subtype;
    }

    /** type/subtype ʬ(άԲĤǡ˶򤬤äƤϤʤ) */
    private void parseMedia(String media)
            throws ParseException {
        int slash = media.indexOf('/');
        if (slash == -1) {
            throw new ParseException(Resource.getMessage("contenttype.error.media",
                                                         new String[]{media}), 0);
        }
        this.type    = media.substring(0, slash ).toLowerCase();
        this.subtype = media.substring(slash + 1).toLowerCase();
        if (type.length() == 0 || subtype.length() == 0) {
            throw new ParseException(Resource.getMessage("contenttype.error.media",
                                                         new String[]{media}), 0);
        }
    }

    /**
     * פ֤ޤ
     *
     * @return 
     */
    public String getType() {
        return type;
    }

    /**
     * ֥פ֤ޤ
     *
     * @return ֥
     */
    public String getSubType() {
        return subtype;
    }

    /**
     * ꤷѥ᡼б֤ͤޤ
     *
     * @param  key ѥ᡼°̾
     *
     * @return ѥ᡼͡¸ߤʤ <code>null</code>
     */
    public String getParameter(String key) {
        return (String) parameters.get(key.toLowerCase());
    }

    /**
     * ѥ᡼°ͤΥꥹȤ֤ޤ
     *
     * @return ѥ᡼°ͤΥꥹ
     */
    public Enumeration getParameterKeys() {
        return parameters.keys();
    }

    /**
     * ѥ᡼˿°̾ͤɲäޤ
     *
     * @param  key ѥ᡼°̾
     * @param  value ѥ᡼
     *
     * @return ꤵƤѥ᡼͡
     *         ¸ߤƤʤä <code>null</code>
     */
    public String putParameter(String key, String value) {
        return (String) parameters.put(key, value);
    }

    /**
     * ѥ᡼򥯥ꥢޤ
     */
    public void clearParameter() {
        parameters.clear();
    }

    /**
     * פפ뤫Ĵ٤ޤ
     *
     * @param  type    
     * @param  subtype ֥
     *
     * @return פ <code>true</code>
     *         ʤ <code>false</code>
     */
    public boolean equalsType(String type, String subtype) {
        if (this.type == type && this.type == subtype) {
            return true;
        }

        if (/*---*/this.type    == null || type    == null
                || this.subtype == null || subtype == null) {
            return false;
        }

        return (this.type   .compareTo(type   ) == 0
             && this.subtype.compareTo(subtype) == 0);
    }

    /**
     * ʸϥפʸɽ֤ޤ
     *
     * @return Content-Type
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(type);
        sb.append('/');
        sb.append(subtype);

        if (parameters.size() > 0) {
            String key, value;
            char c;
            boolean quote = false;
            StringBuffer v = new StringBuffer();

            for (Enumeration e = parameters.keys(); e.hasMoreElements();) {
                key   = (String) e.nextElement();
                value = (String) parameters.get(key);
                sb.append("; ");
                sb.append(key);
                sb.append('=');

                v.setLength(0);
                for (int i = 0; i < value.length(); i++) {
                    c = value.charAt(i);
                    if (quote) {
                        if (c == '"' || c == '\\') {
                            v.append('\\');
                        }
                    } else {
                        switch (c) {
                        case '(':
                        case ')':
                        case '<':
                        case '>':
                        case '@':
                        case ',':
                        case ';':
                        case ':':
                        case '/':
                        case '[':
                        case ']':
                        case '?':
                        case '=':
                        case '{':
                        case '}':
                            quote = true;
                            break;
                        case '\\':
                        case '"':
                            quote = true;
                            v.append('\\');
                            break;
                        default:
                            if (c <= 0x20) {
                                quote = true;
                            }
                        }
                    }
                    v.append(c);
                }
                if (quote) {
                    sb.append('"');
                }
                sb.append(v.toString());
                if (quote) {
                    sb.append('"');
                }
            }
        }

        return sb.toString();
    }

    /**
     * ꤵ줿ʸ񥿥פ URL 顢ʸ񥿥פޤ
     *
     * @param  ct  Content-Type
     * @param  url URL
     *
     * @return Content-Type
     *
     * @throws ParseException ǽ Content-Type ꤵ줿
     */
    public static ContentType valueOf(String ct, URL url)
            throws ParseException {
        if (ct == null) {
            NETWrapper netWrapper = NETWrapper.getInstance();
            if (netWrapper != null) {
                FileNameMap fnm = NETWrapper.getInstance().getFileNameMap();
                if (fnm != null) {
                    ct = fnm.getContentTypeFor(url.toString());
                }
            }
        }

        if (ct != null && ct.compareTo("content/unknown") != 0) {
            return new ContentType(ct);
        }

        int p;
        String ext;

        ext = StringUtils.getFile(url.getFile());

        // ĥҤ
        if ((p = ext.lastIndexOf('.')) == -1) {
            return CT_CONTENT_UNKNOWN;
        }

        ext = ext.substring(p + 1).toLowerCase();

        if (ext.compareTo("html" ) == 0) { return CT_TEXT_HTML ; }
        if (ext.compareTo("htm"  ) == 0) { return CT_TEXT_HTML ; }
        if (ext.compareTo("xhtml") == 0) { return CT_TEXT_HTML ; }
        if (ext.compareTo("css"  ) == 0) { return CT_TEXT_CSS  ; }
        if (ext.compareTo("text" ) == 0) { return CT_TEXT_PLAIN; }
        if (ext.compareTo("txt"  ) == 0) { return CT_TEXT_PLAIN; }
        //### TODO mime.types 

        return CT_CONTENT_UNKNOWN;
    }
}
