/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.token.table;

import net.hizlab.kagetaka.rendering.Background;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Render;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Status;
import net.hizlab.kagetaka.token.Attribute;
import net.hizlab.kagetaka.token.EndToken;
import net.hizlab.kagetaka.token.StartToken;
import net.hizlab.kagetaka.token.Token;
import net.hizlab.kagetaka.token.TokenTypes;
import net.hizlab.kagetaka.token.Value;

import java.awt.Color;

/**
 * ơ֥ϤΥȡγϤɽ륹ѡ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
public class TableStartToken extends StartToken {
    private TableStartToken tableToken = null;

    /**
     * ơ֥ϤΥȡγϤޤ
     *
     * @param  content  ƥ
     * @param  line     ֹ (<code>1</code> )
     *                  Ǥʤ <code>0</code>
     * @param  column    (<code>1</code> )
     *                  Ǥʤ <code>0</code>
     * @param  reporter 顼ݡ
     * @param  type     ȡγϤɽ
     * @param  complete 䴰ƺ줿 <code>true</code>
     *                  ʳξ <code>false</code>
     */
    public TableStartToken(Content content, int line, int column, Reporter reporter, int type, boolean complete) {
        super(content, line, column, reporter, type, complete);
    }

    /** {@inheritDoc} */
    protected boolean setAttributeImple(String key, String value)
            throws IllegalArgumentException {
        if (attribute == null) {
            switch (type) {
            case TokenTypes.TABLE_START  : attribute = new TableAttribute  (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.CAPTION_START: attribute = new CaptionAttribute(content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.THEAD_START  :
            case TokenTypes.TBODY_START  :
            case TokenTypes.TFOOT_START  : attribute = new TbodyAttribute  (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.TR_START     : attribute = new TrAttribute     (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.TH_START     :
            case TokenTypes.TD_START     : attribute = new TdAttribute     (content, lineNumber, columnNumber, reporter); break;
            default                      : attribute = new Attribute       (content, lineNumber, columnNumber, reporter); break;
            }
        }

        return attribute.setAttribute(key, value);
    }

    /** {@inheritDoc} */
    protected StartToken getDefaultParentTokenImple() {
        int tokenType = TokenTypes.UNKNOWN;

        switch (type) {
        case TokenTypes.TABLE_START  : return new StartToken(content, lineNumber, columnNumber, reporter, TokenTypes.BODY_START, true);
        case TokenTypes.CAPTION_START: tokenType = TokenTypes.TABLE_START; break;
        case TokenTypes.THEAD_START  : tokenType = TokenTypes.TABLE_START; break;
        case TokenTypes.TBODY_START  : tokenType = TokenTypes.TABLE_START; break;
        case TokenTypes.TFOOT_START  : tokenType = TokenTypes.TABLE_START; break;
        case TokenTypes.TR_START     : tokenType = TokenTypes.TABLE_START; break;
        case TokenTypes.TH_START     : tokenType = TokenTypes.TR_START   ; break;
        case TokenTypes.TD_START     : tokenType = TokenTypes.TR_START   ; break;
        default:
            return null;
        }

        return new TableStartToken(content, lineNumber, columnNumber, reporter, tokenType, true);
    }

    /** {@inheritDoc} */
    protected EndToken getEndTokenImpl(Token current) {
        if (current == null) {
            return new TableEndToken(content, 0, 0, reporter, getType() + 1, true);
        }

        return new TableEndToken(content, current.getLineNumber(), current.getColumnNumber(), reporter, getType() + 1, true);
    }

    /** {@inheritDoc} */
    protected boolean isContentsImple(int childToken) {
        switch (type) {
        case TokenTypes.TABLE_START:
            switch (childToken) {
            case TokenTypes.CAPTION_START:
            case TokenTypes.THEAD_START  :
            case TokenTypes.TBODY_START  :
            case TokenTypes.TFOOT_START  :
            case TokenTypes.TR_START     :
                return true;
            default: // AVOID
            }
            return (TokenTypes.TABLE_START != childToken
                 && TokenTypes.isBlock(childToken));
        case TokenTypes.CAPTION_START:          // same P, but...
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));
        case TokenTypes.THEAD_START:
        case TokenTypes.TBODY_START:
        case TokenTypes.TFOOT_START:
            return (childToken == TokenTypes.TR_START
                 || (TokenTypes.TABLE_START != childToken
                  && TokenTypes.isBlock(childToken)));
        case TokenTypes.TR_START:
            return (childToken == TokenTypes.TH_START
                 || childToken == TokenTypes.TD_START
                 || (TokenTypes.TABLE_START != childToken
                  && TokenTypes.isBlock(childToken)));
        case TokenTypes.TH_START:               // same LI
        case TokenTypes.TD_START:
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));
        default: // AVOID
        }

        return false;
    }

    /** {@inheritDoc} */
    protected void includeTokenImple(Token token) {
        if (token instanceof TableStartToken) {
            ((TableStartToken) token).tableToken = (type == TokenTypes.TABLE_START ? this : tableToken);
        }
    }

    /** {@inheritDoc} */
    protected void renderImple(Render render) {
        Status status = render.getStatus();

        switch (type) {
        case TokenTypes.TABLE_START  :
            if (attribute != null) {
                TableAttribute attr = (TableAttribute) attribute;
                if (attr.getBorder() != null && attr.getBorder().intValue() > 0) {
                    status.setBorderWidth(Status.TARGET_ALL, new Value(attr.getBorder().intValue(), Value.UNIT_PX));
                    status.setBorderStyle(Status.TARGET_ALL, new Value(Value.TYPE_KEY_OUTSET));
                    status.setBorderColor(Status.TARGET_ALL, Color.gray);
                }
                status.borderVerticalSpacing = status.borderHorizontalSpacing = attr.getCellspacing();
                status.height = attr.getWidth();
                if (attr.getHspace() != null) { status.setMargin(Status.TARGET_TOP | Status.TARGET_BOTTOM, new Value(attr.getHspace().intValue(), Value.UNIT_PX)); }
                if (attr.getVspace() != null) { status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(attr.getVspace().intValue(), Value.UNIT_PX)); }
                switch (attr.getAlign()) {
                case Value.ALIGN_LEFT : status.floatType = Value.FLOAT_LEFT ; break;
                case Value.ALIGN_RIGHT: status.floatType = Value.FLOAT_RIGHT; break;
                default               : status.floatType = Value.FLOAT_NONE ; break;
                }
                if (attr.getBgcolor() != null) { status.setBackground(attr.getBgcolor()); }
            }

            status.setFontWeight(new Value(Value.NONE));
            status.whiteSpace = Value.WHITESPACE_NORMAL;
            status.align      = Value.NONE;
            status.valign     = Value.VALIGN_MIDDLE;
            status.display    = Value.DISPLAY_TABLE;
            break;
        case TokenTypes.CAPTION_START:
        {
            status.setFontWeight(new Value(Value.TYPE_KEY_NORMAL));
            status.align       = Value.ALIGN_CENTER;
            status.valign      = Value.VALIGN_BASELINE;
            status.display     = Value.DISPLAY_TABLE_CAPTION;
            status.captionSide = Value.ALIGN_TOP;
            if (attribute != null) {
                CaptionAttribute attr = (CaptionAttribute) attribute;
                if (attr.getAlign() != Value.INHERIT) { status.captionSide = attr.getAlign(); }
            }

            int target;
            switch (status.captionSide) {
            case Value.ALIGN_BOTTOM: target = Status.TARGET_RIGHT ; break;
            case Value.ALIGN_LEFT  : target = Status.TARGET_BOTTOM; break;
            case Value.ALIGN_RIGHT : target = Status.TARGET_TOP   ; break;
            default                : target = Status.TARGET_LEFT  ; break;
            }
            status.setPadding(target, new Value(0.5, Value.UNIT_EM));

            break;
        }
        case TokenTypes.THEAD_START  :
        case TokenTypes.TBODY_START  :
        case TokenTypes.TFOOT_START  :
        {
            // Ƥ°򥳥ԡ
            Status parent     = status.getParent();
            status.valign     = parent.valign;
            status.background = (parent.background != null ? (Background) parent.background.clone() : null);

            if (attribute != null) {
                TbodyAttribute attr = (TbodyAttribute) attribute;
                if (attr.getAlign  () != Value.INHERIT) { status.align  = attr.getAlign (); }
                if (attr.getVAlign () != Value.INHERIT) { status.valign = attr.getVAlign(); }
            }

            break;
        }
        case TokenTypes.TR_START     :
        {
            // Ƥ°򥳥ԡ
            Status parent     = status.getParent();
            status.valign     = parent.valign;
            status.background = (parent.background != null ? (Background) parent.background.clone() : null);

            if (attribute != null) {
                TrAttribute attr = (TrAttribute) attribute;
                if (attr.getAlign  () != Value.INHERIT) { status.align  = attr.getAlign  (); }
                if (attr.getVAlign () != Value.INHERIT) { status.valign = attr.getVAlign (); }
                if (attr.getBgcolor() != null         ) { status.setBackground(attr.getBgcolor()); }
            }

            break;
        }
        case TokenTypes.TH_START     :
        case TokenTypes.TD_START     :
        {
            int   border  = 0;
            Value padding = new Value(1, Value.UNIT_PX);

            // Ƥ°򥳥ԡ
            Status parent     = status.getParent();
            status.valign     = parent.valign;
            status.background = (parent.background != null ? (Background) parent.background.clone() : null);

            // TH/TD ѽ
            if (type == TokenTypes.TH_START) { status.setFontWeight(new Value(Value.TYPE_KEY_BOLD)); }
            if (status.align == Value.NONE) { status.align = (type == TokenTypes.TH_START ? Value.ALIGN_CENTER : Value.ALIGN_LEFT); }

            // TH/TD °
            if (attribute != null) {
                TdAttribute attr = (TdAttribute) attribute;
                if (attr.getAlign  () != Value.INHERIT) { status.align      = attr.getAlign (); }
                if (attr.getVAlign () != Value.INHERIT) { status.valign     = attr.getVAlign(); }
                if (attr.getNowrap ()                 ) { status.whiteSpace = Value.WHITESPACE_NOWRAP; }
                if (attr.getColspan() != null         ) { status.colSpan    = attr.getColspan().intValue(); }
                if (attr.getRowspan() != null         ) { status.rowSpan    = attr.getRowspan().intValue(); }
                if (attr.getWidth  () != null         ) { status.height     = attr.getWidth (); }
                if (attr.getHeight () != null         ) { status.width      = attr.getHeight(); }
                if (attr.getBgcolor() != null         ) { status.setBackground(attr.getBgcolor()); }
            }

            // TABLE °򥳥ԡ
            if (tableToken.attribute != null) {
                TableAttribute attr = (TableAttribute) tableToken.attribute;
                if (attr.getBorder     () != null) { border  = attr.getBorder().intValue(); }
                if (attr.getCellpadding() != null) { padding = attr.getCellpadding(); }
            }

            // °
            if (border > 0) {
                status.setBorderWidth(Status.TARGET_ALL, new Value(1, Value.UNIT_PX));
                status.setBorderStyle(Status.TARGET_ALL, new Value(Value.TYPE_KEY_INSET));
                status.setBorderColor(Status.TARGET_ALL, Color.gray);
            }
            status.setPadding(Status.TARGET_ALL, padding);
            status.display = Value.DISPLAY_TABLE_CELL;

            break;
        }
        default: // AVOID
        }
    }
}
