/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.token;

import net.hizlab.kagetaka.addin.StyleManager;
import net.hizlab.kagetaka.awt.ColorConverter;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Render;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Status;
import net.hizlab.kagetaka.util.TextFormat;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * ȡγϤɽ륹ѡ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
public class StartToken extends Token {
    /** Υȡ˴ޤޤƤ */
    protected Vector contents = new Vector();

    /** ȡ󥿥 */
    protected final int type;

    /** ° */
    protected Attribute attribute;
    /** ᤷʤ */
    protected String    rawContent;

    // HTML ȡѤѿ
    private boolean htmlHasHead;
    private boolean htmlHasBody;

    /**
     * ǤդΥȡγϤޤ
     *
     * @param  content  ƥ
     * @param  line     ֹ (<code>1</code> )
     *                  Ǥʤ <code>0</code>
     * @param  column    (<code>1</code> )
     *                  Ǥʤ <code>0</code>
     * @param  reporter 顼ݡ
     * @param  type     ȡγϤɽ
     * @param  complete 䴰ƺ줿 <code>true</code>
     *                  ʳξ <code>false</code>
     */
    public StartToken(Content content, int line, int column, Reporter reporter, int type, boolean complete) {
        super(content, line, column, reporter);
        this.type          = type;
        this.completeToken = complete;
    }

    /**
     * °ͤԤޤ
     *
     * @param  pairs  °̾°ͤΥڥ
     *                ¸ߤʤ <code>null</code>
     */
    public final void initAttribute(Hashtable pairs) {
        if (pairs != null) {
            String key, value, keyLC;
            for (Enumeration e = pairs.keys(); e.hasMoreElements();) {
                key   = (String) e.nextElement();
                value = (String) pairs.get(key);
                keyLC = key.toLowerCase();

                try {
                    if (!setAttributeImple(keyLC, value)) {
                        reportMessage(Reporter.WARNING,
                                      "element._.error.attribute.unknown",
                                      new String[]{getName(), key});
                    }
                } catch (IllegalArgumentException ex) {
                    reportMessage(Reporter.WARNING,
                                  "element._.error.attribute.invalid",
                                  new String[]{getName(), key, value});
                }
            }
        }
    }

    /**
     * ꤵ줿°̾°ͤǡ°ꤷޤ
     * °̾¸ߤ <code>true</code> 
     * ¸ߤʤ <code>false</code> ֤ޤ
     * Ϥ <code>key</code> ϡɬʸѴƤޤ
     *
     * @param  key °̾
     * @param  value °
     *
     * @return °̾¸ߤ <code>true</code>
     *         ʳξ <code>false</code>
     *
     * @throws IllegalArgumentException °ͤξ
     */
    protected boolean setAttributeImple(String key, String value)
            throws IllegalArgumentException {
        // 
        if (attribute == null) {
            switch (type) {
            // header
            case TokenTypes.BASE         : attribute = new BaseAttribute (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.SCRIPT_START : return true;          // ̵
            case TokenTypes.META         : attribute = new MetaAttribute (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.LINK         : attribute = new LinkAttribute (content, lineNumber, columnNumber, reporter); break;
            // body
            case TokenTypes.BODY_START   : attribute = new BodyAttribute (content, lineNumber, columnNumber, reporter); break;
            // block
            case TokenTypes.P_START      : attribute = new PAttribute    (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.H1_START     :
            case TokenTypes.H2_START     :
            case TokenTypes.H3_START     :
            case TokenTypes.H4_START     :
            case TokenTypes.H5_START     :
            case TokenTypes.H6_START     : attribute = new HAttribute    (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.UL_START     : attribute = new LAttribute    (content, lineNumber, columnNumber, reporter, LAttribute.TYPE_UL  ); break;
            case TokenTypes.OL_START     : attribute = new LAttribute    (content, lineNumber, columnNumber, reporter, LAttribute.TYPE_OL  ); break;
            case TokenTypes.DIR_START    : attribute = new LAttribute    (content, lineNumber, columnNumber, reporter, LAttribute.TYPE_DIR ); break;
            case TokenTypes.MENU_START   : attribute = new LAttribute    (content, lineNumber, columnNumber, reporter, LAttribute.TYPE_MENU); break;
            case TokenTypes.DIV_START    : attribute = new DivAttribute  (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.HR           : attribute = new HrAttribute   (content, lineNumber, columnNumber, reporter); break;
            // inline
            case TokenTypes.A_START      : attribute = new AAttribute    (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.IMG          : attribute = new ImgAttribute  (content, lineNumber, columnNumber, reporter); break;
            case TokenTypes.FONT_START   : attribute = new FontAttribute (content, lineNumber, columnNumber, reporter); break;
            default:
                if (TokenTypes.isBody(type)) {
                    attribute = new Attribute(content, lineNumber, columnNumber, reporter);
                } else {
                    // °ɬפʤȡξ
                    return false;
                }
            }
        }

        return attribute.setAttribute(key, value);
    }

    /** {@inheritDoc} */
    public String getName() {
        return TokenTypes.getName(type);
    }

    /** {@inheritDoc} */
    public int getType() {
        return type;
    }

    /**
     * Υȡ󤬽°ǥեȤοƥȡ֤ޤ
     * ƥȡȤƹͤȡʣ¸ߤϡ
     * äȤŬڤʥȡ֤ɬפޤ
     * <p>
     * ƥȡ󤬾άƤϡ֤줿ϥȡ
     * ƥȡȤѤΤǡΥ᥽åɤƤӽФӤˡ
     * ̡Υ󥹥󥹤֤ޤ
     *
     * @return ǥեȤοƥȡ
     *         Ǿ̤ΥȡΤ <code>null</code>
     */
    public final StartToken getDefaultParentToken() {
        StartToken token = getDefaultParentTokenImple();

        if (token != null) {
            return token;
        }

        if (type == TokenTypes.HTML_START) {
            return null;
        }

        reportMessage(Reporter.ERROR,
                      "internal.error",
                      new String[]{"StartToken.getDefaultParentToken", "undefine token type", getName()});
        return null;
    }

    /**
     * Υȡ󤬽°ǥեȤοƥȡ󥿥פ֤ޤ
     * ƥȡȤƹͤȡʣ¸ߤϡ
     * äȤŬڤʥȡ֤ɬפޤ
     * <p>
     * ƥȡ󤬾άƤϡ֤줿ϥȡ
     * ƥȡȤѤΤǡΥ᥽åɤƤӽФӤˡ
     * ̡Υ󥹥󥹤֤ɬפޤ
     *
     * @return ǥեȤοƥȡ
     *         ǥեȤǤʤ <code>null</code>
     */
    protected StartToken getDefaultParentTokenImple() {
        int tokenType = TokenTypes.UNKNOWN;

        switch (type) {
        case TokenTypes.HTML_START      : return null;
        // header
        case TokenTypes.HEAD_START      : tokenType = TokenTypes.HTML_START    ; break;

        case TokenTypes.TITLE_START     : tokenType = TokenTypes.HEAD_START    ; break;
        case TokenTypes.BASE            : tokenType = TokenTypes.HEAD_START    ; break;
        case TokenTypes.SCRIPT_START    : tokenType = TokenTypes.HEAD_START    ; break;
        case TokenTypes.META            : tokenType = TokenTypes.HEAD_START    ; break;
        case TokenTypes.LINK            : tokenType = TokenTypes.HEAD_START    ; break;
        // body
        case TokenTypes.BODY_START      : tokenType = TokenTypes.HTML_START    ; break;
        // block
        case TokenTypes.P_START         : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H1_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H2_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H3_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H4_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H5_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.H6_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.UL_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.OL_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.DIR_START       : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.MENU_START      : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.LI_START        : tokenType = TokenTypes.X_BLOCK_START ; break; //  UL_START 
        case TokenTypes.DL_START        : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.DT_START        : tokenType = TokenTypes.X_BLOCK_START ; break; //  DL_START 
        case TokenTypes.DD_START        : tokenType = TokenTypes.X_BLOCK_START ; break; //  DL_START 
        case TokenTypes.PRE_START       : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.DIV_START       : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.NOSCRIPT_START  : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.BLOCKQUOTE_START: tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.HR              : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.ADDRESS_START   : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.CENTER_START    : tokenType = TokenTypes.BODY_START    ; break;
        case TokenTypes.X_BLOCK_START   : tokenType = TokenTypes.BODY_START    ; break;
        // inline
        case TokenTypes.A_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.IMG             : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.BR              : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.SPAN_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.EM_START        : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.STRONG_START    : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.DFN_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.CODE_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.Q_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.SUB_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.SUP_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.SAMP_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.KBD_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.VAR_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.CITE_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.ABBR_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.ACRONYM_START   : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.TT_START        : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.I_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.B_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.BIG_START       : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.SMALL_START     : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.U_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.S_START         : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.STRIKE_START    : tokenType = TokenTypes.X_BLOCK_START ; break;
        case TokenTypes.FONT_START      : tokenType = TokenTypes.X_BLOCK_START ; break;
        default:
            return null;
        }

        return new StartToken(content, lineNumber, columnNumber, reporter, tokenType, true);
    }

    /**
     * γϥȡб뽪λȡ֤ޤ
     * Υȡ󤬶ȡξϡ<code>null</code> ֤ޤ
     * <p>
     * λȡ󤬾άƤˡ֤줿λȡ
     * λȡȤѤޤ
     * Υ᥽åɤϸƤӽФ뤿Ӥˡ̡Υ󥹥󥹤֤ޤ
     *
     * @param  current ߤΥ֤Υȡ
     *                 ֤ξ <code>null</code>
     *
     * @return б뽪λȡ󡢶ȡξ <code>null</code>
     */
    public final EndToken getEndToken(Token current) {
        if (TokenTypes.isEmpty(type)) {
            return null;
        }

        return getEndTokenImpl(current);
    }

    /**
     * γϥȡб뽪λȡ֤ޤ
     * Υ᥽åɤϸƤӽФ뤿Ӥˡ̡Υ󥹥󥹤֤ޤ
     * ȡξϸƤӽФޤ
     *
     * @param  current ߤΥ֤Υȡ
     *                 ֤ξ <code>null</code>
     *
     * @return б뽪λȡ
     */
    protected EndToken getEndTokenImpl(Token current) {
        if (current == null) {
            return new EndToken(content, 0, 0, reporter, getType() + 1, true);
        }

        return new EndToken(content, current.lineNumber, current.columnNumber, reporter, getType() + 1, true);
    }

    /**
     * γϥȡб뽪λȡΡȡ󥿥פ֤ޤ
     * Υȡ󤬶ȡξϡ{@link TokenTypes#UNKNOWN} 
     * ֤ޤ
     * <p>
     * ֤뽪λȡΥפϡλȡ
     * άƤΥåѤޤ
     *
     * @return б뽪λȡΥס
     *         ȡξ {@link TokenTypes#UNKNOWN}
     */
    public int getEndTokenType() {
        if (TokenTypes.isEmpty(type)) {
            return TokenTypes.UNKNOWN;
        }
        return getType() + 1;
    }

    /**
     * ꤵ줿ȡޤळȤ뤫ƥȤޤ
     * <code>token</code> ˤäƻꤵ줿ȡ
     * Υȡ󤬴ޤळȤǽξ <code>true</code> 
     * Բǽʾ <code>false</code> ֤ޤ
     * <p>
     * 㤨СʸˡޤळȤʤȡꤵ줿䡢
     * ĤǤޤळȤʤȡǡ
     * ˥ȡޤǤʤɤ <code>false</code> ֤ޤ
     *
     * @param  token ƥоݤΥȡ
     *
     * @return ޤळȤǽʾ <code>true</code>
     *         Ǥʤ <code>false</code>
     *
     * @see    #includeToken(Token)
     */
    public final boolean isContents(StartToken token) {
        int childToken = token.getType();

        // ɤˤǤ⸽륿
        switch (childToken) {
        case TokenTypes.SCRIPT_START:
        case TokenTypes.META        :
            return true;
        default: // AVOID
        }

        //### BUGS եϤɤˤǤ֤ʻ
        if (TokenTypes.isForm(childToken)
                && TokenTypes.isBody(type)
                && type       != TokenTypes.BODY_START
                && childToken != TokenTypes.OPTGROUP_START
                && childToken != TokenTypes.OPTION_START) {
            return true;
        }

        return isContentsImple(childToken);
    }

    /**
     * ꤵ줿ȡޤळȤ뤫ƥȤޤ
     * <code>token</code> ˤäƻꤵ줿ȡ
     * Υȡ󤬴ޤळȤǽξ <code>true</code> 
     * Բǽʾ <code>false</code> ֤ޤ
     * <p>
     * 㤨СʸˡޤळȤʤȡꤵ줿䡢
     * ĤǤޤळȤʤȡǡ
     * ˥ȡޤǤʤɤ <code>false</code> ֤ޤ
     *
     * @param  childToken ƥоݤΥȡ󥿥
     *
     * @return ޤळȤǽʾ <code>true</code>
     *         Ǥʤ <code>false</code>
     *
     * @see    #includeToken(Token)
     * @see    #isContents(StartToken)
     * @see    #isContentsParent(int)
     */
    protected boolean isContentsImple(int childToken) {
        switch (type) {
        case TokenTypes.HTML_START:
            switch (childToken) {
            case TokenTypes.HEAD_START:
                if (htmlHasHead) { return false; }
                htmlHasHead = true;
                return true;
            case TokenTypes.BODY_START:
                if (htmlHasBody) { return false; }
                htmlHasHead = true;
                htmlHasBody = true;
                return true;
            case TokenTypes.FRAMESET_START:
                return true;
            case TokenTypes.NOFRAMES_START:
                return true;
            default: // AVOID
            }
            return false;

        // header
        case TokenTypes.HEAD_START:
            switch (childToken) {
            case TokenTypes.TITLE_START :
            case TokenTypes.BASE        :
            case TokenTypes.SCRIPT_START:
            case TokenTypes.STYLE_START :
            case TokenTypes.META        :
            case TokenTypes.LINK        :
                return true;
            default: // AVOID
            }
            return false;

        case TokenTypes.TITLE_START :
        case TokenTypes.BASE        :
            return false;
        case TokenTypes.SCRIPT_START:
            return true;
        case TokenTypes.META        :
        case TokenTypes.LINK        :
            return false;

        // body
        case TokenTypes.BODY_START:
            switch (childToken) {
            case TokenTypes.LI_START        :
            case TokenTypes.DT_START        :
            case TokenTypes.DD_START        :
                return false;
            default: // AVOID
            }
            return TokenTypes.isBlock(childToken);

        // block
        case TokenTypes.P_START      :
            return (childToken != TokenTypes.P_START
                 && (TokenTypes.isBlock (childToken)
                  || TokenTypes.isInline(childToken)));

        case TokenTypes.H1_START     :
        case TokenTypes.H2_START     :
        case TokenTypes.H3_START     :
        case TokenTypes.H4_START     :
        case TokenTypes.H5_START     :
        case TokenTypes.H6_START     :
        case TokenTypes.PRE_START    :
        case TokenTypes.ADDRESS_START:
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));

        case TokenTypes.UL_START  :
        case TokenTypes.OL_START  :
        case TokenTypes.DIR_START :
        case TokenTypes.MENU_START:
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));

        case TokenTypes.LI_START :
            return ((TokenTypes.isBlock (childToken) && (childToken != TokenTypes.LI_START))
                 || (TokenTypes.isInline(childToken)));

        case TokenTypes.DL_START:
            return (childToken == TokenTypes.DT_START || childToken == TokenTypes.DD_START);

        case TokenTypes.DT_START:
            return TokenTypes.isInline(childToken);

        case TokenTypes.DD_START:
            return ((TokenTypes.isBlock (childToken)
                  && (childToken != TokenTypes.DT_START)
                  && (childToken != TokenTypes.DD_START))
                 || (TokenTypes.isInline(childToken)));

        case TokenTypes.DIV_START       :       // ϤʤɤäѤ
        case TokenTypes.NOSCRIPT_START  :
        case TokenTypes.BLOCKQUOTE_START:
        case TokenTypes.CENTER_START    :
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));

        case TokenTypes.HR   :
            return false;

        case TokenTypes.X_BLOCK_START   :
            return (TokenTypes.isInline(childToken)
                 || childToken == TokenTypes.LI_START
                 || childToken == TokenTypes.DT_START
                 || childToken == TokenTypes.DD_START);

        // inline
        case TokenTypes.A_START   :
            return TokenTypes.isInline(childToken) && (childToken != TokenTypes.A_START);

        case TokenTypes.SPAN_START   :
        case TokenTypes.EM_START     :
        case TokenTypes.STRONG_START :
        case TokenTypes.DFN_START    :
        case TokenTypes.CODE_START   :
        case TokenTypes.Q_START      :
        case TokenTypes.SUB_START    :
        case TokenTypes.SUP_START    :
        case TokenTypes.SAMP_START   :
        case TokenTypes.KBD_START    :
        case TokenTypes.VAR_START    :
        case TokenTypes.CITE_START   :
        case TokenTypes.ABBR_START   :
        case TokenTypes.ACRONYM_START:
        case TokenTypes.TT_START     :
        case TokenTypes.I_START      :
        case TokenTypes.B_START      :
        case TokenTypes.BIG_START    :
        case TokenTypes.SMALL_START  :
        case TokenTypes.U_START      :
        case TokenTypes.S_START      :
        case TokenTypes.STRIKE_START :
        case TokenTypes.FONT_START   :
            // Ϥʤɡ
            return (TokenTypes.isBlock (childToken)
                 || TokenTypes.isInline(childToken));

        case TokenTypes.IMG:
        case TokenTypes.BR :
            return false;

        default:
            reportMessage(Reporter.ERROR,
                          "internal.error",
                          new String[]{"StartToken.isContents", "undefine token type", getName()});
            return false;
        }
    }

    /**
     * ꤵ줿ȡޤळȤ뤫ƤǥƥȤޤ
     *
     * @param  childToken ޤȡ
     *
     * @return ޤळȤǽʾ <code>true</code>
     *         Ǥʤ <code>false</code>
     *
     * @see    #includeToken(Token)
     * @see    #isContents(StartToken)
     * @see    #isContentsImple(int)
     */
    protected final boolean isContentsParent(int childToken) {
        StartToken parent = getParent();
        if (parent != null) {
            return parent.isContentsImple(childToken);
        }
        return false;
    }

    /**
     * ꤵ줿ȡ󤬡ǤȤɲä줿˸ƤӽФޤ
     * Υ᥽åɤˤɲäǤϡľܤλǤǡ
     * ¹ǤʤɤξϸƤӽФޤ
     * <p>
     * Υ᥽åɤˤɲä줿Ǥϡ{@link #contents} ˳Ǽޤ
     * ͤ򸫤 {@link #isContents(StartToken)} ᥽åɤʤɤ
     * ŬڤȽꤹ뤳ȤǽǤ
     *
     * @param  token ɲä줿
     *
     * @see    #isContents(StartToken)
     */
    public final void includeToken(Token token) {
        contents.addElement(token);
        token.setParent(this);

        includeTokenImple(token);
    }

    /**
     * ꤵ줿ȡ󤬡ǤȤɲä줿˸ƤӽФ
     * ޤ
     * Υ᥽åɤˤɲäǤϡľܤλǤǡ
     * ¹ǤʤɤξϸƤӽФޤ
     *
     * @param  token ɲä줿
     *
     * @see    #includeToken(Token)
     * @see    #includeTokenParent(Token)
     */
    protected void includeTokenImple(Token token) {
    }

    /**
     * ꤵ줿ȡ󤬡ǤȤɲä줿ˡƤΤޤ
     * ʬ¸ߤ̵뤷Ƥ˽ޤ˸ƤӽФޤ
     *
     * @param  token ɲä줿
     *
     * @see    #includeToken(Token)
     * @see    #includeTokenImple(Token)
     */
    protected final void includeTokenParent(Token token) {
        StartToken parent = getParent();
        if (parent != null) {
            parent.includeTokenImple(token);
        }
    }

    /**
     * ΥȡƱ٥ˤΥȡ֤ޤ
     *
     * @return Ʊ٥Υȡ̵ <code>null</code>
     *
     * @kagetaka.memo ˤ˸ƤӽФ褦ʤС
     *                ⤦ᤤˡƤɬפȻפ
     */
    public StartToken getBeforeStartToken() {
        StartToken parent = getParent();
        if (parent == null) {
            return null;
        }

        int p = parent.contents.indexOf(this);
        if (p <= 0) {
            return null;
        }

        Object o;
        while (--p >= 0) {
            if ((o = parent.contents.elementAt(p)) instanceof StartToken) {
                return (StartToken) o;
            }
        }

        return null;
    }

    /**
     * ƤᤷƤϤʤȡȤ֤ޤ
     *
     * @return б뽪λȡޤǤΥǡ
     */
    public String getContent() {
        return rawContent;
    }

    /**
     * ƤᤷƤϤʤȡȤꤷޤ
     *
     * @param  content б뽪λȡޤǤΥǡ
     */
    public void setContent(String content) {
        this.rawContent = content;
    }

    /**
     * Υȡʸɽ֤ޤ
     * <p>
     * <code>StartToken</code> 饹 <code>toString</code> ᥽åɤϡ
     * {@link #getName()} ᥽åɤˤä֤ȡ̾ &lt; &gt;
     * ǰϤ͡åȥޡ (@)ӥ֥ȤΥϥå女ɤ
     * ʤ 16 ɽ鹽ʸ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("<");
        sb.append(getName());
        if (TokenTypes.isEmpty(type)) {
            sb.append(" /");
        }
        sb.append(':');
        sb.append(lineNumber);
        sb.append(':');
        sb.append(columnNumber);
        sb.append('>');
        return sb.toString();
    }

    /**
     * ƥȡ°֤ޤ
     *
     * @return °ꤵƤʤ䡢°ɬפʤ
     *         <code>null</code>
     */
    public Attribute getAttribute() {
        return attribute;
    }

    /**
     * γϥȡΥפ°ͤ򸵤ˡԤޤ
     *
     * @param  render 
     * @param  pseudo 饹
     */
    public final void render(Render render, int pseudo) {
        Status  status   = render.getStatus();
        boolean hasStyle = false;

        if (pseudo == Render.PSEUDO_NONE) {
            renderImple(render);

            if (attribute != null) {
                // ID ޡ֤ȤϿ
                String id = attribute.getBaseId();
                if (id != null) {
                    render.setMark(id);
                }
                // 
                hasStyle = (attribute.getStyle() != null);
            }
        }

        // Ŭ
        StyleManager sm = render.getStyleManager(hasStyle);
        if (sm != null) {
            sm.changeStatus(render.media, this, status, pseudo);
        }
    }

    /**
     * γϥȡΥפ°ͤ򸵤ˡԤޤ
     *
     * @param  render 
     */
    protected void renderImple(Render render) {
        Status status = render.getStatus();

        switch (type) {
        case TokenTypes.HTML_START      : break;               // ƤФʤ
        // header
        case TokenTypes.HEAD_START      : break;               // ƤФʤ

        case TokenTypes.TITLE_START     :
            render.setTitle(TextFormat.convertXhtml(rawContent, true, true, true, true));
            break;
        case TokenTypes.BASE            :
            if (attribute != null) {
                BaseAttribute attr = (BaseAttribute) attribute;
                render.setBase(attr.getHref(), attr.getTarget());
            }
            break;
        case TokenTypes.SCRIPT_START    : break;
        case TokenTypes.META            :
            if (attribute != null) {
                MetaAttribute attr = (MetaAttribute) attribute;
                String he      = attr.getHttpEquiv();
                String content = attr.getContent  ();
                if (he != null && content != null) {
                    he = he.trim().toLowerCase();

                    // եå
                    if (he.compareTo("refresh") == 0) {
                        int p = content.indexOf(';');
                        String time = content;
                        String href = null;
                        if (p != -1) {
                            time = content.substring(0, p);
                            href = content.substring(p + 1).trim();
                            if (href.toLowerCase().startsWith("url=")) {
                                href = href.substring(4).trim();
                            }
                        }
                        // äȤøɡʸäƾ˲Ȥб
                        render.setRefresh(Integer.parseInt("0" + time), href);
                    } else

                    // å
                    if (he.compareTo("set-cookie") == 0) {
                        render.setCookie(content);
                    }
                }
            }
            break;
        case TokenTypes.LINK            :
            if (attribute != null) {
                LinkAttribute attr = (LinkAttribute) attribute;
                String rel = attr.getRel();
                if (rel != null && rel.toLowerCase().compareTo("stylesheet") == 0) {
                    render.setStyleSheet(attr.getHref(), attr.getType());
                }
            }
            break;
        // body
        case TokenTypes.BODY_START      :
            if (attribute != null) {
                BodyAttribute attr = (BodyAttribute) attribute;
                status.setBackground(attr.getBackground(), null, null, null);
                if (attr.getBgcolor() != null) { status.setBackground(attr.getBgcolor()); }
                if (attr.getText   () != null) { status.foreColor  = attr.getText (); }
                if (attr.getLink   () != null) { status.linkColor  = attr.getLink (); }
                if (attr.getVLink  () != null) { status.vlinkColor = attr.getVLink(); }
                if (attr.getALink  () != null) { status.alinkColor = attr.getALink(); }
            }
            status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));
            status.lineHeight    = new Value(Value.TYPE_KEY_NORMAL);
//            status.letterSpacing = new Value(1  , Value.UNIT_PX);
            break;
        // block
        case TokenTypes.P_START         :
            if (attribute != null) {
                status.align = ((PAttribute) attribute).getAlign();
            }
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(1, Value.UNIT_EM));
//            status.whiteSpace = Value.WHITESPACE_NORMAL;
            break;
        case TokenTypes.H1_START        :
        case TokenTypes.H2_START        :
        case TokenTypes.H3_START        :
        case TokenTypes.H4_START        :
        case TokenTypes.H5_START        :
        case TokenTypes.H6_START        :
            switch (type) {
            case TokenTypes.H1_START: status.setFontWeight(new Value(Value.TYPE_KEY_BOLD  )); status.setFontSize(new Value(207, Value.UNIT_PERCENT), Status.TOKEN); break;
            case TokenTypes.H2_START: status.setFontWeight(new Value(Value.TYPE_KEY_BOLD  )); status.setFontSize(new Value(173, Value.UNIT_PERCENT), Status.TOKEN); break;
            case TokenTypes.H3_START: status.setFontWeight(new Value(Value.TYPE_KEY_BOLD  )); status.setFontSize(new Value(144, Value.UNIT_PERCENT), Status.TOKEN); break;
            case TokenTypes.H4_START: status.setFontWeight(new Value(Value.TYPE_KEY_BOLD  )); status.setFontSize(new Value(120, Value.UNIT_PERCENT), Status.TOKEN); break;
            case TokenTypes.H5_START: status.setFontWeight(new Value(Value.TYPE_KEY_BOLD  )); break;
            case TokenTypes.H6_START: status.setFontStyle (new Value(Value.TYPE_KEY_ITALIC)); break;
            default: // AVOID
            }
            if (attribute != null) {
                status.align = ((HAttribute) attribute).getAlign();
            }
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(0.5, Value.UNIT_EM));
            break;
        case TokenTypes.UL_START        :
        case TokenTypes.OL_START        :
        case TokenTypes.DIR_START       :
        case TokenTypes.MENU_START      :
        {
            LAttribute attr = (LAttribute) attribute;
            status.setList((attr != null)
                           ? attr.getType()
                           : ((type == TokenTypes.OL_START)
                              ? Value.LIST_DECIMAL
                              : Value.LIST_MIX    ));
            if (attr != null && attr.getStart() != null) {
                status.counterNo = attr.getStart().intValue();
            }
            status.setMargin(Status.TARGET_TOP, new Value(40, Value.UNIT_PX));
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(1.12, Value.UNIT_EM));

            break;
        }
        case TokenTypes.LI_START        :
            status.hasMarkerBlock = true;
            break;
        case TokenTypes.DL_START        :
            break;
        case TokenTypes.DT_START        :
            status.setFontWeight(new Value(Value.TYPE_KEY_BOLD));
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(2, Value.UNIT_PX));
            break;
        case TokenTypes.DD_START        :
            status.setMargin(Status.TARGET_TOP, new Value(40, Value.UNIT_PX));
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(2, Value.UNIT_PX));
            break;
        case TokenTypes.PRE_START       :
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(1, Value.UNIT_EM));
            status.whiteSpace = Value.WHITESPACE_PRE;
            status.setFontFamily(new Value[]{new Value(Value.FONT_MONOSPACE)});
            break;
        case TokenTypes.DIV_START       :
            if (attribute != null) {
                status.align = ((DivAttribute) attribute).getAlign();
            }
            break;
        case TokenTypes.NOSCRIPT_START  :
            break;
        case TokenTypes.BLOCKQUOTE_START:
            status.setMargin(Status.TARGET_TOP                       , new Value(20, Value.UNIT_PX));
            status.setMargin(Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value( 5, Value.UNIT_PX));
            break;
        case TokenTypes.HR              :
        {
            int     size  = 3;
            boolean shade = true;

            status.isHorizontalRule = true;
            status.height = new Value(100, Value.UNIT_PERCENT);
            if (status.align == Value.NONE) {
                status.align = Value.ALIGN_CENTER;
            }

            if (attribute != null) {
                HrAttribute attr = (HrAttribute) attribute;
                size  = Math.max((attr.getSize() != null ? attr.getSize().intValue() : size), 2);
                shade = !attr.getNoshade();
                if (attr.getAlign() != Value.INHERIT) { status.align  = attr.getAlign(); }
                if (attr.getWidth() != null         ) { status.height = attr.getWidth(); }
            }
            status.width = new Value(size - 2, Value.UNIT_PX);

            if (shade) {
                status.setBorderStyle(Status.TARGET_ALL, new Value(Value.TYPE_KEY_INSET));
            }
            status.setBorderWidth(Status.TARGET_ALL, new Value(1, Value.UNIT_PX));
            status.setBorderColor(Status.TARGET_ALL, ColorConverter.getDarker(status.foreColor));
            status.setMargin     (Status.TARGET_RIGHT | Status.TARGET_LEFT, new Value(0.7, Value.UNIT_EM));

            break;
        }
        case TokenTypes.ADDRESS_START   :
            break;
        case TokenTypes.CENTER_START    :
            status.align = Value.ALIGN_CENTER;
            break;
        case TokenTypes.X_BLOCK_START   :
            break;
        // inline
        case TokenTypes.A_START         :
            if (attribute != null) {
                AAttribute attr = (AAttribute) attribute;
                render.setLink(attr.getHref(), attr.getTarget());
                render.setMark(attr.getName());
                if (attr.getHref() != null) {
                    status.setTextDecoration(Value.TD_UNDERLINE);
                }
            }
            break;
        case TokenTypes.IMG             :
            if (attribute != null) {
                ImgAttribute attr = (ImgAttribute) attribute;
                int floatType = Value.FLOAT_NONE;
                switch (attr.getAlign()) {
                case Value.ALIGN_LEFT : floatType = Value.FLOAT_LEFT ; break;
                case Value.ALIGN_RIGHT: floatType = Value.FLOAT_RIGHT; break;
                default: // AVOID
                }
                render.drawImage(attr.getSrc(), attr.getAlt(), attr.getWidth(), attr.getHeight(), attr.getBorder(), floatType);
            }
            break;
        case TokenTypes.BR              :
            render.doBr();
            break;
        case TokenTypes.SPAN_START      : break;
        case TokenTypes.EM_START        :
            status.setFontStyle(new Value(Value.TYPE_KEY_ITALIC));
            break;
        case TokenTypes.STRONG_START    :
            status.setFontWeight(new Value(Value.TYPE_KEY_BOLD));
            break;
        case TokenTypes.DFN_START       :
            status.setFontStyle(new Value(Value.TYPE_KEY_ITALIC));
            break;
        case TokenTypes.CODE_START      :
            status.setFontFamily(new Value[]{new Value(Value.FONT_MONOSPACE)});
            break;
        case TokenTypes.Q_START         :
            render.drawText("\"");
            break;
        case TokenTypes.SUB_START       :
            status.setFontSize(new Value(70, Value.UNIT_PERCENT), Status.TOKEN);
            status.offsetX = new Value(-50, Value.UNIT_PERCENT);
            break;
        case TokenTypes.SUP_START       :
            status.setFontSize(new Value(70, Value.UNIT_PERCENT), Status.TOKEN);
            status.offsetX = new Value( 50, Value.UNIT_PERCENT);
            break;
        case TokenTypes.SAMP_START      :
            status.setFontFamily(new Value[]{new Value(Value.FONT_MONOSPACE)});
            break;
        case TokenTypes.KBD_START       : break;
        case TokenTypes.VAR_START       :
            status.setFontStyle(new Value(Value.TYPE_KEY_ITALIC));
            break;
        case TokenTypes.CITE_START      :
            status.setFontStyle(new Value(Value.TYPE_KEY_ITALIC));
            break;
        case TokenTypes.ABBR_START      :
        case TokenTypes.ACRONYM_START   :
            if (attribute != null && attribute.getBaseTitle() != null) {
                render.setPopup(attribute.getBaseTitle());
                //### BUGS ۥȤ border-bottom 
                status.setTextDecoration(Value.TD_UNDERLINE);
            }
            break;
        case TokenTypes.TT_START        :
            status.setFontFamily(new Value[]{new Value(Value.FONT_MONOSPACE)});
            break;
        case TokenTypes.I_START         :
            status.setFontStyle(new Value(Value.TYPE_KEY_ITALIC));
            break;
        case TokenTypes.B_START         :
            status.setFontWeight(new Value(Value.TYPE_KEY_BOLD));
            break;
        case TokenTypes.BIG_START       :
            status.setFontSize(new Value(Value.TYPE_KEY_LARGER), Status.TOKEN);
            break;
        case TokenTypes.SMALL_START     :
            status.setFontSize(new Value(Value.TYPE_KEY_SMALLER), Status.TOKEN);
            break;
        case TokenTypes.U_START         :
            status.setTextDecoration(Value.TD_UNDERLINE);
            break;
        case TokenTypes.S_START         :
        case TokenTypes.STRIKE_START    :
            status.setTextDecoration(Value.TD_STRIKE);
            break;
        case TokenTypes.FONT_START      :
            if (attribute != null) {
                FontAttribute attr = (FontAttribute) attribute;
                if (attr.getColor() != null) { status.foreColor = attr.getColor(); }
                Integer fontSize = attr.getSize(3); // BASEFONT 򥵥ݡȤʤΤǡ 3
                if (fontSize != null) {
                    int size = fontSize.intValue();
                    if (size != 3) {
                        status.setFontSize(new Value(Value.TYPE_KEY_MEDIUM), Status.TOKEN);
                        if (fontSize.intValue() < 3) {
                            Value v = new Value(Value.TYPE_KEY_SMALLER);
                            for (int i = fontSize.intValue(); i < 3; i++) {
                                status.setFontSize(v, Status.TOKEN);
                            }
                        } else if (fontSize.intValue() > 3) {
                            Value v = new Value(Value.TYPE_KEY_LARGER);
                            for (int i = fontSize.intValue(); i > 3; i--) {
                                status.setFontSize(v, Status.TOKEN);
                            }
                        }
                    }
                }

                String fontFace = attr.getFace();
                if (fontFace != null && fontFace.length() > 0) {
                    StringTokenizer st = new StringTokenizer(fontFace, ",");
                    Value[]         vs = new Value[st.countTokens()];
                    int index = 0;
                    String name, nameLC;
                    Value  v;
                    while (st.hasMoreTokens()) {
                        name   = st.nextToken().trim();
                        nameLC = name.toLowerCase();
                        if (nameLC.compareTo("serif"     ) == 0) { v = new Value(Value.FONT_SERIF     ); } else
                        if (nameLC.compareTo("sans-serif") == 0) { v = new Value(Value.FONT_SANS_SERIF); } else
                        if (nameLC.compareTo("cursive"   ) == 0) { v = new Value(Value.FONT_CURSIVE   ); } else
                        if (nameLC.compareTo("fantasy"   ) == 0) { v = new Value(Value.FONT_FANTASY   ); } else
                        if (nameLC.compareTo("monospace" ) == 0) { v = new Value(Value.FONT_MONOSPACE ); } else
                        {
                            v = new Value(name);
                        }
                        vs[index++] = v;
                    }
                    status.setFontFamily(vs);
                }
            }
            break;
        default: // AVOID
        }
    }
}
