/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.token;

import java.awt.Color;
import java.awt.Font;
import java.awt.SystemColor;
import java.text.DecimalFormat;
import java.text.ParsePosition;
import java.util.Vector;

/**
 * °ͤʸ󤫤Ŭڤʥ󥹥󥹤뤿Υ桼ƥƥǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.2 $
 */
public final class AttributeConverter {
    /**
     * ͣΥ󥹥ȥ饯Ǥ󥹥󥹤κϽޤ
     */
    private AttributeConverter() { }

    /**
     * ɽʸ󤫤顢{@link java.awt.Color} Υ󥹥󥹤ޤ
     * <p>
     * ʸΥեޥåȤϡʲ̤Ǥ
     * <pre>
     * #f00     (#rgb    - 16 ʿ)
     * #ff0000  (#rrggbb - 16 ʿ)
     * red      (̾)
     * </pre>
     * 16 ʿ 0-9a-f ޤǤʸǡʸʸ϶̤ޤ
     * rgb  3 ηϡºݤˤ 6 ηѴޤ
     * Ѵϡ0 դäΤǤϤʤƷͤ򷫤֤Ȥˤꡢ
     * 6 Ѵޤ㤨С#fb0פȤͤϡ#ffbb00פѴޤ
     * <p>
     * ݡȤƤ뿧̾ϡ<code>aqua, black, blue, fuchsia, gray, green,
     * lime, maroon, navy, olive, purple, red, silver, teal, white, yellow</code>
     *  16 ǡϡHTML 4.0 Ƥ뿧ƱǤ
     *
     * @param  value ɽʸ
     *
     * @return Υ󥹥
     *
     * @throws IllegalArgumentException ͤξ
     *
     * @see    <a href="http://www.w3.org/TR/html401/types.html#h-6.5">HTML 4.01 Specification: 6.5 Colors</a>
     */
    public static Color convertToColor(String value)
            throws IllegalArgumentException {
        Color color = convertToColorSilent(value);
        if (color == null) {
            throw new IllegalArgumentException();
        }

        return color;
    }

    /**
     * ɽʸ󤫤顢{@link java.awt.Color} Υ󥹥󥹤ޤ
     * <p>
     * {@link #convertToColor(String)} Ȥΰ㤤ϡͤѴ褦
     * ˡ㳰ȯˡ<code>null</code> ֤Ǥ
     *
     * @param  value ɽʸ
     *
     * @return Υ󥹥󥹡ͤξ <code>null</code>
     *
     * @see    #convertToColor(String)
     */
    public static Color convertToColorSilent(String value) {
        int length  = value.length();
        int current = 0;

        if (length == 0) {
            return null;
        }

        if (value.charAt(0) == '#') {
            // rgb ɽ
            length--;
            current = 1;
        } else {
            // ̾
            value = value.toLowerCase();
            // ɸ࿧
            if ("black"  .compareTo(value) == 0) { return new Color(0x000000); }
            if ("silver" .compareTo(value) == 0) { return new Color(0xc0c0c0); }
            if ("gray"   .compareTo(value) == 0) { return new Color(0x808080); }
            if ("white"  .compareTo(value) == 0) { return new Color(0xffffff); }
            if ("maroon" .compareTo(value) == 0) { return new Color(0x800000); }
            if ("red"    .compareTo(value) == 0) { return new Color(0xff0000); }
            if ("purple" .compareTo(value) == 0) { return new Color(0x800080); }
            if ("fuchsia".compareTo(value) == 0) { return new Color(0xff00ff); }
            if ("green"  .compareTo(value) == 0) { return new Color(0x008000); }
            if ("lime"   .compareTo(value) == 0) { return new Color(0x00ff00); }
            if ("olive"  .compareTo(value) == 0) { return new Color(0x808000); }
            if ("yellow" .compareTo(value) == 0) { return new Color(0xffff00); }
            if ("navy"   .compareTo(value) == 0) { return new Color(0x000080); }
            if ("blue"   .compareTo(value) == 0) { return new Color(0x0000ff); }
            if ("teal"   .compareTo(value) == 0) { return new Color(0x008080); }
            if ("aqua"   .compareTo(value) == 0) { return new Color(0x00ffff); }
            // ƥ५顼
            if ("activeborder"       .compareTo(value) == 0) { return SystemColor.activeCaptionBorder;   }
            if ("activecaption"      .compareTo(value) == 0) { return SystemColor.activeCaption;         }
            if ("appworkspace"       .compareTo(value) == 0) { return SystemColor.control;               }
            if ("background"         .compareTo(value) == 0) { return SystemColor.desktop;               }
            if ("buttonface"         .compareTo(value) == 0) { return SystemColor.control;               }
            if ("buttonhighlight"    .compareTo(value) == 0) { return SystemColor.controlHighlight;      }
            if ("buttonshadow"       .compareTo(value) == 0) { return SystemColor.controlShadow;         }
            if ("buttontext"         .compareTo(value) == 0) { return SystemColor.controlText;           }
            if ("captiontext"        .compareTo(value) == 0) { return SystemColor.activeCaptionText;     }
            if ("graytext"           .compareTo(value) == 0) { return SystemColor.inactiveCaptionText;   }
            if ("highlight"          .compareTo(value) == 0) { return SystemColor.textHighlight;         }
            if ("highlighttext"      .compareTo(value) == 0) { return SystemColor.textHighlightText;     }
            if ("inactiveborder"     .compareTo(value) == 0) { return SystemColor.inactiveCaptionBorder; }
            if ("inactivecaption"    .compareTo(value) == 0) { return SystemColor.inactiveCaption;       }
            if ("inactivecaptiontext".compareTo(value) == 0) { return SystemColor.inactiveCaptionText;   }
            if ("infobackground"     .compareTo(value) == 0) { return SystemColor.info;                  }
            if ("infotext"           .compareTo(value) == 0) { return SystemColor.infoText;              }
            if ("menu"               .compareTo(value) == 0) { return SystemColor.menu;                  }
            if ("menutext"           .compareTo(value) == 0) { return SystemColor.menuText;              }
            if ("scrollbar"          .compareTo(value) == 0) { return SystemColor.scrollbar;             }
            if ("threeddarkshadow"   .compareTo(value) == 0) { return SystemColor.controlDkShadow;       }
            if ("threedface"         .compareTo(value) == 0) { return SystemColor.controlHighlight;      }
            if ("threedhighlight"    .compareTo(value) == 0) { return SystemColor.controlLtHighlight;    }
            if ("threedlightshadow"  .compareTo(value) == 0) { return SystemColor.controlShadow;         }
            if ("threedshadow"       .compareTo(value) == 0) { return SystemColor.controlShadow;         }
            if ("window"             .compareTo(value) == 0) { return SystemColor.window;                }
            if ("windowframe"        .compareTo(value) == 0) { return SystemColor.windowBorder;          }
            if ("windowtext"         .compareTo(value) == 0) { return SystemColor.windowText;            }
            //### ERROR # ̵ο꤫⤷ʤ
        }

        if (length != 3 && length != 6) {
            return null;
        }

        // 3 ɽ 6 ɽѴ
        if (length == 3) {
            StringBuffer sb = new StringBuffer(7);
            sb.append('#'            );
            sb.append(value.charAt(current    ));
            sb.append(value.charAt(current    ));
            sb.append(value.charAt(current + 1));
            sb.append(value.charAt(current + 1));
            sb.append(value.charAt(current + 2));
            sb.append(value.charAt(current + 2));
            value = sb.toString();
            length  = 6;
            current = 1;
        }

        // 6  16 ʿͤѴ
        int color   = 0;
        char c;
        while (--length >= 0) {
            c = value.charAt(current++);
            if ('0' <= c && c <= '9') {
                color += (c - '0') * Math.pow(0x10, length);
                continue;
            }
            if ('A' <= c && c <= 'F') {
                color += (c - 'A' + 10) * Math.pow(0x10, length);
                continue;
            }
            if ('a' <= c && c <= 'f') {
                color += (c - 'a' + 10) * Math.pow(0x10, length);
                continue;
            }
            return null;
        }

        return new Color(color);
    }

    /**
     * ե̾ɽʸ󤫤顢{@link java.awt.Font} Υ󥹥󥹤
     * ޤ
     *
     * @param  value եȤɽʸ
     *
     * @return եȤΥ󥹥
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static Font convertToFont(String value)
            throws IllegalArgumentException {
        return new Font(value, 12, Font.PLAIN);
    }

    /**
     * ͤɽʸ󤫤顢{@link java.lang.Integer} Υ󥹥󥹤
     * ޤ
     *
     * @param  value ͤɽʸ
     *
     * @return 
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static Integer convertToInteger(String value)
            throws IllegalArgumentException {
        ParsePosition pp = new ParsePosition(0);
        DecimalFormat df = new DecimalFormat("0;-0");
        df.setMaximumFractionDigits(0);
        Number n = df.parse(value, pp);
        if (/*---*/!(n instanceof Long)
                || pp.getIndex() < value.length()) {
            throw new IllegalArgumentException();
        }

        long l = n.longValue();
        if (l < Integer.MIN_VALUE || Integer.MAX_VALUE < l) {
            throw new IllegalArgumentException();
        }

        return new Integer((int) l);
    }

    /**
     * ALIGN °ɽʸ󤫤顢ALIGN ͤޤ
     *
     * @param  value ALIGN °ɽʸ
     *
     * @return ALIGN °
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static int convertToAlign(String value)
            throws IllegalArgumentException {
        value = value.toLowerCase();
        if (value.compareTo("left"  ) == 0) { return Value.ALIGN_LEFT;   }
        if (value.compareTo("center") == 0) { return Value.ALIGN_CENTER; }
        if (value.compareTo("right" ) == 0) { return Value.ALIGN_RIGHT;  }

        throw new IllegalArgumentException();
    }

    /**
     * VALIGN °ɽʸ󤫤顢VALIGN ͤޤ
     *
     * @param  value VALIGN °ɽʸ
     *
     * @return VALIGN °
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static int convertToVAlign(String value)
            throws IllegalArgumentException {
        value = value.toLowerCase();
        if (value.compareTo("top"   ) == 0) { return Value.VALIGN_TOP;    }
        if (value.compareTo("middle") == 0) { return Value.VALIGN_MIDDLE; }
        if (value.compareTo("bottom") == 0) { return Value.VALIGN_BOTTOM; }

        throw new IllegalArgumentException();
    }

    /**
     * ѡȤޤʸ󤫤顢ѡͤޤ
     *
     * @param  value ѡȤޤʸ
     *
     * @return ѡ
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static Value convertToPercent(String value)
            throws IllegalArgumentException {
        if (value.length() == 0) {
            throw new IllegalArgumentException();
        }

        boolean isPercent = false;

        if (value.charAt(value.length() - 1) == '%') {
            isPercent = true;
            value = value.substring(0, value.length() - 1);
        }

        return new Value(convertToInteger(value).intValue(), (isPercent ? Value.UNIT_PERCENT : Value.UNIT_PX));
    }

    /**
     * ʣĹꥹȤ顢ͤޤ
     *
     * @param  value ʣĹꥹȤʸ
     *
     * @return ͤΰ
     *
     * @throws IllegalArgumentException ͤξ
     */
    public static Value[] convertToMultiLengthList(String value)
            throws IllegalArgumentException {
        int length = value.length();
        int comma = -1;
        int start, end;
        Value  v;
        int    unit;
        Vector list = new Vector();

        while (++comma < length) {
            if ((comma = value.indexOf(',', (start = comma))) == -1) {
                comma = length;
            }
            end = comma;

            ML:
            {
                // Υڡ
                for (;;) {
                    if (start == end) { // ̵ξ
                        v = (end == comma)
                            ? new Value(1, Value.UNIT_ASTERISK)
                            : new Value(0, Value.UNIT_PX      );
                        break ML;
                    }
                    if (!Character.isWhitespace(value.charAt(end - 1))) {
                        break;
                    }
                    end--;
                }

                // Υڡ
                while (start < end && Character.isWhitespace(value.charAt(start))) {
                    start++;
                }

                switch (value.charAt(end - 1)) {
                case '%':
                    end--;
                    unit = Value.UNIT_PERCENT;
                    break;
                case '*':
                    end--;
                    if (start == end) {
                        v = new Value(1, Value.UNIT_ASTERISK);
                        break ML;
                    }
                    unit = Value.UNIT_ASTERISK;
                    break;
                default:
                    unit = Value.UNIT_PX;
                }

                // ʸ
                while (start < end && !Character.isDigit(value.charAt(end - 1))) {
                    end--;
                }

                if (start < end) {
                    try {
                        v = new Value(Integer.parseInt(value.substring(start, end)), unit);
                        break ML;
                    } catch (Exception e) { }
                }

                v = new Value(0, Value.UNIT_PX);
            }

            list.addElement(v);
        }

        Value[] values = new Value[list.size()];
        list.copyInto(values);
        return values;
    }
}
