/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering.block;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.image.RotateFilter;
import net.hizlab.kagetaka.rendering.Canvas;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Drawkit;
import net.hizlab.kagetaka.rendering.HawkContext;
import net.hizlab.kagetaka.rendering.ImageItem;
import net.hizlab.kagetaka.rendering.Option;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Status;
import net.hizlab.kagetaka.rendering.StopException;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.ContentType;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.FilteredImageSource;
import java.awt.image.ImageObserver;
import java.io.IOException;
import java.net.URL;
import java.text.ParseException;

/**
 * ݻԤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.10 $
 */
class ImageHolder implements ImageItem {
    private static final int IMAGE_WAIT     = 60000;         // Ԥ
    private static final int IMG_WIDTH      =    26;         // 
    private static final int IMG_HEIGHT     =    26;         // ι⤵
    private static final int D_IMG_PITCH    =     4;         // ߡ

    // ߡݻƤ
    private static final int DUMMY_NOTLOADED = 1;
    private static final int DUMMY_NOTFOUND  = 2;
    private static Image     dummyNotLoaded;
    private static Image     dummyNotFound;
    private static Dimension dummyNotLoadedSize = new Dimension(0, 0);
    private static Dimension dummyNotFoundSize  = new Dimension(0, 0);

    private HawkContext context;
    private Drawkit     drawkit;
    private Status      status;

    Dimension size;
    int       border;

    private URL      url;
    private String   src;
    private String   alt;
    private Observer observer;
    private Image    image;
    private boolean  hasError = false;

    /**
     * ۥޤ
     *
     * @param  drawkit   ɥå
     * @param  status    ơ
     * @param  src       
     * @param  alt       ʸ
     * @param  width     
     * @param  height    ⤵
     * @param  border    ܡ
     */
    ImageHolder(Drawkit drawkit, Status status, String src, String alt,
                Value width, Value height, int border) {
        this.context = drawkit.context;
        this.status  = status;
        this.drawkit = drawkit;
        this.url     = drawkit.createURL(src);
        this.src     = src;
        this.alt     = alt;
        this.size    = new Dimension((width  == null || width .getUnit() == Value.UNIT_PERCENT
                                      ? 0 : width .getNumber().intValue()),
                                     (height == null || height.getUnit() == Value.UNIT_PERCENT
                                      ? 0 : height.getNumber().intValue()));
        this.border  = border;

        Option option = context.getOption();

        // 
        boolean     loadImage = option.getLoadImage();
        Content     content   = drawkit.getImageContent(url, src, loadImage);
        ContentType ct        = null;
        if (content != null) {
            try {
                this.image = (Image) content.getObject(Content.TYPE_IMAGE);
            } catch (IOException e) { }
            ct = content.getContentType();
        } else {
            try {
                ct = ContentType.valueOf(null, url);
            } catch (ParseException e) { }
        }
        String type = (ct != null && "image".equals(ct.getType()) ? ct.getSubType() : "");

        if (image == null) {
            if (loadImage) {
                hasError = true;
            }
        } else if (size.width <= 0 || size.height <= 0) {
            // ΰ֤
            Observer observer = new Observer();
            try {
                // ̡˥å򤫤ʤȡimageUpdate ˤޤʤ
                synchronized (size) {
                    if (size.width  <= 0) { size.width  = image.getWidth (observer); }
                }
                synchronized (size) {
                    if (size.height <= 0) { size.height = image.getHeight(observer); }
                }

                // ǤʤϡޤԵ
                synchronized (size) {
                    if ((!hasError) && (size.width < 0 || size.height < 0)) {
                        try {
                            size.wait(IMAGE_WAIT);
                        } catch (InterruptedException e) {
                            throw new StopException("wait to get a position");
                        }
                    }
                }
            } finally {
                observer.dispose();
            }
        }

        // ʬʤä
        if (size.width <= 0 || size.height <= 0) {
            if (size.width  <= 0) { size.width  = IMG_WIDTH;  }
            if (size.height <= 0) { size.height = IMG_HEIGHT; }
            image    = null;
            hasError = true;
        } else {
            // % ξν
            //### BUGS % δͤʤ
            //### BUGS %  width  height Τ줫̵꤬äĲݤ
            if (width  != null && width .getUnit() == Value.UNIT_PERCENT) {
                size.width  = (size.width  * width .getNumber().intValue()) / 100;
            }
            if (height != null && height.getUnit() == Value.UNIT_PERCENT) {
                size.height = (size.height * height.getNumber().intValue()) / 100;
            }
            // % η 0 ˤʤ礬ꤨ
            if (size.width == 0 || size.height == 0) {
                image = null;
            }

            // ž뤫ɤ
            if (option.getSpinImage() && option.getSpinImage(type) && url != null) {
                int    minWidth    = option.getSpinMinWidth   (type);
                double aspectRatio = option.getSpinAspectRatio(type);

                // ˤž
                if (size.width <= minWidth
                        || (size.width  / (double) size.height >= aspectRatio
                         || size.height / (double) size.width  >= aspectRatio)) {
                    // ѹ
                    int n = size.width;
                    size.width  = size.height;
                    size.height = n;

                    // žβ
/*
  //### BUGS ɤƤߤʤƤ⤦ޤԤʡ
                    if (image != null
                            && drawkit.loadImage(image, "wait to load a source of spin image") == MediaTracker.COMPLETE) {
*/
                    if (image != null) {
                        image = drawkit.toolkit.createImage(new FilteredImageSource(image.getSource(),
                                                                                    new RotateFilter()));
                    } else {
                        image = null;
                    }
                }
            }
        }
    }


    /**
     * 褷ޤ
     *
     * @param  canvas 襭Х
     * @param  x X
     * @param  y Y
     */
    void draw(Canvas canvas, int x, int y) {
        // ֤
        x += border;
        y += border;
        int offset = canvas.width - x - size.width;
        Image    canvasImage = canvas.image;
        Observer observer    = new Observer(canvas, canvasImage, x, y, offset);

        Graphics g = canvasImage.getGraphics();
        try {
            // ܡ
            observer.drawBorder(g);

            // 
            if (status.anchor != null) {
                canvas.itemMap.addLink(status.anchor,
                                       offset - border, y,
                                       offset + size.width + border, y + size.height + border);
            }

            //  URL 
            if (url != null) {
                canvas.itemMap.addURL(url,
                                      offset             , y              ,
                                      offset + size.width, y + size.height);
            }

            // TIP 
            if (alt != null || status.tip != null) {
                canvas.itemMap.addInfo((alt != null ? alt : status.tip),
                                       offset             , y              ,
                                       offset + size.width, y + size.height);
            }

            if (image != null) {
                if (observer.drawImage(g)) {
                    observer.dispose();
                    return;
                }
            } else {
                if (size.width > 0 && size.height > 0) {
                    observer.drawDummy(g, (hasError ? DUMMY_NOTFOUND : DUMMY_NOTLOADED));
                }
                observer.dispose();
                return;
            }

            // ֥Ф
            synchronized (this) {
                if (this.observer != null) {
                    this.observer.dispose();
                }
                this.observer = observer;
            }
        } finally {
            g.dispose();
        }
    }

    /**
     * 椫ɤ֤ޤ
     *
     * @return ξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public synchronized boolean isLoading() {
        if (observer != null) {
            return observer.increment;
        }
        return false;
    }

    /**
     * ꥽ޤ
     */
    public synchronized void dispose() {
        if (observer != null) {
            observer.dispose();
            observer = null;
        }
    }

    /**
     * ۥʸɽ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        if (url != null) {
            return url.toString();
        }
        return src;
    }

//### Observer
    /** ᡼򹹿 */
    private final class Observer implements ImageObserver {
        private Canvas    canvas;
        private Image     canvasImage;
        private int       x;
        private int       y;
        private int       offset;
        private Image     original;
        private Dimension imageSize = new Dimension();
        private boolean   sameSize;         // ꥵȼºݤβƱ

        private boolean enabled   = true;   // Υ֥Фͭɤ
        private boolean increment = false;  // 󥿤򥤥󥯥Ȥ

        /** ѤΥ󥹥󥹤 */
        private Observer(Canvas canvas, Image canvasImage, int x, int y, int offset) {
            this.canvas      = canvas;
            this.canvasImage = canvasImage;
            this.x           = x;
            this.y           = y;
            this.offset      = offset;

            // ɥ󥿡䤹
            drawkit.imageCounter.increment();
            increment = true;
        }

        /** ѤΥ󥹥󥹤 */
        private Observer() {
        }

        /** ܡ */
        private void drawBorder(Graphics g) {
            if (border <= 0) { return; }

            g.setColor(status.foreColor);
            for (int i = 1; i <= border; i++) {
                g.drawRect(x - i,
                           y - i,
                           size.width  + i * 2 - 1,
                           size.height + i * 2 - 1);
            }
        }

        /**  */
        private boolean drawImage(Graphics g) {
            // 󥿡쥹 + ƩβեѤˡߤΰХåå
            original    = drawkit.createImage(size.width, size.height);
            Graphics og = original.getGraphics();
            try {
                canvas.drawImageSync(og, canvas.image,
                                     0, 0, size.width, size.height,
                                     x             , y              ,
                                     x + size.width, y + size.height);
            } finally {
                og.dispose();
            }

            try {
                return g.drawImage(image, x, y, size.width, size.height, this);
            } catch (Exception e) {
                // JVM μˤäƤϡʥե Exception ȯ
                drawkit.reportMessage(Reporter.WARNING, "render.warning.image.draw", new String[]{src, e.toString()});
                image = null;
            }
            return true;
        }

        /** ߡ */
        private synchronized void overwriteDummy(Graphics g, int type) {
            // ߤ褵Ƥ뤫⤷ʤΤǡβǾ
            if (original != null) {
                canvas.drawImageSync(g, original, x, y, size.width, size.height);
            }

            drawDummy(g, type);
            context.repaint(offset + canvas.x, y, size.width, size.height);
        }

        /** ߡɽ */
        private void drawDummy(Graphics g, int type) {
            // ߡ򥭥åɤ߹
            loadDummy(type);

            // Ȥ
            g.setColor(Color.black);
            g.drawLine(x                 , y                  ,
                       x + size.width - 1, y                  );
            g.drawLine(x                 , y                  ,
                       x                 , y + size.height - 1);
            g.setColor(Color.lightGray);
            g.drawLine(x + size.width - 1, y + 1              ,
                       x + size.width - 1, y + size.height - 1);
            g.drawLine(x + 1             , y + size.height - 1,
                       x + size.width - 1, y + size.height - 1);

            Image     dummyImage;
            Dimension dummySize;
            if (type == DUMMY_NOTLOADED) {
                dummyImage = dummyNotLoaded;
                dummySize  = dummyNotLoadedSize;
            } else {
                dummyImage = dummyNotFound;
                dummySize  = dummyNotFoundSize;
            }

            // ߡ
            int w = Math.min(dummySize.width , size.width  - D_IMG_PITCH);
            int h = Math.min(dummySize.height, size.height - D_IMG_PITCH);
            canvas.drawImageSync(g, dummyImage,
                                 x + D_IMG_PITCH    , y + D_IMG_PITCH    ,
                                 x + D_IMG_PITCH + w, y + D_IMG_PITCH + h,
                                 0, 0, w, h);
        }

        /** ߡɤ߹ */
        private void loadDummy(int type) {
            Dimension size = (type == DUMMY_NOTLOADED
                              ? dummyNotLoadedSize
                              : dummyNotFoundSize);
            synchronized (size) {
                Image image = (type == DUMMY_NOTLOADED
                               ? dummyNotLoaded
                               : dummyNotFound);

                if (image != null) {
                    return;
                }

                image = Resource.getImageResource((type == DUMMY_NOTLOADED
                                                   ? "render.image.notloaded"
                                                   : "render.image.notfound"),
                                                  context.getToolkit());
                if (image == null) {
                    return;
                }

                drawkit.loadImage(image, "wait to load a dummy image");

                size.width  = image.getWidth (null);
                size.height = image.getHeight(null);
                if (type == DUMMY_NOTLOADED) {
                    dummyNotLoaded = image;
                } else {
                    dummyNotFound  = image;
                }
            }
        }

        /** ᡼ */
        public synchronized boolean imageUpdate(Image img, int infoflags,
                                                int x, int y, int width, int height) {
//Debug.out.println("src=" + src + ",alt=" + alt + ",iu=" + infoflags + "," + x+"," + y+"," + width + "," + height);
            if (!enabled) { return false; }

            // 顼λ
            if ((infoflags & (ImageObserver.ABORT | ImageObserver.ERROR)) != 0) {
                synchronized (size) {
                    hasError = true;
                    if (size.width <= 0 || size.height <= 0) {
                        size.notify();
                    }
                }
                if ((infoflags & ImageObserver.ERROR) != 0) {
                    if (canvasImage != null) {
                        Graphics g = canvasImage.getGraphics();
                        try {
                            overwriteDummy(g, DUMMY_NOTFOUND);
                        } finally {
                            g.dispose();
                        }
                    }
                    drawkit.reportMessage(Reporter.WARNING, "render.warning.image.load.error", new String[]{src});
                } else {
                    drawkit.reportMessage(Reporter.WARNING, "render.warning.image.load.abort", new String[]{src});
                }
                dispose();

                return false;
            }

            // 
            if ((infoflags & (ImageObserver.WIDTH | ImageObserver.HEIGHT)) != 0) {
                synchronized (size) {
                    if (((infoflags & ImageObserver.WIDTH ) != 0) && (size.width  <= 0)) {
                        size.width  = width;
                    }
                    if (((infoflags & ImageObserver.HEIGHT) != 0) && (size.height <= 0)) {
                        size.height = height;
                    }

                    if (size.width > 0 && size.height > 0) {
                        size.notify();
                    }
                }

                // 󥿡쥹ɽ뤫ɤꤹ뤿ˡºݤβ¸
                if ((infoflags & ImageObserver.WIDTH ) != 0) { imageSize.width  = width ; }
                if ((infoflags & ImageObserver.HEIGHT) != 0) { imageSize.height = height; }
                sameSize = (imageSize.width  == size.width
                         && imageSize.height == size.height);
            }

            // ʹߤ
            if (canvasImage != null) {
                Graphics g = canvasImage.getGraphics();
                try {
                    // ʬ
                    if ((infoflags & ImageObserver.SOMEBITS) != 0) {
                        if (sameSize) {
                            // 󥿡쥹 + ƩβեѤˡХååפ᤹
                            if (original != null) {
                                g.drawImage(original,
                                            this.x + x        , this.y + y         ,
                                            this.x + x + width, this.y + y + height,
                                            x, y, x + width, y + height,
                                            this);
                            }
                            g.drawImage(img,
                                        this.x + x        , this.y + y         ,
                                        this.x + x + width, this.y + y + height,
                                        x, y, x + width, y + height,
                                        this);
                            context.repaint(offset + canvas.x + (size.width - (x + width)), this.y + y, width, height);
                        }
                    }

                    // 
                    if ((infoflags & (ImageObserver.FRAMEBITS | ImageObserver.ALLBITS)) != 0) {
                        g.drawImage(img, this.x, this.y, size.width, size.height, this);
                        context.repaint(offset + canvas.x, this.y, size.width, size.height);
                    }

                    // ˥᡼ΰ쥳ޤ轪λ
                    if ((infoflags & ImageObserver.FRAMEBITS) != 0) {
                        loaded();
                    }

                    // λ
                    if ((infoflags & ImageObserver.ALLBITS) != 0) {
                        dispose();
                    }
                } catch (Exception e) {
                    // JVM μˤäƤϡʥե Exception ȯ
                    drawkit.reportMessage(Reporter.WARNING, "render.warning.image.draw", new String[]{src, e.toString()});
                    overwriteDummy(g, DUMMY_NOTFOUND);
                    dispose();
                    return false;
                } finally {
                    g.dispose();
                }
            }

            return ((infoflags & (ImageObserver.ALLBITS | ImageObserver.ABORT | ImageObserver.ERROR)) == 0);
        }

        /** Υɴλ */
        private synchronized void loaded() {
            // ɥ󥿡򸺤餹
            if (increment) {
                drawkit.imageCounter.decrement();
                increment = false;
            }
        }

        /** ˴ */
        private synchronized void dispose() {
            if (!enabled) {
                return;
            }

            enabled = false;
            loaded();

            // ˴
            if (original != null) {
                original.flush();
                original = null;
            }
        }
    }
}
