/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering.block;

import net.hizlab.kagetaka.addin.Table;
import net.hizlab.kagetaka.rendering.Canvas;
import net.hizlab.kagetaka.rendering.Constant;
import net.hizlab.kagetaka.rendering.Drawkit;
import net.hizlab.kagetaka.rendering.FormItem;
import net.hizlab.kagetaka.rendering.Status;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.CharBuffer;

import java.awt.Dimension;
import java.awt.Insets;
import java.util.Vector;

/**
 * ֥åǤɽޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.9 $
 */
public abstract class Block {
    /** ޡ */
    public static final int MARGIN_TOP    = 0;
    /** ޡ */
    public static final int MARGIN_RIGHT  = 1;
    /** ޡ */
    public static final int MARGIN_BOTTOM = 2;
    /** ޡ */
    public static final int MARGIN_LEFT   = 3;

    /** before ѥޡ */
    public static final int MARKER_BEFORE = 0;
    /** after ѥޡ */
    public static final int MARKER_AFTER  = 1;

    /** ư */
    protected static final int SIZE_AUTO    = -1;
    /** ѡȻ */
    protected static final int SIZE_PERCENT = -2;
    /** ̵ */
    protected static final int SIZE_NONE    = -3;

    /** ɥå */
    protected Drawkit        drawkit;
    /** ƥʥ֥å */
    protected ContainerBlock container;
    /** ƥ֥å */
    protected Block          parent;
    /** ơ */
    public    Status         status;
    /** Ƥƿ֥̾åɤ */
    protected boolean        isAdoptee;
    /** Υ֥åɤ */
    protected boolean        isEmpty;

    /** ޡ */
    public    Insets margin  = new Insets(0, 0, 0, 0);     // * validate
    /** ܡ */
    protected Insets border;
    /** ѥǥ */
    protected Insets padding = new Insets(0, 0, 0, 0);     // * validate
    /** Ȥ礭 */
    protected Insets frame;                 // * validate

    ///// å
    /** ե졼<code>frame.left + frame.right</code> */
    protected int frameWidth;               // * validate
    /** ե졼ι⤵<code>frame.top + frame.bottom</code> */
    protected int frameHeight;              // * validate

    ///// ⤵ξ
    /** ֥åΤκǾι⤵ */
    public int minHeight;                   // * analyze
    /** ֥åΤκι⤵ */
    public int maxHeight;                   // * analyze
    /** Ǥι⤵ */
    protected int contentHeight;            // * validate

    /** ֥åʪο侩 */
    protected int preferredWidth;           // calculatePreferred (<init>) or validate ޤǤ
    /** ֥åʪο侩ι⤵ */
    protected int preferredHeight;          // calculatePreferred (<init>) or validate ޤǤ
    /** ֥åʪ */
    protected int width;                    // * layoutBlock
    /** ֥åʪι⤵ */
    protected int height;                   // * validate

    /** ƥ֥åФƤΥ֥åξ¦ΰ */
    public int top;                         // * validate
    /** ƥ֥åФƤΥ֥åα¦ΰ */
    public int right;                       // * validate
    /** ƥ֥åФƤΥ֥åβ¦ΰ */
    public int bottom;                      // * layoutBlock
    /** ƥ֥åФƤΥ֥åκ¦ΰ */
    public int left;                        // * layoutBlock

    /** ɬפʥ֥åξ¦ΰ */
    public int drawTop;                     // * layoutBlock
    /** ɬפʥ֥åα¦ΰ */
    public int drawRight;                   // * layoutBlock
    /** ɬפʥ֥åβ¦ΰ */
    public int drawBottom;                  // * layoutBlock
    /** ɬפʥ֥åκ¦ΰ */
    public int drawLeft;                    // * layoutBlock

    /** ҤΥ֥åꥹ */
    protected Vector contents = new Vector();
    /** ǥեȤɲ֥å */
    protected Block defaultBlock;

    //// 
    private boolean   dontIndent;           // ǥȤƤϤʤ
    private boolean   invalidate;           // Ʒ׻ɬפ뤫

    //// append Τ
    private Status currentStatus;           // ߤΥơʻҤΥƥȥ֥åѡ

    //// ¾
    private CharBuffer idList;              // Υ֥åκǸդ ID ꥹ

    /**
     * ֥åޤ
     *
     * @param  drawkit   ɥå
     * @param  status    ơ
     * @param  container ƥʥ֥å
     * @param  parent    ƥ֥å
     */
    protected Block(Drawkit drawkit, Status status,
                    ContainerBlock container, Block parent) {
        this.drawkit       = drawkit;
        this.status        = status;
        this.container     = container;
        this.parent        = parent;
        this.currentStatus = status;

        // ܡϥƥʥ˴طʤΤǡǷ׻
        if (status.border != null) {
            status.border.setBaseSize(status.fontData);
            border = status.border.getWidths();
        }

        // 侩򻻽
        calculatePreferred();
    }

    /**
     * ֥åʪο侩򻻽Фޤ
     * Ф̤ϡ
     * {@link #preferredWidth}{@link #preferredHeight} ꤷޤ
     */
    protected void calculatePreferred() {
        // ά AUTOwidth  % ϿƤ FIXED ξΤͭ
        preferredWidth  = (status.width  == null
                           ? SIZE_AUTO
                           : (status.width .getUnit() == Value.UNIT_PERCENT
                              ? (parent.preferredWidth >= 0
                                 ? SIZE_PERCENT
                                 : SIZE_AUTO)
                              : status.width .getValue(1, 0,
                                                       status.fontData,
                                                       Value.DATA_HORIZONTAL)));
        preferredHeight = (status.height == null
                           ? SIZE_AUTO
                           : (status.height.getUnit() == Value.UNIT_PERCENT
                              ? SIZE_PERCENT
                              : status.height.getValue(1, 0,
                                                       status.fontData,
                                                       Value.DATA_VERTICAL  )));
    }

    /**
     * ֥å֥å֤ޤ
     *
     * @param  status       ơ
     * @param  markerStatus ޡơ
     *                      ޡ̵ <code>null</code>
     *
     * @return ֥å
     */
    public Block createBlock(Status status, Status markerStatus) {
        if (status.floatType == Value.FLOAT_NONE) {
            // եȤǤϤʤ֥åξ

            // 򥳥ߥå
            commitDefaultBlock();

            Block block = createBlockInternal(status, markerStatus);
            contents.addElement(block);
            return block;
        }

        // եȥ֥åξ
        FloatBlock block = new FloatBlock(drawkit, status, this, markerStatus);
        appendFloat(block);
        return block;
    }

    /**
     * ֥å֥å֤ޤ
     *
     * @param  status       ơ
     * @param  markerStatus ޡơ
     *                      ޡ̵ <code>null</code>
     *
     * @return ֥å
     */
    protected final Block createBlockInternal(Status status, Status markerStatus) {
        Block  block;

        // ޡĤξ
        if (markerStatus != null) {
            block = new MarkedBlock(drawkit, status, container, this, markerStatus);
        } else
        // HR ξ
        if (status.isHorizontalRule) {
            block = new HrBlock(drawkit, status, container, this);
        } else
        // ơ֥ǤϤʤ䡢ơ֥եȥǺǤʤ
        if (status.display != Value.DISPLAY_TABLE
                || (block = Table.createBlock(drawkit, status, container, this)) == null) {
            block = new BasicBlock(drawkit, status, container, this);
        }

        return block;
    }

    /**
     * ֥å򥳥ߥåȤοƤΥ֥å֤ޤ
     *
     * @return ƥ֥å
     */
    public Block commitBlock() {
        // 򥳥ߥå
        commitDefaultBlock();

        // ID 塼˻ĤäƤ硢֥åκǸؤ褦ˤ
        if (!drawkit.idQueue.isEmpty()) {
            idList = new CharBuffer();
            Object id;
            synchronized (drawkit.ids) {
                while ((id = drawkit.idQueue.get()) != null) {
                    idList.append((char) drawkit.ids.size());
                    drawkit.ids.addElement(id);
                }
            }
        }

        return parent.commitChild();
    }

    /**
     * ǸλҤΥ֥åߥåȤȤ˸ƤӽФ졢
     * ҤΥ֥åФơϼʬȡˤ֤ɬפޤ
     *
     * @return ǸλҤΥ֥åο
     */
    protected Block commitChild() {
        return (isAdoptee ? parent.commitChild() : this);
    }

    /**
     * ʬΥκǾȺι⤵򻻽Фޤ
     * Ф̤ϡ
     * {@link #minHeight}{@link #maxHeight}
     * ꤷޤ
     *
     * @param  minParentHeight ƥե졼κǾ
     *                         ʿޥ֥åκǾ <code>height</code>
     * @param  maxParentHeight ƥե졼κ
     *                         ʿޥ֥åκ <code>height</code>
     */
    public abstract void analyze(int minParentHeight, int maxParentHeight);

    /**
     * Ԥޤ
     * Υ᥽åɤǡ{@link #height}{@link #top}{@link #right}
     * {@link #contentHeight} եɤ
     * {@link #calculateFrame(int)}{@link #layoutBlock(int, int, int)}
     * ᥽åɤꤹեɤꤹɬפޤ
     *
     * @param  parentHeight ե졼ޤ᤿⤵
     *                      ʿޥ֥å <code>height</code>ˡ
     *                      ι⤵ˤʤ٤Ƥޤ褦˺Ԥ
     * @param  prevLeftMargin Υ֥åκޡ
     * @param  x              ޥ֥å鼫֥åФ X
     * @param  y              ޥ֥å鼫֥åФ Y
     * @param  containerX     ƥʥ֥åƥ֥åФ X
     * @param  containerY     ƥʥ֥åƥ֥åФ Y
     */
    public void validate(int parentHeight, int prevLeftMargin, int x, int y,
                         int containerX, int containerY) {
//Debug.out.println(this + ":Block.validate: " + parentHeight + "," + prevLeftMargin + "," + x + "," + y + "," + containerX + "," + containerY + "," + minHeight + "," + maxHeight);
        int newHeight;

        // ե졼򻻽
        calculateFrame(parentHeight);

        // ΰַ
        top   = y;
        right = x + getMarginOffset(prevLeftMargin, margin.right);   // ޡǤä

        // ι⤵׻
        // ⤵Ǿͤ꾮ϡǾͤ˹碌IE 
        switch (preferredHeight) {
        case SIZE_AUTO:
            newHeight = Math.max(minHeight, parentHeight) - frameHeight;
            break;
        case SIZE_PERCENT:
            newHeight = Math.max(minHeight,
                                 (parentHeight * status.height.getNumber().intValue()) / 100);
            break;
        case SIZE_NONE:
            newHeight = parentHeight;
            break;
        default:
            newHeight = Math.max(minHeight, preferredHeight);
        }

        // եȤξϡ礭ǽʸ¤깭
        if (status.floatType != Value.FLOAT_NONE) {
            newHeight = Math.min(newHeight, maxHeight + frameHeight);
        }

        // ⤵¸
        this.contentHeight = this.height = newHeight;

        // 
        layoutBlock(newHeight, containerX + right + frame.right, containerY + top + frame.top);

//Debug.out.println(this + ":Block.validate2: " + width + "x" + height + ", " + top + "," + right + "," + bottom + "," + left);
    }

    /**
     * ܡʳΥե졼طΥ򻻽Фޤ
     * Ф̤ϡ
     * {@link #margin}{@link #padding}{@link #frame}
     * {@link #frameWidth}{@link #frameHeight} ꤷޤ
     *
     * @param  parentHeight ƥ֥åʪι⤵
     */
    protected void calculateFrame(int parentHeight) {
        Status.Box box;

        frame = (border != null ? (Insets) border.clone()
                                : new Insets(0, 0, 0, 0));

        if (status .margin != null) {
            box = status.margin;
            if (box.top    != null) { frame.top    += (margin .top    = box.top   .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.right  != null) { frame.right  += (margin .right  = box.right .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.bottom != null) { frame.bottom += (margin .bottom = box.bottom.getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.left   != null) { frame.left   += (margin .left   = box.left  .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
        }
        if (status.padding != null) {
            box = status.padding;
            if (box.top    != null) { frame.top    += (padding.top    = box.top   .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.right  != null) { frame.right  += (padding.right  = box.right .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.bottom != null) { frame.bottom += (padding.bottom = box.bottom.getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
            if (box.left   != null) { frame.left   += (padding.left   = box.left  .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL)); }
        }
        frameWidth  = frame.left + frame.right;
        frameHeight = frame.top + frame.bottom;
    }

    /**
     * ꤵ줿⤵ˤʤ褦ˡʬΥ֥åڤӲ̤Υ֥å
     * ĴᤷޤĴᤷ̤ϡ
     * {@link #width}{@link #bottom}{@link #left}
     * {@link #drawTop}{@link #drawRight}
     * {@link #drawBottom}{@link #drawLeft}
     * ꤷʪι⤵ <code>newHeight</code>
     * ʾˤʤäƤϤޤ
     * Υ᥽åɤϥ֥å礭Ѥ뤿Ӥ˸ƤӽФޤ
     *
     * @param  newHeight  ˷׻Ѥߤʪι⤵
     * @param  containerX ƥʥ֥å鼫֥åФ X
     * @param  containerY ƥʥ֥å鼫֥åФ Y
     */
    protected void layoutBlock(int newHeight, int containerX, int containerY) {
    }

    /**
     * ߤη׻̵ˤơƷ׻оݤˤޤ
     */
    protected void invalidate() {
        invalidate = true;
        parent.invalidate();
    }

    /**
     * ֥å褷ޤ
     *
     * @param  canvas 褹륭Х
     * @param  x      ƥ֥åʪΡХ夫 X 
     * @param  y      ƥ֥åʪΡХ夫 Y 
     */
    protected void draw(Canvas canvas, int x, int y) {
//Debug.out.println(this + ":Block.draw: " + x + "," + y + " : " + frameHeight + "," + maxHeight + "," + height + "," + width);
        // ID ꥹȤ硢ID ΰ֤
        if (idList != null) {
            int idX = canvas.width - (x - right - frame.right - width + Constant.ID_GAP);
            for (int i = idList.length() - 1; i >= 0; i--) {
                drawkit.setIdPosition(idList.get(i), canvas, idX, 0);
            }
        }

        // 礭 0 ξά
        if (frameHeight == 0 && maxHeight == 0 && height == 0) {
            return;
        }

        x -= right;
        y += top;

        // طʤ
        if (status.background != null) {
            drawBackground(canvas, x, y);
        }

        // ܡ
        if (border != null) {
            drawBorder(canvas, x, y);
        }

        // ե졼ʬ򸺤餷ơʬʪα˰ư
        x -= frame.right;
        y += frame.top;

        drawContent(canvas, x, y);
    }

    /**
     * طʲطʿ褷ޤ
     *
     * @param  canvas Х
     * @param  x      ֥åΡХ夫 X 
     * @param  y      ֥åΡХ夫 Y 
     */
    protected void drawBackground(Canvas canvas, int x, int y) {
        int w = width  + padding.left + padding.right;
        int h = height + padding.top  + padding.bottom;
        if (w > 0 && h > 0) {
            canvas.drawBackground(x - frame.right - width - padding.left,
                                  y + (frame.top - padding.top),
                                  w, h, w, 0, status);
        }
    }

    /**
     * ܡ褷ޤ
     *
     * @param  canvas Х
     * @param  x      ֥åΡХ夫 X 
     * @param  y      ֥åΡХ夫 Y 
     */
    protected void drawBorder(Canvas canvas, int x, int y) {
        status.border.draw(canvas.g,
                           x - frame.right - width - padding.left - border.left,
                           y + margin.top,
                           x - margin.right,
                           y + frame.top + height + padding.bottom + border.bottom,
                           status.foreColor);
    }

    /**
     * ʪ褷ޤ
     *
     * @param  canvas Х
     * @param  x      ֥åʪΡХ夫 X 
     * @param  y      ֥åʪΡХ夫 Y 
     */
    protected void drawContent(Canvas canvas, int x, int y) {
        int contentsSize = contents.size();

        for (int i = 0; i < contentsSize; i++) {
            ((Block) contents.elementAt(i)).draw(canvas, x, y);
        }
    }

    /**
     * ޡƱΤǤä򤷤ƺʬ֤ޤ
     *
     * @param     left  Υ֥åκޡ
     * @param     right Υ֥åαޡ
     *
     * @return    ޡǤäκʬ
     */
    protected final int getMarginOffset(int left, int right) {
        if ((left >= 0 && right >= 0)
                || (left < 0 && right < 0)) {
            return (left > right ? -right : -left);
        }

        return 0;
    }

//### 
    /**
     * ǥեȤɲ֥å {@link #defaultBlock} ¸ߤ뤫
     * åޤ̵Ϻɬפޤ
     */
    protected void ensureDefaultBlock() {
        if (defaultBlock != null) {
            return;
        }

        if (!dontIndent) {
            dontIndent = true;
            defaultBlock  = new TextBlock(drawkit, currentStatus, container, this,
                                          (status.textIndent != null
                                           ? status.textIndent.getValue(1,
                                                                        status.fontFullSize.height,
                                                                        status.fontData,
                                                                        Value.DATA_VERTICAL)
                                           : 0));
        } else {
            defaultBlock = new TextBlock(drawkit, currentStatus, container, this, 0);
        }
    }

    /**
     * ǥեȤɲ֥å {@link #defaultBlock} 򥳥ߥåȤޤ
     */
    protected void commitDefaultBlock() {
        if (defaultBlock == null) {
            return;
        }

        defaultBlock.commitBlock();
        if (!defaultBlock.isEmpty) {
            contents.addElement(defaultBlock);
        }
        defaultBlock = null;
    }

    /**
     * ʸ褷ޤ
     *
     * @param  text ʸ
     */
    public void appendString(String text) {
        ensureDefaultBlock();
        defaultBlock.appendString(text);
    }

    /**
     * Ѥ߹Ԥ褷ޤ
     */
    public void appendNewLine() {
        ensureDefaultBlock();
        defaultBlock.appendNewLine();
    }

    /**
     * ɲäޤ
     *
     * @param  src       
     * @param  alt       ʸ
     * @param  width     
     * @param  height    ⤵
     * @param  border    ܡ
     * @param  floatType եȥ
     */
    public void appendImage(String src, String alt, Value width, Value height,
                            int border, int floatType) {
        if (floatType == Value.FLOAT_NONE) {
            ensureDefaultBlock();
            defaultBlock.appendImage(src, alt, width, height, border, floatType);
        } else {
            // եȤβϡեȥ֥åɲä
            Status floatStatus = this.status.createChild();
            floatStatus.floatType = floatType;
            Block floatBlock = createBlock(floatStatus, null);
            floatBlock.appendImage(src, alt, width, height, border, Value.FLOAT_NONE);
            floatBlock.commitBlock();
        }
    }

    /**
     * եॢƥɲäޤ
     *
     * @param  item ƥ
     */
    public void appendForm(FormItem item) {
        ensureDefaultBlock();
        defaultBlock.appendForm(item);
    }

    /**
     * 饤֥åɲäޤ
     *
     * @param  block 饤֥å
     */
    public void appendBlock(Block block) {
        ensureDefaultBlock();
        defaultBlock.appendBlock(block);
    }

    /**
     * եȥ֥åɲäޤ
     *
     * @param  block եȥ֥å
     */
    public void appendFloat(FloatBlock block) {
        ensureDefaultBlock();
        defaultBlock.appendFloat(block);
    }

    /**
     * ӤΥ⡼ɤѹޤ
     *
     * @param  mode ӤΥ⡼
     */
    public void setRuby(int mode) {
        ensureDefaultBlock();
        defaultBlock.setRuby(mode);
    }

    /**
     * ơѹ줿硣
     *
     * @param  status ơ
     */
    public void statusChanged(Status status) {
        currentStatus = status;
        if (defaultBlock != null) {
            defaultBlock.statusChanged(status);
        }
    }

//### ơ֥
    /**
     * ơ֥ιԤλޤ
     */
    public void commitRow() {
        ensureDefaultBlock();
        defaultBlock.commitRow();
    }

//### Utils
    /**
     * ե졼ʥޡ + ܡ + ѥǥ󥰡ˤι⤵֤ޤ
     * <p>
     * Υ᥽åɤϤĤǤѤǤޤ
     *
     * @param  parentHeight ƥ֥åʪι⤵
     *
     * @return ե졼ι⤵
     */
    protected int getFrameHeight(int parentHeight) {
        Status.Box box;
        int height = (border != null ? border.top + border.bottom : 0);

        if (status .margin != null) {
            box = status.margin;
            if (box.top    != null) { height += box.top   .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); }
            if (box.bottom != null) { height += box.bottom.getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); }
        }
        if (status.padding != null) {
            box = status.padding;
            if (box.top    != null) { height += box.top   .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); }
            if (box.bottom != null) { height += box.bottom.getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); }
        }

        return height;
    }

    /**
     * {@link #validate(int, int, int, int, int, int)} ˡ
     * ޡΤꤿˡޡ֤ޤ
     * <p>
     * Υ᥽åɤϤĤǤѤǤޤ
     *
     * @param  parentHeight ƥ֥åʪι⤵
     * @param  sense        
     *
     * @return ޡ
     */
    public int getMargin(int parentHeight, int sense) {
        if (status.margin != null) {
            Status.Box box = status.margin;
            switch (sense) {
            case MARGIN_TOP   : if (box.top    != null) { return box.top   .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); } break;
            case MARGIN_RIGHT : if (box.right  != null) { return box.right .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); } break;
            case MARGIN_BOTTOM: if (box.bottom != null) { return box.bottom.getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); } break;
            case MARGIN_LEFT  : if (box.left   != null) { return box.left  .getValue(1, parentHeight, status.fontData, Value.DATA_VERTICAL); } break;
            default: // AVOID
            }
        }
        return 0;
    }

    /**
     * 侩֤ޤ
     * <p>
     * Υ᥽åɤϤĤǤѤǤޤ
     *
     * @return 侩
     */
    public final Dimension getPreferredSize() {
        return new Dimension(preferredWidth, preferredHeight);
    }

    /**
     * ǽιԤ֤ޤ
     * <p>
     * Υ᥽åɤ <code>validate</code> ˻ѤǤޤ
     *
     * @return ǽι
     */
    protected Line getFirstLine() {
        Line line;
        int  contentsSize = contents.size();

        for (int i = 0; i < contentsSize; i++) {
            if ((line = ((Block) contents.elementAt(i)).getFirstLine()) != null) {
                line.right += right + frame.right;
                return line;
            }
        }

        return null;
    }

//### ¾
    /**
     * ʸɽ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        StringBuffer sb = new StringBuffer(40);
        String       s  = getClass().getName();
        int          p  = s.lastIndexOf('.');
        if (p != -1) {
            sb.append(s.substring(p + 1));
        } else {
            sb.append(s);
        }
        sb.append('@');
        sb.append(Integer.toHexString(hashCode()));
        sb.append('[');
        sb.append(net.hizlab.kagetaka.token.TokenTypes.getName(status.type));
        sb.append(']');
        return sb.toString();
    }
}
