/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.addin.FormManager;
import net.hizlab.kagetaka.io.CausedIOException;
import net.hizlab.kagetaka.net.URLUtils;

import java.awt.Point;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.net.SocketException;

/**
 * ɤ߹ߥꥯȤɽ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.10 $
 */
public class Request {
    /** TARGET ˽ */
    public static final int OPEN_DEFAULT   = 0;
    /** ɥǳ */
    public static final int OPEN_NEWWINDOW = 1;
    /** ֤ǳ */
    public static final int OPEN_NEWTAB    = 2;

    /** ɬåȤ */
    public static final int CACHE_MUST   = 4;
    /** åȤ */
    public static final int CACHE_SOFT   = 3;
    /** ξϥåȤڤʤåԤ */
    public static final int CACHE_NORMAL = 2;
    /** ɬåԤ */
    public static final int CACHE_CHECK  = 1;
    /** åȤʤ */
    public static final int CACHE_NONE   = 0;

    /** ꥯ URL */
    public final URL      url;
    /** POST ǡ */
    public final PostData postData;
    /** ΥꥯȤθˤʤäꥯ */
    public final Request  parent;
    /** 󥯸 URL */
    public final URL      referer;
    /** å */
    public final String   target;
    /** ˡ */
    public final int      openMode;

    // 
    boolean isChain;
    Request next;
    Request prev;

    // ɥȾ
    private boolean     isActive;
    private Document    document;
    private FormManager formManager;

    // ¸
    private Point    position;
    private FormData formData;

    // ꥯȾ
    private int      useCache;
    private URL      contentUrl;
    private boolean  isIgnored;

    /**
     * ꥯȤޤ
     *
     * @param  url      URL
     * @param  pd       POST Ϥǡ
     *                  POST ǤϤʤ <code>null</code>
     * @param  referer  󥯸 URL
     *                  󥯤éäΤǤϤʤ <code>null</code>
     * @param  target   å
     *                  åȤꤵƤʤ <code>null</code>
     * @param  openMode ⡼
     * @param  cache    åѥ
     */
    public Request(URL url, PostData pd, URL referer,
                   String target, int openMode, int cache) {
        this(url, pd, null, referer, target, openMode, cache);
    }

    /**
     * ꥯȤޤ
     *
     * @param  url      URL
     * @param  pd       POST Ϥǡ
     *                  POST ǤϤʤ <code>null</code>
     * @param  parent   ΥꥯȤθˤʤäƥꥯȡ
     *                  ̵ <code>null</code>
     * @param  referer  󥯸 URL
     *                  󥯤éäΤǤϤʤ <code>null</code>
     * @param  target   å
     *                  åȤꤵƤʤ <code>null</code>
     * @param  openMode ⡼
     * @param  cache    åѥ
     */
    protected Request(URL url, PostData pd, Request parent, URL referer,
                      String target, int openMode, int cache) {
        this.url        = url;
        this.postData   = pd;
        this.parent     = parent;
        this.referer    = referer;
        this.target     = target;
        this.openMode   = openMode;
        this.useCache   = cache;
        this.contentUrl = url;
    }

    /**
     * ΥꥯȤ򸵤˿ꥯȤޤ
     *
     * @param  url URL
     *
     * @return ꥯ
     */
    public Request createRequest(URL url) {
        return createRequest(url, useCache);
    }

    /**
     * ΥꥯȤ򸵤ˡꤵ줿å奿פ
     * ꥯȤޤ
     *
     * @param  url   URL
     * @param  cache åѥ
     *
     * @return ꥯ
     */
    public Request createRequest(URL url, int cache) {
        return new Request(url, null, this, contentUrl,
                           null, Request.OPEN_DEFAULT, cache);
    }

    /**
     * ꤵ줿ɥȤǽޤ
     *
     * @param  context 륳ƥ
     *
     * @return Ǥ <code>ture</code>
     *         ʤä <code>false</code>
     */
    boolean initialize(HawkContext context) {
        Content content = getContent(context, null, false);
        if (content == null) {
            return false;
        }

        synchronized (this) {
            cleanup();
            // ʹߤǡå夫ɤ߹ߤǤϤʤ
            //### BUGS file: ξǤ isCached ֤褦ˤ
            if (this.document != null
                    && !content.isCached()
                    && !content.url.getProtocol().equalsIgnoreCase("file")) {
                this.formData = null;
            }

            this.isActive = true;
            this.document = new Document(context, content);
        }
        return true;
    }

    /**
     * ꤵ줿ꥯȤб륳ƥĤ֤ޤ
     *
     * @param  context        ƥ
     * @param  useLoadMessage Υåɽ
     *                        <code>true</code>
     *                        ɽʤ <code>false</code>
     *
     * @return ƥġ
     *         顼ȯꡢ¸ߤʤ <code>null</code>
     */
    public Content getContent(HawkContext context,
                              boolean useLoadMessage) {
        Document document;

        // ɥȤ򸡺
        Request r = this;
        while ((document = r.getDocument()) == null) {
            if (r.parent == null) {
                break;
            }
            r = r.parent;
        }

        return getContent(context, document, useLoadMessage);
    }

    /**
     * ꤵ줿ꥯȤб륳ƥĤ֤ޤ
     *
     * @param  context        ƥ
     * @param  document       ɥ
     * @param  useLoadMessage Υåɽ
     *                        <code>true</code>
     *                        ɽʤ <code>false</code>
     *
     * @return ƥġ
     *         顼ȯꡢ¸ߤʤ <code>null</code>
     */
    private Content getContent(HawkContext context,
                               Document document,
                               boolean useLoadMessage) {
        String   urlString = URLUtils.getFullPath(url);
        Reporter reporter  = context.getReporter();
        Content  content;

        // ɥȥå夫Ƥߤ
        if (document != null && postData == null) {
            if ((content = document.getContentFromCache(urlString)) != null) {
                contentUrl = URLUtils.copyRef(content.url, url);
                return content;
            }
        }

        try {
            if (useLoadMessage && reporter != null) {
                reporter.report(Reporter.RENDERING, Reporter.INFO, Reporter.FYI, document.content, 0, 0, "Engine",
                                Resource.getMessage("engine.status.load.start", new String[]{url.toExternalForm()}));
            }

            try {
                if ((content = getContent(context)) != null) {
                    // ɥȤХåɲ
                    if (document != null) {
                        document.addContentToCache(urlString, content);
                    }
                    contentUrl = URLUtils.copyRef(content.url, url);
                    return content;
                }
            } catch (CausedIOException e) {
                // CausedIOException ξ硢ʤ٤ͳꤲ
                Throwable t = e.getCause();
                if (t instanceof IOException) {
                    throw (IOException) t;
                }
                if (t instanceof RuntimeException) {
                    throw (RuntimeException) t;
                }
                throw e;
            }
        } catch (SecurityException e) {
            reportMessage(reporter, document,
                          "engine.status.connect.error.security",
                          new String[]{urlString},
                          null);
        } catch (FileNotFoundException e) {
            reportMessage(reporter, document,
                          "engine.status.connect.error.nofile",
                          new String[]{urlString},
                          null);
        } catch (UnknownHostException  e) {
            reportMessage(reporter, document,
                          "engine.status.connect.error.hostname",
                          new String[]{url.getHost()},
                          null);
        } catch (ConnectException e) {
            reportMessage(reporter, document,
                          "engine.status.connect.error.service",
                          new String[]{url.getHost()},
                          null);
        } catch (NoRouteToHostException e) {
            reportMessage(reporter, document,
                          "engine.status.connect.error.route",
                          new String[]{url.getHost()},
                          null);
        } catch (SocketException e) {
            String s = e.getMessage();
            reportMessage(reporter, document,
                          "engine.status.connect.error.socket",
                          new Object[]{url.getHost(),
                                       new Integer(s != null ? 1 : 0),
                                       s,
                                       e.getClass().getName()},
                          null);
        } catch (IOException e) {
            String s = e.getMessage();
            reportMessage(reporter, document,
                          "engine.status.connect.error",
                          new Object[]{urlString,
                                       new Integer(s != null ? 1 : 0),
                                       s,
                                       e.getClass().getName()},
                          null);
        } catch (InterruptedException e) {
            throw new StopException("wait to load (" + urlString + ")");
        } catch (RuntimeException e) {
            String s = e.getMessage();
            reportMessage(reporter, document,
                          "engine.status.connect.error",
                          new Object[]{urlString,
                                       new Integer(s != null ? 1 : 0),
                                       s,
                                       e.getClass().getName()},
                          e);
        }

        return null;
    }

    /** ݡ */
    private void reportMessage(Reporter reporter, Document document,
                               String key, Object[] args, Throwable t) {
        if (reporter == null) {
            return;
        }

        reporter.report(Reporter.NETWORK, Reporter.WARNING, Reporter.FYI,
                        (document != null ? document.content : null),
                        0, 0, "Engine", Resource.getMessage(key, args));

        if (t != null) {
            reporter.report(t, this);
        }
    }

    /**
     * ΥꥯȤ饳ƥĤޤ
     * {@link HawkContext#getContent(Request)} ؤΥåѡ᥽åɤǡ
     * Υ᥽åɤͳȡͥåȥ³Ǥ⤹
     * ߤ뤳ȤǤޤ
     *
     * @param  context 륳ƥ
     *
     * @return ƥ
     *
     * @throws IOException IO 顼ȯ
     * @throws InterruptedException ߤȯ
     */
    Content getContent(HawkContext context)
            throws IOException, InterruptedException {
        ConnectWrapper cw = new ConnectWrapper(context);
        cw.start();

        try {
            cw.join();
        } catch (InterruptedException e) {
            cw.interrupt();
            throw e;
        }

        if (cw.content              != null) { return cw.content;              }
        if (cw.ioException          != null) { throw  cw.ioException;          }
        if (cw.interruptedException != null) { throw  cw.interruptedException; }
        if (cw.runtimeException     != null) { throw  cw.runtimeException;     }
        if (cw.error                != null) { throw  cw.error;                }

        return null;
    }

    /**
     * åѥפ֤ޤ
     *
     * @return åѥ
     */
    public int getUseCache() {
        return useCache;
    }

    /**
     * åѥפꤷޤ
     *
     * @param  useCache åѥ
     */
    public void setUseCache(int useCache) {
        this.useCache = useCache;
    }

    /**
     * ꤷƥĤޤǤ뤫֤ޤ
     *
     * @param  content ƥ
     *
     * @return ޤǤ <code>true</code>
     *         ޤǤʤ <code>false</code>
     */
    public boolean contains(Content content) {
        return (document != null && document.contains(content));
    }

    /**
     * ɥȤ֤ޤ
     *
     * @return ɥ
     */
    public Document getDocument() {
        return document;
    }

    /**
     * ᥤ󥳥ƥĤ URL ֤ޤ
     *
     * @return ᥤ󥳥ƥĤ URL
     */
    public URL getContentURL() {
        return contentUrl;
    }

    /**
     * ꥯȤηƤ򥳥ԡޤ
     *
     * @param  src ԡΥꥯ
     */
    public void copyFrom(Request src) {
        synchronized (this) {
            cleanup();

            isActive = true;
            document = src.getDocument();
            document.increment();
        }
    }

    /**
     * ΥꥯȤˤ̤ΥڡΡߤξ֤¸ޤ
     *
     * @param  position ɽΰ
     */
    public void save(Point position) {
        synchronized (this) {
            this.position = position;

            // եͤ¸
            if (formManager != null) {
                this.formData = formManager.save();
                formManager = null;
            }
        }
    }

    /**
     * ΥꥯȤˤ̤Υڡǡ
     * ǸɽƤ֤֤ޤ
     *
     * @return 
     */
    public Point getPosition() {
        return position;
    }

    /**
     * ΥꥯȤˤ̤Υڡǡ
     * ǸϤƤեǡ֤ޤ
     *
     * @return եǡ
     */
    public FormData getFormData() {
        return formData;
    }

    /**
     * ꥯȤǼºݤˤϥɤԤʤɤ֤ޤ
     *
     * @return ɤԤʤ <code>true</code>
     *         ̾ξ <code>false</code>
     */
    public boolean isIgnored() {
        return isIgnored;
    }

    /**
     * ꥯȤǼºݤ˥ɤԤʤꤷޤ
     *
     * @param  isIgnored ɤԤʤ <code>true</code>
     *                   ̾ξ <code>false</code>
     */
    public void setIgnored(boolean isIgnored) {
        this.isIgnored = isIgnored;
    }

    /**
     * Ūʥ꥽ޤ
     */
    public void cleanup() {
        synchronized (this) {
            if (isActive) {
                isActive = false;
                document.decrement();
                formManager = null;
            }
        }
    }

    /**
     * μΥꥯȤ֤ޤ
     *
     * @return Υꥯ
     */
    public Request getNext() {
        return next;
    }

    /**
     * ΥꥯȤ֤ޤ
     *
     * @return Υꥯ
     */
    public Request getPrevious() {
        return prev;
    }

    /**
     * ΥФɤ֤ޤ
     *
     * @return ΥФξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean isChain() {
        return isChain;
    }

    /**
     * ΥꥯȤˤ̤ΥڡѤƤ
     * եޥ͡ꤷޤ
     *
     * @param  formManager եޥ͡
     */
    void setFormManager(FormManager formManager) {
        this.formManager = formManager;
    }

//### ConnectWrapper
    /** ³åѡ */
    private final class ConnectWrapper extends Thread {
        private HawkContext          context;
        private Content              content;
        private IOException          ioException;
        private InterruptedException interruptedException;
        private RuntimeException     runtimeException;
        private Error                error;

        /** 󥹥󥹤 */
        private ConnectWrapper(HawkContext context) {
            this.context = context;
        }

        /**  */
        public void run() {
            try {
                content = context.getContent(Request.this);
            } catch (IOException e) {
                ioException = e;
            } catch (InterruptedException e) {
                interruptedException = e;
            } catch (RuntimeException e) {
                runtimeException = e;
            } catch (Error e) {
                error = e;
            }
        }
    }
}
