/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.addin.FormManager;
import net.hizlab.kagetaka.addin.FrameManager;
import net.hizlab.kagetaka.addin.StyleManager;
import net.hizlab.kagetaka.rendering.block.Block;
import net.hizlab.kagetaka.rendering.block.RootBlock;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.token.Token;
import net.hizlab.kagetaka.token.TokenTypes;
import net.hizlab.kagetaka.token.Value;

import java.awt.Color;
import java.awt.Dimension;
import java.io.IOException;
import java.net.URL;
import java.text.ParseException;

/**
 * ߤΥ֤°ɽ饹Ǥ
 * 襨󥸥ϡ°򸵤Ԥޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.15 $
 */
public class Render {
    /** ǤǤϤʤ */
    public static final int PSEUDO_NONE         = 0x0000000;
    /** :first-child 饹 */
    public static final int PSEUDO_FIRST_CHILD  = 0x0000001;
    /** :link 饹 */
    public static final int PSEUDO_LINK         = 0x0000002;
    /** :visited 饹 */
    public static final int PSEUDO_VISITED      = 0x0000004;
    /** :hover 饹 */
    public static final int PSEUDO_HOVER        = 0x0000008;
    /** :active 饹 */
    public static final int PSEUDO_ACTIVE       = 0x0000010;
    /** :focus 饹 */
    public static final int PSEUDO_FOCUS        = 0x0000020;
    /** :lang 饹 */
    public static final int PSEUDO_LANG         = 0x0000040;

    /** :first-line  */
    public static final int PSEUDO_FIRST_LINE   = 0x0010000;
    /** :first-letter  */
    public static final int PSEUDO_FIRST_LETTER = 0x0020000;
    /** :before  */
    public static final int PSEUDO_BEFORE       = 0x0040000;
    /** :after  */
    public static final int PSEUDO_AFTER        = 0x0080000;
    /** :marker  */
    public static final int PSEUDO_MARKER       = 0x0100000;

    // ǥեȥơ
    private static final int   DEFAULT_ALIGN       = Value.ALIGN_LEFT;
    private static final Color DEFAULT_FORE_COLOR  = Color.black;
    private static final Color DEFAULT_BACK_COLOR  = Color.white;
    private static final Color DEFAULT_LINK_COLOR  = Color.blue;
    private static final Color DEFAULT_VLINK_COLOR = new Color(0x55, 0x1A, 0x8B);
    private static final Color DEFAULT_ALINK_COLOR = Color.red;
    private static final int   DEFAULT_TD          = 0;

    /** ǥ */
    public final int media;

    private HawkContext   context;
    private Drawkit       drawkit;
    private StatusManager statusManager;
    private Status        status;

    // 
    private boolean      styleManagerCreate = false;
    private StyleManager styleManager       = null;

    // ե
    private boolean      formManagerCreate  = false;
    private FormManager  formManager        = null;

    // ե졼
    private boolean      frameManagerCreate = false;
    private FrameManager frameManager       = null;

    // եå
    private int refreshTime = -1;
    private URL refreshUrl  = null;

    // ط
    private RootBlock rootBlock;            // ȥåץ٥֥åBODYб
    private Block     bottomBlock;          // == null ? wait  : ̾
    private FormItem  formItem;             // ɲäեॢƥ
    private int       currentTag;           // ߤѥֹ
    private int       redrawTag;            // ѥֹ
    private boolean   disposed;             // ̵ˤ줿ɤ

    /**
     * 襹ơޤ
     *
     * @param  context 륳ƥ
     * @param  request ꥯ
     * @param  media   ǥ
     */
    Render(HawkContext context, Request request, int media) {
        this.context       = context;
        this.media         = media;
        this.drawkit       = new Drawkit(context, request);
        this.statusManager = new StatusManager(drawkit);

        Status status = this.status = statusManager.baseStatus;

        // BODY ơνͤ
        status.setBackground(DEFAULT_BACK_COLOR);
        status.align      = DEFAULT_ALIGN      ;
        status.foreColor  = DEFAULT_FORE_COLOR ;
        status.linkColor  = DEFAULT_LINK_COLOR ;
        status.vlinkColor = DEFAULT_VLINK_COLOR;
        status.alinkColor = DEFAULT_ALINK_COLOR;
        status.decoration = DEFAULT_TD         ;
    }

//### Public
    /**
     * ʸ褷ޤ
     *
     * @param  text 褹ʸ
     */
    public void drawText(String text) {
        if (status.isBlackHole) {
            return;
        }
        bottomBlock.appendString(text);
    }

    /**
     * ԤԤޤ
     */
    public void doBr() {
        if (status.isBlackHole) {
            return;
        }
        bottomBlock.appendNewLine();
    }

    /**
     * 褷ޤ
     *
     * @param  src 
     * @param  alt ʸ
     * @param  width  
     * @param  height ⤵
     * @param  border ܡ
     * @param  floatType 
     */
    public void drawImage(String src, String alt, Value width, Value height, Integer border, int floatType) {
        if (status.isBlackHole) {
            return;
        }
        if (src == null) {
            return;
        }

        int b = 0;

        if (border != null) {
            b = border.intValue();
        } else if (status.anchor != null) {
            b = Constant.IMG_BORDER;
        } else {
            b = 0;
        }

        bottomBlock.appendImage(src, alt, width, height, b, floatType);
    }

    /**
     * եॢƥ褷ޤ
     *
     * @param  item ƥ
     */
    public void drawForm(FormItem item) {
        formItem = item;
    }

    /**
     * ȥꤷޤ
     *
     * @param  title ȥ
     */
    public void setTitle(String title) {
        drawkit.document.setTitle(title);
    }

    /**
     * ߤ襹ơ֤ޤ
     *
     * @return 襹ơ
     */
    public Status getStatus() {
        return status;
    }

    /**
     * ١ URL ȥåȤꤷޤ
     *
     * @param  href   ١ URL
     * @param  target ١å
     */
    public void setBase(String href, String target) {
        if (href != null) {
            URL url = drawkit.createURL(href);
            if (url != null) {
                drawkit.document.setBaseURL(url);
            }
        }
        if (target != null) {
            drawkit.document.setBaseTarget(target);
        }
    }

    /**
     * ֥åޡꤷޤ
     *
     * @param  name NAME
     */
    public void setMark(String name) {
        if (name != null) {
            IdMap.Item item = drawkit.idMap.addId(name);
            if (item == null) {
                drawkit.reportMessage(Reporter.WARNING, "render.warning.id.duplicate", new String[]{name});
                return;
            }
            drawkit.idQueue.put(item);
        }
    }

    /**
     * 󥯤ꤷޤ
     *
     * @param  href   
     * @param  target å
     */
    public void setLink(String href, String target) {
        if (href != null) {
            URL url = drawkit.createURL(href);
            if (url == null) {
                return;
            }

            status.anchor = new Link(url, target);

            if (context.haveEverVisited(url)) {
                status.foreColor = status.vlinkColor;
            } else {
                status.foreColor = status.linkColor;
            }
        }
    }

    /**
     * ݥåץåʸꤷޤ
     *
     * @param  tip ݥåץåʸ
     */
    public void setPopup(String tip) {
        if (tip != null) {
            status.tip = tip;
        }
    }

    /**
     * ơ֥ιԤλޤ
     */
    public void commitRow() {
        bottomBlock.commitRow();
    }

    /**
     * Ӥ򳫻Ϥޤ
     */
    public void startRuby() {
        bottomBlock.setRuby(Constant.RUBY_START);
    }

    /**
     * оʸ򳫻Ϥޤ
     */
    public void startRubyRb() {
        bottomBlock.setRuby(Constant.RUBY_RB);
    }

    /**
     * ʸ򳫻Ϥޤ
     */
    public void startRubyRp() {
        bottomBlock.setRuby(Constant.RUBY_RP);
    }

    /**
     * ʸ򳫻Ϥޤ
     */
    public void startRubyRt() {
        bottomBlock.setRuby(Constant.RUBY_RT);
    }

    /**
     * Ӥλޤ
     */
    public void endRuby() {
        bottomBlock.setRuby(Constant.RUBY_END);
    }

    /**
     * ưեåϿޤ
     *
     * @param  time եå夹ޤǤԵÿ
     * @param  href եå URL
     *              <code>null</code> ξϼʬ
     */
    public void setRefresh(int time, String href) {
        refreshTime = time;
        if (href == null) {
            refreshUrl = drawkit.document.getBaseURL();
        } else {
            refreshUrl = drawkit.createURL(href);
            if (refreshUrl == null) {
                refreshTime = -1;
            }
        }
    }

    /**
     * åꤷޤ
     *
     * @param  value åɽʸ
     */
    public void setCookie(String value) {
        context.setCookie(value, drawkit.document.content.url);
    }

//### 
    /**
     * 륷ȤϿޤ
     *
     * @param  href 륷Ȥ URL
     * @param  contentType ƥĥ
     */
    public void setStyleSheet(String href, String contentType) {
        if (href == null) {
            return;
        }

        URL url = drawkit.createURL(href);
        if (url == null) {
            return;
        }

        // ޥ͡㤬̤ξϺƤߤ
        getStyleManager(true);
        if (styleManager == null) {
            return;
        }

        Request request = drawkit.request.createRequest(url);
        Content content = request.getContent(context, true);
        if (content == null) {
            return;
        }

        // 饯åȤ
        ContentType ct = null;
        if (contentType != null) {
            try {
                ct = ContentType.valueOf(contentType, url);
            } catch (ParseException e) {
                //### ERROR
            }
        }
        if (ct == null) {
            ct = content.getContentType();
        }
        if (ct == null || !ct.equalsType("text", "css")) {
            //### ERROR
            return;
        }

        try {
            styleManager.addStyle(content, request);
        } catch (IOException e) {
            String s = e.getMessage();
            drawkit.reportMessage(Reporter.WARNING,
                                  "engine.status.load.error.io",
                                  new Object[]{url.toString(),
                                               new Integer(s != null ? 1 : 0),
                                               s,
                                               e.getClass().getName()});
            drawkit.reportMessage(e);
        }
    }

    /**
     * ޥ֤͡ޤ
     *
     * @param  create ޥ͡㤬̵˺
     *                <code>true</code>ʳξ <code>false</code>
     *
     * @return ޥ͡㤬̵䡢
     *         뤬ݡȤƤʤ <code>null</code>
     */
    public StyleManager getStyleManager(boolean create) {
        if (styleManager == null && !styleManagerCreate && create) {
            styleManager = StyleManager.createInstance(context,
                                                       drawkit.document,
                                                       drawkit.request);
            drawkit.document.setStyleManager(styleManager);
            styleManagerCreate = true;
        }

        return styleManager;
    }

//### ե
    /**
     * եޥ֤͡ޤ
     *
     * @return եबݡȤƤʤ <code>null</code>
     */
    public FormManager getFormManager() {
        if (formManager == null && !formManagerCreate) {
            formManager = FormManager.createInstance(context,
                                                     drawkit.document,
                                                     drawkit.request);
            formManagerCreate = true;
            drawkit.request.setFormManager(formManager);
        }

        return formManager;
    }

//### ե졼
    /**
     * ե졼ޥ֤͡ޤ
     *
     * @return ե졼बݡȤƤʤ <code>null</code>
     */
    public FrameManager getFrameManager() {
        if (frameManager == null && !frameManagerCreate) {
            frameManager = FrameManager.createInstance(context,
                                                       drawkit.document,
                                                       drawkit.request);
            frameManagerCreate = true;
        }

        return frameManager;
    }

    /**
     * ե졼򥳥ߥåȤޤ
     */
    public void commitFrame() {
        if (frameManager != null) {
            frameManager.commit();
        }
    }

//### HawkEngine 
    /**
     * ʸ򳫻Ϥޤ
     *
     * @param  token BODY ϥȡ
     */
    void startBody(Token token) {
        Status status = this.status;

        // ȡγϽ
        token.render(this, PSEUDO_NONE);

        // Document 򥻥åȥå
        Document document = drawkit.document;

        int scaleWidth  = ((status.lineHeight != null
                         && status.lineHeight.getType() != Value.TYPE_KEY_NORMAL)
                           ? status.lineHeight.getValue(status.fontFullSize.width,
                                                        status.fontFullSize.width,
                                                        status.fontData,
                                                        Value.DATA_HORIZONTAL)
                           : (int) Math.round(status.fontFullSize.width * Constant.LINE_HEIGHT));
        int scaleHeight = status.fontFullSize.height;
        if (status.letterSpacing != null) {
            scaleHeight += status.letterSpacing.getValue(1,
                                                         status.fontFullSize.height,
                                                         status.fontData, Value.DATA_VERTICAL);
        }
        document.setCharScale(new Dimension(scaleWidth, scaleHeight));

        if (status.background != null) { document.setBackground(status.background); }

        // ȥåץ٥ BOX 
        synchronized (this) {
            bottomBlock = rootBlock = new RootBlock(drawkit, status, redrawTag);
        }
    }

    /**
     * ʸĤޤ
     *
     * @param  token BODY ϥȡ
     */
    void endBody(Token token) {
        // ĤFORM Ĥ̵뤷Ƥ뤿
        if (formManager != null) {
            formManager.endForm();
        }

        token.render(this, PSEUDO_NONE);

        rootBlock.commitBlock();
        context.commitPanel(drawkit.document, currentTag);

        status = null;
    }

    /**
     * ȡν򳫻Ϥޤ
     * ߤΥơѾƿơ
     * 򸵤˳ϽԤޤ
     *
     * @param  token   ϥȡ
     * @param  isBlock ȡ󤬥֥åξ <code>true</code>
     *                 ʳξ <code>false</code>
     */
    void startToken(Token token, boolean isBlock) {
        int type = token.getType();

        // ơ
        status = statusManager.push(type);

        // ȡγϽ
        token.render(this, PSEUDO_NONE);

        // ơå
        status.checkStatus();
        ensureFormItem();

        if (isBlock) {
            Status markerStatus = null;

            // ޡ֥åɬפ
            if (status.hasMarkerBlock) {
                status = statusManager.push(type);
                token.render(this, PSEUDO_MARKER);
                status.checkStatus();
                ensureFormItem();

                markerStatus = status;

                status = statusManager.pop();
            }

            bottomBlock = bottomBlock.createBlock(status, markerStatus);
        } else {
            bottomBlock.statusChanged(status);
        }

        // ξϡΤޤ޽λƤޤ
        if (TokenTypes.isEmpty(type)) {
            endToken(token, isBlock, true);
        }
    }

    /**
     * ȡνλޤ
     * λԤäˡߤΥơȡ󳫻ξ֤ᤷޤ
     *
     * @param  token   λȡ
     * @param  isBlock ȡ󤬥֥åξ <code>true</code>
     *                 ʳξ <code>false</code>
     * @param  isEmpty ȡ󤬶ξ <code>true</code>
     *                 ʳξ <code>false</code>
     */
    void endToken(Token token, boolean isBlock, boolean isEmpty) {
        // ȡνλ
        if (!isEmpty) {
            token.render(this, PSEUDO_NONE);
            ensureFormItem();
        }

        // ơ᤹
        status = statusManager.pop();

        if (isBlock) {
            // ֥å򥳥ߥå
            bottomBlock = bottomBlock.commitBlock();

            // ߤΥ֥åǾ̤ξ硢ɬפǤкɽ
            if (bottomBlock == rootBlock) {
                int tag;
                synchronized (this) {
                    if (redrawTag == currentTag) {
                        return;
                    }
                    tag = redrawTag;
                }
                rootBlock.redraw(tag);
                currentTag = tag;
            }
        } else {
            bottomBlock.statusChanged(status);
        }
    }

    /**
     * ɤ߹ߤλΤԤޤ
     */
    void waitImageLoad() {
        drawkit.waitImageLoad();
    }

    /**
     * ɤ߹ߤ򥭥󥻥뤷ޤ
     */
    void stopImageLoad() {
        drawkit.stopImageLoad();
    }

    /**
     * ׵ԤĤԵޤ
     * meta-refresh ꤵƤϡԵ
     *  URL ֤ޤmeta-refresh ꤵƤʤϡ
     * ǤޤԵޤǤȡ{@link StopException}
     * 㳰ޤ
     *
     * @return URL
     *
     * @throws StopException Ǥ줿
     */
    URL waitForRedraw()
           throws StopException {
        long startTime = System.currentTimeMillis();
        long waitTime  = 0;
        int  tag;

        for (;;) {
            synchronized (this) {
                // meta-refresh θԤ֤
                if (refreshTime >= 0) {
                    waitTime = refreshTime * 1000
                             - (System.currentTimeMillis() - startTime);
                    if (waitTime <= 0) {
                        waitTime    = 0;
                        refreshTime = 0;
                        return refreshUrl;
                    }
                }

                tag = redrawTag;
            }

            // ɽ
            if (tag != currentTag && rootBlock != null) {
                rootBlock.redraw(tag);
                currentTag = tag;
//Debug.out.println("> render.redraw done");
                context.commitPanel(drawkit.document, tag);
                continue;
            }

            try {
                synchronized (this) {
                    if (disposed) {
                        return null;
                    }

                    // wait ιѤ bottomBlock  null ˤ wait 
                    bottomBlock = null;
//Debug.out.println("> render.wait");
                    wait(waitTime);
                    bottomBlock = rootBlock;
                }
            } catch (InterruptedException e) {
                throw new StopException("wait for refresh");
            }
        }
    }

    /**
     * ѤߤΥ֥å褷ޤ
     *
     * @param  tag ѹ̤륿
     */
    void redraw(int tag) {
        synchronized (this) {
            redrawTag = tag;
            if (rootBlock == null) {
                // ޤ褬ϤޤäƤʤ
                return;
            }
            if (bottomBlock == null) {
                // 轪λѤ
                notify();
            }
        }
    }

    /**
     * ꥽ޤ
     */
    void dispose() {
        drawkit.dispose();
        if (rootBlock != null) {            // ե졼ॻåȤξ
            rootBlock.dispose();
        }
        synchronized (this) {
            disposed = true;
        }
    }

    /** եॢƥºݤɲ */
    private void ensureFormItem() {
        if (formItem != null) {
            formItem.setup();
            bottomBlock.appendForm(formItem);
            formItem = null;
        }
    }
}
