/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.addin.java2.NETWrapper;
import net.hizlab.kagetaka.net.URLUtils;
import net.hizlab.kagetaka.util.Charset;
import net.hizlab.kagetaka.util.ContentType;

import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.ImageProducer;
import java.io.InputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.text.ParseException;

/**
 * ƥĤɽ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.9 $
 */
public class Content {
    private static NETWrapper netWrapper = NETWrapper.getInstance();

    /**  */
    public static final int TYPE_ALL   = 0xFFFFFFFF;
    /** text  */
    public static final int TYPE_TEXT  = 0x00000001;
    /** image  */
    public static final int TYPE_IMAGE = 0x00000002;
    /** audio  */
    public static final int TYPE_AUDIO = 0x00000004;

    /** ɥȤ URL */
    public final URL         url;
    /** ɥȤΥƥĥ */
    public final int         contentLength;
    /** ɥȤμ */
    public final long        date;
    /** ɥȤͭ */
    public final long        expiration;
    /** ɥȤκǽ */
    public final long        lastModified;
    /** ɥȤκǽ */
    public final String      eTag;
    /** ɥȤ˴ */
    public final long        destroy;

    /** 򤹤뤫ɤ */
    protected boolean timeAdjustment;

    /** ͥ */
    protected URLConnection connection;
    /** ץåȥȥ꡼ */
    protected InputStream inputStream;
    /** ƥĥ֥ */
    protected Object      object;
    /** ɥȤΥƥĥ */
    protected ContentType contentType;

    private String encoding;

    private int referenceCount = 1;         // ȿ

    /**
     * ƥĤޤ
     *
     * @param  connection ͥ
     */
    public Content(URLConnection connection) {
        this.connection = connection;
        this.url        = URLUtils.removeRef(connection.getURL());

        ContentType ct;
        try {
            ct = ContentType.valueOf(connection.getContentType(), url);
        } catch (ParseException e) {
            ct = ContentType.CT_TEXT_HTML;
            //### ERROR
        }
        this.contentType   = ct;
        this.contentLength = connection.getContentLength();
        long date          = connection.getDate         ();
        long expiration    = connection.getExpiration   ();
        long lastModified  = connection.getLastModified ();
        //### BUGS ֤ϡꥯȤߥ󥰤 Date Ӥ򤹤٤RFC2616
        if (timeAdjustment && date > 0) {
            long diff = System.currentTimeMillis() - date;
            date         += diff;
            if (lastModified > 0) { lastModified += diff; }
            if (expiration   > 0) { expiration   += diff; }
        }
        this.date          = date;
        this.expiration    = expiration;
        this.lastModified  = lastModified;
        this.eTag          = connection.getHeaderField("ETag");

        String charset  = contentType.getParameter("charset");
        String encoding = null;
        if (charset != null && (encoding = Charset.toEncoding(charset)) == null) {
            //### ERROR
Debug.out.println("invalid charset=[" + charset + "]");
//            reporter.report(Reporter.NETWORK, Reporter.WARNING, Reporter.NONE, this, 0, 0, "Charset",
//                            Resource.getMessage("charset.warning.invalid", new Object[]{charset}));
        }
        this.encoding = encoding;

        String cache = connection.getHeaderField("Cache-Control");
        if (cache != null
                || (cache = connection.getHeaderField("Pragma")) != null) {
            cache = cache.toLowerCase();

            //### BUGS ä RFC2616 14.9 򥵥ݡȤɬפ
            this.destroy = (cache.startsWith("no-cache") ? 0 : -1);
        } else {
            this.destroy = -1;
        }
    }

    /**
     * ƥĤɤ߼ϥȥ꡼֤ޤ
     *
     * @param  reporter ݡ
     *
     * @return ϥȥ꡼
     *
     * @throws IOException IO 顼ȯ
     */
    public InputStream getInputStream(Reporter reporter)
            throws IOException {
        //### BUGS ǥХå
        if (inputStream != null) {
            throw new IllegalStateException("can not use 2nd time");
        }

        try {
            return (inputStream = connection.getInputStream());
        } catch (IOException e) {
            if (connection instanceof HttpURLConnection && netWrapper != null) {
                HttpURLConnection httpConnection = (HttpURLConnection) connection;
                InputStream is = netWrapper.getConnectionErrorStream(httpConnection);
                if (is != null) {
                    return (inputStream = is);
                }
            }

            throw e;
        }
    }

    /**
     * ƥĤȤ򥪥֥ȤȤȤ֤ޤ
     *
     * @param  type 륳ƥĤΥ
     *
     * @return ƥĤ
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized Object getObject(int type)
            throws IOException {
        if (object != null) {
            return object;
        }

        if (connection instanceof HttpURLConnection) {
            if (((HttpURLConnection) connection).getResponseCode() >= 300) {
                return null;
            }
        }

        object = connection.getContent();

        if (object instanceof ImageProducer) {
            object = Toolkit.getDefaultToolkit().createImage((ImageProducer) object);
        }

        return object;
    }

    /**
     * ƥĥפ֤ޤ
     *
     * @return ƥĥ
     */
    public synchronized ContentType getContentType() {
        return contentType;
    }

    /**
     * ɥȤΥ󥳡ǥ󥰤֤ޤ
     *
     * @return ɥȤΥ󥳡ǥ
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * ɥȤΥ󥳡ǥ󥰤ꤷޤ
     *
     * @param  encoding ɥȤΥ󥳡ǥ
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * å夵줿ƥĤɤ֤ޤ
     *
     * @return å夵줿ƥĤξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean isCached() {
        return false;
    }

    /**
     * ƥĤ򤬺Ѳǽɤ֤ޤ
     *
     * @return ƥĤѲǽʾ <code>true</code>
     *         ʳξ <code>false</code>
     */
    boolean canReUse() {
        // Object פξ礫
        // ޤ InputStream ɤ߹ޤƤʤʲΤߤɽξ
        return (object != null || inputStream == null);
    }

    /**
     * ȥ󥿡äޤ
     */
    synchronized void increment() {
        referenceCount++;
    }

    /**
     * ȥ󥿡򸺾ޤ
     * Ȥ¸ߤʤʤäϡ꥽ޤ
     *
     * @return Ȥ¸ߤʤʤä <code>true</code>
     *         ޤȤ¸ߤ <code>false</code>
     */
    synchronized boolean decrement() {
        if (--referenceCount > 0) {
            return false;
        }

        // ꥽
        if (inputStream != null) {
            try {
                inputStream.close();
            } catch (IOException e) { }
            inputStream = null;
        }
        if (object != null) {
            if (object instanceof Image) {
                ((Image) object).flush();
            }
            object = null;
        }
        return true;
    }

    /**
     * ȥ󥿡֤ޤ
     *
     * @return ȥ󥿡
     */
    int getReferenceCount() {
        return referenceCount;
    }
}
