/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.rendering;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.parser.Parser;
import net.hizlab.kagetaka.parser.ParserManager;
import net.hizlab.kagetaka.parser.TokenCoordinator;
import net.hizlab.kagetaka.token.EndToken;
import net.hizlab.kagetaka.token.StartToken;
import net.hizlab.kagetaka.token.TextToken;
import net.hizlab.kagetaka.token.Token;
import net.hizlab.kagetaka.token.TokenManager;
import net.hizlab.kagetaka.token.TokenTypes;
import net.hizlab.kagetaka.token.Value;
import net.hizlab.kagetaka.util.ContentType;
import net.hizlab.kagetaka.util.StringUtils;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.SocketException;

/**
 * ԤåɤǤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.15 $
 */
class Conductor extends Thread implements EngineListener {
    private static int threadNumber = 0;

    private static final int LT_TEXT   = 0;
    private static final int LT_HTML   = 1;
    private static final int LT_IMAGE  = 2;
    private static final int LT_BINARY = 3;

    private HawkContext context;
    private Reporter    reporter;
    private Option      option;
    private String      name;

    private EngineListener listener;

    // ɥ塼
    private Request loadRequest;
    private boolean isWaiting;
    private boolean isDrawing;

    // draw ǻ
    private URL      url;
    private String   urlString;
    private boolean  active = true;
    private Render   render;

    /**
     * 襹åɤޤ
     *
     * @param  context 륳ƥ
     */
    Conductor(HawkContext context) {
        this.context  = context;
        this.reporter = context.getReporter();
        this.option   = context.getOption  ();
        this.name     = "HawkEngine-" + (threadNumber++);

        setName(name);
        start();
    }

    /**
     * 󥸥ꥹʤϿޤ
     *
     * @param  listener 󥸥ꥹ
     */
    void setEngineListener(EngineListener listener) {
        this.listener = listener;
    }

    /**
     * 򳫻Ϥޤ
     *
     * @param  request ꥯ
     */
    void load(Request request) {
        synchronized (this) {
            this.loadRequest = request;

            if (isWaiting) {
                notify();
            } else if (isDrawing) {
                interrupt();
            }
        }
    }

    /**
     * ߥνߤޤ
     * ˤäƤϡߤǤ櫓ǤϤޤ
     */
    void abort() {
        synchronized (this) {
            if (isDrawing) {
                interrupt();
            }
        }
    }

    /**
     * 褵Ƥʬ褹褦˥󥸥˰ꤷޤ
     *
     * @param  tag ѹ̤륿
     */
    void redraw(int tag) {
        Render render = this.render;
        if (render != null) {
           render.redraw(tag);
        }
    }

    /**
     * Τ٤ƤΥ꥽˴ޤ
     */
    void dispose() {
        interrupt();
        active = false;
    }

    /**
     * ꥯȤԤ롼פ¹Ԥޤ
     */
    public void run() {
        Request nextRequest;

        for (;;) {
            // ꥯȤƤӽФޤԵ
            synchronized (this) {
                isDrawing = false;

                while (loadRequest == null) {
                    // ˴줿
                    if (!active) {
                        return;
                    }

                    try {
                        isWaiting = true;
                        wait();
                    } catch (InterruptedException e) { }
                }

                // ˴줿
                if (!active) {
                    return;
                }

                // Υꥯ
                nextRequest = loadRequest;
                loadRequest = null;
                isWaiting   = false;
                isDrawing   = true;
            }

            // 
            try {
                draw(nextRequest);
            } catch (RuntimeException e) {
                try {
                    String s = e.getMessage();
                    report(Reporter.NETWORK, Reporter.WARNING, Reporter.FYI, null,
                           "engine.error.stop", new Object[]{new Integer(s != null ? 1 : 0), s, e.getClass().getName()});
                    report(Reporter.NETWORK, e, nextRequest);
                } catch (RuntimeException ex) {
                    // ˤ⥨顼̿Ū
                    ex.printStackTrace(Debug.err);
                }
            }

            // ߤ򥯥ꥢ
            Thread.interrupted();
        }
    }

    /** ºݤ¹ */
    private void draw(Request request) {
        this.url       = request.url;
        this.urlString = url.toString();
        setName(name + "-(" + urlString + ")");

        int     loadType = LT_HTML;
        boolean noerror  = false;
        Content content  = null;

        try {
            ////// ե򳫤³Ƽ
            CONNECT:
            try {
                report(Reporter.NETWORK, Reporter.INFO, Reporter.FYI, content,
                       "engine.status.connect.open", new String[]{urlString});

                if (request.isIgnored()) {
                    noerror = true;
                    break CONNECT;
                }
                connecting(request);

                // ³
                if (!request.initialize(context)) {
                    return;
                }
                content = request.getDocument().content;

                // 쥯Ȥ줿ǽΤǡURL Ƽ
                this.url       = content.url;
                this.urlString = url.toString();

                // 饯åȤ
                ContentType ct = content.getContentType();
                String media   = ct.getType   ();
                String subtype = ct.getSubType();

                // פȽ
                if (media.compareTo("text") == 0) {
                    if (subtype != null && subtype.compareTo("html") == 0) {
                        loadType = LT_HTML;
                    } else {
                        loadType = LT_TEXT;
                    }
                } else if (media.compareTo("image") == 0) {
                    // 
                    loadType = LT_IMAGE;
                } else {
                    loadType = LT_BINARY;
                }

                report(Reporter.NETWORK, Reporter.INFO, Reporter.FYI, content,
                       "engine.status.connect.opened", new String[]{urlString});
                noerror = true;
            } catch (StopException e) {
            } finally {
                connected(request, noerror);

                // 顼ȯƤꡢߤäƤ齪λ
                if (!noerror || isInterrupted()) {
                    renderingSkipped(request);

                    // ߡפ줿ϡˤ
                    if (isInterrupted()) {
                        noerror = true;
                    }
                    return;
                }
            }

            ////// Хʥξϥ󥰤򤷤ʤΤǤäü
            if (loadType == LT_BINARY) {
                try {
                    noerror = false;
                    report(Reporter.RENDERING, Reporter.INFO, Reporter.FYI, content,
                           "engine.status.connect.download", new String[]{urlString});

                    doBinary(request, content);

                    noerror = true;
                } finally {
                    renderingSkipped(request);
                }
                return;
            }

            ///// 
            RENDER:
            try {
                noerror = false;
                report(Reporter.RENDERING, Reporter.INFO, Reporter.FYI, content,
                       "engine.status.load.start", new String[]{urlString});
                renderingStarted(request);

                if (request.isIgnored()) {
                    noerror = true;
                    break RENDER;
                }

                // 
                render = new Render(context, request, Value.MEDIA_SCREEN);

                switch (loadType) {
                case LT_TEXT:
                    doText(request, content);
                    break;
                case LT_HTML:
//if (!urlString.startsWith("about:"))
//throw new RuntimeException("test");
                    doHtml(request, content);
                    break;
                case LT_IMAGE:
                    doImage(request, content);
                    break;
                default: // AVOID
                }

                report(Reporter.RENDERING, Reporter.INFO, Reporter.FYI, content,
                       "engine.status.load.complete", null);
                noerror = true;
            } catch (StopException e) {
                report(Reporter.RENDERING, Reporter.INFO, Reporter.FYI, content,
                       "engine.warning.stop", null);
                render.stopImageLoad();
            } catch (SocketException e) {
                String s = e.getMessage();
                report(Reporter.RENDERING, Reporter.WARNING, Reporter.FYI, content,
                       "engine.status.load.error.socket", new Object[]{urlString, new Integer(s != null ? 1 : 0), s, e.getClass().getName()});
            } catch (IOException e) {
                String s = e.getMessage();
                report(Reporter.RENDERING, Reporter.WARNING, Reporter.FYI, content,
                       "engine.status.load.error.io", new Object[]{urlString, new Integer(s != null ? 1 : 0), s, e.getClass().getName()});
//                report(Reporter.RENDERING, e, request);
            } catch (OutOfMemoryError e) {
                report(Reporter.RENDERING, Reporter.WARNING, Reporter.FYI, content,
                       "engine.status.load.error.oom", new String[]{urlString});
//                report(Reporter.RENDERING, e, request);
            } catch (RuntimeException e) {
                String s = e.getMessage();
                report(Reporter.RENDERING, Reporter.WARNING, Reporter.FYI, content,
                       "engine.status.load.error", new Object[]{urlString, new Integer(s != null ? 1 : 0), s, e.getClass().getName()});
                report(Reporter.RENDERING, e, request);
            } finally {
                renderingStopped(request, noerror);
            }
        } finally {
            ////// Υ᥽åɤνλ
            URL refreshUrl = null;

            // 顼̵ text/html ξ
            if (noerror && (loadType == LT_HTML || loadType == LT_TEXT || loadType == LT_IMAGE)) {
                // եåν
                //### BUGS MS ΥץåȤξ硢waitForRedraw  waite Ƥ interrupt ˥å
                if (active && render != null
                        && !net.hizlab.kagetaka.util.Environment.isApplet) {
                    try {
                        refreshUrl = render.waitForRedraw();
                    } catch (StopException e) { }
                }
            }

            if (render != null) {
                render.dispose();
            }

            if (refreshUrl != null) {
                refresh(request, new Request(refreshUrl, null,
                                             content.url, null,
                                             Request.OPEN_DEFAULT,
                                             Request.CACHE_NORMAL));
            }
        }
    }

    /** ȡ */
    private Token createToken(int type, boolean start, Content content) {
        return (start
                ? (Token) TokenManager.createStartToken(content, 0, 0,
                                                        reporter, type, true)
                : (Token) TokenManager.createEndToken  (content, 0, 0,
                                                        reporter, type, true));
    }

    /** ݡ */
    private void report(int facility, int level, int emphasis,
                        Content content, String key, Object[] args) {
        if (reporter != null) {
            try {
                reporter.report(facility, level, emphasis, content, 0, 0,
                                "Engine", Resource.getMessage(key, args));
            } catch (RuntimeException e) {
                // ˤ⥨顼̿Ū
                e.printStackTrace(Debug.err);
            }
        }
    }

    /** ݡ */
    private void report(int facility, Throwable t, Request request) {
        if (reporter != null && reporter.isReported(facility, Reporter.ERROR)) {
            try {
                reporter.report(t, request);
            } catch (RuntimeException e) {
                // ˤ⥨顼̿Ū
                e.printStackTrace(Debug.err);
            }
        }
    }

//### EngineListener
    /* ³ */
    /** {@inheritDoc} */
    public void connecting(Request request) {
        if (active && listener != null) {
            listener.connecting(request);
        }
    }

    /* ³줿 */
    /** {@inheritDoc} */
    public void connected(Request request, boolean noerror) {
        if (active && listener != null) {
            listener.connected(request, noerror);
        }
    }

    /* 򥹥å */
    /** {@inheritDoc} */
    public void renderingSkipped(Request request) {
        if (active && listener != null) {
            listener.renderingSkipped(request);
        }
    }

    /* 򳫻 */
    /** {@inheritDoc} */
    public void renderingStarted(Request request) {
        if (active && listener != null) {
            listener.renderingStarted(request);
        }
    }

    /* 褬 */
    /** {@inheritDoc} */
    public void renderingStopped(Request request, boolean noerror) {
        if (active && listener != null) {
            listener.renderingStopped(request, noerror);
        }
    }

    /*  */
    /** {@inheritDoc} */
    public void refresh(Request baseRequest, Request newRequest) {
        if (active && listener != null) {
            listener.refresh(baseRequest, newRequest);
        }
    }

//### doXXX
    /** ƥȷϤν */
    private void doText(Request request, Content content)
            throws IOException {
        InputStream is = null;

        try {
            is = content.getInputStream(reporter);

            BufferedReader br = null;
            String encoding = content.getEncoding();
            if (encoding != null) {
                try {
                    br = new BufferedReader(new InputStreamReader(is, encoding));
                } catch (UnsupportedEncodingException e) { }
            }
            if (br == null) {
                br = new BufferedReader(new InputStreamReader(is));
            }

            Status status = render.getStatus();

            status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));
            status.lineHeight = new Value(1.2, Value.UNIT_EM);
            status.whiteSpace = Value.WHITESPACE_PRE;
            status.reference  = false;
            status.setFontFamily(new Value[]{new Value(Value.FONT_MONOSPACE)});

            // ȥɽʸ
            render.setTitle(StringUtils.getFile(content.url));

            render.startBody(createToken(TokenTypes.BODY_START, true, content));

            try {
                String line;
                int    num;
                do {
                    num = 0;
                    render.startToken(createToken(TokenTypes.X_BLOCK_START, true, content), true);

                    try {
                        while ((line = br.readLine()) != null) {
                            render.drawText(line);
                            render.drawText("\n");

                            // 100 
                            if (++num >= 100) {
                                break;
                            }
                        }
                    } finally {
                        render.endToken(createToken(TokenTypes.X_BLOCK_END, false, content), true, false);
                    }
                } while (line != null);
            } finally {
                render.endBody(createToken(TokenTypes.BODY_END, false, content));
            }
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) { }
            }
        }
    }

    /** HTML Ϥν */
    private void doHtml(Request request, Content content)
            throws IOException {
        // ѡѰդ
        String inputParser = option.getInputStreamParser();
        if (inputParser == null) {
            inputParser = Resource.getMessage("parser.default", null);
        }

        Parser parser = null;

        try {
            parser = ParserManager.createInstance(inputParser, content, reporter);

            // ե륿Ѱդ
            String[] filters = option.getFilterParsers();
            if (filters != null && filters.length > 0) {
                for (int i = 0; i < filters.length; i++) {
                    Parser p = ParserManager.createInstance(filters[i], content, reporter, parser);
                    if (p != null) {
                        parser = p;
                    } else {
                        ParserManager.ParserInfo pi = ParserManager.getParser(filters[i]);
                        String name = (pi != null ? pi.getName() : "?");
                        report(Reporter.PARSER, Reporter.WARNING, Reporter.FYI, content,
                               "engine.error.filter", new String[]{name, filters[i]});
                    }
                }
            }

            // ȡʬ
            TokenCoordinator tc      = new TokenCoordinator(request, content, reporter, parser);
            Token            token   = null;
            int              type    = TokenTypes.UNKNOWN;
            boolean          isBlock = false;
            boolean          inHead  = false;
            boolean          inBody  = false;
            boolean          inFrame = false;
            boolean          isSet   = false;
            boolean          debug   = (reporter != null
                                     && reporter.isReported(Reporter.PARSER,
                                                            Reporter.DEBUG));

            READ:
            while ((token = tc.next()) != null) {
                if (debug) {
                    reporter.report(Reporter.PARSER, Reporter.DEBUG, Reporter.NONE,
                                    content, token.getLineNumber(), token.getColumnNumber(),
                                    "Engine", "[" + token + "]");
                }

                if (Thread.interrupted()) {
                    break READ;
                }

                type = token.getType();

                // ֥åɤȽ
                if (/*---*/type == TokenTypes.FORM_START
                        || type == TokenTypes.FORM_END) {
                    isBlock = TokenTypes.canHaveBlock(token.getParent().getType());
                } else {
                    isBlock = TokenTypes.isBlockEx(type);
                }

                // Υȡ̵뤹
                switch (type) {
                case TokenTypes.UNKNOWN:
                case TokenTypes.DTD    :
                case TokenTypes.COMMENT:
                case TokenTypes.PI     :
                case TokenTypes.MISC   :
                    token.render(render, Render.PSEUDO_NONE);
                    continue;
                default: // AVOID
                }

                // BODY γξ
                if (!inBody) {
                    // ƥȤ̵
                    if (type == TokenTypes.TEXT) {
                        continue;
                    }

                    if (!isSet) {
                        // BODY Ϥ줿
                        if (!inFrame && type == TokenTypes.BODY_START) {
                            inBody = true;
                            isSet  = true;
                            render.startBody(token);
                            continue;
                        }

                        if (!inHead) {
                            // HEAD Ϥ줿
                            if (type == TokenTypes.HEAD_START) {
                                inHead = true;
                            } else
                            // FRAMESET Ϥ줿
                            if (!inFrame && type == TokenTypes.FRAMESET_START) {
                                inFrame = true;
                                isSet   = true;
                            }
                        } else
                        // HEAD λ
                        if (type == TokenTypes.HEAD_END) {
                            inHead = false;
                        }
                    }

                    // ե졼ǤϤʤ礫ե졼 BODY ʳΥͭ
                    if (!inFrame || !TokenTypes.isBody(type)) {
                        token.render(render, Render.PSEUDO_NONE);
                    }
                    continue;
                }

                ///////////////////////
                // BODY ν

                // BODY λ
                if (type == TokenTypes.BODY_END) {
                    inBody = false;
                    render.endBody(token);
                    continue;
                }

                // ϥȡ
                if (token instanceof StartToken) {
                    render.startToken(token, isBlock);
                    continue;
                }

                // λȡ
                if (token instanceof EndToken) {
                    render.endToken(token, isBlock, false);
                    continue;
                }

                // ƥȥȡ
                if (token instanceof TextToken) {
                    token.render(render, Render.PSEUDO_NONE);
                    continue;
                }
            }

            // ե졼ब¸ߤϥߥåȤ
            if (inFrame) {
                render.commitFrame();
            }

            render.waitImageLoad();
        } finally {
            if (parser != null) {
                parser.close();
            }
        }
    }

    /** Ϥν */
    private void doImage(Request request, Content content) {
        Status status = render.getStatus();

        status.setMargin(Status.TARGET_ALL, new Value(5, Value.UNIT_PX));

        // ȥɽʸ
        render.setTitle(StringUtils.getFile(content.url));

        render.startBody (createToken(TokenTypes.BODY_START   , true , content));
        render.startToken(createToken(TokenTypes.X_BLOCK_START, true , content), true);

        render.drawImage(content.url.toString(), null, null, null, new Integer(0), Value.FLOAT_NONE);

        render.endToken  (createToken(TokenTypes.X_BLOCK_END  , false, content), true, false);
        render.endBody   (createToken(TokenTypes.BODY_END     , false, content));
    }

    /** ХʥϤν */
    private void doBinary(Request request, Content content) {
        context.download(content);
    }
}
