/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.protocol.http;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.net.HttpClient;
import net.hizlab.kagetaka.net.InputStreamExtension;
import net.hizlab.kagetaka.net.MessageHeader;
import net.hizlab.kagetaka.net.PosterOutputStream;
import net.hizlab.kagetaka.net.StreamMonitor;
import net.hizlab.kagetaka.net.URLUtils;
import net.hizlab.kagetaka.protocol.CacheFile;
import net.hizlab.kagetaka.protocol.CacheManager;
import net.hizlab.kagetaka.protocol.CacheSupported;
import net.hizlab.kagetaka.protocol.ProxySupported;
import net.hizlab.kagetaka.protocol.URLConnectionCache;
import net.hizlab.kagetaka.rendering.PostData;
import net.hizlab.kagetaka.io.CausedIOException;
import net.hizlab.kagetaka.util.CausedRuntimeException;

import java.io.InputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * HTTP 饤ȤδŪʵǽ󶡤ޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
public class HawkHttpURLConnection extends HttpURLConnection
        implements CacheSupported, ProxySupported {
    /** HTTP С */
    protected static final String HTTP_VERSION  = "HTTP/1.1";
    /** դ륳ƥĥ */
    protected static final String ACCEPT_STRING = "text/html, image/gif, image/jpeg, *; q=.2, */*; q=.2";

    private static final int    POSTER_SIZE   = 256;

    /** ʤإå */
    static final String[] EXCLUDE_HEADERS = {"Proxy-Authorization",  "Authorization"};

    private static SimpleDateFormat modifiedSinceFormat;
    static {
        modifiedSinceFormat = new SimpleDateFormat ("EEE, dd MMM yyyy HH:mm:ss 'GMT'", Locale.US);
        modifiedSinceFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
    }

    /** ž */
    protected static int     maxRedirects;
    private   static boolean strictPostRedirect;
    static {
        String s;

        // 쥯ȿ
        int max = 20;
        if ((s = System.getProperty("kagetaka.http.maxRedirects")) == null) {
            s = System.getProperty("http.maxRedirects");
        }
        if (s != null) {
            try {
                max = Integer.parseInt(s);
            } catch (NumberFormatException e) { }
        }
        maxRedirects = max;

        // POST Υ쥯Ȥػߤ뤫
        if ((s = System.getProperty("kagetaka.http.strictPostRedirect")) == null) {
            System.getProperty("http.strictPostRedirect");
        }
        strictPostRedirect = (s != null ? Boolean.valueOf(s).booleanValue() : false);
    }

    /** 쥯ȤԤɤ */
    protected boolean instanceFollowRedirects = getFollowRedirects();

    /** ץѤ뤫ɤ */
    protected boolean usingProxy;
    /** ץۥ */
    protected String  proxyHost;
    /** ץݡ */
    protected int     proxyPort;
    /** å奨ȥ꡼ */
    protected URLConnectionCache.Entry cacheEntry;

    /** HTTP 쥹ݥ󥹤 400 ʾ㳰ȯ뤫ɤ */
    protected boolean       responseException;
    /** HTTP 饤 */
    protected HttpClient    http;
    /** ꥯ */
    protected MessageHeader requests  = new MessageHeader();
    /** 쥹ݥ */
    protected MessageHeader responses = new MessageHeader();
    /** ץåȥȥ꡼ */
    protected InputStream   inputStream;
    /** ȯƤ㳰 */
    protected Exception     rememberedException;

    private PosterOutputStream poster;
    private boolean            setupRequests;

    // å
    private CacheManager cacheManager;
    private CacheFile    cacheFile;
    private boolean      validCacheFile;
    private PostData     postData;


    /**
     * HTTP URL ͥޤ
     *
     * @param  url        ³ URL
     * @param  proxyHost  ץۥ
     * @param  proxyPort  ץݡ
     * @param  cacheEntry å奨ȥ
     */
    HawkHttpURLConnection(URL url, String proxyHost, int proxyPort,
                          URLConnectionCache.Entry cacheEntry) {
        super(url);
Debug.out.println("++ URL : " + url + " : " + proxyHost + ":" + proxyPort);

        setupProxy(proxyHost, proxyPort);

        // إå
        cacheEntry.setupURLConnection(this);
        this.cacheEntry = cacheEntry;
    }

//### CacheSupported
    /** {@inheritDoc} */
    public void setupCache(CacheManager cm, PostData pd) {
        this.cacheManager = cm;
        this.postData     = pd;

        cacheEntry.setCacheManager(cm);
    }

    /** {@inheritDoc} */
    public void removeCache() {
        cacheEntry.dispose();
        if (validCacheFile && cacheManager != null) {
            cacheManager.removeEntry(this, postData);
        }
    }

    /** {@inheritDoc} */
    public String getCachePath() {
        if (cacheFile != null) {
            return cacheFile.getPath();
        }

        return null;
    }

//### URLConnection
    /**
     * ³Ԥޤ
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized void connect() throws IOException {
        if (connected) {
            return;
        }

        // Connector ƤФ줿ȤʳΤߡå夫ɤ߹
        if (cacheManager == null
                && (cacheFile = cacheEntry.getCacheFile()) != null) {
            try {
                if ((inputStream = cacheFile.getInputStream()) != null) {
Debug.out.println("* ruse=" + cacheFile.getName() + "," + url);
                    validCacheFile  = true;
                    connected       = true;
                    responseCode    = HTTP_OK;
                    responseMessage = "OK";
                    return;
                }
            } catch (IOException e) {
                //### ERROR
Debug.out.println(e);
            }
        }

        if (usingProxy) {
            proxiedConnect();
        } else {
            plainConnect();
        }

        connected = true;
    }

    /**
     * HTTP 쥹ݥ󥹥ɤ֤ޤ
     *
     * @return HTTP 쥹ݥ󥹥
     *
     * @throws IOException IO 顼ȯ
     */
    public int getResponseCode() throws IOException {
        if (responseCode != -1) {
            return responseCode;
        }

        try {
            getInputStream();
        } catch (Exception e) { }

        String line = responses.get(0);
        if (line == null) {
            rethrowException();
            throw new IOException("Invalid response");           // ʤϤ
        }

        // ơ饤
        if (line.startsWith("HTTP/1.")) {
            // HTTP/1.x 000 MESSAGE
            int length = line.length();
            int start  = line.indexOf(' ');
            if (start > 0) {
                while (++start < length && line.charAt(start) == ' ') {
                    // AVOID
                }
                int end = line.indexOf(' ', start);
                if (end != -1) {
                    if (end < length) {
                        responseMessage = line.substring(end + 1);
                    }
                } else {
                    end = length;
                }

                try {
                    return (responseCode = Integer.parseInt(line.substring(start, end)));
                } catch (NumberFormatException e) { }
            }
        }

        return -1;
    }

    /**
     * ꥯȤǽϤϥȥ꡼ᤷޤ
     *
     * @return ϥȥ꡼
     *
     * @throws IOException IO 顼ȯ
     */
    public OutputStream getOutputStream() throws IOException {
        if (validCacheFile) {
            throw new ProtocolException("Cannot write output by cache.");
        }

        try {
            if (!doOutput) {
                throw new ProtocolException("cannot write to a URLConnection if doOutput=false - call setDoOutput(true)");
            }

            if (method.equals("GET")) {
                method = "POST";
            }

            if (!"POST".equals(method) && !"PUT".equals(method) && "http".equals(url.getProtocol())) {
                throw new ProtocolException("HTTP method " + method + " doesn't support output");
            }

            if (inputStream != null) {
                throw new ProtocolException("Cannot write output after reading input.");
            }

            if (poster == null) {
                poster = new PosterOutputStream(POSTER_SIZE);
            }

            return poster;
        } catch (RuntimeException e) {
            release();
            throw e;
        } catch (IOException e) {
            release();
            throw e;
        }
    }

    /**
     * ϥȥ꡼֤ޤ
     *
     * @return ϥȥ꡼
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized InputStream getInputStream() throws IOException {
        if (!doInput) {
            throw new ProtocolException("Cannot read from URLConnection if doInput=false (call setDoInput(true))");
        }

        // 㳰κ
        if (rememberedException != null) {
            rethrowException();
        }

        if (inputStream != null) {
            return inputStream;
        }

        boolean releaseWhenException = true; // 㳰ȯȤ release 뤫
        try {
            int redirects = 0;

            // 쥯ѤΥ롼
            do {
                if (!connected) {
                    connect();
                    // åξ硢connect  inputStream åȤ
                    if (inputStream != null) {
                        return inputStream;
                    }
                }

                // ƻԥ롼
                for (;;) {
                    try {
                        // ꥯȤ
                        inputStream = http.send(setupRequests(),
                                                poster,
                                                responses,
                                                (method.equals("HEAD") || method.equals("TRACE")),
                                                true);
                        break;
                    } catch (IOException e) {
                        try {
                            // ƻԤɬפ̵ϡ顼
                            if (!http.needRetry()) {
                                throw e;
                            }
                        } finally {
                            release();
                        }
                        checkIOException(e);

                        // ƻԤ
                        connect();
                    }
                }

                int code = getResponseCode();

                // 200 Τߥå夷Ƥߤ
                CacheManager cm = getCacheManager();
                if (cm != null
                        && 200 <= code && code < 300
                        && (inputStream instanceof InputStreamExtension)
                        && (cacheFile = cm.getFile(this)) != null) {
                    ((InputStreamExtension) inputStream).setStreamMonitor(new CacheOutputStream(cacheFile));
                }
if (responseCode == HTTP_NOT_MODIFIED) {
Debug.out.println("* nmod=," + url);
} else {
Debug.out.println("* miss=" + (cacheFile != null ? cacheFile.getName() : "") + "," + url);
}

                if (followRedirect(code)) {
                    release();
                    redirects++;
                    continue;
                }

                // 400 ʾϥ顼ˤ
                if (responseException && code >= 400) {
                    releaseWhenException = false; // 㳰ȯ release ʤ褦ˤ
                    if (code == 404 || code == 410) {
                        throw new FileNotFoundException(url.toString());
                    }
                    throw new IOException("Server returned HTTP response code: " + code + " for URL: " + url.toString());
                }

                return inputStream;
            } while (redirects < maxRedirects);

            throw new ProtocolException("Server redirected too many times (" + redirects + ")");
        } catch (RuntimeException e) {
Debug.err.println("### ERROR ### HawkHttpURLConnection.getInputStream : " + e);
e.printStackTrace(Debug.err);
            release();
            rememberedException = e;
            throw e;
        } catch (IOException e) {
            if (releaseWhenException) {
Debug.out.println("### WARNING ### HawkHttpURLConnection.getInputStream : " + e);
e.printStackTrace(Debug.out);
                release();
            }
            rememberedException = e;
            throw e;
        }
    }

    /** 쥯Ȥ뤫ɤ֤ */
    private boolean followRedirect(int code) throws IOException {
        if (!getInstanceFollowRedirects()) {
            return false;
        }

        if (code < 300 || code == HTTP_NOT_MODIFIED || code == 306 || code > 307) {
            return false;
        }

        String location = getHeaderField("Location");
        if (location == null) {
            return false;
        }

        URL locationUrl;
        try {
            locationUrl = new URL(location);
            if (!url.getProtocol().equalsIgnoreCase(locationUrl.getProtocol())) {
                return false;
            }
        } catch (MalformedURLException e) {
            locationUrl = new URL(url, location);
        }

        release();
        responses = new MessageHeader();

        if (code == HTTP_USE_PROXY) {
            setupProxy(locationUrl.getHost(), locationUrl.getPort());
            requests.set(0, method + " " + getConnectURL() + " "  + HTTP_VERSION, null);
        } else {
            url = locationUrl;
            if (method.equals("POST") && !strictPostRedirect && code != 307) {
                // ꥯȤľ
                setupRequests = false;
                requests      = new MessageHeader();
                poster        = null;
                setRequestMethod("GET");
            } else {
                requests.set(0, method + " " + getConnectURL() + " "  + HTTP_VERSION, null);
                requests.set("Host", getHostHeader(false));
            }
        }

        return true;
    }

    /**
     * HTTP 顼ȯʥ쥹ݥ󥹥ɤ 400 ʾˤˡ
     * ϥȥ꡼֤ޤ
     *
     * @return 顼ξϥȥ꡼
     */
    public InputStream getErrorStream() {
        if (connected && responseCode >= 400 && inputStream != null) {
            return inputStream;
        }

        return null;
    }

    /**
     * ꥯȥץѥƥ֤ޤ
     *
     * @param  key   
     *
     * @return ꥽
     */
    public String getRequestProperty(String key) {
        if (key != null) {
            for (int i = 0; i < EXCLUDE_HEADERS.length; i++) {
                if (key.equalsIgnoreCase(EXCLUDE_HEADERS[i])) {
                    return null;
                }
            }
        }

        return requests.get(key);
    }

    /**
     * ꥯȥץѥƥꤷޤ
     *
     * @param  key   
     * @param  value 
     */
    public void setRequestProperty(String key, String value) {
        super.setRequestProperty(key, value);
        checkMessageHeader(key, value);
        requests.set(key, value);

        // 󥹥ȥ饯 cacheEntry.setupURLConnection(this);
        // ƤФ줿ϡϿʤ褦ˤ
        if (cacheEntry != null) {
            cacheEntry.setRequestProperty(key, value);
        }
    }

    /**
     * إåեɤ֤ޤ
     *
     * @param  name  
     *
     * @return إåե
     */
    public String getHeaderField(String name) {
        try {
            getInputStream();
        } catch (IOException e) { }

        return responses.get(name);
    }

    /**
     * إåեɤ֤ޤ
     *
     * @param  n إå
     *
     * @return إåե
     */
    public String getHeaderField(int n) {
        try {
            getInputStream();
        } catch (IOException e) { }

        return responses.get(n);
    }

    /**
     * إåեɤ֤ޤ
     *
     * @param  n إå
     *
     * @return إåե
     */
    public String getHeaderFieldKey(int n) {
        try {
            getInputStream();
        } catch (IOException e) { }

        return responses.keyAt(n);
    }

//### HttpURLConnection
    /**
     * Ǥޤ
     */
    public void disconnect() {
        release();
    }

    /**
     * ץѤ뤫ɤ֤ޤ
     *
     * @return ץѤ <code>true</code>
     *         Ѥʤ <code>false</code>
     */
    public boolean usingProxy() {
        return usingProxy;
    }

    /**
     * 쥯ȤԤɤ֤ޤ
     *
     * @return 쥯Ȥ <code>true</code>
     *         ʤ <code>false</code>
     */
    public boolean getInstanceFollowRedirects() {
        return instanceFollowRedirects;
    }

    /**
     * 쥯ȤԤɤꤷޤ
     *
     * @param  followRedirects 쥯Ȥ <code>true</code>
     *                         ʤ <code>false</code>
     */
    public void setInstanceFollowRedirects(boolean followRedirects) {
        instanceFollowRedirects = followRedirects;
    }

//### original
    /**
     * ľ³ޤ
     *
     * @throws IOException IO 顼ȯ
     */
    protected synchronized void plainConnect() throws IOException {
        http = HttpClient.getInstance(url.getProtocol(), url.getHost(), url.getPort(), false);
    }

    /**
     * ץͳ³ޤ
     *
     * @throws IOException IO 顼ȯ
     */
    protected synchronized void proxiedConnect() throws IOException {
        http = HttpClient.getInstance(url.getProtocol(), proxyHost, proxyPort, true);
    }

    /**
     * ꥽ޤ
     */
    protected synchronized void release() {
        responseCode = -1;

        if (inputStream != null) {
            InputStream is = inputStream;
            inputStream = null;
            try {
                is.close();
            } catch (IOException e) { }
        }

        if (http != null) {
            http.dispose();
            http      = null;
            connected = false;
        }
    }

    /**
     * ץꤷޤ
     *
     * @param  host ۥȡѤʤ <code>null</code>
     * @param  port ݡȡѤʤ <code>-1</code>
     */
    protected void setupProxy(String host, int port) {
        this.usingProxy = (host != null);
        this.proxyHost  = host;
        this.proxyPort  = port;
    }

    /**
     * åإååޤ
     *
     * @param  key   
     * @param  value 
     *
     * @see    IllegalArgumentException ʥإåξ
     */
    protected void checkMessageHeader(String key, String value) {
        int  index;

        if ((index = key.indexOf('\n')) != -1) {
            throw new IllegalArgumentException("Illegal character(s) in message header field: " + key);
        }

        if (value == null) {
            return;
        }

        index = 0;
        while ((index = value.indexOf('\n', index)) != -1) {
            if (++index < value.length()) {
                char c = value.charAt(index);
                if (c == ' ' || c == '\t') {
                    continue;
                }
            }
            throw new IllegalArgumentException("Illegal character(s) in message header value: " + value);
        }
    }

    /**
     * HTTP ꥯȤ˻Ѥꥯʸ֤ޤ
     *
     * @return ꥯʸ
     */
    protected String getConnectURL() {
        return (!usingProxy
                ? url.getFile()
                : URLUtils.getFullPath(url));
    }

    /**
     * {@link #url} 򸵤ˤ Host إåѤʸ֤ޤ
     *
     * @param  mustPort ݡȤɬղä뤫ɤ
     *
     * @return إåѤʸ
     */
    protected String getHostHeader(boolean mustPort) {
        String host        = url.getHost();
        int    port        = url.getPort();
        int    defaultPort = getDefaultPort();
        if (mustPort || (port != -1 && port != defaultPort)) {
            host += ":" + String.valueOf((port != -1 ? port : defaultPort));
        }
        return host;
    }

    /**
     * ǥեȥݡȤ֤ޤ
     *
     * @return ǥեȥݡ
     */
    protected int getDefaultPort() {
        return HttpClient.DEFAULT_PORT;
    }

    /**
     * HTTP ꥯȥإå򥻥åȥåפ
     * ³˻Ѥꥯȥإå֤ޤ
     * <p>
     * Υ᥽åǡ{@link #requests} եɤꤷ
     * ºݤΥꥯȤǻѤꥯȥإåͤȤ
     * ֤ɬפޤˤꡢURLConnection ѤƤ
     * 饤Ȥ֤ꥯȥإåͤȡ
     * ºݤꥯȥإå̤ˤ뤳Ȥޤ
     *
     * @return ºݤ˻Ѥꥯȥإå
     */
    protected MessageHeader setupRequests() {
        if (setupRequests) {
            return requests;
        }

        requests.prepend(method + " " + getConnectURL() + " "  + HTTP_VERSION, null);

        requests.setIfNotSet("Host"  , getHostHeader(false));
        requests.setIfNotSet("Accept", ACCEPT_STRING);

        if (http.candoHttpKeepAlive()) {
            if (usingProxy) {
                requests.setIfNotSet("Proxy-Connection", "keep-alive");
            } else {
                requests.setIfNotSet("Connection"      , "keep-alive");
            }
        }

        long modifiedSince = getIfModifiedSince();
        if (modifiedSince > 0) {
            requests.setIfNotSet("If-Modified-Since", modifiedSinceFormat.format(new Date(modifiedSince)));
        }

        if (!getUseCaches()) {
            requests.setIfNotSet ("Cache-Control", "no-cache");
            requests.setIfNotSet ("Pragma"       , "no-cache");
        }

        // POST ǡ
        if (poster != null) {
            synchronized (poster) {
                try {
                    poster.close();
                } catch (IOException e) { }
                if (!method.equals("PUT")) {
                    requests.setIfNotSet("Content-Type", "application/x-www-form-urlencoded");
                }

                requests.set("Content-Length",  String.valueOf(poster.size()));
            }
        }

        setupRequests = true;
        return requests;
    }

    /**
     * ꤵ줿㳰˺ƻԤƤϤʤˡ
     * 㳰򥹥ޤ
     *
     * @param  exception 㳰
     *
     * @throws IOException 㳰
     */
    protected void checkIOException(IOException exception) throws IOException {
    }

    /** 㳰 */
    private void rethrowException() throws IOException {
        if (rememberedException != null) {
            if (rememberedException instanceof RuntimeException) {
                throw new CausedRuntimeException(rememberedException);
            }
            throw new CausedIOException(rememberedException);
        }
    }

    /** åޥ͡ */
    private CacheManager getCacheManager() {
        // Connector ƤФ줿Ȥʳϥåޥ͡㤬
        // ꤵƤʤΤǡѤߤΤΤѤ
        return (cacheManager != null
                ? cacheManager
                : cacheEntry.getCacheManager());
    }

//### CacheOutputStream
    /** åϥȥ꡼ */
    private final class CacheOutputStream
            extends FileOutputStream
            implements StreamMonitor {
        private File    file;               // ե
        private boolean error;              // 顼ȯɤ

        /** 󥹥󥹤 */
        private CacheOutputStream(File file) throws IOException {
            super(file);
            this.file = file;
        }

        /* ȥ꡼ǡ̲ᤷ */
        /** {@inheritDoc} */
        public void pass(int b) {
            if (error) {
                return;
            }

            try {
                write(b);
            } catch (IOException e) {
                error = true;
            }
        }

        /* ȥ꡼ǡ̲ᤷ */
        /** {@inheritDoc} */
        public void pass(byte[] b, int off, int len) {
            if (error) {
                return;
            }

            try {
                write(b, off, len);
            } catch (IOException e) {
                error = true;
            }
        }

        /* ȥ꡼̲᤬λ */
        /** {@inheritDoc} */
        public void stop(int cause) {
            synchronized (HawkHttpURLConnection.this) {
                if (file == null) {
                    return;
                }

                try {
                    try {
                        close();

                        if (cause == END_OF_STREAM || cause == CLOSE) {
Debug.out.println("* save=" + (cacheFile != null ? cacheFile.getName() : "") + "," + file.length() + "," + url);
                            validCacheFile = true;

                            // åɲ
                            cacheEntry.setCacheFile(cacheFile);
                            CacheManager cm = getCacheManager();
                            if (cm != null) {
                                cm.addEntry(HawkHttpURLConnection.this, postData, cacheFile);
                            }
                            return;
                        }
                    } catch (IOException e) {
                        // ̵
                    }
Debug.out.println("* drop=" + (cacheFile != null ? cacheFile.getName() : "") + "," + file.length() + "," + cause + "," + url);
                    cacheFile.delete();
                } finally {
                    file = null;
                }
            }
        }
    }
}
